#!/usr/bin/perl

#    mkpatch - Create patches against the Linux kernel
#    Copyright (c) 1999  Frodo Looijaard <frodol@dds.nl>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use strict;

use vars qw($temp);
$temp = "mkpatch/.temp";

# Forward declaration
sub gen_drivers_char_Makefile;

# Generate a diff between the old kernel file and the new I2C file. We
# arrange the headers to tell us the old tree was under directory 
# `linux-old', and the new tree under `linux'.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
# $_[2]: Name of the kernel file
# $_[3]: Name of the patched file
sub print_diff
{
  my ($package_root,$kernel_root,$kernel_file,$package_file) = @_;
  my ($diff_command,$dummy);

  $diff_command = "diff -u2";
  if ( -e "$kernel_root/$kernel_file") {
    $diff_command .= " $kernel_root/$kernel_file ";
  } else {
    $diff_command .= " /dev/null ";
  }
  if ( -e "$package_root/$package_file") {
    $diff_command .= " $package_root/$package_file ";
  } else {
    $diff_command .= " /dev/null";
  }
  open INPUT, "$diff_command|" or die "Can't call `$diff_command'";
  $dummy = <INPUT>;
  $dummy = <INPUT>;
  print "--- linux-old/$kernel_file\t".`date`;
  print "+++ linux/$kernel_file\t".`date`;
    
  while (<INPUT>) {
    print;
  }
  close INPUT;
}

# Find all the lm_sensors code in a file
# $_[0]: Linux kernel tree (like /usr/src/linux)
# $_[1]: Name of the kernel file
# Returns a list of strings with the sensors codes
sub find_sensors_code
{
  my ($kernel_root,$kernel_file) = @_;
  my @res;
  open INPUT, "$kernel_root/$kernel_file" 
       or return @res;
  while (<INPUT>) {
    if (m@sensors code starts here@) {
      push @res,"";
      while (<INPUT>) {
        last if m@sensors code ends here@;
        $res[$#res] .= $_;
      }
    }
  }
  return @res;    
} 

# Here we generate diffs for all kernel files mentioned in OLDI2C 
# which change the invocation # `#include <linux/i2c.h>' to 
# `#include <linux/i2c-old.h>'. But first, we generate diffs to copy
# file <linux/i2c.h> to <linux/i2c-old.h>, if the kernel does not have
# this file yet.
# $_[0]: sensors package root (like /tmp/sensors)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub patch_old_i2c
{
  my ($package_root,$kernel_root) = @_;
  my (@files,$file,$f);
  # If i2c.c does not exist, either we renamed it earlier, or there is no
  # i2c support in this kernel at all.
  return if not -e "$kernel_root/drivers/char/i2c.c";

  print_diff $kernel_root,$kernel_root,"include/linux/i2c-old.h", 
             "include/linux/i2c.h";


  open INPUT, "$package_root/mkpatch/OLDI2C" 
        or die "Can't open `$package_root/mkpatch/OLDI2C'";
  @files = <INPUT>;
  close INPUT;

  foreach $f (@files,"drivers/char/i2c-old.c") {
    $file = $f; # Ugly, but seemingly needed to run on Perl 5.6.
    chomp $file;
    if ($file eq "drivers/char/i2c-old.c") {
      open INPUT, "$kernel_root/drivers/char/i2c.c"
            or next;
    } else { 
      open INPUT, "$kernel_root/$file"
           or next;
    }
    open OUTPUT, ">$package_root/$temp"
           or die "Can't open `$package_root/$temp'";
    while (<INPUT>) {
      s@(\s*#\s*include\s*)<linux/i2c.h>@\1<linux/i2c-old.h>@;
      print OUTPUT;
    }
    close INPUT;
    close OUTPUT;
    print_diff $package_root,$kernel_root,$file,$temp;
  }
  print_diff "/dev",$kernel_root,"drivers/char/i2c.c","null";
  gen_drivers_char_Makefile $package_root, $kernel_root;
}

# This generates diffs for kernel file Documentation/Configure.help. This
# file contains the help texts that can be displayed during `make *config'
# for the kernel.
# The new texts are put at the end of the file, or just before the
# lm_sensors texts.
# Of course, care is taken old lines are removed.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_Documentation_Configure_help
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "Documentation/Configure.help";
  my $package_file = $temp;
  my $printed = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@^I2C support$@ or m@^I2C bit-banging interfaces@ or
           m@^Philips style parallel port adapter@ or
           m@^ELV adapter@ or m@^Velleman K9000 adapter@ or
           m@^I2C PCF 8584 interfaces@ or m@^Elektor ISA card@ or
           m@^I2C device interface@ ) {
      $_ = <INPUT>;
      $_ = <INPUT>;
      $_ = <INPUT> while not m@^\S@ and not eof(INPUT);
      redo MAIN;
    }
    if (not $printed and (eof(INPUT) or m@I2C mainboard interfaces@ or
                          m@A couple of things I keep forgetting@ or
                          m@Bus Mouse Support@)) {
      print OUTPUT <<'EOF';
I2C support
CONFIG_I2C
  I2C (pronounce: I-square-C) is a slow serial bus protocol used in
  many micro controller applications and developed by Philips. SMBus,
  or System Management Bus is a subset of the I2C protocol. More
  information is contained in the directory Documentation/i2c/,
  especially in the file called "summary" there.

  Both I2C and SMBus are supported here. You will need this for
  hardware sensors support, and also for Video for Linux support.
  Specifically, if you want to use a BT848 based frame grabber/overlay
  boards under Linux, say Y here and also to "I2C bit-banging
  interfaces", below.

  If you want I2C support, you should say Y here and also to the
  specific driver for your bus adapter(s) below. If you say Y to
  "/proc file system" below, you will then get a /proc interface which
  is documented in Documentation/i2c/proc-interface.

  This I2C support is also available as a module. If you want to
  compile it as a module, say M here and read
  Documentation/modules.txt. The module will be called i2c-core.o.

I2C bit-banging interfaces
CONFIG_I2C_ALGOBIT
  This allows you to use a range of I2C adapters called bit-banging
  adapters. Say Y if you own an I2C adapter belonging to this class
  and then say Y to the specific driver for you adapter below.

  This support is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-algo-bit.o.

Philips style parallel port adapter
CONFIG_I2C_BITLP
  This supports parallel-port I2C adapters made by Philips. Say Y if
  you own such an adapter.

  This driver is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-philips-par.o.

  Note that if you want support for different parallel port devices,
  life will be much easier if you compile them all as modules.

ELV adapter
CONFIG_I2C_BITELV
  This supports parallel-port I2C adapters called ELV. Say Y if you
  own such an adapter.

  This driver is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-elv.o.

Velleman K9000 adapter
CONFIG_I2C_BITVELLE
  This supports the Velleman K9000 parallel-port I2C adapter. Say Y if
  you own such an adapter.

  This driver is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-velleman.o.

I2C PCF 8584 interfaces
CONFIG_I2C_ALGOPCF
  This allows you to use a range of I2C adapters called PCF adapters.
  Say Y if you own an I2C adapter belonging to this class and then say
  Y to the specific driver for you adapter below.

  This support is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-algo-pcf.o.

Elektor ISA card
CONFIG_I2C_PCFISA
  This supports the PCF8584 ISA bus I2C adapter. Say Y if you own such
  an adapter.

  This driver is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-elektor.o.

I2C device interface
CONFIG_I2C_CHARDEV
  Say Y here to use i2c-* device files, usually found in the /dev
  directory on your system. They make it possible to have user-space
  programs use the I2C bus. Information on how to do this is contained
  in the file Documentation/i2c/dev-interface.

  This code is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-dev.o.

I2C /proc support
CONFIG_I2C_PROC
  This provides support for i2c device entries in the /proc filesystem.
  The entries will be found in /proc/sys/dev/sensors.

  This code is also available as a module. If you want to compile
  it as a module, say M here and read Documentation/modules.txt. The
  module will be called i2c-proc.o.

EOF
      $printed = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for the main Linux Makefile.
# Three lines which add drivers/i2c/i2.a to the DRIVERS list are put just
# before the place where the architecture Makefile is included.
# Of course, care is taken old lines are removed.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "Makefile";
  my $package_file = $temp;
  my $printed = 0;
  my $new_style = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@^DRIVERS :=@) {
       $new_style = 1;
    }
    if (m@DRIVERS-\$\(CONFIG_I2C\)@) {
      $_ = <INPUT>;
      redo MAIN;
    } elsif (m@CONFIG_I2C@) {
      $_ = <INPUT> while not m@endif@;
      $_ = <INPUT>;
      $_ = <INPUT> if m@^$@;
      redo MAIN;
    }
    if (not $printed and m@DRIVERS-\$\(CONFIG_PHONE\)@) {
      if ($new_style) {
        print OUTPUT << 'EOF';
DRIVERS-$(CONFIG_I2C) += drivers/i2c/i2c.a
EOF
      } else {
        print OUTPUT << 'EOF';
DRIVERS-$(CONFIG_I2C) += drivers/i2c/i2c.o
EOF
      }
      $printed = 1;
    } elsif (not $printed and 
        (m@include arch/\$\(ARCH\)/Makefile@ or m@CONFIG_SENSORS@ or
         m@CONFIG_PHONE@ )) {
      print OUTPUT <<'EOF';
ifeq ($(CONFIG_I2C),y)
DRIVERS := $(DRIVERS) drivers/i2c/i2c.a
endif

EOF
      $printed = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `Makefile' failed.\n".
      "See our home page http://www.lm-sensors.nu for assistance!" if $printed == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for drivers/Makefile
# First, `i2c' is added to the ALL_SUB_DIRS list. Next, a couple of lines
# to add i2c to the SUB_DIRS and/or MOD_SUB_DIRS lists is put right before
# Rules.make is included.
# Of course, care is taken old lines are removed.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_drivers_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "drivers/Makefile";
  my $package_file = $temp;
  my $i2c_present;
  my $printed = 0;
  my $added = 0;
  my $new_style = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@^mod-subdirs\s*:=@) {
       $new_style = 1;
    }
    if ((! $new_style and m@^ALL_SUB_DIRS\s*:=@) or m@^mod-subdirs\s*:=@ ) {
      $added = 1;
      $i2c_present = 0;
      while (m@\\$@) {
        $i2c_present = 1 if m@i2c@;
        print OUTPUT;
        $_ = <INPUT>;
      }
      $i2c_present = 1 if m@i2c@;
      s@$@ i2c@ if (not $i2c_present);
      print OUTPUT;
      $_ = <INPUT>;
      redo MAIN;
    } 
    if (m@^ifeq.*CONFIG_I2C@) {
      $_ = <INPUT> while not m@^endif@;
      $_ = <INPUT>;
      $_ = <INPUT> if m@^$@;
      redo MAIN;
    } 
    if (m@^subdir.*CONFIG_I2C@) {
      $_ = <INPUT>;
      redo MAIN;
    }
    if (not $printed and
        (m@^include \$\(TOPDIR\)/Rules.make$@ or
         m@^ifeq \(\$\(CONFIG_ACPI\),y\)$@ or
         m@^ifeq \(\$\(CONFIG_SENSORS\),y\)@) or
         m@^subdir-\$\(CONFIG_ACPI\)@) {
      if ($new_style) {
        print OUTPUT <<'EOF';
subdir-$(CONFIG_I2C)		+= i2c
EOF
      } else {
        print OUTPUT <<'EOF';
ifeq ($(CONFIG_I2C),y)
SUB_DIRS += i2c
MOD_SUB_DIRS += i2c
else
  ifeq ($(CONFIG_I2C),m)
  MOD_SUB_DIRS += i2c
  endif
endif

EOF
      }
      $printed = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `Makefile' failed.\n".
      "See our home page http://www.lm-sensors.nu for assistance!" if $printed == 0 or $added == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for drivers/char/Makefile
# It changes all occurences of `i2c.o' to `i2c-old.o'.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_drivers_char_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "drivers/char/Makefile";
  my $package_file = $temp;
  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  while(<INPUT>) {
    s@i2c\.o@i2c-old\.o@;
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

sub gen_drivers_i2c_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "drivers/i2c/Makefile";
  my $package_file = $temp;
  my $use_new_format = 0;
  if (-e "$kernel_root/$kernel_file") {
    `grep -q -s 'i2c\.o' "$kernel_root/$kernel_file"`;
     $use_new_format = ! $?;
  }

  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  if ($use_new_format) {
    print OUTPUT <<'EOF';
#
# Makefile for the kernel i2c bus driver.
#

O_TARGET := i2c.o

export-objs	:= i2c-core.o i2c-algo-bit.o i2c-algo-pcf.o i2c-proc.o

obj-$(CONFIG_I2C)		+= i2c-core.o
obj-$(CONFIG_I2C_CHARDEV)	+= i2c-dev.o
obj-$(CONFIG_I2C_ALGOBIT)	+= i2c-algo-bit.o
obj-$(CONFIG_I2C_BITLP)	+= i2c-philips-par.o
obj-$(CONFIG_I2C_BITELV)	+= i2c-elv.o
obj-$(CONFIG_I2C_BITVELLE)	+= i2c-velleman.o
obj-$(CONFIG_I2C_ALGOPCF)	+= i2c-algo-pcf.o
obj-$(CONFIG_I2C_PCFISA)	+= i2c-elektor.o
obj-$(CONFIG_I2C_PROC)		+= i2c-proc.o

# This is needed for automatic patch generation: sensors code starts here
# This is needed for automatic patch generation: sensors code ends here

include $(TOPDIR)/Rules.make

EOF
  } else {
    print OUTPUT <<'EOF';
#
# Makefile for the kernel i2c bus driver.
#

SUB_DIRS     :=
MOD_SUB_DIRS := $(SUB_DIRS)
ALL_SUB_DIRS := $(SUB_DIRS)
MOD_LIST_NAME := I2C_MODULES

L_TARGET := i2c.a
MX_OBJS :=  
M_OBJS  := 
LX_OBJS :=
L_OBJS  := 

# -----
# i2c core components
# -----

ifeq ($(CONFIG_I2C),y)
  LX_OBJS += i2c-core.o
else
  ifeq ($(CONFIG_I2C),m)
    MX_OBJS += i2c-core.o
  endif
endif

ifeq ($(CONFIG_I2C_CHARDEV),y)
  L_OBJS += i2c-dev.o
else
  ifeq ($(CONFIG_I2C_CHARDEV),m)
    M_OBJS += i2c-dev.o
  endif
endif

ifeq ($(CONFIG_I2C_PROC),y)
  LX_OBJS += i2c-proc.o
else
  ifeq ($(CONFIG_I2C_PROC),m)
    MX_OBJS += i2c-proc.o
  endif
endif

# -----
# Bit banging adapters...
# -----

ifeq ($(CONFIG_I2C_ALGOBIT),y)
  LX_OBJS += i2c-algo-bit.o
else
  ifeq ($(CONFIG_I2C_ALGOBIT),m)
    MX_OBJS += i2c-algo-bit.o
  endif
endif

ifeq ($(CONFIG_I2C_BITLP),y)
  L_OBJS += i2c-philips-par.o
else
  ifeq ($(CONFIG_I2C_BITLP),m)
    M_OBJS += i2c-philips-par.o
  endif
endif

ifeq ($(CONFIG_I2C_BITELV),y)
  L_OBJS += i2c-elv.o
else
  ifeq ($(CONFIG_I2C_BITELV),m)
    M_OBJS += i2c-elv.o
  endif
endif

ifeq ($(CONFIG_I2C_BITVELLE),y)
  L_OBJS += i2c-velleman.o
else
  ifeq ($(CONFIG_I2C_BITVELLE),m)
    M_OBJS += i2c-velleman.o
  endif
endif



# -----
# PCF components
# -----

ifeq ($(CONFIG_I2C_ALGOPCF),y)
  LX_OBJS += i2c-algo-pcf.o
else
  ifeq ($(CONFIG_I2C_ALGOPCF),m)
    MX_OBJS += i2c-algo-pcf.o
  endif
endif

ifeq ($(CONFIG_I2C_PCFISA),y)
  L_OBJS += i2c-elektor.o
else
  ifeq ($(CONFIG_I2C_PCFISA),m)
    M_OBJS += i2c-elektor.o
  endif
endif

# This is needed for automatic patch generation: sensors code starts here
# This is needed for automatic patch generation: sensors code ends here

include $(TOPDIR)/Rules.make

EOF
  }
  close OUTPUT;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for drivers/char/Config.in
# It adds a line just before CONFIG_APM or main_menu_option lines to include
# the I2C Config.in.
# Of course, care is taken old lines are removed.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_drivers_char_Config_in
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "drivers/char/Config.in";
  my $package_file = $temp;
  my $ready = 0;
  my $printed = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@i2c@) {
      $_ = <INPUT>;
      $_ = <INPUT> if (m@^$@);
      redo MAIN;
    }
    if ($ready and not $printed and 
        (m@^mainmenu_option@ or m@CONFIG_APM@ or m@CONFIG_ALPHA_BOOK1@ or
         m@source drivers/sensors/Config.in@)) {
      $printed = 1;
      print OUTPUT <<'EOF';
source drivers/i2c/Config.in

EOF
    }
    $ready = 1 if (m@^mainmenu_option@);
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `drivers/char/Config.in' failed.\n".
      "See our home page http://www.lm-sensors.nu for assistance!" if $printed == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}
 

# This generates diffs for drivers/char/mem.c They are a bit intricate.
# Lines are generated at the beginning to declare i2c_init and i2c_init_all.
# The first is the invocation for the old I2C driver, the second for the
# new driver. At the bottom, a call to i2c_init_all is added when the
# new I2C stuff is configured in.
# Of course, care is taken old lines are removed.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_drivers_char_mem_c
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "drivers/char/mem.c";
  my $package_file = $temp;
  my $right_place = 0;
  my $done = 0;
  my $atstart = 1;
  my $pr1 = 0;
  my $pr2 = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@#include <linux/i2c.h>@) {
       $_=<INPUT>;
       redo MAIN;
    }
    if ($atstart and m@#ifdef@) {
      print OUTPUT << 'EOF';
#ifdef CONFIG_I2C
extern int i2c_init_all(void);
#endif
EOF
      $atstart = 0;
      $pr1 = 1;
    }
    while (not $right_place and (m@CONFIG_I2C@ or m@CONFIG_VIDEO_BT848@)) {
      $_ = <INPUT> while not m@#endif@;
      $_ = <INPUT>;
      redo MAIN;
    }
    $right_place = 1 if (m@chr_dev_init@);
    if ($right_place and m@CONFIG_I2C@) {
      $_ = <INPUT> while not m@#endif@;
      $_ = <INPUT>;
      $_ = <INPUT> if m@^$@;
      redo MAIN;
    }
    if ($right_place and not $done and
        (m@CONFIG_VIDEO_BT848@ or m@return 0;@ or m@CONFIG_SENSORS@ or
         m@CONFIG_FB@)) {
      print OUTPUT <<'EOF';
#ifdef CONFIG_I2C
	i2c_init_all();
#endif
EOF
      $done = 1;
      $pr2 = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `drivers/char/mem.c' failed.\n".
      "See our home page http://www.lm-sensors.nu for assistance!" if $pr1 == 0 or $pr2 == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}
 
# Generate the diffs for the list of MAINTAINERS
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_MAINTAINERS
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "MAINTAINERS";
  my $package_file = $temp;
  my $done = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@I2C DRIVERS@) {
       $_=<INPUT> while not m@^$@;
       $_=<INPUT>;
       redo MAIN;
    }
    if (not $done and (m@i386 BOOT CODE@ or m@IBM MCA SCSI SUBSYSTEM DRIVER@)) {
      print OUTPUT <<'EOF';
I2C DRIVERS
P:	Simon Vogl
M:	simon@tk.uni-linz.ac.at
P:	Frodo Looijaard
M:	frodol@dds.nl
L:	linux-i2c@pelican.tk.uni-linz.ac.at
W:	http://www.tk.uni-linz.ac.at/~simon/private/i2c
S:	Maintained

EOF
      $done = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `MAINTAINERS' failed.\n".
      "See our home page http://www.lm-sensors.nu for assistance!" if $done == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# Main function
sub main
{
  my ($package_root,$kernel_root,%files,%includes,$package_file,$kernel_file);
  my ($diff_command,$dummy,$data0,$data1,$sedscript,@sensors_subs);

  # --> Read the command-line
  $package_root = $ARGV[0];
  die "Package root `$package_root' is not found\n" 
        unless -d "$package_root/mkpatch";
  $kernel_root = $ARGV[1];
  die "Kernel root `$kernel_root' is not found\n" 
        unless -f "$kernel_root/Rules.make";

  patch_old_i2c $package_root, $kernel_root;
         

  # --> Read FILES
  open INPUT, "$package_root/mkpatch/FILES" 
        or die "Can't open `$package_root/mkpatch/FILES'";
  while (<INPUT>) {
    ($data0,$data1) = /(\S+)\s+(\S+)/;
    $files{$data0} = $data1;
  } 
  close INPUT;

  # --> Read INCLUDES
  open INPUT, "$package_root/mkpatch/INCLUDES" 
        or die "Can't open `$package_root/mkpatch/INCLUDES'";
  while (<INPUT>) {
    ($data0,$data1) = /(\S+)\s+(\S+)/;
    $includes{$data0} = $data1;
    $sedscript .= 's,(#\s*include\s*)'.$data0.'(\s*),\1'."$data1".'\2, ; ';
  } 
  close INPUT;

  # --> Start generating
  foreach $package_file (sort keys %files) {
    $kernel_file = $files{$package_file};
    @sensors_subs = find_sensors_code "$kernel_root","$kernel_file";
    open INPUT, "$package_root/$package_file"
         or die "Can't open `$package_root/$package_file'";
    open OUTPUT, ">$package_root/$temp"
         or die "Can't open `$package_root/$temp'";
    while (<INPUT>) {
      eval $sedscript;
      if (m@sensors code starts here@) {
        print OUTPUT;
        while (<INPUT>) {
           last if m@sensors code ends here@;
        }
        print OUTPUT $sensors_subs[0];
        shift @sensors_subs
      }
      print OUTPUT;
    }
    close INPUT;
    close OUTPUT;
    print_diff "$package_root","$kernel_root","$kernel_file","$temp";
  }

  gen_Makefile $package_root, $kernel_root;
  gen_drivers_Makefile $package_root, $kernel_root;
  gen_drivers_i2c_Makefile $package_root, $kernel_root;
  gen_drivers_char_Config_in $package_root, $kernel_root;
  gen_drivers_char_mem_c $package_root, $kernel_root;
  gen_Documentation_Configure_help $package_root, $kernel_root;
  gen_MAINTAINERS $package_root, $kernel_root;
}

main;

