/*
 * ion/function.c
 *
 * Copyright (c) Tuomo Valkonen 1999-2001. 
 * See the included file LICENSE for details.
 */

#include <string.h>
#include "common.h"
#include "function.h"
#include "exec.h"
#include "frame.h"
#include "clientwin.h"
#include "client.h"
#include "focus.h"
#include "workspace.h"
#include "pointer.h"
#include "wedln.h"
#include "edln.h"
#include "query.h"
#include "resize.h"


/*{{{ Definitions and function classes */


static void callhnd_direct(WThing *thing, WFunction *func,
						   int n, const Token *args);
static void callhnd_generic_void(WThing *thing, WFunction *func,
								 int n, const Token *args);
static void callhnd_generic_l(WThing *thing, WFunction *func,
							  int n, const Token *args);
static void callhnd_generic_s(WThing *thing, WFunction *func,
							  int n, const Token *args);
static void callhnd_generic_ss(WThing *thing, WFunction *func,
							   int n, const Token *args);
static void callhnd_global_void(WThing *thing, WFunction *func,
								int n, const Token *args);
static void callhnd_global_l(WThing *thing, WFunction *func,
							 int n, const Token *args);
static void callhnd_global_ll(WThing *thing, WFunction *func,
							  int n, const Token *args);
static void callhnd_global_s(WThing *thing, WFunction *func,
							 int n, const Token *args);
static void callhnd_edln_void(WThing *thing, WFunction *func,
							  int n, const Token *args);
extern void callhnd_button_void(WThing *thing, WFunction *func,
								int n, const Token *args);
extern void callhnd_drag_void(WThing *thing, WFunction *func,
							  int n, const Token *args);
static void callhnd_cclient_void(WThing *thing, WFunction *func,
								 int n, const Token *args);

extern WDragHandler p_tabdrag_handler;
extern WDragHandler p_resize_handler;


/*}}}*/


/*{{{ Wrappers/helpers */


static void wrap_switch_client(WClient *client)
{
	frame_switch_client(CLIENT_FRAME(client), client);
}


/*}}}*/


/*{{{ Function table */


#define FN_(CALLHND, ARGTSTR, OBJT, NAME, FUN) \
	{callhnd_##CALLHND, &OBJDESCR(OBJT), ARGTSTR, NAME, (void*)FUN}

#define FN(ARGT, HND, OBJT, NAME, FUN) \
	FN_(HND##_##ARGT, #ARGT, OBJT, NAME, FUN)

#define FN_VOID(HND, OBJT, NAME, FUN) \
	FN_(HND##_void, "", OBJT, NAME, FUN)

#define FN_GLOBAL(ARGT, NAME, FUN) \
	FN(ARGT, global, WObj, NAME, FUN)

#define FN_GLOBAL_VOID(NAME, FUN) \
	FN_VOID(global, WObj, NAME, FUN)

#define FN_SCREEN(ARGT, NAME, FUN) \
	FN(ARGT, generic, WScreen, NAME, FUN)

#define FN_SCREEN_VOID( NAME, FUN) \
	FN_VOID(generic, WScreen, NAME, FUN)

static WFunction funcs[]={
	/* frame */
	FN(l,	generic, WFrame,	"switch_nth",		frame_switch_nth),
	FN_VOID(generic, WFrame, 	"switch_next",		frame_switch_next),
	FN_VOID(generic, WFrame, 	"switch_prev",		frame_switch_prev),
	FN_VOID(generic, WFrame,	"split_vert",		split_vert),
	FN_VOID(generic, WFrame,	"split_horiz",		split_horiz),
	FN_VOID(generic, WFrame,	"destroy_frame",	destroy_frame),
	FN_VOID(generic, WFrame,	"closedestroy",		closedestroy),
	FN_VOID(generic, WFrame,	"attach_tagged",	frame_attach_tagged),
	
	FN(ss, 	generic, WFrame,	"query_runfile",	query_runfile),
	FN(ss, 	generic, WFrame,	"query_runwith",	query_runwith),
	FN(ss,	generic, WFrame,	"query_yesno",		query_yesno),
	FN_VOID(generic, WFrame,	"query_function",	query_function),
	FN_VOID(generic, WFrame,	"query_exec",		query_exec),
	FN_VOID(generic, WFrame,	"query_attachclient", query_attachclient),
	FN_VOID(generic, WFrame,	"query_gotoclient", query_gotoclient),
	FN_VOID(generic, WFrame,	"query_workspace",	query_workspace),
	FN_VOID(generic, WFrame,	"query_workspace_with",	query_workspace_with),
	
	FN_VOID(generic, WWindow,	"resize_vert",		resize_vert),
	FN_VOID(generic, WWindow,	"resize_horiz",		resize_horiz),
	FN_VOID(generic, WWindow,	"maximize_vert", 	maximize_vert),
	FN_VOID(generic, WWindow,	"maximize_horiz", 	maximize_horiz),

	/* screen */
	FN_SCREEN(l,				"switch_ws_nth",	switch_workspace_nth),
	FN_SCREEN_VOID(				"switch_ws_next",	switch_workspace_next),     
	FN_SCREEN_VOID(				"switch_ws_prev",	switch_workspace_prev),     
	FN_SCREEN(s,				"exec",				wm_exec),

	/* client */
	FN_VOID(cclient, WFrame,	"close",			close_client),
	FN_VOID(cclient, WFrame,	"close_main",		close_client_main),
	FN_VOID(cclient, WFrame,	"kill",				kill_client),
	FN_VOID(cclient, WFrame,	"toggle_tagged",	client_toggle_tagged),
	FN_VOID(generic, WClient,	"switch_tab",		wrap_switch_client),
	
	/* global */
	FN_GLOBAL_VOID(				"goto_above_frame",	goto_above_frame),
	FN_GLOBAL_VOID(				"goto_below_frame",	goto_below_frame),
	FN_GLOBAL_VOID(				"goto_right_frame",goto_right_frame),
	FN_GLOBAL_VOID(				"goto_left_frame",	goto_left_frame),
	FN_GLOBAL_VOID(				"goto_previous",	goto_previous),
	FN_GLOBAL_VOID(				"restart",			wm_restart),
	FN_GLOBAL_VOID(				"exit",				wm_exit),
	FN_GLOBAL(s,				"restart_other",	wm_restart_other),
	FN_GLOBAL(s,				"switch_ws_name",	switch_workspace_name),
	FN_GLOBAL(ll,				"switch_ws_nth2",	switch_workspace_nth2),
	FN_GLOBAL_VOID(				"clear_tags",		clear_tags),

	/* mouse move/resize and tab drag */
	FN_VOID(drag, WFrame,		"p_resize",			&p_resize_handler),
	FN_VOID(drag, WClient,		"p_tabdrag", 		&p_tabdrag_handler),
	{NULL, NULL, NULL, NULL, NULL}
};


static WFunction moveres_funcs[]={
	FN_VOID(generic, WWindow,	"end_resize",		end_resize),
	FN_VOID(generic, WWindow,	"cancel_resize",	cancel_resize),
	FN_VOID(generic, WWindow,	"grow",				grow),
	FN_VOID(generic, WWindow,	"shrink",			shrink),
	{NULL, NULL, NULL, NULL, NULL}
};


static WFunction input_funcs[]={
	FN_VOID(edln, WEdln,		"back",				edln_back),
	FN_VOID(edln, WEdln,		"forward",			edln_forward),
	FN_VOID(edln, WEdln,		"bol",				edln_bol),
	FN_VOID(edln, WEdln,		"eol",				edln_eol),
	FN_VOID(edln, WEdln,		"skip_word",		edln_skip_word),
	FN_VOID(edln, WEdln,		"bskip_word",		edln_bskip_word),
	FN_VOID(edln, WEdln,		"delete",			edln_delete),
	FN_VOID(edln, WEdln,		"backspace",		edln_backspace),
	FN_VOID(edln, WEdln,		"kill_to_eol",		edln_kill_to_eol),
	FN_VOID(edln, WEdln,		"kill_to_bol",		edln_kill_to_bol),
	FN_VOID(edln, WEdln,		"kill_line",		edln_kill_line),
	FN_VOID(edln, WEdln,		"kill_word",		edln_kill_word),
	FN_VOID(edln, WEdln,		"bkill_word",		edln_bkill_word),
	FN_VOID(edln, WEdln,		"set_mark",			edln_set_mark),
	FN_VOID(edln, WEdln,		"cut",				edln_cut),
	FN_VOID(edln, WEdln,		"copy",				edln_copy),
	FN_VOID(edln, WEdln,		"complete",			edln_complete),
	FN_VOID(edln, WEdln,		"history_next",		edln_history_next),
	FN_VOID(edln, WEdln,		"history_prev",		edln_history_prev),
	FN_VOID(generic, WEdln,		"paste",			wedln_paste),
	FN_VOID(generic, WEdln,		"finish",			wedln_finish),	
	FN_VOID(generic, WInput,	"cancel",			input_cancel),	
	FN_VOID(generic, WInput,	"scrollup",			input_scrollup),
	FN_VOID(generic, WInput,	"scrolldown",		input_scrolldown),
	{NULL, NULL, NULL, NULL, NULL}
};


static WFunction *funtabs[]={
	funcs,
	moveres_funcs,
	input_funcs
};


/*}}}*/


/*{{{ lookup_func */


WFunction *lookup_func(const char *name, int funtab)
{
	WFunction *func=funtabs[funtab];
	
	while(func->callhnd!=NULL){
		if(strcmp(func->name, name)==0)
			return func;
		func++;
	}
	
	return NULL;
}


int complete_simplemainfunc(char *nam, char ***cp_ret, char **beg)
{
	char *name;
	char **cp;
	int n=0, l=strlen(nam);
	WFunction *func=funtabs[FUNTAB_MAIN];

	*cp_ret=NULL;
	
	for(; func->callhnd!=NULL; func++){
		
		if((name=func->name)==NULL)
			continue;
		
		if(func->argtypes!=NULL && strcmp(func->argtypes, ""))
			continue;
		
		if(l && strncmp(name, nam, l))
			continue;
		
		name=scopy(name);
		
		if(name==NULL){
			warn_err();
			continue;
		}
		
		cp=REALLOC_N(*cp_ret, char*, n, n+1);
		
		if(cp==NULL){
			warn_err();
			free(name);
		}else{
			cp[n]=name;
			n++;
			*cp_ret=cp;
		}
	}
	
	return n;
}

/*}}}*/


/*{{{ Call handlers */


static void callhnd_direct(WThing *thing, WFunction *func,
						   int n, const Token *args)
{
	typedef void Func(WThing*, int, const Token*);
	((Func*)func->fn)(thing, n, args);
}


static void callhnd_generic_void(WThing *thing, WFunction *func,
								 int n, const Token *args)
{
	typedef void Func(WThing*);
	thing=find_parent(thing, func->objdescr);
	if(thing!=NULL)
		((Func*)func->fn)(thing);
}


static void callhnd_generic_l(WThing *thing, WFunction *func,
							  int n, const Token *args)
{
	typedef void Func(WThing*, int);
	thing=find_parent(thing, func->objdescr);
	if(thing!=NULL)
		((Func*)func->fn)(thing, TOK_LONG_VAL(args));
}


static void callhnd_generic_s(WThing *thing, WFunction *func,
							  int n, const Token *args)
{
	typedef void Func(WThing*, char*);
	thing=find_parent(thing, func->objdescr);
	if(thing!=NULL)
		((Func*)func->fn)(thing, TOK_STRING_VAL(args));
}


static void callhnd_generic_ss(WThing *thing, WFunction *func,
							   int n, const Token *args)
{
	typedef void Func(WThing*, char*, char*);
	thing=find_parent(thing, func->objdescr);
	if(thing!=NULL)
		((Func*)func->fn)(thing, TOK_STRING_VAL(args), TOK_STRING_VAL(args+1));
}


static void callhnd_global_void(WThing *thing, WFunction *func,
								int n, const Token *args)
{
	typedef void Func();
	((Func*)func->fn)();
}


static void callhnd_global_l(WThing *thing, WFunction *func,
							 int n, const Token *args)
{
	typedef void Func(int);
	((Func*)func->fn)(TOK_LONG_VAL(args));
}


static void callhnd_global_ll(WThing *thing, WFunction *func,
							  int n, const Token *args)
{
	typedef void Func(int, int);
	((Func*)func->fn)(TOK_LONG_VAL(args), TOK_LONG_VAL(args+1));
}


static void callhnd_global_s(WThing *thing, WFunction *func,
							 int n, const Token *args)
{
	typedef void Func(char*);
	((Func*)func->fn)(TOK_STRING_VAL(args));
}


static void callhnd_edln_void(WThing *thing, WFunction *func,
							  int n, const Token *args)
{
	WEdln *wedln;
	typedef void Func(Edln*);
	
	if(!WTHING_IS(thing, WEdln))
		return;
	
	wedln=(WEdln*)thing;
	
	((Func*)func->fn)(&(wedln->edln));
}

static void callhnd_cclient_void(WThing *thing, WFunction *func,
								 int n, const Token *args)
{
	typedef void Func(WClient*);
	WFrame *frame;
	frame=FIND_PARENT(thing, WFrame);
	if(frame!=NULL && frame->current_client!=NULL)
		((Func*)func->fn)(frame->current_client);
}


/*}}}*/


