;;; jde-gen.el -- Integrated Development Environment for Java.
;; $Revision: 1.26 $ 

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Keywords: java, tools

;; Copyright (C) 1997, 1998 Paul Kinnucan.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; Gnu Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.
;;

(require 'tempo)

(defgroup jde-gen nil
  "JDE Autocoder"
  :group 'jde
  :prefix "jde-gen-")

(defcustom jde-gen-k&r t
  "*If non-nil, use braces in Original Kerningham & Ritchie Style.
The Creators of C started using brace placement style:

    class Some {

    }

But there is also alternative line-up style

    class Some
    {

    }

Setting this variable to t, uses k&R style in skeletons and tempaltes."
  :group 'jde-gen
  :type  'boolean)

(defun jde-gen-lookup-named (name)
  "Lookup some saved data under the name NAME.
Returns the data if NAME was found, and nil otherwise."
  (cdr (assq name tempo-named-insertions)))

(defun jde-gen-read-template (strings)
  "Converts an autocode template represented as a list
of strings to a list of Lisp objects as required by
tempo."
  (let ((template-string "")
	(n (length strings))
	(i 0))
    (while (< i n)
      (setq template-string
		  (concat template-string " " (nth i strings)))
      (setq i (1+ i)))
      (setq template-string
	    (concat "'(" template-string ")"))
      (eval (car (read-from-string template-string)))))

(defcustom jde-gen-buffer-boilerplate nil
"*Lines of boilerplate text to put at the head of a buffer template."
  :group 'jde-gen
  :type '(repeat (string :tag "Line")))

(defcustom jde-gen-boilerplate-function 'jde-gen-create-buffer-boilerplate
"*Specifes buffer boilerplate text function.
This variable specifies a function to create boilerplate text for
insertion at the head of Java source buffers generated by JDE
templates. The specified function should take no arguments and should
return a text string.  The default value is
`jde-gen-create-buffer-boilerplate', which returns the lines of text
specified by `jde-gen-buffer-boilerplate'."
  :group 'jde-gen
  :type 'function)

(defun jde-gen-create-buffer-boilerplate ()
"This function creates buffer boilerplate from the
variable `jde-gen-buffer-boilerplate."
  (if jde-gen-buffer-boilerplate
      (let ((bp "")
	    (n (length jde-gen-buffer-boilerplate))
	    (i 0))
	(while (< i n)
	  (setq bp
		(concat bp (elt jde-gen-buffer-boilerplate i) "\n"))
	  (setq i (1+ i)))
	bp)))

(defun jde-gen-get-super-class ()
  (let ((super-class (read-from-minibuffer "extends: "))
	(interface (read-from-minibuffer "implements: ")))
    (cond
     ((and
       (not (string= super-class ""))
       (not (string= interface "")))
      (concat "extends " super-class "\n"
	      "  implements " interface))
     ((not (string= super-class ""))
      (concat "extends " super-class))
     ((not (string= interface ""))
      (concat "implements " interface)))))

(defcustom jde-gen-class-buffer-template
  (list
    "(funcall jde-gen-boilerplate-function) '>'n"
    "\"/**\" '>'n"
    "\" * \""
    "(file-name-nondirectory buffer-file-name) '>'n"
    "\" *\" '>'n"
    "\" *\" '>'n"
    "\" * Created: \" (current-time-string) '>'n"
    "\" *\" '>'n"
    "\" * @author <a href=\\\"mailto: \\\"\" (user-full-name) \"</a>\"'>'n"
    "\" * @version\" '>'n"
    "\" */\" '>'n'"
    "'>'n"
    "\"public class \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\" \" (jde-gen-get-super-class)"

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"public \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\" ()\""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'p'n"
    "\"}\">"
    "'>'n"
    "\"}\">"
    "\"// \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "'>'n")
  "*Template for new Java class.
Setting this variable defines a template instantiation
command `jde-gen-class', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-class
	    (tempo-define-template "java-class-buffer-template"
		       (jde-gen-read-template val)
                       nil
                       "Insert a generic Java class buffer skeleton."))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-class-buffer (file)
  "Create a new Java buffer containing a class of the same name.
This command inserts the class template generated by `jde-gen-class'.
It then moves the point to the location to the constructor."
  (interactive "F")
  (find-file file)
  (jde-gen-class)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))



(defcustom jde-gen-console-buffer-template
  (list
    "(funcall jde-gen-boilerplate-function) '>'n"
    "\"/**\" '>'n"
    "\" * \""
    "(file-name-nondirectory buffer-file-name) '>'n"
    "\" *\" '>'n"
    "\" *\" '>'n"
    "\" * Created: \" (current-time-string) '>'n"
    "\" *\" '>'n"
    "\" * @author <a href=\\\"mailto: \\\"\" (user-full-name) \"</a>\"'>'n"
    "\" * @version\" '>'n"
    "\" */\" '>'n"
    "'>'n"
    "\"public class \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"public \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\" ()\""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n"
    "\"}\"'>'n"
    "'>'n"
    "\"public static void main(String[] args)\""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'p'n"
    "\"}\"'>'n"
    "\"} // \"'>"
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "'>'n")
  "*Template for new Java console app main class buffer.
Setting this variable defines a template instantiation
command `jde-gen-console', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-console
	    (tempo-define-template "java-console-buffer-template"
		       (jde-gen-read-template val)
                       nil
                       "Insert skeleton for a new Java console buffer"))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-console-buffer (file)
  "Create a new Java buffer containing a console class of the same name.
This command inserts the class template generated by `jde-gen-class'.
It then moves the point to the location to the constructor."
  (interactive "F")
  (find-file file)
  (jde-gen-console)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))


(defcustom jde-gen-jfc-app-buffer-template
  (list
    "(funcall jde-gen-boilerplate-function) '>'n"
    "\"import java.awt.Dimension;\" '>'n"
    "\"import java.awt.Graphics;\" '>'n"
    "\"import java.awt.Graphics2D;\" '>'n"
    "\"import java.awt.Color;\" '>'n"
    "\"import java.awt.geom.Ellipse2D;\" '>'n"
    "\"import java.awt.event.WindowAdapter;\" '>'n"
    "\"import java.awt.event.WindowEvent;\" '>'n"
    "\"import javax.swing.JFrame;\" '>'n"
    "\"import javax.swing.JPanel;\" '>'n"
    "\"import javax.swing.JScrollPane;\" '>'n"
    "\"import javax.swing.JMenuBar;\" '>'n"
    "\"import javax.swing.JMenu;\" '>'n"
    "\"import java.awt.event.ActionEvent;\" '>'n"
    "\"import javax.swing.AbstractAction;\" '>'n '>'n"
    "\"/**\" '>'n"
    "\" * \""
    "(file-name-nondirectory buffer-file-name) '>'n"
    "\" *\" '>'n"
    "\" *\" '>'n"
    "\" * Created: \" (current-time-string) '>'n"
    "\" *\" '>'n"
    "\" * @author <a href=\\\"mailto: \\\"\" (user-full-name) \"</a>\"'>'n"
    "\" * @version\" '>'n"
    "\" */\" '>'n"
    "'>'n"
    "\"public class \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\" extends JFrame\""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"class Canvas extends JPanel\""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"public Canvas () \""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"setSize(getPreferredSize());\" '>'n"
    "\"Canvas.this.setBackground(Color.white);\" '>'n"
    "\"}\"'>'n '>'n"

    "\"public Dimension getPreferredSize() \""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"


    "\"return new Dimension(600, 600);\" '>'n"
    "\"}\"'>'n '>'n"

    "\"public void paintComponent(Graphics g) \""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"super.paintComponent(g);\" '>'n"
    "\"Graphics2D g2d = (Graphics2D) g;\" '>'n"
    "\"Ellipse2D circle = new Ellipse2D.Double(0d, 0d, 100d, 100d);\" '>'n"
    "\"g2d.setColor(Color.red);\" '>'n"
    "\"g2d.translate(10, 10);\" '>'n"
    "\"g2d.draw(circle);\" '>'n"
    "\"g2d.fill(circle);\" '>'n"
    "\"}\"'>'n "

    "\"}\"'>'n '>'n"


    ;; Constructor
    "\"public \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\"()\""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"


    "\"super(\\\"\" (P \"Enter app title: \") \"\\\");\" '>'n"
    "\"setSize(300, 300);\" '>'n"
    "\"addWindowListener(new WindowAdapter() \""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"


    "\"public void windowClosing(WindowEvent e) {System.exit(0);}\" '>'n"
    "\"public void windowOpened(WindowEvent e) {}\" '>'n"
    "\"});\"'>'n"


    "\"setJMenuBar(createMenu());\" '>'n"
    "\"getContentPane().add(new JScrollPane(new Canvas()));\" '>'n"
    "\"}\"'>'n"
    "'>'n"

    ;; Main method
    "\"public static void main(String[] args) \""


    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n"
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\" f = new \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\"();\" '>'n"
    "\"f.show();\" '>'n"
    "\"}\"'>'n '>'n"
    ;; createMenu method
    "\"protected JMenuBar createMenu() \""


    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"


    "\"JMenuBar mb = new JMenuBar();\" '>'n"
    "\"JMenu menu = new JMenu(\\\"File\\\");\" '>'n"
    "\"menu.add(new AbstractAction(\\\"Exit\\\") \""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"


    "\"public void actionPerformed(ActionEvent e) \""

    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"


    "\"System.exit(0);\" '>'n"
    "\"}\" '>'n"
    "\"});\" '>'n"
    "\"mb.add(menu);\" '>'n"
    "\"return mb;\" '>'n"
    "\"}\"'>'n "
    "\"} // \"'>"
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "'>'n")
  "*Template for JFC (Swing) application buffer.
Setting this variable defines a template instantiation
command `jde-gen-jfc-app', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-jfc-app
	    (tempo-define-template "java-jfc-app-buffer-template"
		       (jde-gen-read-template val)
                       nil
                       "Insert skeleton for a JFC app buffer"))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-jfc-app-buffer (file)
  "Create a new Java buffer containing a JFC application class.
This command inserts the class template generated by `jde-gen-jfc-app'.
It then moves the point to the location to the constructor."
  (interactive "F")
  (find-file file)
  (jde-gen-jfc-app)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))


(defcustom jde-gen-buffer-templates
  (list (cons "Class" 'jde-gen-class)
	(cons "Console" 'jde-gen-console)
	(cons "Swing App" 'jde-gen-jfc-app))
  "*Specifies available autocode templates for creating buffers.
The value of this variable is an association list. The car of
each element specifies the template's title. The cdr specifies
a command that inserts the template into a buffer. See the function
`tempo-define-template' for any easy way to create a template
insertion command."
  :group 'jde-gen
  :type '(repeat
	  (cons :tag "Template"
	   (string :tag "Title")
	   (function :tag "Command")))
  :set '(lambda (sym val)
	  (let ((n (length val))
		(i 0))
	    (setq jde-gen-buffer-template-names (list))
	    (while (< i n)
	      (setq jde-gen-buffer-template-names
		    (append
		     jde-gen-buffer-template-names
		     (list (cons (car (nth i val)) (1+ i)))))
	      (setq i (1+ i))))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-buffer (template file)
  "Create a new Java buffer containing a code template.
This command inserts the specified template at the beginning
of the buffer."
  (interactive
   (list (completing-read "Template: " jde-gen-buffer-template-names)
	 (read-file-name "File: ")))
  (find-file file)
  (funcall (cdr (assoc template jde-gen-buffer-templates)))
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp))

(defun jde-gen-init-cap (name)
  (concat (upcase (substring name 0 1)) (substring name 1)))


(defcustom jde-gen-get-set-var-template
  '(
    "(end-of-line) '&"
    "(P \"Variable type: \" type) \" \""
    "(P \"Variable name: \" name) \";\" '>'n '>'n"
    ;;we begin by the getter
    "\"/**\" '>'n"
    "\"* Get the value of \" (s name) \".\" '>'n"
    "\"* @return value of \" (s name) \".\" '>'n"
    "\"*/\" '>'n"
    " \"public \" (s type)"
    ;;we must check type cause boolean getters
    ;;start with "is" not "get"
    "(if (string= \"boolean\" (jde-gen-lookup-named 'type) ) "
    "\" is\" "
    "\" get\" ) "
    "(jde-gen-init-cap (jde-gen-lookup-named 'name))"
    "\"() \""
    
    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\" '>'n"
 
    "\"return \" (s name) \";\" '>'n \"}\""
    "'>'n '>'n"
    ;;we continue with the setter
    "\"/**\" '>'n"
    "\"* Set the value of \" (s name) \".\" '>'n"
    "\"* @param v  Value to assign to \" (s name) \".\" '>'n"
    "\"*/\" '>'n"
    "\"public void set\" (jde-gen-init-cap (jde-gen-lookup-named 'name))"
    "\"(\" (s type) \"  v) \" "

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\" '>'n"

    "'>'n \"this.\" (s name) \" = v;\" '>'n \"}\" '>'n'>"
    )
  "*Template for creating a get/set method pair.
Setting this variable defines a template instantiation
command `jde-gen-get-set', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
   (defalias 'jde-gen-get-set
     (tempo-define-template
      "java-get-set-pair"
      (jde-gen-read-template val)
      nil
      "Insert variable get-set method pair."))
   (set-default sym val)))


(defcustom jde-gen-inner-class-template
  '(
    "(end-of-line) '& \"class \" (P \"Class name: \" class)"
    "(P \"Superclass: \" super t)"
    "(let ((parent (jde-gen-lookup-named 'super)))"
        "(if (not (string= parent \"\"))"
           "(concat \" extends \" parent ))) "

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\" '>'n"

    "\"public \" (s class) \"() \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\"'>'n"
    "\"}\" '>'n'>"
    )
"*Template that creates an empty private class at point."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-inner-class
	    (tempo-define-template
	     "java-inner-class"
	     (jde-gen-read-template val)
	     nil
	     "Insert inner class."))
	  (set-default sym val)))

(defcustom jde-gen-action-listener-template
  '(
    "'& (P \"Component name: \")"
    "\".addActionListener(new ActionListener() \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"public void actionPerformed(ActionEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\"'>'n \"});\"'>'n'>"
    )
  "*Template for generating an action listener.
Setting this variable defines a template instantiation
command, `jde-gen-action-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-action-listener
	    (tempo-define-template
	     "java-action-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton action listener."))
	  (set-default sym val)))

(defcustom jde-gen-window-listener-template
  '(
    "(end-of-line) '& (P \"Window name: \")"
    "\".addWindowListener(new WindowAdapter() \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'> \"public void windowActivated(WindowEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\""
    "'>'n \"public void windowClosed(WindowEvent e)\""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n \"}\""
    "'>'n \"public void windowClosing(WindowEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n \"System.exit(0);\" '>'n \"}\""

    "'>'n \"public void windowDeactivated(WindowEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n \"}\""
    "'>'n \"public void windowDeiconified(WindowEvent e) \""
 
    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n \"}\""
    "'>'n \"public void windowIconified(WindowEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n \"}\""
    "'>'n \"public void windowOpened(WindowEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "'>'n \"}\""
    "'>'n \"});\" '>'n'>"
    )
  "*Template for generating a window listener.
Setting this variable defines a template instantiation
command, `jde-gen-window-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-window-listener
	    (tempo-define-template
	     "java-window-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton window listener."))
	  (set-default sym val)))



(defcustom jde-gen-mouse-listener-template
  '(
    "(end-of-line) '& (P \"Component name: \")"
    "\".addMouseListener(new MouseAdapter() \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'> "

    "'>'n \"public void mouseClicked(MouseEvent e) \" "

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\" '>"
    "'>'n \"public void mouseEntered(MouseEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\" '>"
    "'>'n \"public void mouseExited(MouseEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\"'>"
    "'>'n \"public void mousePressed(MouseEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\" '>"
    "'>'n \"public void mouseReleased(MouseEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\"'>"
    "'>'n \"});\"'>'n'>"
    )
  "*Template for generating a mouse listener.
Setting this variable defines a template instantiation
command, `jde-gen-mouse-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-mouse-listener
	    (tempo-define-template
	     "java-mouse-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton mouse listener."))
	  (set-default sym val)))

(defcustom jde-gen-mouse-motion-listener-template
  '(
    "(end-of-line) '& (P \"Component name: \")"
    "\".addMouseMotionListener(new MouseMotionAdapter() \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>"

    "'>'n \"public void mouseDragged(MouseEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\"'>"
    "'>'n \"public void mouseMoved(MouseEvent e) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"

    "\"}\"'>"
    "'>'n \"});\"'>'n'>"
    )
  "*Template for generating a mouse listener.
Setting this variable defines a template instantiation
command, `jde-gen-mouse-motion-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-mouse-motion-listener
	    (tempo-define-template
	     "java-mouse-motion-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton mouse motion listener."))
	  (set-default sym val)))



(defcustom jde-gen-to-string-method-template
  '(
    "(end-of-line) '&"
    "\"public String toString() \""
    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
     "\"}\"'>'n'>"
    )
  "*Template for generating a toString method.
Setting this variable defines a template instantiation
command, `jde-gen-to-string-method', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-to-string-method
	    (tempo-define-template
	     "toString method"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton toString method."))
	  (set-default sym val)))

(defcustom  jde-gen-println
  '(
    "(end-of-line) '&"
    "\"System.out.println(\" (P \"Print out: \") \");\" '>'n'>"
   )
  "*Template for generating a System.out.println statement."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-println
	    (tempo-define-template
	     "println statement"
	     (jde-gen-read-template val)
	     nil
	     "Insert println statement."))
	  (set-default sym val)))

(defcustom  jde-gen-beep
  '(
    "(end-of-line) '&"
    "\"Toolkit.getDefaultToolkit().beep();\"'>'n'>"
   )
  "*Template for generating a Toolkit.getDefaultToolkit().beep() statement."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-beep
	    (tempo-define-template
	     "beep statement"
	     (jde-gen-read-template val)
	     nil
	     "Insert beep statement."))
	  (set-default sym val)))

(defcustom  jde-gen-property-change-support
  '(
    "(end-of-line) '&"

    "\"protected PropertyChangeSupport pcs =  new PropertyChangeSupport(this);\" '>'n '>'n"


    "\"/**\" '>'n"
    "\"* Adds a PropertyChangeListener to the listener list.\" '>'n"
    "\"* The listener is registered for all properties.\" '>'n"
    "\"*\" '>'n"
    "\"* @param listener The PropertyChangeListener to be added\" '>'n"
    "\"*/\" '>'n"
    "\"public void addPropertyChangeListener(PropertyChangeListener listener) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n"
    
    "\"pcs.addPropertyChangeListener(listener);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Removes a PropertyChangeListener from the listener list.\" '>'n"
    "\"* This removes a PropertyChangeListener that was registered for all properties.\" '>'n"
    "\"*\" '>'n "
    "\"* @param listener The PropertyChangeListener to be removed\" '>'n"
    "\"*/\" '>'n"
    "\"public void removePropertyChangeListener(PropertyChangeListener listener) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n"
    
    "'>\"pcs.removePropertyChangeListener(listener);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Adds a PropertyChangeListener for a specific property.\" '>'n"
    "\"* The listener will be invoked only when a call on firePropertyChange\" '>'n"
    "\"* names that specific property.\" '>'n"
    "\"*\" '>'n \"* @param propertyName The name of the property to listen on\" '>'n"
    "\"* @param listener The PropertyChangeListener to be added\" '>'n \"*/\" '>'n"
    "\"public void addPropertyChangeListener(String propertyName,\" '>'n"
    "\"PropertyChangeListener listener) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"pcs.addPropertyChangeListener(propertyName, listener);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Removes a PropertyChangeListener for a specific property.\" '>'n"
    "\"*\" '>'n \"* @param propertyName The name of the property that was listened on\" '>'n"
    "\"* @param listener The PropertyChangeListener to be removed\" '>'n \"*/\" '>'n"
    "\"public void removePropertyChangeListener(String propertyName,\" '>'n"
    "\"PropertyChangeListener listener) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"pcs.removePropertyChangeListener(propertyName, listener);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Reports a bound property update to any registered listeners. \" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"*\" '>'n \"* @param propertyName The programmatic name of the property that was changed\" '>'n"
    "\"* @param oldValue The old value of the property\" '>'n"
    "\"* @param newValue The new value of the property.\" '>'n \"*/\" '>'n"
    "\"public void firePropertyChange(String propertyName, Object oldValue, Object newValue) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"pcs.firePropertyChange(propertyName, oldValue, newValue);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Reports a bound property update to any registered listeners. \" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"* This is merely a convenience wrapper around the more general\" '>'n"
    "\"* firePropertyChange method that takes Object values.\" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"*\" '>'n \"* @param propertyName The programmatic name of the property that was changed\" '>'n"
    "\"* @param oldValue The old value of the property\" '>'n"
    "\"* @param newValue The new value of the property.\" '>'n \"*/\" '>'n"
    "\"public void firePropertyChange(String propertyName, int oldValue, int newValue) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"pcs.firePropertyChange(propertyName, oldValue, newValue);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Reports a bound property update to any registered listeners. \" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"* This is merely a convenience wrapper around the more general\" '>'n"
    "\"* firePropertyChange method that takes Object values.\" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"*\" '>'n \"* @param propertyName The programmatic name of the property that was changed\" '>'n"
    "\"* @param oldValue The old value of the property\" '>'n"
    "\"* @param newValue The new value of the property.\" '>'n \"*/\" '>'n"
    "\"public void firePropertyChange(String propertyName, boolean oldValue, boolean newValue) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"pcs.firePropertyChange(propertyName, oldValue, newValue);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Fires an existing PropertyChangeEvent to any registered listeners.\" '>'n"
    "\"* No event is fired if the given event's old and new values are equal and non-null. \" '>'n"
    "\"*\" '>'n \"* @param evt The PropertyChangeEvent object.\" '>'n\"*/\" '>'n"
    "\"public void firePropertyChange(PropertyChangeEvent evt) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"pcs.firePropertyChange(evt);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n\"* Checks if there are any listeners for a specific property.\" '>'n"
    "\"*\" '>'n \"* @param evt The PropertyChangeEvent object.\" '>'n"
    "\"* @return <code>true</code>if there are one or more listeners for the given property\" '>'n"
    "\"*/\" '>'n"
    "\"public boolean hasListeners(String propertyName) \""

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "'> \"return pcs.hasListeners(propertyName);\" '>'n \"}\" '>'n '>'n'>"
   )
  "*Template for adding property change support to a class."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-property-change-support
	    (tempo-define-template
	     "property change support template"
	     (jde-gen-read-template val)
	     nil
	     "Insert property change support template."))
	  (set-default sym val)))

(defcustom jde-gen-entity-bean-template
  '(
    "(jde-wiz-insert-imports-into-buffer (list \"javax.ejb.*\"
\"java.rmi.RemoteException\"))"
;;   "(jde-wiz-update-implements-clause \"EntityBean\")"
    "'> \"public void ejbActivate() throws RemoteException \""
    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n"  
    "'> \"public void ejbPassivate() throws RemoteException \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n"  

    "'> \"public void ejbLoad() throws RemoteException \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n"  

    "'> \"public void ejbStore() throws RemoteException \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n"  

    "'> \"public void ejbRemove() throws RemoteException \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n"  

    "'> \"public void setEntityContext(EntityContext ctx) throws RemoteException \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n"  

    "'> \"public void unsetEntityContext() throws RemoteException \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n"
    "\"}\"'>'n '>'n'>"  

    )
  "*Template that creates an empty implementation of an EJB Entity Bean."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-entity-bean
	    (tempo-define-template
	     "ejb-entity-bean"
	     (jde-gen-read-template val)
	     nil
	     "Insert EJB Entity Bean."))
	  (set-default sym val)))

(defcustom jde-gen-session-bean-template
  '(
    "(jde-wiz-insert-imports-into-buffer (list \"javax.ejb.*\"
\"java.rmi.RemoteException\"))"
    "(jde-wiz-update-implements-clause \"SessionBean\")"
    "'> \"public void ejbActivate() throws RemoteException {\"'>'n \"}\"'>'n
'>'n"
    "'> \"public void ejbPassivate() throws RemoteException {\"'>'n \"}\"'>'n
'>'n"
    "'> \"public void ejbRemove() throws RemoteException {\"'>'n \"}\"'>'n '>'n"
    "'> \"public void setSessionContext(SessionContext ctx) throws
RemoteException {\""
    "'>'n \"}\"'>'n '>'n"
    "'> \"public void unsetSessionContext() throws RemoteException {\"'>'n
\"}\"'>'n '>'n'>"
    )
  "*Template that creates an empty implementation of an EJB Session Bean."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-session-bean
	    (tempo-define-template
	     "ejb-session-bean"
	     (jde-gen-read-template val)
	     nil
	     "Insert EJB Session Bean."))
	  (set-default sym val)))


(defcustom jde-gen-code-templates
  (list (cons "Get Set Pair" 'jde-gen-get-set)
	(cons "toString method" 'jde-gen-to-string-method)
	(cons "Action Listener" 'jde-gen-action-listener)
	(cons "Window Listener" 'jde-gen-window-listener)
	(cons "Mouse Listener" 'jde-gen-mouse-listener)
	(cons "Mouse Motion Listener" 'jde-gen-mouse-motion-listener)
	(cons "Inner Class" 'jde-gen-inner-class)
	(cons "println" 'jde-gen-println)
        (cons "beep" 'jde-gen-beep)
	(cons "property change support" 'jde-gen-property-change-support)
	(cons "EJB Entity Bean" 'jde-gen-entity-bean)
	(cons "EJB Session Bean" 'jde-gen-session-bean)
	)
  "*Specifies available autocode templates.
The value of this variable is an association list. The car of
each element specifies a template name. The cdr specifies
a command that inserts the template into a buffer. See the function
`tempo-define-template' for any easy way to create a template
insertion command."
  :group 'jde-gen
  :type '(repeat
	  (cons :tag "Template"
	   (string :tag "Name")
	   (function :tag "Command")))
  :set '(lambda (sym val)
	  (let ((n (length val))
		(i 0))
	    (setq jde-gen-template-names (list))
	    (while (< i n)
	      (setq jde-gen-template-names
		    (append
		     jde-gen-template-names
		     (list (cons (car (nth i val)) (1+ i)))))
	      (setq i (1+ i))))
	  (set-default sym val)))

(defun jde-gen-code (name)
  "Insert the code template specified by NAME at point.
The template must be one of those specified by the
variable `jde-gen-code-templates'."
  (interactive
   (list
    (completing-read "Template name: " jde-gen-template-names)))
  (funcall (cdr (assoc name jde-gen-code-templates))))



;;; Control Flow Templates
;;; Contributed by Eric D. Friedman <friedman@lmi.net>

(defvar jde-tempo-tags nil
  "Tempo tags for JDE mode")

(defcustom jde-gen-cflow-enable t
  "Enables abbreviations for Java control flow constructs."
  :group 'de-gen
  :type 'boolean)

(defcustom jde-gen-cflow-if
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"if\")"
    "'(l '> \"if (\" (p \"if-clause: \" clause) \") \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of if (\" (s clause) \")\"'>'n'>)"
    ")"
   )
  "Skeleton if statement. To insert the if statement at point, type if 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode'"
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-if"
	     (jde-gen-read-template val)
	     "if"
	     "Insert a Java if statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-else
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"else\")"
    "'(l '> \"else \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of else\"'>'n'>)"
    ")"
   )
  "Skeleton else statement. To insert the statement at point, type else 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-else"
	     (jde-gen-read-template val)
	     "else"
	     "Insert a Java else statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-if-else
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"ife\")"
    "'(l '> \"if (\" (p \"if-clause: \" clause) \") \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of if (\" (s clause) \")\"'> n"
    "'> \"else \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of if (\" (s clause) \")else\"'>'n'>)"
    ")"
   )
  "Skeleton if-else statement. To insert the statement at point, type ife 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-if-else"
	     (jde-gen-read-template val)
	     "ifelse"
	     "Insert a Java if else statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-while
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"while\")"
    "'(l '> \"while (\" (p \"while-clause: \" clause) \") \""
    "(if jde-gen-k&r "
     "()"
     "'>'n)"
     "\"{\"'>'n'>'r'n"
    "\"} // end of while (\" (s clause) \")\"'>'n'>)"
    ")"
   )
  "Skeleton while statement. To insert the statement at point, type while 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-while"
	     (jde-gen-read-template val)
	     "while"
	     "Insert a Java while statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-for
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"for\")"
    "'(l '> \"for (\" (p \"for-clause: \" clause) \") \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of for (\" (s clause) \")\"'>'n'>)"
    ")"
   )
  "Skeleton for statement. To insert the statement at point, type for 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-for"
	     (jde-gen-read-template val)
	     "for"
	     "Insert a Java for statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-for-i
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"fori\")"
    "'(l '> \"for (int \" (p \"variable: \" var) \" = 0; \""
    "(s var)"
    "\" < \"(p \"upper bound: \" ub)\"; \" (s var) \"++) \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of for (int \" (s var) \" = 0; \""
    "(s var) \" < \" (s ub) \"; \" (s var) \"++)\"'>'n'>)"
    ")"
  )
  "Skeleton for i statement. To insert the statement at point, type fori 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-for-i"
	     (jde-gen-read-template val)
	     "fori"
	     "Insert a Java for i statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-main
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"main\")"
    "'(l '> \"public static void main (String[] args) \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'>'r'n"
    "\"} // end of main ()\"'>'n'>)"
    ")"       
  )
  "Skeleton main method. To insert the method at point, type main 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-main"
	     (jde-gen-read-template val)
	     "main"
	     "Insert a Java main statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-switch
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"switch\")"
    "'(l '> \"switch (\" (p \"switch-condition: \" clause) \") \""
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n'"
    "\"case \" (p \"first value: \") \":\"'>'n'>'p'n"
    "\"break;\"'>'n'>'p'n"
    "\"default:\"'>'n'>'p'n"
    "\"break;\"'>'n"
    "\"} // end of switch (\" (s clause) \")\"'>'n'>)"
    ")"
    )
  "Skeleton switch statement. To insert the statement at point, type switch 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-switch"
	     (jde-gen-read-template val)
	     "switch"
	     "Insert a Java switch statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defcustom jde-gen-cflow-case
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    "'(l \"case\")"
    "'(l 'n \"case \" (p \"value: \") \":\"'>'n'>'p'n"
    "\"break;\"'>'n'>'p)"
    ")"
    )
  "Skeleton case statement. To insert the statement at point, type case 
and then space. Note that abbrev mode must be enabled. See 
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template
	     "jde-case"
	     (jde-gen-read-template val)
	     "case"
	     "Insert a Java case statement"
	     'jde-tempo-tags)
	  (set-default sym val)))

(defun jde-gen-load-cflow-abbrevs ()
  "Defines jde-mode abbrevs for the control flow templates."
  (define-abbrev local-abbrev-table "if" "" 'tempo-template-jde-if 0)
  (define-abbrev local-abbrev-table "else" "" 'tempo-template-jde-else 0)
  (define-abbrev local-abbrev-table "ife" "" 'tempo-template-jde-if-else 0)
  (define-abbrev local-abbrev-table "while" "" 'tempo-template-jde-while 0)
  (define-abbrev local-abbrev-table "for" "" 'tempo-template-jde-for 0)
  (define-abbrev local-abbrev-table "fori" "" 'tempo-template-jde-for-i 0)
  (define-abbrev local-abbrev-table "switch" "" 'tempo-template-jde-switch 0)
  (define-abbrev local-abbrev-table "case" "" 'tempo-template-jde-case 0)
  (define-abbrev local-abbrev-table "main" "" 'tempo-template-jde-main 0))


(provide 'jde-gen)

;; $Log: jde-gen.el,v $
;; Revision 1.26  2000/08/19 05:10:05  paulk
;; Adds jde-gen-cflow-enable variable.
;;
;; Revision 1.25  2000/07/23 02:44:44  paulk
;; Templates now indent correctly when inserted in a buffer. Thanks to el mono <mono@utp.edu.co> for this enhancement.
;;
;; Revision 1.24  2000/07/20 06:08:59  paulk
;; Extended K&R coding style to all templates. Thanks to Stephane Nicolas <s.nicolas@videotron.ca> for doing this.
;;
;; Revision 1.23  2000/06/28 02:46:48  paulk
;; Get/set pair template now generates correct method name for getting the value of boolean variables. Thanks to Stephane <s.nicolas@videotron.ca> for contributing this fix.
;;
;; Revision 1.22  2000/06/01 06:43:01  paulk
;; Added control flow templates contributed by Eric D. Friedman <friedman@lmi.net>.
;;
;; Revision 1.21  2000/02/01 05:22:51  paulk
;; Provide choice of coding styles for code generated by templates. Thanks to Jari Aalto for this enhancement.
;;
;; Revision 1.20  1999/09/23 03:23:41  paulk
;; Added code templates implementing EJB EntityBean and SessionBean
;; interfaces. Thanks to Brendan.Burns@tfsma-ims.tfn.com for contributing
;; the templates.
;;
;; Revision 1.19  1999/08/29 04:29:18  paulk
;; Patches provided by Michael Ernst <mernst@alum.mit.edu>
;;
;; Revision 1.18  1999/02/11 17:03:00  paulk
;; Updated the Swing application template to the JDK 1.2 Swing package
;; scheme and expanded the template to provide a menu and scrollable
;; canvas.
;;
;; Revision 1.17  1998/09/16 22:55:51  paulk
;; Added template for Java bean property change support.
;;
;; Revision 1.16  1998/09/13 00:34:28  paulk
;; Added a template for generating a System.out.println statement.
;;
;; Revision 1.15  1998/07/22 00:28:04  paulk
;; Modified class buffer creation templates to use tempo-marks
;; to mark initial position for user to insert code. Thanks
;; to David Hull <david.hull@trw.com> for suggesting this.
;;
;; Revision 1.14  1998/07/06 06:39:42  paulk
;; class buffer template now prompts for super class and
;; interface
;;
;; Revision 1.13  1998/07/06 05:06:13  paulk
;; Added boilerlate to other buffer generation templates.
;;
;; Revision 1.12  1998/07/01 03:54:40  paulk
;; Added source file boilerplate support.
;;
;; Revision 1.11  1998/06/27 03:04:46  paulk
;; Fixed capitalization on get-set method pair. Thanks to Jere_McDevitt@HomeDepot.COM
;;
;; Revision 1.10  1998/06/17 03:49:21  paulk
;; Fixed bug that caused jfc-app to be generated instead of console app.
;; Added a mouse motion listener template.
;; Added a toString method template.
;;
;; Revision 1.9  1998/05/27 05:55:20  paulk
;; Added autoload comments.
;;
;; Revision 1.8  1998/05/27 05:51:20  paulk
;; *** empty log message ***
;;
;; Revision 1.7  1998/05/17 06:20:37  paulk
;; Added templates for a Swing application and an inner class.
;;
;; Fixed a bug in jde-gen-buffer
;;
;; Revision 1.6  1998/04/18 14:08:55  kinnucan
;; Fixes some bugs in the generated code.
;;
;; Revision 1.5  1998/04/10 02:55:00  kinnucan
;; * Updated some of the doc strings.
;;
;; Revision 1.4  1998/04/09 04:51:09  kinnucan
;; * Added the capability to define your own custom autocode templates.
;;   The facility consists of the following items:
;;
;;   - jde-gen-code-templates
;;
;;     Defines a list of templates for code inserted at point. The
;;     list by default contains the templates defined by the JDE.
;;     You can define your own templates and add them to the list,
;;     using the Emacs customization feature. See tempo.el for
;;     information on creating templates.
;;
;;   - jde-gen-buffer-templates
;;
;;     Defines a list of templates for code to be inserted in a
;;     newly created Java buffer.
;;
;;   - jde-gen-code (JDE->Generate->Custom)
;;
;;     This command inserts a specified code template at point.
;;
;;   - jde-gen-buffer (Files->JDE New->Custom)
;;
;;     This command creates the specified buffer and inserts
;;     a specified template at the beginning of the buffer.
;;
;; Revision 1.3  1998/04/08 04:38:16  kinnucan
;; * Provided each template variable with a set function that regenerates
;;   the corresponding template command whenever the template is changed.
;;
;; Revision 1.2  1998/04/06 03:47:20  kinnucan
;; * Added jde-gen-class-buffer and jde-gen-console-buffer functions.
;;
;; Revision 1.1  1998/04/01 05:33:43  kinnucan
;; Initial revision
;;

;; End of jde-gen.el
