%%% File    : wae_man.erl
%%% Author  : Johan Blom <johblo@dragon.cellpt.se>
%%% Purpose : Handling of WAP Content Types (binary encoding etc)
%%% Created : 31 Aug 2000 by Johan Blom <johblo@dragon.cellpt.se>

-module(wae_man).
-author('johblo@dragon.cellpt.se').
-revision('$Revision: 1.3 $ ').
-rcsid('@(#) $Id: wae_man.erl,v 1.3 2001/07/17 12:53:34 johblo Exp $ ').
-modified('$Date: 2001/07/17 12:53:34 $ ').
-modified_by('$Author: johblo $ ').
-vsn("1").

-export([encode_body/3]).
-export([dtd2tab/1,detect_charset/2]).
-export([get_accept_vals/1]).

-include("utilslog.hrl").
-include("wae.hrl").

%%% ===========================================================================
%%% Used to encode body to be sent with method_result_reg or
%%% unit_method_result_req
encode_body({ContentType,Par},Body,{CliHdlist,DTDrules}) ->
    Accept=get_accept_vals(CliHdlist),
    case acceptable_media(Accept#accept.accept,ContentType) of
	ok ->
	    {ContentType,Body};
	_ ->
	    convert_content({ContentType,Par},Body,Accept,DTDrules)
    end.


%%% Check for acceptable content by examining the 'accept' and 'accept-charset'
%%% headers for acceptable content.
%%% FIXME! The 'accept-language' and 'accept-encoding' is tested elsewhere (?)
%%%        Also 
%%% Note:
%%% - Missing 'accept' header or with value '*/*' means all content is allowed.
%%% - Missing 'accept-chraset' header or with value '*' means all charsets are
%%%   allowed.
get_accept_vals(CliHd) ->
    Acc=case string:tokens(http:getHeaderValue(accept,CliHd),",") of
	    [] ->
		[];
	    [["*/*"]] ->
		[];
	    V0 ->
		lists:map(fun(X) -> list_to_atom(X) end,V0)
	end,
    ?debug("Accept media:~p",[Acc],get_accept_vals),
    ACh=case string:tokens(http:getHeaderValue('accept-charset',CliHd),
		       ",") of
	    [] ->
		[];
	    [["*"]] ->
		[];
	    V1 ->
		lists:map(fun(X) -> list_to_atom(X) end,V1)
	end,
    ?debug("Accept charset:~p",[ACh],get_accept_vals),
    #accept{accept=Acc,accept_charset=ACh}.



%% Convert Body dependent on acceptable ContentType.
convert_content(CT,Body,Accept,DTDrules) ->
    case CT of
	{'text/vnd.wap.wml',Par} -> wml_encode(Body,Par,Accept,DTDrules);
%	{'text/vnd.wap.wmlscript',_} ->
%	    wmlscript_encode(Body,Par,Accept,DTDrules);
%	{'text/vnd.wap.channel',_} -> {status,{501,"WTA Channel"}};
%	{'text/vnd.wap.si',_} -> {status,{501,"Push SI"}};
%	{'text/vnd.wap.sl',_} -> {status,{501,"Push SL"}};
%	{'text/vnd.wap.connectivity-xml',_} -> {status,{501,"E2E Navigation"}};
	{'application/vnd.wap.connectivity-wbxml',_} ->
	    {status,{501,"E2E Navigation"}};
	{'application/vnd.wap.uaprof',_} -> {status,{501,"UA Prof"}};
	{'application/vnd.wap.wtls-ca-certificate',_} -> {status,{501,"WTLS"}};
	{'application/vnd.wap.wtls-user-certificate',_}->{status,{501,"WTLS"}};
	{'application/vnd.wap.co',_}->{status,{501,"Cache"}};
	{'application/vnd.wap.wmlsc',_}->{status,{501,"WML Script"}};

%%% If image conversion is on, convert to WBMP
%	{'image/gif',_} -> {status,{501,"Conversion GIF->WBMP"}};
%	{'image/jpeg',_} -> {status,{501,"Conversion JPEG->WBMP"}};
%	{'image/tiff',_} -> {status,{501,"Conversion TIFF->WBMP"}};
%	{'image/png',_} -> {status,{501,"Conversion PNG->WBMP"}};

%%% If format conversion is on, convert to WML
%	{'text/plain',_} -> {status,{501,"Conversion plain text->WML"}};
%	{'text/html',_} -> {status,{501,"Conversion HTML->WML"}};
%	{'text/x-hdml',_} -> {status,{501,"Conversion HDML->WML"}};
%	{'text/x-ttml',_} -> {status,{501,"Conversion TTML->WML"}};

	{OtherType,_} when Accept#accept.accept==[] ->
	    {OtherType,Body};
	{OtherType,_} ->
	    {status,{501,"Conversion of "++atom_to_list(OtherType)}}
    end.

acceptable_media([],_) ->
    ok;
acceptable_media(MediaList,Media) ->
    case lists:member(Media,MediaList) of
	true -> 
	    ok;
	false ->
	    {error,non_acceptable_media}
    end.

%%% Don't bother checking if 'text/vnd.wap.wml' is an acceptable media for a
%%% WAP Client.
wml_encode(Body,Parlist,Accept,DTDRules) ->
    case acceptable_media(Accept#accept.accept,'application/vnd.wap.wmlc') of
	ok ->
	    ExtCharset=http:getParameterValue(charset,Parlist),
	    case catch wbxml:encode(Body,Accept,ExtCharset,DTDRules) of
		{ok,Bytes} ->
		    {'application/vnd.wap.wmlc', Bytes};
		Error2 -> 
		    ?error("Couldn't encode WML ~p got ~p",
			[Body,Error2],wml_encode),
		    {status,500}
	    end;
	Error ->
	    ?error("Couldn't encode WML ~p got ~p",[Body,Error],wml_encode),
	    {status,500}
    end.

wmlscript_encode(Body,Parlist,Accept,DTDRules) ->
    Charset=http:getParameterValue(charset,Parlist),
    ?debug("Character set provided ~p",[{Parlist,Charset}],wml_encode),
    ?error("Coudn't encode WML Script",[],wmlscript_encode),
    {status,501}.


%%% ============================================================================
%%% This should probably not be here...
%%% - detect_charset/2 for detecting Charsets in XML documents
%%% - dtd2tab/2 for transforming DTD to a tab file, using xmerl
%% -----------------------------------------------------------------------------

% FIXME! Whatabout aliases etc?
detect_charset(ExtCharset,Content) ->
    case autodetect(ExtCharset,Content) of	    
	{auto,Content1} ->
	    {auto,'iso-10646-utf-1',Content};
	{external,Content1} ->
	    {external,'iso-10646-utf-1',Content};
	{undefined,_} ->
	    {undefined,undefined,Content};
	{ExtCharset, Content} ->
	    {external,ExtCharset,Content}
    end.

%%------------------------------------------------------------------------------
%% Auto detect what kind of character set we are dealing with and transform
%% to Erlang integer Unicode format if found.
%% Appendix F, Page 56-57, XML 1.0 W3C Recommendation 6 October 2000
%% (http://www.w3.org/TR/REC-xml)
%% 00 00 00 3C ( "<" in UCS-4 big-endian)
%% 3C 00 00 00 ( "<" in UCS-4 little-endian)
%% FE FF (UTF-16 - big-endian Mark)
%% FF FE (UTF-16 - little-endian Mark)
%% 00 3C 00 3F ( "<?" in UTF-16 big-endian)
%% 3C 00 3F 00 ( "<?" in UTF-16 big-endian)
%% 3C 3F (7-bit,8-bit or mixed width encoding)
%% 4C 6F A7 94 (EBCDIC) - Not Implemented!!!!

%% Check byte-order mark and transform to Unicode, Erlang integer
%%% --- With byte-order mark
autodetect(undefined,[0,0,16#fe,16#ff | Input]) ->
    {auto, ucs:from_ucs4be(Input)};
autodetect('iso-10646-utf-1',[0,0,16#fe,16#ff | Input]) ->
    {external, ucs:from_ucs4be(Input)};
autodetect(undefined,[16#ff,16#fe,0,0 | Input]) ->
    {auto, ucs:from_ucs4le(Input)};
autodetect('iso-10646-utf-1',[16#ff,16#fe,0,0 | Input]) ->
    {external, ucs:from_ucs4le(Input)};

autodetect(undefined,[16#fe,16#ff | Input]) ->
    {auto, ucs:from_utf16be(Input)};
autodetect('utf-16be',[16#fe,16#ff | Input]) ->
    {external, ucs:from_utf16be(Input)};
autodetect(undefined,[16#ff,16#fe | Input]) ->
    {auto, ucs:from_utf16le(Input)};
autodetect('utf-16le',[16#ff,16#fe | Input]) ->
    {external, ucs:from_utf16le(Input)};

autodetect(undefined,[16#ef,16#bb,16#bf | Input]) ->
    {auto, ucs:from_utf8(Input)};
autodetect('utf-8',[16#ef,16#bb,16#bf | Input]) ->
    {external, ucs:from_utf8(Input)};

%%% --- Without byte-order mark
autodetect(undefined,[0,0,0,16#3c|Input]) ->
    {auto, ucs:from_ucs4be([0,0,0,16#3c|Input])};
autodetect('iso-10646-utf-1',[0,0,0,16#3c|Input]) ->
    {external, ucs:from_ucs4be([0,0,0,16#3c|Input])};
autodetect(undefined,[16#3c,0,0,0|Input]) ->
    {auto, ucs:from_ucs4le([16#3c,0,0,0|Input])};
autodetect('iso-10646-utf-1',[16#3c,0,0,0|Input]) ->
    {external, ucs:from_ucs4le([16#3c,0,0,0|Input])};

autodetect(undefined,[0,16#3c,0,16#3f | Input]) ->
    {auto, ucs:from_utf16be([0,16#3c,0,16#3f|Input])};
autodetect('utf-16be',[0,16#3c,0,16#3f | Input]) ->
    {external, ucs:from_utf16be([0,16#3c,0,16#3f|Input])};
autodetect(undefined,[16#3c,0,16#3f,0 | Input]) ->
    {auto, ucs:from_utf16le([16#3c,0,16#3f,0|Input])};
autodetect('utf-16le',[16#3c,0,16#3f,0 | Input]) ->
    {external, ucs:from_utf16le([16#3c,0,16#3f,0|Input])};

autodetect(ExtCharset,Content) ->
    {ExtCharset, Content}.


%% Change the byte order of the input stream from little to big endian,
%% or vice versa.
%% Modes:
%% 2 - 21212121... => 12121212...
%% 4 - 43214321... => 12341234...
change_byte_order([N4,N3,N2,N1 | Input], 4) ->
    [N1,N2,N3,N4 | change_byte_order(Input, 4)];
change_byte_order([N2,N1,N4,N3 | Input], 5) ->
    [N1,N2,N3,N4 | change_byte_order(Input, 5)];
change_byte_order([N3,N4,N1,N2 | Input], 6) ->
    [N1,N2,N3,N4 | change_byte_order(Input, 6)];
change_byte_order([N2,N1 | Input], 2) ->
    [N1,N2 | change_byte_order(Input, 2)];
change_byte_order([], _) ->
    [].

%%------------------------------------------------------------------------------
%% Convert files containing DTD to *.tab files suitable for xmerl
dtd2tab(DTD) when atom(DTD) ->
    Dir=filename:dirname(code:which(?MODULE))++"../priv/",
    Rules=ets:new(DTD,[set, public]),
    case xmerl_scan:file(Dir++atom_to_list(DTD),[{rules,Rules}]) of
	{ok,[]} ->
	    ets:tab2file(Rules,Dir++atom_to_list(DTD)++".tab");
	A ->
	    io:format("ERROR: in parsing DTD:~p~n",
		      [[Dir++atom_to_list(DTD),A]])
    end,
    halt();
dtd2tab([DTD])  when atom(DTD) ->
    dtd2tab(DTD);
dtd2tab(DTD) ->
    io:format("ERROR: DTD must be an atom:~p~n",[DTD]),
    halt().
