%%% File    : wap_gateway_man.erl
%%% Author  : Johan Blom <johblo@dragon.cellpt.se>
%%% Purpose : WAP Gateway manager
%%% Created : 29 Aug 2000 by Johan Blom <johblo@dragon.cellpt.se>

-module(wap_gateway_man).
-author('johblo@dragon.cellpt.se').
-behaviour(gen_server).
-revision('$Revision: 1.3 $ ').
-rcsid('@(#) $Id: wap_gateway_man.erl,v 1.3 2001/07/17 12:53:34 johblo Exp $ ').
-modified('$Date: 2001/07/17 12:53:34 $ ').
-modified_by('$Author: johblo $ ').
-vsn("1").

-export([gen_status_error/3]).

%% External exports
-export([start_link/0,stop/0]).
-export([load_dtd_tabs/0]).

%% Internal gen_server callbacks
-export([init/1, handle_call/3, handle_cast/2, handle_info/2, terminate/2,
	 code_change/3]).

-define(START_OPTIONS,[{timeout,100000}]).

-define(GMACALL,             {local,?MODULE}).
-define(HTTP_REQUEST_TIMEOUT,   5000).
-define(WAE_MAN, wae_man).

%% Names derived from the WBXML spec (Public Identifiers, Section 7.2)
% -define(SUPPORTED_XML_DOCS,[wml_10,wml_11,wml_12,wml_13,si_10,sl_10,pap_10]).
-define(SUPPORTED_XML_DOCS,[wml_11]).

-record(state,{
	  methods,  % (ets) Pending HTTP requests
	  dtd_rules % (ets) XML DTD parsing table
	 }).

-include("utilslog.hrl").


%%% ----------------------------------------------------------------------------
start_link() ->
    gen_server:start_link(?GMACALL, ?MODULE, [], ?START_OPTIONS).

stop() ->
    gen_server:call(?GMACALL,stop).

init(Args) ->
    ucs:start_server(),
    register_in_stack(self()),
    DTDdb=load_dtd_tabs(),
    ?trace("Started gateway manager ok",[],init),
    {ok, #state{methods=ets:new(wapgw_sessions,[set, private]),
		dtd_rules=DTDdb
	       }}.

terminate(Reason, State) ->
    ?trace("Stopped gateway manager ~p",[Reason],init).


%%----------------------------------------------------------------------

%% Register all known applications in the WAP stack application
register_in_stack(This) ->
    register_in_stack(wap_gateway_db:find_appref(),This).
				   
register_in_stack([],This) ->
    [];
register_in_stack([AppRef|AppRefList],This) ->
    case catch wap_stack_man:reapp_reg(AppRef,This) of
	ok ->
	    register_in_stack(AppRefList,This);
	Error ->	    
	    ?debug("Couldn't register ~p got ~p",[AppRef,Error],
		   register_in_stack),
	    throw({error,cannot_register_app})
    end.

%% Load predefined DTD tables with data suitable for xmerl, to be used when
%% parsing received XML data.
load_dtd_tabs() ->
    Tab=ets:new(dtd_tabs,[set, private]),
    load_dtd_tabs(?SUPPORTED_XML_DOCS,Tab).

load_dtd_tabs([],Tab) ->
    Tab;
load_dtd_tabs([DTD|Rest],Tab) ->
    Filename=filename:join([filename:dirname(code:which(?WAE_MAN)),
			    "../priv/",atom_to_list(DTD)++".tab"]),
    ?debug("Filename ~p",[Filename],load_dtd_tabs),
    case ets:file2tab(Filename) of
	{ok,DTDtab} ->
	    ets:insert(Tab,{DTD,DTDtab}),
	    load_dtd_tabs(Rest,Tab);
	{error,Reason} ->
	    ?error("Couldn't load ets tab file ~p",[Reason],load_dtd_tabs),
	    throw({error,Reason})
	end.


%%----------------------------------------------------------------------
%% Func: handle_call/3
%% Returns: {reply, Reply, State}          |
%%          {reply, Reply, State, Timeout} |
%%          {noreply, State}               |
%%          {noreply, State, Timeout}      |
%%          {stop, Reason, Reply, State}   | (terminate/2 is called)
%%          {stop, Reason, State}            (terminate/2 is called)
%%----------------------------------------------------------------------
%% --------- Recieved Push Proxy Gateway requests --------- 
handle_call({push_req}, _, State) ->
%%    wap_stack_man:unit_push_req(StRef,URef)
    Reply = ok,
    {reply, Reply, State};
handle_call({confirmed_push_req}, _, State) ->
%%    wap_stack_man:confirmed_push_req(StRef,URef)
    Reply = ok,
    {reply, Reply, State}.

%%----------------------------------------------------------------------
%% Func: handle_cast/2
%% Returns: {noreply, State}          |
%%          {noreply, State, Timeout} |
%%          {stop, Reason, State}            (terminate/2 is called)
%%----------------------------------------------------------------------
%% --------- WSP Session handling --------- 
handle_cast({connect_ind,WSPses,Vers,Headlist,Cap}, State) ->
    case Vers of
	{Major,Minor} when Major=<1 ->
	    wsp_session_s:connect_res(WSPses,[],Cap); % Accept whatever caps
	_ ->
	    wsp_session_s:disconnect_req(WSPses,{200,[],[],[]}) % OBS BUG
    end,
    {noreply, State};
handle_cast({disconnect_ind,WSPses,FailReason}, State) ->
    ?trace("Gateway received disconnect_ind ~p",[FailReason],handle_cast),
    {noreply, State};
handle_cast({suspend_ind,WSPses,Reason}, State) ->
    ?trace("Gateway received suspend_ind ~p",[Reason],handle_cast),
    {noreply, State};
handle_cast({resume_ind,WSPses}, State) ->
    ?trace("Gateway received resume_ind",[],handle_cast),
    wsp_session_s:resume_res(WSPses),
    {noreply, State};

%% --------- Recieved HTTP request --------- 
handle_cast({unit_method_invoke_ind,StRef,URef,HTTPMethod,HTTPCont}, State) ->
    ?debug("unit_method_invoke_ind ~p ~p",[HTTPMethod,HTTPCont],handle_cast),
    Pids=handle_method_invoke({unit_rcv_result,StRef,URef},HTTPMethod,HTTPCont),
    add_method(State#state.methods,URef,Pids),
    {noreply, State};
handle_cast({method_invoke_ind,WSPmet,HTTPMethod,HTTPCont}, State) ->
    wsp_method_s:method_invoke_res(WSPmet),
    Pids=handle_method_invoke({rcv_result,WSPmet},HTTPMethod,HTTPCont),
    add_method(State#state.methods,WSPmet,Pids),
    {noreply, State};
handle_cast({method_abort_ind,WSPmet,FailReason}, State) ->
    remove_method_abort(State#state.methods,WSPmet),
    {noreply, State};
handle_cast({method_result_cnf,WSPmet,AckHeaders}, State) ->
    {noreply, State};

%% --------- Returned HTTP response --------- 
handle_cast({{{unit_rcv_result,StRef,URef},CliHd},Reply},State) ->
    ContInfo={CliHd,State#state.dtd_rules},
    case Reply of
	{status,Status} ->
	    HTTPCont=gen_status_error(Status,[],ContInfo),
	    wap_stack_man:unit_method_result_req(StRef,URef,HTTPCont);
	HTTPCont ->
	    wap_stack_man:unit_method_result_req(StRef,URef,
						   handle_rcv_result(HTTPCont,
								     ContInfo))
    end,
    remove_method(State#state.methods,URef),
    {noreply, State};
handle_cast({{{rcv_result,WSPmet},CliHd},Reply},State) ->
    ContInfo={CliHd,State#state.dtd_rules},
    case Reply of
	{status,Status} ->
	    HTTPCont=gen_status_error(Status,[],ContInfo),
	    wsp_method_s:method_result_req(WSPmet,HTTPCont);
	HTTPCont ->
	    wsp_method_s:method_result_req(WSPmet,
					   handle_rcv_result(HTTPCont,ContInfo))
    end,
    remove_method(State#state.methods,WSPmet),
    {noreply, State};


%% --------- Returned Push Proxy Gateway requests --------- 
handle_cast({confirmed_push_cnf,WSPpus,AckHeaders}, State) ->
    {noreply, State};

handle_cast({exception_ind, Sref, Reason}, State) ->
    {noreply, State}.


%% Add additional headers for features in the WAP Gateway and
%% make the HTTP request (this will spawn another process)
handle_method_invoke(Ref,HTTPMethod,{Url,Headers}) ->
    ReqHeaders=add_headers(Headers),
    ?debug("~p(~p,~p)",[HTTPMethod,Url,ReqHeaders],handle_method_invoke),
    http:request({Ref,Headers},
		 HTTPMethod,{Url,ReqHeaders,'',[]},
		 ?HTTP_REQUEST_TIMEOUT);
handle_method_invoke(Ref,HTTPMethod,{Url,Headers,CT,Body}) ->
    ReqHeaders=add_headers(Headers),
    ?debug("~p(~p,~p,~p,~p)",
	   [HTTPMethod,Url,ReqHeaders,CT,Body],handle_method_invoke),
    http:request({Ref,Headers},
		 HTTPMethod,{Url,ReqHeaders,CT,binary_to_list(Body)},
		 ?HTTP_REQUEST_TIMEOUT).

%%% Processes the received package in the requested way (e.g. WBXML encoding) 
%%% If Status received is not ok, check if the received body is WML
handle_rcv_result({Status,Headers,CT,Body},ContInfo) when 200=<Status,
							  Status=<206->
    ContentType=wsp_headers_page1:single_token_field(CT),
    case wae_man:encode_body(ContentType,Body,ContInfo) of
	{status,NewStatus} ->
	    gen_status_error(NewStatus,Headers,ContInfo);
	{ResH,ResCT,ResBo} ->
	    ResHd = remove_headers(Headers++ResH),
	    ?debug("Status:~p Headers:~p CT:~p",
		   [Status,ResHd,ResCT],handle_rcv_result),
	    {Status,ResHd,ResCT,list_to_binary(ResBo)};
	{ResCT,ResBo} ->
	    ResHd = remove_headers(Headers),
	    ?debug("Status:~p Headers:~p CT:~p",
		   [Status,ResHd,ResCT],handle_rcv_result),
	    {Status,ResHd,ResCT,list_to_binary(ResBo)}
    end;
handle_rcv_result({Status,Headers,{'text/vnd.wap.wml',X},Body},ContInfo) ->
    case wae_man:encode_body({'text/vnd.wap.wml',X},Body,ContInfo) of
	{status,NewStatus} ->
	    gen_status_error(NewStatus,Headers,ContInfo);
	{ResH,ResCT,ResBo} ->
	    ResHd = remove_headers(Headers++ResH),
	    ?debug("Status:~p Headers:~p CT:~p",
		   [Status,ResHd,ResCT],handle_rcv_result),
	    {Status,ResHd,ResCT,list_to_binary(ResBo)};
	{ResCT,ResBo} ->
	    ResHd = remove_headers(Headers),
	    ?debug("Status:~p Headers:~p CT:~p",
		   [Status,ResHd,ResCT],handle_rcv_result),
	    {Status,ResHd,ResCT,list_to_binary(ResBo)}
    end;
handle_rcv_result({Status,Headers,{'application/vnd.wap.wmlc',X},Body},_) ->
    {Status,Headers,{'application/vnd.wap.wmlc',X},list_to_binary(Body)};
handle_rcv_result({Status,Headers,_,_},ContInfo) ->
    gen_status_error(Status,Headers,ContInfo).

%%----------------------------------------------------------------------
%% Func: handle_info/2
%% Returns: {noreply, State}          |
%%          {noreply, State, Timeout} |
%%          {stop, Reason, State}            (terminate/2 is called)
%%----------------------------------------------------------------------
handle_info(Info, State) ->
    {noreply, State}.

%%----------------------------------------------------------------------
%% Func: code_change/3
%% Purpose: Convert process state when code is changed
%% Returns: {ok, NewState}
%%----------------------------------------------------------------------
code_change(OldVsn, State, Extra) ->
    {ok, State}.

%%%----------------------------------------------------------------------
%%% Internal functions
%%%----------------------------------------------------------------------


%% The Gateway also currently support conversion of WML to binary
%% So add..
%% {accept, text/vnd.wap.wml}
add_headers(Headers) ->
    case lists:keysearch(accept,1,Headers) of
	{value,{accept,V}} ->
	    lists:keyreplace(accept,1,Headers,{accept,V++",text/vnd.wap.wml"});
	false ->
	    Headers ++ [{accept, "text/vnd.wap.wml"}]
    end.


%%% Filter away some headers
%%% That is headers
%% - that are *NOT* end-to-end headers
%% - 'set-cookie', for now as some terminals (Nokia 7110) cannot handle them
%% - 'transfer-encoding', does not apply (handled by the HTTP client)
remove_headers(Headers) ->
    H1=lists:keydelete('content-length', 1, Headers),
    H2=lists:keydelete('set-cookie', 1, H1),
    lists:keydelete('transfer-encoding', 1, H2).


%%% ----------------------------------------------------------------------------
%%% Use WML 1.1 (most widely supported) for the error response.
gen_status_error({StatusCode,StatusText},Headers,ContInfo) ->
    Body=xml_version_string()++doctype_str(wml_11)++
	"<wml><card title='Status"++ integer_to_list(StatusCode)++"'>"++
	"<do type=\"prev\" label=\"Back\"><prev/></do>"++
	"<p>"++get_status_string(StatusCode)++":"++StatusText++"</p>"++
	"</card></wml>",
    {ResCT,ResBo}=
	wae_man:encode_body({'text/vnd.wap.wml',undefined},Body,ContInfo),
    {200,Headers,ResCT,list_to_binary(ResBo)};
gen_status_error(Status,Headers,ContInfo) when list(Status) ->
    Body=xml_version_string()++doctype_str(wml_11)++
	"<wml><card title='Status'>"++
	"<do type=\"prev\" label=\"Back\"><prev/></do>"++
	"<p>"++get_status_string(Status)++"</p>"++
	"</card></wml>",
    {ResCT,ResBo}=
	wae_man:encode_body({'text/vnd.wap.wml',undefined},Body,ContInfo),
    {200,Headers,ResCT,list_to_binary(ResBo)};
gen_status_error(Status,Headers,ContInfo) ->
    Body=xml_version_string()++doctype_str(wml_11)++
	"<wml><card title='Status"++ integer_to_list(Status)++"'>"++
	"<do type=\"prev\" label=\"Back\"><prev/></do>"++
	"<p>"++get_status_string(Status)++"</p>"++
	"</card></wml>",
    {ResCT,ResBo}=
	wae_man:encode_body({'text/vnd.wap.wml',undefined},Body,ContInfo),
    {200,Headers,ResCT,list_to_binary(ResBo)}.

xml_version_string() ->
    "<?xml version='1.0'?>".
doctype_str(wml_11) ->
    "<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\">".


%% Additional definitions added when no status code is received
get_status_string(Status) when integer(Status) ->
    http:format_status(Status);
get_status_string(Reason) when list(Reason) -> Reason;
get_status_string(X) -> "Unknown".

%% -----------------------------------------------------------------------------
%% Simple database interface for pending HTTP requests
add_method(Tab,Key,HTTPpid) ->
    ets:insert(Tab,{Key,HTTPpid}).

remove_method(Tab,Key) ->
    ets:delete(Tab,Key).

remove_method_abort(Tab,Key) ->
    case ets:lookup(Tab,Key) of
	[{_,HTTPpid}] ->
	    http:cancel_request(HTTPpid),
	    ets:delete(Tab,Key);
	Other ->
	    Other
    end.
