%%% The contents of this file are subject to the Erlang Public License,
%%% Version 1.0, (the "License"); you may not use this file except in
%%% compliance with the License. You may obtain a copy of the License at
%%% http://www.erlang.org/license/EPL1_0.txt
%%%
%%% Software distributed under the License is distributed on an "AS IS"
%%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%%% the License for the specific language governing rights and limitations
%%% under the License.
%%%
%%% The Original Code is xmerl-0.6
%%%
%%% The Initial Developer of the Original Code is Ericsson Telecom
%%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%%% Telecom AB. All Rights Reserved.
%%%
%%% Contributor(s): ______________________________________.
%%%
%%%----------------------------------------------------------------------
%%% #0.    BASIC INFORMATION
%%%----------------------------------------------------------------------
%%% File:       xmerl_xml.erl
%%% Author       : Ulf Wiger <ulf.wiger@ericsson.com>
%%% Description  : Callback module for exporting 
%%%		   complete or simple forms to XML.
%%% 
%%% Modules used : lists, io_lib
%%% 
%%%----------------------------------------------------------------------
%%% 

-module(xmerl_xml).
-vsn('0.6').
-date('00-09-22').
-author('ulf.wiger@ericsson.com').


-export(['#xml-inheritance#'/0]).

-export(['#root#'/4,
	 '#element#'/5]).

-include("xmerl.hrl").


'#xml-inheritance#'() -> [].



%% The '#root#' tag is called when the entire structure has been exported.
%% It does not appear in the structure itself.
'#root#'(Data, Attrs, [], E) ->
    ["<?xml version=\"1.0\"?>\n", Data].

'#element#'(Tag, [], Attrs, Parents, E) ->
    TagStr = mk_string(Tag),
    ["<", tag_and_attrs(TagStr, Attrs), "/>\n"];
'#element#'(Tag, Data, Attrs, Parents, E) ->
    TagStr = mk_string(Tag),
    ["<", tag_and_attrs(TagStr, Attrs), ">\n", 
     Data, opt_newline(Data),
     "</", TagStr, ">\n"].




tag_and_attrs(TagStr, []) ->
    TagStr;
tag_and_attrs(TagStr, [Attr|Attrs]) ->
    Pad = lists:duplicate(length(TagStr) + 2, $\s),
    Tail = [["\n", Pad, attr_string(A)] ||
	       A <- Attrs],
    [TagStr, " ", attr_string(Attr) | Tail].

attr_string({K, V}) ->
    [mk_string(K), "=\"", mk_string(V), "\""];
attr_string(#xmlAttribute{name = K, value = V}) ->
    [mk_string(K), "=\"", mk_string(V), "\""].
    

mk_string(I) when integer(I) ->
    integer_to_list(I);
mk_string(A) when atom(A) ->
    atom_to_list(A);
mk_string(L) when list(L) ->
    %% again, we can't regognize a string without "parsing" it
    case catch list_to_binary(L) of
	{'EXIT',_} ->
	    io_lib:format("~p", [L]);
	_ ->
	    L
    end;
mk_string(Term) ->
    io_lib:format("~p", [Term]).



opt_newline(Data) ->
    B = list_to_binary(Data),
    Sz = size(B),
    case binary_to_list(B, Sz, Sz) of
	"\n" ->
	    [];
	Other ->
	    io:format("Data end = ~p~n", [Other]),
	    "\n"
    end.
