/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <qlayout.h>
#include <qlist.h>

#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "deventry.h"
#include "devicelist.h"
#include "devmantab.h"
#include "propdialog.h"


extern DeviceList  *devicelist;


DeviceManagerTab::DeviceManagerTab(QWidget *parent, const char *name) : QWidget(parent, name)
{
  _listview = new QListView(this, "listview");
  _listview->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  _listview->setLineWidth(2);
  _listview->setAllColumnsShowFocus(true);
  _listview->setRootIsDecorated(true);
  _listview->setItemMargin(2);
  _listview->addColumn(i18n("Device"));
  connect(_listview, SIGNAL(currentChanged(QListViewItem *)), SLOT(slotCurrentChanged(QListViewItem *)));
  
  QVBoxLayout  *toplayout = new QVBoxLayout(this, 16, 4, "toplayout");
  toplayout->addWidget(_listview);
  toplayout->addSpacing(4);
  
  _pushprop = new QPushButton(i18n("&Properties"), this, "pushprop");
  _pushprop->adjustSize();
  connect(_pushprop, SIGNAL(clicked()), SLOT(slotProperties()));
  
  _pushrefresh = new QPushButton(i18n("&Refresh"), this, "pushrefresh");
  _pushrefresh->adjustSize();
  connect(_pushrefresh, SIGNAL(clicked()), SLOT(slotRefresh()));  
  
  QHBoxLayout  *butlayout = new QHBoxLayout(toplayout, 4, "butlayout");
  butlayout->addWidget(_pushprop);
  butlayout->addWidget(_pushrefresh);
  butlayout->addStretch(1);
  
  fillIn();
  slotCurrentChanged(_listview->firstChild());
  
  toplayout->activate();
}


/*
 * Private methods
 ******************/
 
void  DeviceManagerTab::fillIn()
{
  DeviceEntry  *deventry, *parent, *newentry;
  QPixmap      parentxpm, childxpm;
  Device       *device;
  
  _listview->clear();
  
  for (device = devicelist->first(); device; device = devicelist->next())
  {
    for (deventry = (DeviceEntry *) _listview->firstChild(); deventry; deventry = (DeviceEntry *) deventry->nextSibling())
      if (!deventry->device() && (device->deviceBaseClass() == deventry->deviceBaseClass())) break;
 
    parent    = (deventry) ? deventry : new DeviceEntry(_listview, device->deviceBaseClass(),
  								   device->deviceBaseClassName());
    newentry  = new DeviceEntry(parent, device);
    parentxpm = childxpm = BarIcon(device->smallPixmapFile());
 
    if (device->conflict() || parent->conflict()) 
    {
      visualizeConflict(parentxpm);
      parent->setConflict(true);
      parent->setOpen(true);
      
      if (device->conflict()) visualizeConflict(childxpm);
    }
    
    parent->setPixmap(0,   parentxpm);
    newentry->setPixmap(0, childxpm);
  }
}


void  DeviceManagerTab::visualizeConflict(QPixmap& pixmap)
{
  QPixmap  confmark = BarIcon("confmark");
  
  bitBlt(&pixmap, 0, 0, &confmark, 0, 0, 10, 10, CopyROP);
}


/*
 * Private slots
 ****************/
 
void  DeviceManagerTab::slotCurrentChanged(QListViewItem *item)
{
  _pushprop->setEnabled(((DeviceEntry *) item)->device());
} 


void  DeviceManagerTab::slotProperties()
{
  PropertiesDialog  *propdialog;
  DeviceEntry       *deventry = (DeviceEntry *) _listview->currentItem();
  Device            *device;
  
  if (deventry)
  {
    propdialog = new PropertiesDialog(device = deventry->device(), this, "propdialog");  
    
    if (propdialog->exec() == QDialog::Accepted)
    {
      QString  tmp;
      
      tmp  = i18n("You are about to change device resource\n"
                  "settings which may affect stability of\n"
                  "your system.\n"
                  "Do you really want the changes to take\n"
                  "effect?");
      
      if (KMessageBox::warningYesNo(this, tmp, i18n("Change resource settings"))
		== KMessageBox::Yes)
      {
        device->changePhysicalResources(propdialog->resourceList());      
        devicelist->rescan(false);
        fillIn();
      }
    }
    
    delete propdialog;
  }
}


void  DeviceManagerTab::slotRefresh()
{
  devicelist->rescan(true);
  fillIn();
}

 
