//////////////////////////////////////////////////////////////         
//      $Id: debInterface.cpp,v 1.36 2000/08/30 17:01:09 pfeiffer Exp $ 
//
// Author: Toivo Pedaste
//
#include "../config.h"

#include <unistd.h>
#include <stdlib.h>		// for getenv
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>		// for O_RDONLY
#include <time.h>		// for localtime
#include <setjmp.h>
#include <iostream.h>   

#include <kurl.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "packageInfo.h"
#include "debInterface.h"
#include "updateLoc.h"
#include "kpackage.h"
#include "managementWidget.h"
#include "utils.h"
#include "procbuf.h"
#include "options.h"
#include "cache.h"
#include <klocale.h>


extern KApplication *app;
extern Params *params;

#define AVAIL          "/var/lib/dpkg/available"
#define STATUS          "/var/lib/dpkg/status"

#define INFODIR         "/var/lib/dpkg/info/"

static param deb_pinstall[] =  {
  {"Allow Downgrade",TRUE,TRUE},
  {"Check Conflicts",TRUE,TRUE},
  {"Check Dependencies",TRUE,TRUE},
  {"Test (do not install)",FALSE,FALSE},
  {0,FALSE,FALSE}
};

static param deb_puninstall[] =  {
  {"Purge Config Files",TRUE,FALSE},
  {"Check Dependencies",TRUE,TRUE},
  {"Test (do not uninstall)",FALSE,FALSE},
  {0,FALSE,FALSE}
};


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
DEB::DEB():pkgInterface()
{
  head = "DEB";
  icon = "deb";  

  pict = UserIcon(icon);
  bad_pict = UserIcon("dbad");
  updated_pict = UserIcon("dupdated");
  new_pict = UserIcon("dnew");

  packagePattern = "*.deb";
  queryMsg = i18n("Querying DEB package list: ");
  typeID = "/deb";
  procMsg = i18n("Kpackage: Waiting on DPKG");

  locatedialog = new Locations(i18n("Location of Debian package archives"));
  locatedialog->cLocations(3, 2,  this, i18n("L"),
			   "Deb", "*.deb",
			   i18n("Version\nArchitecture"),
			   i18n("Location of base directory of Debian distribution"),
			   "stable frozen unstable\ni386 alpha sparc powerpc arm m68k");
  locatedialog->pLocations(3, 6, this, i18n("P"),
			   "Deb", "*.deb Packages Packages.gz status available",
			   i18n("Location of 'Packages' files for sections of Debian distributions"),
			   i18n("Location of base directory of Debian distribution"));
  locatedialog->dLocations(2, 6, this, i18n("D"),
			   "Deb", "*.deb",
			   i18n("Location of directories containg Debian packages"));
  connect(locatedialog,SIGNAL(returnVal(LcacheObj *)),
	  this,SLOT(setAvail(LcacheObj *)));
  locatedialog->apply_slot();

  deb_pinstall[0].name = i18n("Allow Downgrade");
  deb_pinstall[1].name = i18n("Check Conflicts");
  deb_pinstall[2].name = i18n("Check Dependencies");
  deb_pinstall[3].name = i18n("Test (do not install)");

  deb_puninstall[0].name = i18n("Purge Config Files");
  deb_puninstall[1].name = i18n("Check Dependencies");
  deb_puninstall[2].name = i18n("Test (do not uninstall)");
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
DEB::~DEB()
{
}

//////////////////////////////////////////////////////////////////////////////
param *DEB::initinstallOptions()
{
  return &(deb_pinstall[0]);
}

param *DEB::inituninstallOptions()
{
  return &(deb_puninstall[0]);
}

// check if debian file
bool DEB::isType(char *buf, const QString &)
{
  if  (!strcmp(buf,"!<arch>\n")) {
    return true;
  } else if (!strncmp(buf,"0.9",3)) {
    return true;
  } else
    return false;
}

void DEB::distPackages(QList<packageInfo> *pki, cacheObj *cp)
{ 
  LcacheObj *cList = new LcacheObj();
  QString loc = cp->base;

  QStringList opt = QStringList::split('\n',cp->option);
  QStringList::Iterator ocIt = opt.begin();
  QString rel = *ocIt;
  QString arch = *(++ocIt);

  QString parts[3] = {"main", "contrib", "non-free"};
  for (int i = 0; i < 3; i++) {
    QString file = loc + "/dists/";
    file += rel;
    file += "/";
    file += parts[i];
    file += "/binary-";
    file += arch;
    file += "/Packages";
    QString s;
    QString tmp = cp->cacheFile;
    tmp += s.setNum(i);
    cacheObj *cp = new cacheObj(loc,file,tmp);
    cList->append(cp);
  }
  for (int i = 0; i < 3; i++) {
    QString file = loc + "/dists/";
    file += rel;
    file += "/";
    file += "non-US";
    file += "/";
    file += parts[i];
    file += "/binary-";
    file += arch;
    file += "/Packages";
    QString s;
    QString tmp = cp->cacheFile;
    tmp += s.setNum(i + 3);
    cacheObj *cp = new cacheObj(loc,file,tmp);
    cList->append(cp);
  }
  listUnIPackages(pki, cList);
}

void DEB::listPackages(QList<packageInfo> *pki)
{ 
  listInstalledPackages(pki);
  if (packageLoc) {
    listUnIPackages(pki, packageLoc);
  }
}

void DEB::listUnIPackages(QList<packageInfo> *pki, LcacheObj *pCache)
{ 
  QString s;
  cacheObj *cp;

  for (cp = pCache->first(); cp != 0; cp = pCache->next()) {
    if (!cp->option.isEmpty()) {
      distPackages(pki, cp);
    } else if (!cp->base.isEmpty()) {
      s = getPackList(cp);
      if (!s.isEmpty()) {
	listPackList(pki,s,cp);
      }
    } else { 
      s = getDir(cp);
      if (!s.isEmpty()) {
	listDir(pki,s,cp->location);
      }
    }
  }
}

bool DEB::parseName(QString name, QString *n, QString *v)
{
  int d1, d2, s1;

  s1 = name.findRev('.');
  if (s1 > 0) {
      d2 = name.findRev('-',s1-1);
      if (d2 > 0) {
	d1 = name.findRev('_',d2-1);
	if (d1 < 0)
	  d1 = d2;
	*n = name.left(d1);
	*v = name.mid(d1+1,s1-d1-1);
	return TRUE;
      }
  }
  return FALSE;
}

void DEB::listInstalledPackages(QList<packageInfo> *pki)
{
  listPackList(pki,STATUS,0);
}

void DEB::listPackList(QList<packageInfo> *pki, const QString &fname, cacheObj *cp)
{
  bool local = FALSE;
  packageInfo *p;
  QStringList list;
  QString sline( i18n("Querying DEB package list: ")+fname );

  if (cp) {
    KURL u(cp->base);
    local = u.isLocalFile();
  }

  kpackage->setStatus(sline);
  kpackage->setPercent(0);

  char linebuf[1024];
  FILE *file;
  QString s;

  file= fopen(QFile::encodeName(fname),"r");
  if (file) {
    while (fgets(linebuf,sizeof(linebuf),file)) {
      if (strcmp(linebuf,"\n")) {
	s = linebuf;
	s.truncate(s.length() - 1);
	list << s;
      } else {
	p = collectInfo(list);
	if (p) {
	  if (!p->update(pki, typeID, cp == 0)) {
	    delete p;
	  } else if (cp) {
	    p->info->insert("base",new QString(cp->base));
	  }
	}
	list.clear();
      }
    }
    fclose(file);
  }

  kpackage->setPercent(100);
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
packageInfo *DEB::getPackageInfo(char mode, const QString &name, const QString &)
{
  if (mode == 'i')
    return getIPackageInfo(name);
  else
    return getUPackageInfo(name);
}

packageInfo *DEB::getIPackageInfo( const QString &name)
{
  // query an installed package!
  packageInfo *pki = 0; 
  QString search;
  QStringList list;
  
  search = "Package: "+ name;

  QFile f(STATUS);
  if ( f.open(IO_ReadOnly) ) {  
    QTextStream t( &f );
    QString s;
    while ( !t.eof() ) {
      s = t.readLine();

      if ( s == search) {
	list << s;
	break;
      }
    }

    while ( !t.eof() ) {
      s = t.readLine();
      if (s.length()) {
	list << s;
      } else {
	pki = collectInfo(list);
	break;	
      }
    }
  }
  f.close();
  return pki;
}

packageInfo *DEB::getUPackageInfo( const QString &name)
{
  // query an uninstalled package      
  QStringList list;
  packageInfo *pki = 0; 

  reader.setup("dpkg");
  *reader.proc << "--info"<<name;
  if (reader.start(0)) {
    int p = reader.buf.find("package:",0,FALSE);
    if (p == 0) {
      list = QStringList::split("\n ",reader.buf);
    } else if (p > 0) {
      list = QStringList::split("\n ",reader.buf.mid(p));
    } else
      return 0;

    pki = DEB::collectInfo(list);
    if (pki)
      pki->updated = TRUE;
  }
  return pki;
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
packageInfo *DEB::collectInfo(QStringList &ln)
{
  QDict<QString> *a = new QDict<QString>;
  a->setAutoDelete(TRUE);

  QString key, val;
  bool bad_install = FALSE;
  bool available = FALSE;

  for ( QStringList::Iterator it = ln.begin();  it != ln.end(); ++it ) {
  loop:
    int col = (*it).find(':');
    key = ((*it).left(col)).lower();
    val = (*it).mid(col+2);
    //    val.truncate(val.length() - 1);

    if (key == "conffiles") {
      while (++it != ln.end()) {
	if ((*it)[0] == ' ') {
	} else {
	  goto loop;
	}
      }
    } else if (key == "description") {
      QString desc = val;
      desc += "\n";
      while (++it != ln.end()) {
	if ((*it)[0] == ' ') {
	  desc += *it;
	} else {
	  a->insert("description", new QString(desc));
	  goto loop;
	}
      }
      a->insert("description", new QString(desc));
      break;
    } else if (key == "package") {
      a->insert("name", new QString(val));
    } else if (key == "md5sum") {
      available = TRUE;
      bad_install = FALSE;
    } else if (key == "section") {
      a->insert("group",  new QString(val));
    } else if (key == "status") {
      if (val.find("not-installed") >= 0) {
	delete a;
	return 0;
      }
      if (val != "install ok installed" && 
	  val != "deinstall ok installed" &&
	  val != "purge ok installed") {
	bad_install = TRUE;
      }
      a->insert("status",  new QString(val));
    } else if (key == "size") {
      a->insert("file-size",  new QString(val));
    } else if (key == "installed-size") {
      a->insert("size",  new QString(val + "000"));
    } else {
      //      printf("[%s=%s]\n",key.data(),val.data());
      a->insert(key, new QString(val));
    }
  }

  packageInfo *i = new packageInfo(a,this);
  if (bad_install) {
    i->packageState = packageInfo::BAD_INSTALL;
  } else if (available) {
    i->packageState = packageInfo::AVAILABLE;
  } else {
    i->packageState = packageInfo::INSTALLED;
  }
  i->fixup();
  return i;

}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QStringList *DEB::getFileList(packageInfo *p)
{
  QString name;
  char mode;
  FILE *file;

  QString fn( p->getFilename());
  if(!fn.isEmpty())
    mode = 'u';
  else
    mode = 'i';

  QStringList *filelist = new QStringList();

  switch(mode)
  {
      ////////////////////////////////////////////////////////////////////////
      // query an installed package!
    case 'i':
      {
         name = p->getProperty("name");

         QString vb( INFODIR + name + ".list");
         file= fopen(vb.ascii(),"r");

         if (file) {
           char linebuf[1024];
	   while (fgets(linebuf,sizeof(linebuf),file)) {
	     linebuf[strlen(linebuf) - 1] = 0;    // remove new line
	     filelist->append(linebuf);
	   }
	   fclose(file);
         }
      }
      break;

      ////////////////////////////////////////////////////////////////////
      // query an uninstalled package      
    case 'u':
      {
        name = fn;
        reader.setup("dpkg");
        *reader.proc << "--contents"<<name;

        if (!reader.start(procMsg))
	  return 0;

	*filelist =  QStringList::split("\n",reader.buf);
	for ( QStringList::Iterator it = (*filelist).begin();
	      it != (*filelist).end(); ++it ) {
	  (*it) = (*it).mid((*it).findRev(' ') + 1);
	}
     }
     break;
  } // Switch

  return filelist;
}

//////////////////////////////////////////////////////////////////////////////
// Use the ~/.dpkg_ok file to tell if dpkg has worked or failed
// file created by the install/uninstall scripts
//////////////////////////////////////////////////////////////////////////////
int DEB::dpkgChange(int del)
{
    QDateTime st;
    st.setTime_t(0);

    QString tfile = getenv("HOME");
    if (!tfile.isNull()) {
      tfile += "/.dpkg_ok";
      QFileInfo *fileInfo = new QFileInfo(tfile);
      if (fileInfo->exists()) {
	return st.secsTo(fileInfo->lastModified());
	if (del)
	  unlink(tfile.ascii());
      } else {
	return 0;
      }
    } else
      return 0;
}

//////////////////////////////////////////////////////////////////////////////
// Call the script to uninstall packages setting parameters
// to dpkg dependent on flags, returning whether everyting worked
//////////////////////////////////////////////////////////////////////////////
int DEB::doUninstall(int uninstallFlags, QString packs)
{
  int test = 0, ctime, nctime;

  ctime = dpkgChange(0);

  //  reader.setup("konsole");
  reader.setup("xterm");
  *reader.proc << "-e" << "kpackage_dpkg_rm";

  *reader.proc <<  i18n("'Delete this window to continue'");

  if ((uninstallFlags>>0 & 1) ^ deb_puninstall[0].invert)
    *reader.proc << "--purge";
  else
    *reader.proc << "--remove";
  if ((uninstallFlags>>1 & 1) ^ deb_puninstall[1].invert)
    *reader.proc << "--force-depends";
  if ((uninstallFlags>>2 & 1) ^ deb_puninstall[2].invert) {
    *reader.proc << "--no-act";
    test = 1;
  }

  *reader.proc << packs;

  reader.start(procMsg);
  nctime = dpkgChange(1);
  
  return (nctime == ctime) || test;
}

//////////////////////////////////////////////////////////////////////////////
// Call the script to install packages setting parameters
// to dpkg dependent on flags, returning whether everyting worked
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

int DEB::doInstall(int installFlags, QString packs)
{
  int test = 0, ctime, nctime;

  ctime = dpkgChange(0);

  //  reader.setup("konsole");
  reader.setup("xterm");
  *reader.proc << "-e" << "kpackage_dpkg_ins";
  *reader.proc <<  i18n("'Delete this window to continue'");

  if ((installFlags>>0 & 1) ^ deb_pinstall[0].invert)
    *reader.proc << "--refuse-downgrade";
  if ((installFlags>>1 & 1) ^ deb_pinstall[1].invert)
    *reader.proc << "--force-conflicts";
  if ((installFlags>>2 & 1) ^ deb_pinstall[2].invert)
    *reader.proc << "--force-depends";
  if ((installFlags>>3 & 1) ^ deb_pinstall[3].invert) {
    *reader.proc << "--no-act";
    test = 1;
  }

  *reader.proc <<  packs;

  reader.start(procMsg);
  nctime = dpkgChange(1);
  
  return (nctime == ctime) || test;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QString DEB::FindFile(const QString &name)
{
  reader.setup("dpkg");
  *reader.proc << "-S"<<name;

  if (!reader.start(procMsg,FALSE))
    return "";

  int p = 0,lines = 0;
  while ((p = reader.buf.find(':',p)) >= 0) {
    lines++;
    reader.buf.replace(p,1,"\t");
    p = reader.buf.find('\n',p);
  }

  if (lines == 1) {
    if (reader.buf.find("not found") >= 0)
      reader.buf.truncate(0);
  }


  return reader.buf;
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void DEB::setLocation()
{
    locatedialog->restore();
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void DEB::setAvail(LcacheObj *slist)
{
  if (packageLoc)
    delete packageLoc;
  packageLoc = slist;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
