/*
    Requires the Qt widget libraries, available at no cost at
    http://www.troll.no

    Copyright (C) 1997-2000 Peter Putzer
                            putzer@kde.org

	$Id: TopWidget.cpp,v 1.91.2.1 2000/10/27 16:19:17 putzer Exp $

    This program is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License
    as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/****************************************************************
**
** KSysV
** Toplevel Widget
**
****************************************************************/

#include <ctype.h>

#include <qpopmenu.h>
#include <qkeycode.h>
#include <qmsgbox.h>
// #include <qfontmetrics.h>
// #include <qpainter.h>
#include <qmultilinedit.h>
#include <qdatetime.h>
// #include <qprinter.h>
// #include <qpaintdevicemetrics.h>
#include <qbuttongroup.h>
#include <qclipboard.h>
// #include <qprintdialog.h>
#include <qtooltip.h>
#include <qcheckbox.h>
#include <qhbox.h>
#include <qregexp.h>
#include <qdatastream.h>
#include <qpixmapcache.h>
#include <qtextview.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qwhatsthis.h>

#include <ktoolbar.h>
#include <kkeydialog.h>
#include <kmenubar.h>
#include <kcompletion.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kdebug.h>
#include <kapp.h>
#include <kglobal.h>
#include <khelpmenu.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstdaccel.h>
#include <kedittoolbar.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kaboutdata.h>

#include "ServiceDlg.h"
#include "PreferencesDialog.h"
#include "ActionList.h"
#include "ksv_core.h"
#include "ksv_conf.h"
#include "About.h"
#include "RunlevelAuthIcon.h"
#include "ksvdraglist.h"
#include "Data.h"
#include "ksvdrag.h"
#include "ConfigWizard.h"
#include "OldView.h"
#include "IOCore.h"
#include "TopWidget.h"

namespace Status
{
  enum {
    Changed, Checklist, Writable
  };
}; // namespace Status

KSVTopLevel::KSVTopLevel()
  : KMainWindow(0, 0L, WStyle_ContextHelp|WDestructiveClose),
    mConfig(KSVConfig::self()),
	mView (0L),
    mPreferences (0L),

	mEditUndo (0L), mEditCut (0L), mEditCopy (0L), mEditPaste (0L),
	mFileRevert (0L), mFileLoad (0L), mFilePrint (0L), mFilePrintLog (0L),
    mFileSave (0L), mFileSaveAs (0L), mFileSaveLog(0L), mFileQuit (0L),

	mToolsStartService (0L), mToolsStopService (0L),
    mToolsRestartService (0L), mToolsEditService (0L),

	mOptionsTools (0L), mOptionsStatus (0L), mOptionsToggleLog (0L),
    mUndoList (new ActionList (this, "UndoList")),
	mRedoList (new ActionList (this, "RedoList")),
	mStartDlg (new ServiceDlg (i18n("Start Service"),
							   i18n("&Choose which service to start:"),
							   this)),
	mStopDlg (new ServiceDlg (i18n("Stop Service"),
							  i18n("&Choose which service to stop:"),
							  this)),
	mRestartDlg (new ServiceDlg (i18n("Restart Service"),
								 i18n("&Choose which service to restart:"),
								 this)),
	mEditDlg (new ServiceDlg (i18n("Edit Service"),
							  i18n("&Choose which service to edit:"),
							  this)),
    mVisible (new QCheckBox*[ksv::runlevelNumber])
{
  kapp->setMainWidget (this);

  setCaption(false);
  
  initActions(); // order dependency
  mView = new KSVContent (this, "Content"); // order dependency
  initTools(); // order dependency
  initStatusBar(); // order dependency

  setCentralWidget(mView);

  // ensure that the statusbar gets updated correctly
  connect (mView, SIGNAL(sigRun(const QString&)), this, SLOT(slotUpdateRunning(const QString&)));
  connect (mView, SIGNAL(sigStop()), statusBar(), SLOT(clear()));
  connect (mView, SIGNAL(cannotGenerateNumber()), this, SLOT(catchCannotGenerateNumber()));
  connect (mView, SIGNAL(undoAction(KSVAction*)), this, SLOT(pushUndoAction(KSVAction*)));
  connect (mView, SIGNAL(logChanged()), this, SLOT(enableLogActions()));

  // cut & copy
  connect (mView, SIGNAL (newOrigin()),
		   this, SLOT (dispatchEdit()));

  // undo
  connect (mUndoList, SIGNAL(empty()), this, SLOT(disableUndo()));
  connect (mUndoList, SIGNAL(filled()), this, SLOT(enableUndo()));

  // and redo
  connect (mRedoList, SIGNAL(empty()), this, SLOT(disableRedo()));
  connect (mRedoList, SIGNAL(filled()), this, SLOT(enableRedo()));
  
  setUndo (false); // nothing to undo yet
  setRedo (false); // nor to redo, either

  // paste
  connect (kapp->clipboard(), SIGNAL (dataChanged()),
		   this, SLOT (dispatchEdit()));

  // init mView according to saved preferences
  slotReadConfig();
  initView();
  setMinimumSize(600,400);

  // restore size and position
  move(mConfig->position()); // doesnt seem to work while unmapped
  resize(mConfig->size());

//   // set enabled or disabled, depending on AuthIcon
//   mView->multiplexEnabled (mAuth->status());
  mAuth->setCheckEnabled(true);

  dispatchEdit(); // disable cut & copy on startup
}

KSVTopLevel::~KSVTopLevel()
{
  mUndoList->clear();
  mRedoList->clear();

  delete[] mVisible;
}

void KSVTopLevel::initTools()
{
  connect (mStartDlg, SIGNAL(doAction(const QString&)),
		   mView, SLOT(startService(const QString&)));
  connect (mStopDlg, SIGNAL(doAction(const QString&)),
		   mView, SLOT(editService(const QString&)));
  connect (mRestartDlg, SIGNAL(doAction(const QString&)),
		   mView, SLOT(restartService(const QString&))); 
  connect (mEditDlg, SIGNAL(doAction(const QString&)),
		   mView, SLOT(editService(const QString&)));

  connect (mStartDlg, SIGNAL (display (bool)),
		   this, SLOT (dispatchStartService (bool)));
  connect (mStopDlg, SIGNAL (display (bool)),
		   this, SLOT (dispatchStopService (bool)));
  connect (mRestartDlg, SIGNAL (display (bool)),
		   this, SLOT (dispatchRestartService (bool)));
  connect (mEditDlg, SIGNAL (display (bool)),
		   this, SLOT (dispatchEditService (bool)));

}

void KSVTopLevel::initActions ()
{
  // setup File menu
  mFileRevert = KStdAction::revert (this, SLOT (slotClearChanges()), actionCollection());
  mFileRevert->setText (i18n("Re&vert Configuration..."));

  mFileLoad = KStdAction::open (this, SLOT (load()), actionCollection());
  mFileLoad->setText (i18n ("&Open..."));

  mFileSave = KStdAction::save(this, SLOT(slotAcceptChanges()), actionCollection());
  mFileSave->setText (i18n("&Save Configuration..."));

  mFileSaveAs = KStdAction::saveAs (this, SLOT (saveAs ()), actionCollection());

  mFileSaveLog = KStdAction::save (this, SLOT(slotSaveLog()), actionCollection(), "ksysv_save_log");
  mFileSaveLog->setText (i18n("Save &Log..."));
  mFileSaveLog->setAccel (Key_L+CTRL);
  mFileSaveLog->setEnabled (false);

  // disabled due to complexity
  //  mFilePrint = KStdAction::print (this, SLOT (print()), actionCollection());

//   mFilePrintLog = KStdAction::print(this, SLOT(printLog()), actionCollection(), "ksysv_print_log");
//   mFilePrintLog->setText( i18n("&Print Log..."));
//   mFilePrintLog->setEnabled (false);

  mFileQuit = KStdAction::quit(this, SLOT(close()), actionCollection());
  
  // setup Edit menu
  mEditUndo = KStdAction::undo(this, SLOT(editUndo()), actionCollection());
  mEditRedo = KStdAction::redo(this, SLOT(editRedo()), actionCollection());
  mEditCut = KStdAction::cut(this, SLOT(editCut()), actionCollection());
  mEditCopy = KStdAction::copy(this, SLOT(editCopy()), actionCollection());
  mEditPaste = KStdAction::paste(this, SLOT(editPaste()), actionCollection());
  mPasteAppend = KStdAction::paste (this, SLOT (pasteAppend()),
									actionCollection(), "ksysv_paste_append");
  
  // setup Settings menu
  mOptionsTools = KStdAction::showToolbar(this, SLOT(toggleTools()), actionCollection());
  mOptionsStatus = KStdAction::showStatusbar(this, SLOT(toggleStatus()), actionCollection());
  KStdAction::keyBindings (this, SLOT(configureKeys()), actionCollection());
  KStdAction::configureToolbars (this, SLOT(configureToolbars()), actionCollection());
  KStdAction::saveOptions(this, SLOT(saveOptions()), actionCollection());
  KStdAction::preferences(this, SLOT(slotShowConfig()), actionCollection());
  mOptionsToggleLog = new KToggleAction (i18n("Show &Log"), "toggle_log", 0,
										 this, SLOT (toggleLog()),
										 actionCollection (), "ksysv_toggle_log");

  // setup Tools menu
  mToolsStartService = new KToggleAction (i18n("&Start Service..."), "ksysv_start", 0,
										 mStartDlg, SLOT (show()),
										 actionCollection(), "ksysv_start_service");
  mToolsStopService = new KToggleAction (i18n("&Stop Service..."), "ksysv_stop", 0,
										mStopDlg, SLOT (show()),
										actionCollection(), "ksysv_stop_service");

  mToolsRestartService = new KToggleAction (i18n("&Restart Service..."), 0,
										   mRestartDlg, SLOT (show()),
										   actionCollection(), "ksysv_restart_service");

  mToolsEditService = new KToggleAction (i18n("&Edit Service..."), 0,
										mEditDlg, SLOT (show()),
										actionCollection(), "ksysv_edit_service");

  createGUI(xmlFile());

  KAction* aboutKSysV = actionCollection()->action (KStdAction::stdName (KStdAction::AboutApp));
  if (aboutKSysV)
	{
	  disconnect (aboutKSysV, SIGNAL (activated()), 0, 0);
	  connect (aboutKSysV, SIGNAL (activated()), AboutDlg::self(), SLOT (show()));
	}
}

bool KSVTopLevel::queryExit() {
  uint res = KMessageBox::Yes;

  if (mChanged) {
    res = KMessageBox::warningYesNo(kapp->mainWidget(),
				    i18n("There are unsaved changes. Are you sure you want to quit?"),
				    i18n("Quit"),
				    i18n("&Quit"),
				    i18n("&Cancel"));

  }

  qApp->processEvents();
  return res == KMessageBox::Yes;
}

void KSVTopLevel::slotClearChanges()
{
  if (mChanged && 
      KMessageBox::Yes ==
      KMessageBox::questionYesNo(kapp->mainWidget(),
				 i18n("Do you really want to revert all unsaved changes?"),
				 i18n("Revert Configuration"),
				 i18n("&Revert"),
				 i18n("&Cancel")))
    {
      mUndoList->undoAll();
	  mRedoList->clear();
    }
}

void KSVTopLevel::slotAcceptChanges() {
  if (KMessageBox::Yes ==
      KMessageBox::warningYesNo(kapp->mainWidget(),
				i18n("You're about to save the changes made to your init "
				     "configuration.\nPlease realize that wrong settings can "
				     "make your system hang on startup.\n\n"
				     "Do you wish to continue?"),
				i18n("Save Configuration"),
				i18n("&Save"),
				i18n("&Cancel")))
    {
      mView->slotWriteSysV();
      initView();
    }
}

void KSVTopLevel::initView()
{
  const bool authEnabled = mAuth->isCheckEnabled();
  mAuth->setCheckEnabled(false);

  mUndoList->clear();
  mRedoList->clear();

  mView->initScripts();
  mView->initRunlevels();

  setChanged(false);

  // disable ToolsMenu_ entries when they can't do anything
  ServiceDlg* tmp = new ServiceDlg ("","", this);

  tmp->resetChooser (mView->scripts, false);
  if (!tmp->count())
    {
	  mToolsStartService->setEnabled (false);
	  mToolsStopService->setEnabled (false);
	  mToolsRestartService->setEnabled (false);
    }
  else
    {
	  mToolsStartService->setEnabled (true);
	  mToolsStopService->setEnabled (true);
	  mToolsRestartService->setEnabled (true);
    }

  tmp->resetChooser (mView->scripts, true);
  if (!tmp->count())
    {
	  mToolsEditService->setEnabled (false);
    }
  else
    {

	  mToolsEditService->setEnabled (true);
    }
  delete tmp;

  // reset tools
  mStartDlg->resetChooser (mView->scripts, false);
  mStopDlg->resetChooser (mView->scripts, false);
  mRestartDlg->resetChooser (mView->scripts, false);
  mEditDlg->resetChooser (mView->scripts, true);

  mAuth->setCheckEnabled(authEnabled);
}

void KSVTopLevel::initStatusBar()
{
  KStatusBar* status = statusBar();

  QHBox* visBox = new QHBox (status, "visBox");
  QButtonGroup* group = new QButtonGroup (this, "visButtonGroup");
  group->hide();
  connect (group, SIGNAL (clicked (int)), this, SLOT (toggleRunlevel (int)));

  QWhatsThis::add (visBox, i18n ("<p>Click on the checkboxes to <strong>show</strong> or "\
                                 "<strong>hide</strong> runlevel panes.</p> " \
                                 "<p>The list of currently visible runlevels is saved "\
                                 "when you use the <strong>Save Options command</strong>.</p>"));
  QToolTip::add (visBox, i18n ("Show/Hide Runlevels"));

  new QLabel (i18n(" Show Runlevels "), visBox, "visLabel");
  for (int i = 0; i < ksv::runlevelNumber; ++i)
	{
	  QString label; label.setNum (i);

	  mVisible[i] = new QCheckBox (label, visBox, label.latin1());
	  mVisible[i]->setChecked (mConfig->showRunlevel (i));
	  
	  group->insert (mVisible[i]);
	}

  QHBox* authIconBox = new QHBox (status, "AuthIconBox");
  QWidget* strut = new QWidget (authIconBox, "Strut");
  strut->setFixedWidth (KDialog::spacingHint());
  mAuth = new RunlevelAuthIcon (mConfig->scriptPath(), mConfig->runlevelPath(), authIconBox);
  connect (mAuth, SIGNAL (authChanged(bool)), mView, SLOT(multiplexEnabled(bool)));
  connect (mAuth, SIGNAL (authChanged(bool)), this, SLOT(writingEnabled(bool)));

  QWhatsThis::add (authIconBox, i18n ("<p>When the lock is closed <img src=\"user|ksysv_locked\"/>, "\
                                      "you haven't got the right " \
                                      "<strong>permissions</strong> to edit the init configuration.</p>" \
                                      "<p>Either restart %1 as root (or another more privileged user), " \
                                      "or ask your sysadmin to install %1 <em>suid</em> or " \
                                      "<em>sgid</em>.</p><p>The latter way is <strong>not</strong> "\
                                      "recommended though, due to security issues.</p>")
                   .arg (kapp->aboutData()->programName()).arg(kapp->aboutData()->programName()));

  authIconBox->setMinimumSize (authIconBox->minimumSizeHint());
  visBox->setMinimumSize (visBox->minimumSizeHint());

  status->addWidget (authIconBox, 0, false);
  status->insertItem ("", Status::Changed, 100);
  status->addWidget (visBox, 0, true);
  
  status->setItemAlignment (Status::Changed, AlignLeft|AlignVCenter);
}

void KSVTopLevel::slotShowConfig()
{
  if (!mPreferences)
    {
      mPreferences = KSVPreferences::self();
      
      connect (mPreferences, SIGNAL (updateColors ()),
               this, SLOT (updateColors ()));

      connect (mPreferences, SIGNAL (updateServicesPath ()),
               this, SLOT (updateServicesPath ()));

      connect (mPreferences, SIGNAL (updateRunlevelsPath ()),
               this, SLOT (updateRunlevelsPath ()));
    }

  mPreferences->setInitialSize (QSize (400,300), true);
  mPreferences->exec();
}

void KSVTopLevel::updateColors ()
{
  ksv::serviceCompletion ()->clear ();

  mView->setColors (mConfig->newNormalColor(),
                    mConfig->newSelectedColor(),
                    mConfig->changedNormalColor(),
                    mConfig->changedSelectedColor());
}

void KSVTopLevel::updateServicesPath ()
{
  mView->updateServicesAfterChange (mConfig->scriptPath());
  mAuth->setServicesPath (mConfig->scriptPath());
}

void KSVTopLevel::updateRunlevelsPath ()
{
  mView->updateRunlevelsAfterChange ();
  mAuth->setRunlevelPath (mConfig->runlevelPath());
}

void KSVTopLevel::slotReadConfig() {
  setLog(mConfig->showLog());
  setStatus(mConfig->showStatus());
  setTools(mConfig->showTools());
}

void KSVTopLevel::toggleLog() {
  const bool value = !mConfig->showLog();
  setLog(value);
}

void KSVTopLevel::toggleStatus() {
  setStatus ( mOptionsStatus->isChecked() );
}

void KSVTopLevel::toggleTools() {
  setTools ( mOptionsTools->isChecked() );
}

void KSVTopLevel::saveOptions()
{
  mConfig->writeSettings();
}

void KSVTopLevel::slotUpdateRunning (const QString& text)
{
  statusBar()->changeItem(text, Status::Changed);
}

void KSVTopLevel::editCut() {
  KSVDragList* list = mView->getOrigin();

  if (list && list->currentItem())
	{
	  KSVDrag* mime = new KSVDrag (*list->currentItem()->data(), 0L, 0L);
	  kapp->clipboard()->setData (mime);
	  
	  KSVData data = *list->currentItem()->data();
	  delete list->currentItem();
		  
	  mUndoList->push (new RemoveAction (list, &data));
	  setChanged (true);
	}
}

void KSVTopLevel::editCopy()
{
  KSVDragList* list = mView->getOrigin();

  if (list)
	{
	  KSVDrag* mime = new KSVDrag (*static_cast<KSVItem*> (list->currentItem()), 0L, 0L);
	  kapp->clipboard()->setData (mime);
	}
}

void KSVTopLevel::editPaste()
{
  KSVDragList* list = mView->getOrigin();
  
  if (list)
    {
	  KSVData data;
	  
	  if (KSVDrag::decodeNative (kapp->clipboard()->data(), data))
		{
		  KSVAction* action = 0L;

		  if (list->insert (data, list->currentItem(), action))
			{
			  setChanged (true);
			  mUndoList->push (new AddAction (list, list->match (data)->data()));
			}
		}
	}
  else
    qFatal("Bug: could not get origin of \"Paste\" event.\n" \
		   "Please notify the maintainer of this program,\n" \
		   "Peter Putzer <putzer@kde.org>.");
}

void KSVTopLevel::setChanged (bool val)
{
  mChanged = val;
  setCaption(val);

  mFileRevert->setEnabled (val);
  mFileSave->setEnabled (val);

  // update statusbar
  statusBar()->changeItem(val ? i18n(" Changed") : QString::null, Status::Changed);

  // clear messages
  statusBar()->clear();
}

void KSVTopLevel::properties()
{
  KSVDragList* list = mView->getOrigin();

  if (list)
    mView->infoOnData(list->currentItem());
}

void KSVTopLevel::scriptProperties()
{
  KSVDragList* list = mView->getOrigin();

  if (list)
    mView->slotScriptProperties(list->currentItem());
}

void KSVTopLevel::editUndo ()
{
  KSVAction* action = mUndoList->top();

  mUndoList->undoLast();

  mRedoList->push (action);
}

void KSVTopLevel::editRedo ()
{
  KSVAction* action = mRedoList->top();
  
  mRedoList->redoLast();
  
  setChanged (true);
  mUndoList->push (action);
}

void KSVTopLevel::setCut (bool val)
{
  mEditCut->setEnabled (val);
}

void KSVTopLevel::setCopy (bool val)
{
  mEditCopy->setEnabled (val);
}

void KSVTopLevel::setPaste (bool val)
{
  mEditPaste->setEnabled (val);
  mPasteAppend->setEnabled (val);
}

void KSVTopLevel::setUndo( bool val ) {
  mEditUndo->setEnabled (val);
}

void KSVTopLevel::setRedo (bool val)
{
  mEditRedo->setEnabled (val);
}

void KSVTopLevel::setLog (bool val)
{
  mConfig->setShowLog(val);
  
  mOptionsToggleLog->setChecked (val);
  
  mView->setDisplayScriptOutput(val);
}

void KSVTopLevel::setStatus (bool val)
{
  if (val)
	statusBar()->show();
  else
	statusBar()->hide();

  mConfig->setShowStatus (val);
  mOptionsStatus->setChecked (val);
}

void KSVTopLevel::setTools (bool val)
{
  if (val)
	toolBar("mainToolBar")->show();
  else
	toolBar("mainToolBar")->hide();


  mConfig->setShowTools (val);
  mOptionsTools->setChecked (val);
}

void KSVTopLevel::writingEnabled (bool on)
{
  mFileLoad->setEnabled(on);
}

void KSVTopLevel::print()
{
// #define checkPage if (metrics.height() - y < fm.lineSpacing()) prt.newPage();

//   static QPrinter prt;
//   prt.setDocName(kapp->aboutData()->programName() + " Configuration");
//   prt.setCreator(kapp->aboutData()->programName());

//   static QPrintDialog* dlg = new QPrintDialog (&prt, this, "KSysV Print Dialog");
//   dlg->setCaption(kapp->makeStdCaption (i18n("Print")));
  
//   if (dlg->exec() == QDialog::Accepted)
//     {
//       int y = 10;
//       QPainter p;
//       p.begin( &prt );

//       QPaintDeviceMetrics metrics (&prt);
  
//       p.setFont (QFont("courier", 20, QFont::Bold));
//       QFontMetrics fm = p.fontMetrics();

//       p.drawText (10, y, i18n("%1 Configuration of %2")
//                   .arg (kapp->aboutData()->programName())
//                   .arg (ksv::hostname()));
//       y += fm.lineSpacing();

//       p.drawText (10, y, QDateTime::currentDateTime().toString());
//       y += fm.lineSpacing() * 2; // an extra empty line

//       for (int i = 0; i < ksv::runlevelNumber; ++i)
//         {
//           p.setFont (QFont("courier", 16, QFont::Bold));
//           QFontMetrics fm = p.fontMetrics();
          
//           p.drawText (10, y, i18n ("Runlevel %1").arg(i));
//           y += fm.lineSpacing();

//           checkPage
          
//           p.drawText (10, y, i18n ("Started Services"));
//           y += fm.lineSpacing() * 2; // an extra empty line

//        checkPage
            
//           p.setFont (QFont("courier", 10));
//           fm = p.fontMetrics();

//           for (QListViewItemIterator it (mView->startRL[i]);
//                it.current();
//                ++it)
//             {
//               KSVItem* item = static_cast<KSVItem*> (it.current());

//               y += fm.ascent();
//               p.drawText (10, y, item->toString());
//               y += fm.descent();

//        checkPage

//             }

//           p.setFont (QFont("courier", 16, QFont::Bold));
//           fm = p.fontMetrics();
//           y += fm.lineSpacing(); // an extra empty line
//           p.drawText (10, y, i18n ("Stopped Services"));
//           y += fm.lineSpacing() * 2; // an extra empty line
   
//        checkPage
       
//           p.setFont (QFont("courier", 10));
//           fm = p.fontMetrics();

//           for (QListViewItemIterator it (mView->stopRL[i]);
//                it.current();
//                ++it)
//             {
//               KSVItem* item = static_cast<KSVItem*> (it.current());
//        checkPage

//               y += fm.ascent();
//               p.drawText (10, y, item->toString());
//               y += fm.descent();
//             }
//        checkPage

//           p.setFont (QFont("courier", 16, QFont::Bold));
//           fm = p.fontMetrics();
//           y += fm.lineSpacing() * 3; // two extra empty line
//         }
//       //      QStringList lines = QStringList::split ('\n', mView->log(), true);
//       //       for(QStringList::Iterator it = lines.begin();
//       //           it != lines.end();
//       //           ++it)
//       //         {
//       //           y += fm.ascent();
          
//       //           QString line = *it;
//       //           if (line.isNull())
//       //             line = " ";

//       //           line.replace( QRegExp("\t"), "        " );

//       //           strncpy(buf,line.data(),160);
          
//       //           for (int j = 0 ; j <150; j++)
//       //             {
//       //               if (!isprint(buf[j]))
//       //                 buf[j] = ' ';
//       //             }
          
//       //           buf[line.length()] = '\0';
//       //           p.drawText( 10, y, buf );
//       //           y += fm.descent();
//       //         }
      
//       p.end();        
//     }
}

void KSVTopLevel::printLog()
{  
//   static QPrinter prt;
//   prt.setDocName(kapp->aboutData()->programName() + " Log File");
//   prt.setCreator(kapp->aboutData()->programName());

//   static QPrintDialog* dlg = new QPrintDialog (&prt, this, "KSysV Print Log Dialog");
//   dlg->setCaption(kapp->makeStdCaption (i18n("Print Log")));
  
//   char buf[200];
//   if (dlg->exec() == QDialog::Accepted)
//     {
//       int y = 10;
//       QPainter p;
//       p.begin( &prt );
  
//       p.setFont (QFont("courier", 10, QFont::Bold));
//       QFontMetrics fm = p.fontMetrics();

//       p.drawText (10, y, i18n("%1 Log").arg(kapp->aboutData()->programName()));
//       y += fm.lineSpacing();

//       p.drawText (10, y, QDateTime::currentDateTime().toString());
//       y += fm.lineSpacing() * 2; // an extra empty line

//       p.setFont (QFont("courier", 10));
//       fm = p.fontMetrics();

//       QStringList lines = QStringList::split ('\n', mView->log(), true);
//       for(QStringList::Iterator it = lines.begin();
//           it != lines.end();
//           ++it)
//         {
//           y += fm.ascent();
          
//           QString line = *it;
//           if (line.isNull())
//             line = " ";

//           line.replace( QRegExp("\t"), "        " );

//           strncpy(buf,line.data(),160);
          
//           for (int j = 0 ; j <150; j++)
//             {
//               if (!isprint(buf[j]))
//                 buf[j] = ' ';
//             }
          
//           buf[line.length()] = '\0';
//           p.drawText( 10, y, buf );
//           y += fm.descent();
//         }
      
//       p.end();        
//     }
}

void KSVTopLevel::catchCannotGenerateNumber()
{
  statusBar()->message(i18n("Could not generate sorting number. Please change manually."), 5000);
}

void KSVTopLevel::closeEvent (QCloseEvent* e)
{
  if (mView->isInitializing())
    {
      // do not allow a close during clearing => otherwise we get a segfault
	  e->ignore();
      return;
    }

  KMainWindow::closeEvent (e);
}

void KSVTopLevel::dispatchEdit ()
{
  KSVDragList* list = mView->getOrigin();

  if (!list)
    {
      setCopy (false);
      setCut (false);
      setPaste (false);

      return;
    }

  KSVItem* current = list->currentItem();

  if (current)
	{
	  setCopy (true);
	  setCut (true);
	}
  else
	{
	  setCopy (false);
	  setCut (false);
	}
  
  if (mView->scripts == list)
	{
	  setCut (false);
	  setPaste (false);
	}
  else
	{
	  QMimeSource* mime = kapp->clipboard()->data();

	  if (mime && mime->provides ("application/x-ksysv"))
		setPaste (true);
	  else
		setPaste (false);
	}
}

void KSVTopLevel::enableUndo()
{
  setUndo(true);
}

void KSVTopLevel::disableUndo()
{
  setUndo(false);
  setChanged(false);
}

void KSVTopLevel::enableRedo()
{
  setRedo (true);
}

void KSVTopLevel::disableRedo ()
{
  setRedo (false);
}

void KSVTopLevel::slotSaveLog()
{
  static QString filter = "*.ksysv_log|" + QString(kapp->caption() + i18n(" log files") + " (*.ksysv_log)");

  QString filename = KFileDialog::getSaveFileName(0L, filter, this);

  if (filename.isEmpty())
    return;
  else if (filename.right(4) != ".ksysv_log")
    filename += ".ksysv_log";

  QFile file(filename);

  file.open( IO_WriteOnly | IO_Raw );
  QTextStream s(&file);

  s << "KDE System V Init Editor"
    << endl
    << QDateTime::currentDateTime().toString()
    << endl << endl
    << mView->log()
    << endl;

  file.close();
}

void KSVTopLevel::show()
{
  KMainWindow::show();

#if 0 // we need this in the future...

  // configuration wizard if this is the first start
  static bool first = true;
  
  if (first )//&& !mConfig->isConfigured())
    {
      ConfigWizard* w = new ConfigWizard(this, "ConfigWizard");

      w->exec();
      
      mConfig->setConfigured(true);
      mConfig->writeConfig();
      first = false;
    }

#endif // if 0

}

void KSVTopLevel::enableLogActions ()
{
  mFileSaveLog->setEnabled (true);
  //  mFilePrintLog->setEnabled (true); // disabled due to complexity
}

void KSVTopLevel::checkScriptMenu (QPopupMenu* menu, int row)
{
 //  KSVItem* i = mView->scripts->at(row);

//   if (i)
//     {
//       QFileInfo info (i->filenameAndPath());

//       if (info.isReadable() && info.isWritable())
// 	menu->setItemEnabled (PopMenu::EditService, true);
//       else
// 	menu->setItemEnabled (PopMenu::EditService, false);

//       if (info.isExecutable())
// 	{
// 	  menu->setItemEnabled (PopMenu::StartService, true);
// 	  menu->setItemEnabled (PopMenu::StopService, true);
// 	  menu->setItemEnabled (PopMenu::RestartService, true);
// 	}
//       else
// 	{
// 	  menu->setItemEnabled (PopMenu::StartService, false);
// 	  menu->setItemEnabled (PopMenu::StopService, false);
// 	  menu->setItemEnabled (PopMenu::RestartService, false);
// 	}
//     }
}

void KSVTopLevel::setCaption (bool changed)
{
  setPlainCaption (kapp->makeStdCaption(ksv::hostname(), true, changed));
}

void KSVTopLevel::configureKeys ()
{
  // remove unwanted (internal) actions
  KActionCollection coll = *actionCollection();
  coll.take(mPasteAppend);

  KKeyDialog::configureKeys (&coll, xmlFile(), true, this);
}

void KSVTopLevel::configureToolbars ()
{
  KEditToolbar* dlg = new KEditToolbar (actionCollection(), xmlFile(), true, this);

  if (dlg->exec())
	{
	  createGUI (xmlFile());
	}

  delete dlg;
}

void KSVTopLevel::dispatchEditService (bool val)
{
  mToolsEditService->setChecked (val);
}

void KSVTopLevel::dispatchStartService (bool val)
{
  mToolsStartService->setChecked (val);
}

void KSVTopLevel::dispatchStopService (bool val)
{
  mToolsStopService->setChecked (val);
}

void KSVTopLevel::dispatchRestartService (bool val)
{
  mToolsRestartService->setChecked (val);
}

void KSVTopLevel::pasteAppend ()
{
  mView->pasteAppend ();
}

void KSVTopLevel::toggleRunlevel (int index)
{
  bool state = mVisible[index]->isChecked();

  if (state)
	mView->showRunlevel (index);
  else
	mView->hideRunlevel (index);

  mConfig->setShowRunlevel (index, state);
}

void KSVTopLevel::saveAs ()
{
  static QString filter = "*.ksysv|" + i18n("Saved Init Configurations") + " (*.ksysv)";
  QString filename = KFileDialog::getSaveFileName(0L, filter, this);

  if (filename.isEmpty())
    return;
  else if (filename.right(6) != ".ksysv")
    filename += ".ksysv";

  QFile file(filename);

  file.open (IO_WriteOnly | IO_Raw);
  QDataStream s (&file);

  if (ksv::IO::saveConfiguration (s, mView->startRL, mView->stopRL))
    {
      statusBar()->message(i18n("Configuration package saved successfully."), 5000);
    }
  else
    {
      kdDebug(3000) << "ERROR saving file" << endl;
    }

  file.close();
}

void KSVTopLevel::load ()
{  
  static QString filter = "*.ksysv|" + i18n("Saved Configurations") + " (*.ksysv)";
  QString filename = KFileDialog::getOpenFileName(0L, filter, this);

  if (filename.isEmpty())
    return;

  QFile file(filename);

  file.open (IO_ReadOnly | IO_Raw);
  QDataStream s (&file);
  QValueList<KSVData>* startLists = new QValueList<KSVData>[ksv::runlevelNumber];
  QValueList<KSVData>* stopLists = new QValueList<KSVData>[ksv::runlevelNumber];
  
  if (ksv::IO::loadSavedConfiguration (s, startLists, stopLists))
    {
      statusBar()->message(i18n("Configuration package loaded successfully."), 5000);
      mView->mergeLoadedPackage (startLists, stopLists);
    }
  else
    {
      kdDebug (3000) << "ERROR loading file" << endl;
    }

  file.close();

  delete[] startLists;
  delete[] stopLists;
}

void KSVTopLevel::pushUndoAction (KSVAction* action)
{
  mRedoList->clear();
  mUndoList->push (action);

  setChanged (true);
}

#include "TopWidget.moc"
