/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qtooltip.h>
#include <qdrawutil.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qvaluelist.h>

#include <kwin.h>
#include <kwinmodule.h>
#include <kapp.h>
#include <kstyle.h>
#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <kservice.h>
#include <kdrawutil.h>
#include <kprocess.h>
#include <kstddirs.h>
#include <netwm.h>
#include <dcopclient.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

// kdesktopfile.h includes QVariant, which defines Bool and clashes with X.
#undef Bool
#include <kdesktopfile.h>

#include "taskbar.h"
#include "taskbar.moc"

template class QList<TaskButton>;
template class QList<AppStartButton>;

KWinModule* kwin_module = 0;

extern "C"
{
    KPanelApplet* init(QWidget *parent, const QString& configFile)
    {
	KGlobal::locale()->insertCatalogue("ktaskbarapplet");
	TaskbarApplet *taskbar = new TaskbarApplet(configFile, KPanelApplet::Stretch,
					       KPanelApplet::Preferences, parent, "ktaskbarapplet");
	kwin_module = new KWinModule(taskbar);
	taskbar->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
	taskbar->init();
	return taskbar;
    }
}

TaskbarApplet::TaskbarApplet(const QString& configFile, Type type, int actions,
                             QWidget *parent, const char *name)
  : KPanelApplet(configFile, type, actions, parent, name)
  , DCOPObject("TaskbarApplet"), _activeWidget(0), _layout(0)
{
    KConfig *conf = config();
    conf->setGroup("General");
    show_all = conf->readBoolEntry("ShowAllWindows", false);
    kapp->dcopClient()->connectDCOPSignal("", "clientDied(pid_t)",
					  "TaskbarApplet", "clientDied(pid_t)", false);
    connect(&publishTimer, SIGNAL(timeout()), this, SLOT( publishIconGeometries() ) );
    visible_rows = 1;
}

TaskbarApplet::~TaskbarApplet()
{
    // Disconnecting signals is not yet done automatically.
    kapp->dcopClient()->disconnectDCOPSignal("", "clientDied(pid_t)",
					     "TaskbarApplet", "clientDied(pid_t)");
}

void TaskbarApplet::init()
{
    connect( kwin_module, SIGNAL( windowAdded(WId) ), this, SLOT( windowAdded(WId) ) );
    connect( kwin_module, SIGNAL( windowRemoved(WId) ), this, SLOT( windowRemoved(WId) ) );
    connect( kwin_module, SIGNAL( activeWindowChanged(WId) ), this, SLOT( activeWindowChanged(WId) ) );
    connect( kwin_module, SIGNAL( currentDesktopChanged(int) ), this, SLOT( currentDesktopChanged(int) ) );
    connect( kwin_module, SIGNAL( windowChanged(WId,unsigned int) ),
	     this, SLOT( windowChanged(WId,unsigned int) ) );

    // register existing windows
    const QValueList<WId> windows = kwin_module->windows();
    QValueList<WId>::ConstIterator it;
    for ( it=windows.begin(); it!=windows.end(); ++it )
	windowAdded( *it );

    resetLayout();
}

void TaskbarApplet::killAppStartButton(pid_t pid)
{
    QMap<pid_t, AppStartButton *>::Iterator it = appsStarting.find(pid);

    if (it == appsStarting.end())
      return;

    AppStartButton * b = *it;

    if (0 != b) {
      appsStarting.remove(pid);
      delete b;
      b = 0;
      updateLayout();
    } else {
    }
}

void TaskbarApplet::resizeEvent(QResizeEvent*)
{
    resetLayout();
}

TaskButton* TaskbarApplet::findButton( WId w )
{
    for ( TaskButton* b = buttons.first(); b; b = buttons.next() ) {
        if ( b->window() == w  || b->hasTransient( w ) )
            return b;
    }
    return 0;
}


bool TaskButton::staysOnTop() const
{
    return info.state & NET::StaysOnTop;
}

void TaskbarApplet::windowAdded(WId w )
{
    if (topLevelWidget() && topLevelWidget()->inherits("Panel")) {
	if ( w == topLevelWidget()->winId() )
	    return;
    }

    NETWinInfo info ( qt_xdisplay(),  w, qt_xrootwin(), NET::WMWindowType | NET::WMPid | NET::WMState );
    if ( info.windowType() != NET::Normal && info.windowType() != NET::Override
	 && info.windowType() != NET::Unknown )
	return;
    if ( (info.state() & NET::SkipTaskbar) != 0 )
	return;

    Window transient_for;
    if ( XGetTransientForHint( qt_xdisplay(), (Window) w, &transient_for ) &&
	 (WId) transient_for != qt_xrootwin() && transient_for != 0 ) {
	TaskButton* btn = findButton( (WId) transient_for );
	if ( btn ) {
	    if ( btn->window() != w )
		btn->addTransient( w );
	    return;
	}
    }

    // Now do app-starting-notification stuff before we give the window
    // a taskbar button.

    // Strategy:
    //
    // Is this a NET_WM compliant app ?
    // Yes -> kill relevant app-starting button
    // No  -> Is the WM_CLASS.res_name for this app used by any existing
    //        app-starting buttons ?
    //        Yes -> kill relevant button.
    //        No  -> kill all non-NET_WM-compliant app-starting buttons.

    pid_t pid = info.pid();

    bool hasPid = (0 != pid);

    if (hasPid) {

      // Easy - this app is NET_WM compliant

      killAppStartButton(pid);

    } else {

      // Hard - this app is not NET_WM compliant

      XClassHint hint;
      Status ok = XGetClassHint(qt_xdisplay(), w, &hint);

      bool found = false;

      if (0 != ok) { // We managed to read the class hint

        QString resName   (hint.res_name);
        QString resClass  (hint.res_class);


        QMap<pid_t, AppStartButton *>::Iterator it;

        for (it = appsStarting.begin(); it != appsStarting.end(); ++it) {

          AppStartButton * b = *it;

          if (b->netCompliant()) // Ignore the compliant ones
            continue;

          if (b->bin() == resName || (b->bin().lower() == resClass.lower())) {

            // Found it !
            found = true;
            appsStarting.remove(it);
            delete b;
            b = 0;
            updateLayout();
            break;
          }
        }
      }

      if (!found) {

        // Build a list of all non-compliant buttons.

        QValueList<pid_t> buttonsToKill;

        QMap<pid_t, AppStartButton *>::ConstIterator it;

        for (it = appsStarting.begin(); it != appsStarting.end(); ++it) {

          AppStartButton * b = *it;

          if (!b->netCompliant())
            buttonsToKill << b->pid();
        }

        // Kill all non-compliant buttons.

        QValueList<pid_t>::Iterator killit(buttonsToKill.begin());

        for (; killit != buttonsToKill.end(); ++killit)
          killAppStartButton(*killit);
      }
    }

    TaskButton* b = new TaskButton(w, this);
    buttons.append( b );
    resetLayout();
}

void TaskbarApplet::windowRemoved(WId w )
{
    if (topLevelWidget() && topLevelWidget()->inherits("Panel")) {
	if ( w == topLevelWidget()->winId() )
	    return;
    }
    TaskButton* b = findButton( w );
    if ( b ) {
        if ( b->window() == w ) {
            buttons.removeRef(b);
            delete b;
            resetLayout();
        } else {
            b->removeTransient( w );
        }
    }
    resetLayout();
}

void TaskbarApplet::windowChanged(WId w, unsigned int dirty)
{
    if( !(dirty & (NET::WMVisibleName|NET::WMName|NET::WMState|NET::WMIcon|NET::XAWMState|NET::WMDesktop)) )
        return;

    TaskButton* b = findButton( w );
    if ( !b )
        return;

    b->refresh();
    if ( dirty & NET::WMIcon )
        b->refreshIcon();

    if ( b->onCurrentDesktop() != b->isVisibleTo( this ) ) {
        b->onCurrentDesktop()?b->show():b->hide();
        resetLayout();
    }
}

void TaskbarApplet::activeWindowChanged(WId w )
{
    if (topLevelWidget() && topLevelWidget()->inherits("Panel")) {
	if ( w == topLevelWidget()->winId() )
	    return;
    }
    TaskButton* b = findButton( w );

    if ( b )
        b->setActiveButton();
    else
        TaskButton::noActiveButton();
}

void TaskbarApplet::currentDesktopChanged(int )
{
    resetLayout();
}

int TaskbarApplet::widthForHeight(int) const
{
    return 180; // a default size (we are a stretch applet so this does not really matter)
}

int TaskbarApplet::heightForWidth(int) const
{
    return 180; // a default size (we are a stretch applet so this does not really matter)
}

void TaskbarApplet::resetLayout()
{
    if(_layout) {
        delete _layout;
	_layout = 0;
    }

    bool horiz = orientation() == Horizontal;
    int items, i, col, row, mod = 0;

    TaskButton *b;
    for(items=0, b = buttons.first(); b; b = buttons.next())
        items += (b->onCurrentDesktop() || show_all)?1:0;

    items += appsStarting.count();

    if(items){
	mod = (horiz && height() > 32) ? 2 : 1;
	_layout = new QGridLayout(this, mod ,1 ,lineWidth());
    }

    for(i=0, col=0, row=0, b = buttons.first(); b; b = buttons.next()){
        if(b->onCurrentDesktop() || show_all){
            b->show();
            if(horiz){
                b->setMinimumSize(24, 8);
                b->setMaximumSize(180, 180);
                _layout->addWidget(b, row, col);
		_layout->setColStretch(col, 100);
            }
            else{
                b->setMinimumSize(8, 24);
                _layout->addWidget(b, row, 0);
            }
            ++i;
            if ( horiz && ((i%mod) == 0) ) {
		row = 0;
		++col;
            }
            else
		++row;
        }
        else{
            b->move(0, 0);
            b->hide();
        }
    }

    QMap<pid_t, AppStartButton *>::Iterator it(appsStarting.begin());

    for (; it != appsStarting.end(); ++it) {

	AppStartButton * asb = *it;

	asb->show();

	if (horiz) {
	    asb->setMinimumSize(24, 8);
	    asb->setMaximumSize(180, 180);
	    _layout->addWidget(asb, row, col);
	    _layout->setColStretch(col, 100);
	} else {
	    asb->setMinimumSize(8, 24);
	    _layout->addWidget(asb, row, 0);
	}

	++i;
	if ( horiz && ((i%mod) == 0) ) {
	    row = 0;
	    ++col;
	}
	else
	    ++row;
    }

    if(!_layout) return;

    if(horiz)
    	_layout->setColStretch(++col, 1);
    else
    	_layout->setRowStretch(++row, 100);
    _layout->activate();
    updateGeometry();
    publishTimer.start(0, true);
}


void TaskbarApplet::publishIconGeometries()
{
    QPoint p = mapToGlobal( QPoint(0,0) ); // roundtrip, don't do that too often
    for ( TaskButton* b = buttons.first(); b; b = buttons.next())
        b->publishIconGeometry( p );
}

void TaskbarApplet::preferences()
{
    KProcess proc;
    proc << locate("exe", "kcmshell");
    proc << "LookNFeel/kcmtaskbar";
    proc.start(KProcess::DontCare);
}

TaskButton* TaskButton::activeButton = 0;

void TaskButton::setActiveButton()
{
    setOn( TRUE );
    if ( activeButton && activeButton != this )
        activeButton->setOn( FALSE );
    setOn( TRUE );
    activeButton = this;
}

void TaskButton::noActiveButton()
{
    if ( activeButton )
        activeButton->setOn( FALSE );
    activeButton = 0;
}

TaskButton::TaskButton( WId w, TaskbarApplet * parent, const char *name )
    : QPushButton( parent, name )
{
    setToggleButton( TRUE );
    win = w;

    refresh();
    refreshIcon();
    connect( this, SIGNAL( clicked() ), this, SLOT( toggled() ) );

    popup = 0;
    desk_popup = 0;
}

TaskButton::~TaskButton()
{
    if ( activeButton == this )
        activeButton = 0;
    delete popup; popup = 0;
}

bool TaskButton::onCurrentDesktop() const
{
    return info.onAllDesktops  ||  info.desktop == kwin_module->currentDesktop();
}

void TaskButton::refresh()
{
    info = KWin::info( win );
    QString t = info.visibleNameWithState();
    if(t != text()) {
        setText( t );
        QToolTip::add( this, t );
    }
}

void TaskButton::refreshIcon()
{
    pixmap = KWin::icon( win, 16, 16, true );
    if(pixmap.isNull())
        pixmap = SmallIcon("bx2");
    update();
}

void TaskButton::drawButton(QPainter *p)
{
  bool sunken = isOn() || isDown();

  if (0 != kapp->kstyle()) {
      kapp->kstyle()->drawKickerTaskButton(p, 0, 0, width(), height(),
					   colorGroup(), text(), sunken, &pixmap);
      return;
  }

  kDrawNextButton(p, rect(), colorGroup(), sunken,
		  sunken ? &colorGroup().brush(QColorGroup::Mid) :
		  &colorGroup().brush(QColorGroup::Button));

  int pxWidth = 20;
  int textPos = 0;

  QRect br(style().buttonRect(0, 0, width(), height()));

  if (sunken)
    p->translate(1,1);

  if (!pixmap.isNull()) {

    int dx = (pxWidth   - pixmap.width())   / 2;
    int dy = (height()  - pixmap.height())  / 2;

    p->drawPixmap(br.x() + dx, dy, pixmap);

    textPos += pxWidth;
  }

  QString s(text());

  static QString modStr =
    QString::fromUtf8("[") + i18n("modified") + QString::fromUtf8("]");

  int modStrPos = s.find(modStr);

  if (-1 != modStrPos) {

    // +1 because we include a space after the closing brace.
    s.remove(modStrPos, modStr.length()+1);

    QPixmap modPixmap = SmallIcon("modified");

    int dx = (pxWidth   - modPixmap.width())  / 2;
    int dy = (height()  - modPixmap.height()) / 2;

    p->drawPixmap(br.x() + textPos + dx, dy, modPixmap);

    textPos += pxWidth;
  }

  if (!s.isEmpty())
  {
    if (p->fontMetrics().width(s) > br.width() - textPos) {

      int maxLen = br.width() - textPos - p->fontMetrics().width("...");

      while ((!s.isEmpty()) && (p->fontMetrics().width(s) > maxLen))
        s.truncate(s.length() - 1);

      s.append("...");
    }

    p->setPen(sunken ? colorGroup().light() : colorGroup().buttonText());

    p->drawText(
      br.x() + textPos, -1,
      width() - textPos, height(),
      AlignLeft | AlignVCenter,
      s
    );
  }
}

void TaskButton::toggled()
{
    if ( !isOn() ) {
        TaskButton* btn = 0;
        QValueList<WId>::ConstIterator it;
        for ( it = kwin_module->stackingOrder().fromLast();
              it != kwin_module->stackingOrder().end() && !btn ; --it ) {
            btn = ((TaskbarApplet*)parentWidget() )->findButton( *it );
            if ( btn && btn != this && (btn->staysOnTop() == staysOnTop()) ) {
                // we are not on top, raise rather than iconify. This
                // pleases focusFollowsMouse users
                setOn( TRUE );
                XRaiseWindow( qt_xdisplay(), win );
                break;
            }
        }
    }
    if ( isOn() ) {
        setActiveButton();
        KWin::setActiveWindow( win );
    }
    else {
        XIconifyWindow( qt_xdisplay(), win, qt_xscreen() );
    }
}

void TaskButton::clientPopupAboutToShow()
{
    if ( !popup )
        return;
    popup->setItemChecked( IconifyOp, info.mappingState == NET::Iconic );
    popup->setItemChecked( MaximizeOp, info.state & NET::Max );
}

void TaskButton::desktopPopupAboutToShow()
{
    if ( !desk_popup )
        return;
    desk_popup->clear();
    desk_popup->insertItem( i18n("&All desktops"), 0 );
    desk_popup->insertSeparator();
    if ( info.onAllDesktops )
        desk_popup->setItemChecked( 0, true );
    int id;
    for ( int i = 1; i <= kwin_module->numberOfDesktops(); i++ ) {
        id = desk_popup->insertItem( QString("&%1 %2").arg(i).arg(kwin_module->desktopName(i)), i );
        if ( !info.onAllDesktops && info.desktop == i )
            desk_popup->setItemChecked( id, TRUE );
    }
}

void TaskButton::clientPopupActivated( int id )
{
    switch ( id )
    {
    case MaximizeOp:
        if ( (info.state & NET::Max)  == 0 ) {
            NETWinInfo ni( qt_xdisplay(),  win, qt_xrootwin(), 0);
            ni.setState( NET::Max, NET::Max );
        } else {
            NETWinInfo ni( qt_xdisplay(),  win, qt_xrootwin(), 0);
            ni.setState( 0, NET::Max );
        }
        break;
    case IconifyOp:
        if ( info.mappingState != NET::Iconic ) {
            XIconifyWindow( qt_xdisplay(), win, qt_xscreen() );
        } else {
            KWin::setActiveWindow( win );
        }
        break;
    case CloseOp: {
        NETRootInfo ri( qt_xdisplay(),  0 );
        ri.closeWindowRequest( win );
    } break;
    default:
        break;
    }
}


void TaskButton::publishIconGeometry( const QPoint& global )
{
    NETWinInfo ni( qt_xdisplay(),  win, qt_xrootwin(), 0);
    QPoint p = global + geometry().topLeft();
    NETRect r;
    r.pos.x = p.x();
    r.pos.y = p.y();
    r.size.width = width();
    r.size.height = height();
    ni.setIconGeometry( r );
}

void TaskButton::sendToDesktop( int desk )
{
    NETWinInfo ni( qt_xdisplay(),  win, qt_xrootwin(), 0);
    if ( desk == 0 ) {
        if ( info.onAllDesktops ) {
            ni.setDesktop( kwin_module->currentDesktop() );
        } else {
            ni.setDesktop( NETWinInfo::OnAllDesktops );
        }
        return;
    }

    ni.setDesktop( desk );
}

void TaskButton::mousePressEvent(QMouseEvent *e)
{
    if ( e->button() == QMouseEvent::RightButton )
    {
        if (!popup)
        {
            popup = new QPopupMenu;
            popup->setCheckable( TRUE );
            connect( popup, SIGNAL( aboutToShow() ),
                    this,  SLOT( clientPopupAboutToShow() ) );
            connect( popup, SIGNAL( activated(int) ),
                    this,  SLOT( clientPopupActivated(int) ) );

            if (!desk_popup) desk_popup = new QPopupMenu( popup );
            desk_popup->setCheckable( TRUE );
            connect( desk_popup, SIGNAL( aboutToShow() ),
                    this,       SLOT( desktopPopupAboutToShow() ) );
            connect( desk_popup, SIGNAL( activated(int) ),
                    this,       SLOT( sendToDesktop(int) ) );

            popup->insertItem( i18n("Mi&nimize"), IconifyOp );
            popup->insertItem( i18n("Ma&ximize"), MaximizeOp );

            popup->insertSeparator();

            popup->insertItem(i18n("&To desktop"), desk_popup );

            popup->insertSeparator();

            popup->insertItem(i18n("&Close"), CloseOp);
        }
        popup->popup(e->globalPos());
        return;
    }
    QPushButton::mousePressEvent(e);
}

QSizePolicy TaskButton::sizePolicy() const
{
    return(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
}

QSize TaskButton::sizeHint() const
{
    return(minimumSize());
}

void TaskbarApplet::clientStarted(QString name, QString icon, pid_t pid,
    QString bin, bool compliant)
{
    //kdDebug() << "clientStarted(" << name << ", " << icon << ", " << (long)pid << "d)" << endl;

    if ( (long)pid == 0 )
	return;

    AppStartButton * b = new AppStartButton(name, icon, pid, bin, compliant, this);
    appsStarting[pid] = b;

    connect(
	    b,    SIGNAL(killMe(pid_t)),
	    this, SLOT(killAppStartButton(pid_t)));

    resetLayout();
    b->show();
}

void TaskbarApplet::clientMapped(pid_t pid)
{
    //kdDebug() << "clientMapped " << (long)pid << endl;
    if ( (long)pid != 0 )
	killAppStartButton(pid);
}

void TaskbarApplet::clientDied(pid_t pid)
{
    //kdDebug() << "clientDied(" << (long)pid << "d)" << endl;
    if ( (long)pid != 0 )
	killAppStartButton(pid);
}

void TaskbarApplet::mousePressEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}
void TaskbarApplet::mouseReleaseEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}
void TaskbarApplet::mouseDoubleClickEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}
void TaskbarApplet::mouseMoveEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}

void TaskbarApplet::propagateMouseEvent( QMouseEvent* e )
{
    if (topLevelWidget() && topLevelWidget()->inherits("Panel")) {
	QMouseEvent me(e->type(), mapTo( topLevelWidget(), e->pos() ),
		       e->globalPos(), e->button(), e->state() );
	QApplication::sendEvent( topLevelWidget(), &me );
    }
}

bool TaskbarApplet::x11Event( XEvent* e )
{
    // "Catch" XEvents which occur on the taskbars's edges, and redirect them
    // to the apropriate task buttons. This way the taskbar obeys Fitt's law.
    switch ( e->type ) {

    case ButtonPress:
	{
	    // Only ButtonPress events which occur on the widget's frame need to
	    // be handled.
	    if (contentsRect().contains( QPoint(e->xbutton.x, e->xbutton.y) ) ||
		!rect().contains( QPoint(e->xbutton.x, e->xbutton.y) ))
		return false;

	    // Determine the difference between the catched event's position and
	    // the position of the new event that we will contruct. The new
	    // position may not be on the frame, but has to be in
	    // contentsRect(). Using a difference is easier because it can be
	    // applied it to both the local and global positions.
	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xbutton.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xbutton.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xbutton.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xbutton.y );

	    // The widget which will be the destination of the new event.
	    QWidget* destWidget = QApplication::widgetAt(e->xbutton.x_root + dx,
							 e->xbutton.y_root + dy, true);

	    // If there is no such widget, we leave the event to Qt's event
	    // handler. If destWidget is equal to this widget pass it to Qt's
	    // event handler too to avoid nasty loops.
	    if (!destWidget || destWidget == this)
		return false;

	    // Now construct the new event.
	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xbutton.window = destWidget->winId();
	    Window child; // Not used
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xbutton.x + dx, e->xbutton.y + dy,
				  &ne.xbutton.x, &ne.xbutton.y, &child);
	    ne.xbutton.x_root = e->xbutton.x_root + dx;
	    ne.xbutton.y_root = e->xbutton.y_root + dy;

	    // Pretty obvious... Send the event.
	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    // Make the receiver our active widget. It will receive all events
	    // until the mouse button is released.
	    _activeWidget = destWidget;

	    // We're done with this event.
	    return true;
	}

	// The rest of the cases are more or less a duplication of the first
	// one with off course some minor differences. ButtonRelease is almost
	// the same as MotionNotify, but there's xbutton and xmotion.
    case ButtonRelease:
	{
	    // Handle events outside the widget's rectangle too, since the mouse
	    // can be grabbed.
	    if (contentsRect().contains( QPoint(e->xbutton.x, e->xbutton.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xbutton.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xbutton.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xbutton.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xbutton.y );

	    // If there is a widget active it should receive the new event.
	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xbutton.x_root + dx,
						    e->xbutton.y_root + dy, true);
	
	    if (!destWidget || destWidget == this)
		return false;
	
	    // The event's position can be outside the widget as well, so
	    // there's no need to adjust the position.
	    if (!rect().contains( QPoint(e->xbutton.x, e->xbutton.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xbutton.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xbutton.x + dx, e->xbutton.y + dy,
				  &ne.xbutton.x, &ne.xbutton.y, &child);
	    ne.xbutton.x_root = e->xbutton.x_root + dx;
	    ne.xbutton.y_root = e->xbutton.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    // Turn off the active widget.
	    _activeWidget = 0;

	    return true;
	}

    case MotionNotify:
	{
	    if (contentsRect().contains( QPoint(e->xmotion.x, e->xmotion.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xmotion.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xmotion.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xmotion.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xmotion.y );

	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xmotion.x_root + dx,
						    e->xmotion.y_root + dy, true);
						
	    if (!destWidget || destWidget == this)
		return false;

	    if (!rect().contains( QPoint(e->xmotion.x, e->xmotion.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xmotion.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xmotion.x + dx, e->xmotion.y + dy,
				  &ne.xmotion.x, &ne.xmotion.y, &child);
	    ne.xmotion.x_root = e->xmotion.x_root + dx;
	    ne.xmotion.y_root = e->xmotion.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    return true;
	}

    case EnterNotify:
    case LeaveNotify:
	{
	    if (contentsRect().contains(
					QPoint(e->xcrossing.x, e->xcrossing.y) ) ||
		!rect().contains( QPoint(e->xcrossing.x, e->xcrossing.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xcrossing.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xcrossing.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xcrossing.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xcrossing.y );

	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xcrossing.x_root + dx,
						    e->xcrossing.y_root + dy, true);
	
	    if (!destWidget || destWidget == this)
		return false;

	    if (!rect().contains( QPoint(e->xcrossing.x, e->xcrossing.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xcrossing.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xcrossing.x + dx, e->xcrossing.y + dy,
				  &ne.xcrossing.x, &ne.xcrossing.y, &child);
	    ne.xcrossing.x_root = e->xcrossing.x_root + dx;
	    ne.xcrossing.y_root = e->xcrossing.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    return true;
	}
    default:
	break;
    }

    return false;
}

AppStartButton::AppStartButton(const QString & text, const QString & /*icon*/,
			       pid_t pid, const QString & bin, bool compliant, QWidget * parent)
  : QPushButton(text, parent), frame(0), pid_(pid), bin_(bin),
  compliant_(compliant)
{

    QString appName(text);

    setText(text);

    QToolTip::add(this, i18n("Starting %1").arg(appName));

    anim.setAutoDelete(true);

    for (int i = 1; i < 9; i++)
	anim.append(new QPixmap(UserIcon(QString("disk") + QString::number(i))));

    connect(&animTimer, SIGNAL(timeout()), this, SLOT(animTimerFired()));

    animTimer.start(100);

    // Go away after 20s if we weren't removed before.
    startTimer(20000);
}

void AppStartButton::timerEvent(QTimerEvent *)
{
    killTimers();
    emit(killMe(pid_));
}

void AppStartButton::animTimerFired()
{
    QPainter p(this);

    QPixmap *pix = anim.at(frame);

    p.drawPixmap(4, (height() - pix->height())/2, *pix);

    if (frame == 7)
	frame = 0;
    else
	++frame;
}

void AppStartButton::drawButton(QPainter * p)
{
    bool sunken = isOn() || isDown();

    if (0 != kapp->kstyle()) {
	kapp->kstyle()->drawKickerTaskButton(p, 0, 0, width(), height(), colorGroup(), text(), sunken);
	return;
    }

    kDrawNextButton(p, rect(), colorGroup(), false, &colorGroup().brush(QColorGroup::Button));

    const int pxWidth = 20;

    QRect br(style().buttonRect(0, 0, width(), height()));

    QString s(text());

    if (!s.isEmpty()){
	if (p->fontMetrics().width(s) > br.width() - pxWidth) {

	    int maxLen = br.width() - pxWidth - p->fontMetrics().width("...");

	    while ((!s.isEmpty()) && (p->fontMetrics().width(s) > maxLen))
		s.truncate(s.length() - 1);

	    s.append("...");
	}

	p->setPen(sunken ? colorGroup().light() : colorGroup().buttonText());

	p->drawText(br.x() + pxWidth, 0, width() - pxWidth, height(),
		    AlignLeft | AlignVCenter, s);
    }
}

void AppStartButton::mousePressEvent(QMouseEvent *)
{
    //Don't call default handler ... we literally want to do nothing!
}

QSizePolicy AppStartButton::sizePolicy() const
{
    return(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
}

QSize AppStartButton::sizeHint() const
{
    return(minimumSize());
}
