/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qpainter.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qapp.h>
#include <qstyle.h>

#include <kstddirs.h>
#include <kglobal.h>
#include <kdebug.h>
#include <kimageeffect.h>
#include <kiconloader.h>
#include <kmimetype.h>
#include <kglobalsettings.h>
#include <kcursor.h>
#include <kapp.h>
#include <kipc.h>

#include "panelbuttonbase.h"
#include "panelbuttonbase.moc"

PanelButtonBase::PanelButtonBase(QWidget *parent, const char *name)
  : QButton(parent, name)
  , _dir(Bottom)
  , _drawArrow(false)
  , _highlight(false)
  , _changeCursorOverItem(true)
  , _tile(QString::null)
{
  setBackgroundMode( PaletteBackground );
  oldCursor = cursor();

  slotSettingsChanged(KApplication::SETTINGS_MOUSE);
  connect(kapp, SIGNAL(settingsChanged(int)), SLOT(slotSettingsChanged(int)));
  kapp->addKipcEventMask(KIPC::SettingsChanged);
}

void PanelButtonBase::slotSettingsChanged(int category)
{
  if (category != KApplication::SETTINGS_MOUSE) return;

  _changeCursorOverItem = KGlobalSettings::changeCursorOverIcon();

    if(!_changeCursorOverItem)
      setCursor(oldCursor);
}

void PanelButtonBase::setTile(const QString& tile)
{
  _tile = tile;
  loadTiles();
  repaint();
}

void PanelButtonBase::setIcon(const QString & nm, const QString & fallback)
{
  KIcon::StdSizes sz = // Er, enum names should not be plural
    width() < 32 ? KIcon::SizeSmall :
    (width() < 48 ? KIcon::SizeMedium : KIcon::SizeLarge);

  KIconLoader * ldr = KGlobal::iconLoader();

  _icon = ldr->loadIcon(nm, KIcon::Panel, sz, KIcon::DefaultState, 0L, true);

  if (_icon.isNull())
    _icon = ldr->loadIcon(fallback, KIcon::Panel, sz, KIcon::DefaultState);

  _iconh = ldr->loadIcon(nm, KIcon::Panel, sz, KIcon::ActiveState, 0L, true);

  if (_iconh.isNull())
    _iconh = ldr->loadIcon(fallback, KIcon::Panel, sz, KIcon::ActiveState);

  repaint();
}

void PanelButtonBase::setIcon(const KURL & u)
{
  KIcon::StdSizes sz =
    width() < 32 ? KIcon::SizeSmall :
    (width() < 48 ? KIcon::SizeMedium : KIcon::SizeLarge);

  _icon = KMimeType::pixmapForURL(u, 0, KIcon::Panel, sz, KIcon::DefaultState);
  _iconh = KMimeType::pixmapForURL(u, 0, KIcon::Panel, sz, KIcon::ActiveState);

  repaint();
}


void PanelButtonBase::setDrawArrow(bool v)
{
  if (_drawArrow == v) return;
  _drawArrow = v;
  repaint();
}

void PanelButtonBase::setArrowDirection(Position dir)
{
  if (_dir == dir) return;
  _dir = dir;
  repaint();
}

void PanelButtonBase::setBackground()
{
    KConfig* config = KGlobal::config();
    config->setGroup("General");
    if (config->readBoolEntry("UseBackgroundTheme", false))
	{
	    // Get the pixmap from the container area object
	    QWidget* pContainer = (QWidget*) parent()->parent()->parent();
	    QPalette pal = pContainer->palette();
	    QBrush bgBrush = pal.brush( QPalette::Active, QColorGroup::Background );
	    QPixmap* containerBG = bgBrush.pixmap();

	    // Make sure the background pixmap exists
	    if ( containerBG )
		{
		    // Create a pixmap the same size as the button to use as the bg
		    QPixmap bgPix( width(), height() );

		    // Calculate which part of the container area background should be copied
		    QWidget* p = (QWidget*) parent();

		    int srcx, srcy;
		    if ( _dir == Bottom || _dir == Top )
			{
			    srcx = p->x() % containerBG->width();
			    srcy = 0;
			}
		    else
			{
			    srcx = 0;
			    srcy = p->y() % containerBG->height();
			}

		    // Check if we need to take care of a wrap-around
		    if ( srcx + p->width() <= containerBG->width() &&
			 srcy + p->height() <= containerBG->height() )
			{
				// Nothing funny going on with overlaps - just copy

			    QPoint dp( 0, 0 ); // destination point
			    QRect sr( srcx, srcy, width(), height() );
			    bitBlt( &bgPix, dp, containerBG, sr, CopyROP );
			}
		    else
			{
				// Need to do 2 seperate bitBlts to take care of the overlap

				// Copy the left/top of the image first up to the wrap-
				// around point
			    int x = 0;
			    int y = 0;
			    int w = containerBG->width() - srcx;
			    int h = containerBG->height() - srcy;
			    QPoint dp( x, y ); // destination point
			    QRect sr( srcx, srcy, w, h );
			    bitBlt( &bgPix, dp, containerBG, sr, CopyROP );

				// Now copy the wrap-around bit
			    if ( _dir == Bottom || _dir == Top )
				{
				    x = containerBG->width() - srcx;
				    y = 0;
				    w = srcx + p->width() - containerBG->width();
				    h = p->height();
				}
			    else
				{
				    x = 0;
				    y = containerBG->height() - srcy;
				    w = p->width();
				    h = srcy + p->height() - containerBG->height();
				}
			    dp = QPoint( x, y );
			    sr = QRect( 0, 0, w, h );
			    bitBlt( &bgPix, dp, containerBG, sr, CopyROP );
			}

		    _bg = bgPix;
		}
	    else
		{
		    // Conatainer palette pixmap not set yet
		    _bg = QPixmap();
		}
	}
    else
	{
	    // No background pixmap to use
	    _bg = QPixmap();
	}
}

void PanelButtonBase::resizeEvent(QResizeEvent*)
{
  // optimize: reload only when size really changes
  loadTiles();
  repaint();
}

void PanelButtonBase::enterEvent(QEvent* e)
{
  if (_changeCursorOverItem)
    setCursor(KCursor().handCursor());

  _highlight = true;
  QPainter p(this);
  drawButton(&p);
  QButton::enterEvent(e);
}

void PanelButtonBase::leaveEvent(QEvent* e)
{
  if (_changeCursorOverItem)
    setCursor(oldCursor);

  _highlight = false;
  QPainter p(this);
  drawButton(&p);
  QButton::leaveEvent(e);
}

void PanelButtonBase::loadTiles()
{
  if(_tile == QString::null)
    _up = _down = QPixmap();

  QString uptile, downtile;

  if (height() < 42)
    {
      uptile = _tile + "_tiny_up.png";
      downtile = _tile + "_tiny_down.png";
    }
  else if (height() < 54)
    {
      uptile = _tile + "_normal_up.png";
      downtile = _tile + "_normal_down.png";
    }
  else
    {
      uptile = _tile + "_large_up.png";
      downtile = _tile + "_large_down.png";
    }

  uptile = KGlobal::dirs()->findResource("tiles", uptile);
  downtile = KGlobal::dirs()->findResource("tiles", downtile);

  QPixmap up(uptile);
  QPixmap down(downtile);

  if (!up.isNull())
    _up = up;
  else
    _up = QPixmap();

  if (!down.isNull())
    _down = down;
  else
    _down = QPixmap();
}

void PanelButtonBase::drawButton(QPainter *p)
{
	setBackground();

  if(isDown() || isOn() ){

    if (!_down.isNull())     // draw down tile
      {
        int x = (width() - _down.width())/2;
        int y = (height() - _down.height())/2;
        p->drawPixmap(x, y, _down);
      }
    else // no tile loaded
      {
        if ( isDown() )
	    p->fillRect(rect(), colorGroup().brush(QColorGroup::Mid));
        else
	    p->fillRect(rect(), colorGroup().brush(QColorGroup::Midlight));
        p->setPen(Qt::black);
        p->drawLine(0, 0, width()-1, 0);
        p->drawLine(0, 0, 0, height()-1);
        p->setPen(colorGroup().light());
        p->drawLine(0, height()-1, width()-1, height()-1);
        p->drawLine(width()-1, 0, width()-1, height()-1);
      }
  }
  else {

    if (!_up.isNull())  // draw up tile
      {
        int x = (width() - _up.width())/2;
        int y = (height() - _up.height())/2;
        p->drawPixmap(x, y, _up);
      }
    else // no tile loaded
      {
        // no fallback code needed
	/* antonio: Yes, we need it because the buttons may have an effect
	 * applied that forces us to repaint the whole icon before painting
	 * another icon effect */
		/* Sean: True but we should use a background pixmap if available */
	 	if ( !_bg.isNull() )
		{
			p->drawPixmap( 0, 0, _bg );
		}
		else
		{
		 	p->fillRect(rect(), colorGroup().brush(QColorGroup::Button));
		}

      }
  }
  drawButtonLabel(p);

  int x1, y1, x2, y2;
  rect().coords( &x1, &y1, &x2, &y2 );// get coordinates
  if ( hasFocus() ) {
 	QRect r(x1+2, y1+2, x2-x1-3, y2-y1-3);
 	style().drawFocusRect( p, r , colorGroup(), &colorGroup().button() );
  }
}

void PanelButtonBase::drawButtonLabel(QPainter *p)
{
  bool hl = _highlight;

  // draw icon
  if (!hl && !_icon.isNull())
    {
      int x = (width() - _icon.width())/2;
      int y = (height() - _icon.height())/2;
      if (isDown()||isOn()) {
        x+=2;
        y+=2;
      }
      p->drawPixmap(x, y, _icon);
    }
  else if (hl && !_iconh.isNull())
    {
      int x = (width() - _iconh.width())/2;
      int y = (height() - _iconh.height())/2;
      if (isDown()||isOn()) {
        x+=2;
        y+=2;
      }
      p->drawPixmap(x, y, _iconh);
    }

  int d = 0;
  if(isDown()) d = 2;

  // draw arrow
  if(_drawArrow && width() > 32 && height() > 32)
    {
      if(_dir == Top)
        QApplication::style().drawArrow(p, Qt::UpArrow, isDown(), 3+d, 3+d,
                                        8, 8, colorGroup(), true);
      else if (_dir == Bottom)
        QApplication::style().drawArrow(p, Qt::DownArrow, isDown(),
                                        3+d, height()-12+d, 8, 8,
                                        colorGroup(), true);
      else if (_dir == Right)
        QApplication::style().drawArrow(p, Qt::RightArrow, isDown(),
                                        width()-12+d, 3+d, 8, 8,
                                        colorGroup(), true);
      else
        QApplication::style().drawArrow(p, Qt::LeftArrow, isDown(), 3+d, 3+d,
                                        8, 8, colorGroup(), true);
    }
}
