#include <qtimer.h>

#include <kinstance.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kconfig.h>
#include <kglobalsettings.h>

#include "kgv_view.h"
#include "kgv_miniwidget.h"


extern "C"{
  void *init_libkghostview()
  {
    return new KGVFactory;
  }
};

KInstance *KGVFactory::s_instance = 0L;

KGVFactory::KGVFactory()
{
}

KGVFactory::~KGVFactory()
{
    if ( s_instance )
        delete s_instance;

    s_instance = 0;
}

KParts::Part *KGVFactory::createPart( QWidget *parentWidget, const char *widgetName, QObject *parent, const char *name, const char *className, const QStringList & )
{
    bool bBrowserView = ( strcmp( className, "Browser/View" ) == 0 );
    KParts::Part *obj = new KGVPart( bBrowserView, parentWidget, widgetName, parent, name );
    emit objectCreated( obj );
    return obj;
}

KInstance *KGVFactory::instance()
{
    if ( !s_instance )
        s_instance = new KInstance( "kghostview" );
    return s_instance;
}

KGVPart::KGVPart( bool bBrowserView, QWidget *parentWidget, 
		  const char *, QObject *parent, const char *name )
 : KParts::ReadOnlyPart( parent, name )
{
  setInstance( KGVFactory::instance() );
  w = new KGVMiniWidget( parentWidget ); // ### obey widgetName (Simon)
  // Clicking or tabbing on it should make it active (required by KParts)
  w->setFocusPolicy( QWidget::StrongFocus );
  setWidget( w );

  //Put state-dependent actions in this collection.
  statedep = new KActionCollection;
  
  if ( !bBrowserView )
    statedep->insert( KStdAction::saveAs ( miniWidget(), SLOT (saveAs()),
					   actionCollection()) );
  
  //Edit Menu
  statedep->insert 
    ( new KAction(i18n("&Info"), 0, miniWidget(), SLOT (info()),
      actionCollection(), "info") );

  
  QStringList orientations;
  orientations.append (i18n("Portrait"));
  orientations.append (i18n("Landscape"));
  orientations.append (i18n("Upside Down"));
  orientations.append (i18n("Seascape"));
  
  orientation =
    new KSelectAction (i18n("&Orientation"), 0, 0L, 0L, actionCollection(),
		       "orientation_menu");
  statedep->insert (orientation);
  connect ( orientation, SIGNAL (activated (int)),
	    this, SLOT (slotOrientation(int)) );
  
  orientation->setItems (orientations);
  
  //TODO -- zoom (maybe remove media from toolbar)
  media =
    new KSelectAction (i18n("Paper &Size"), 0, 0L, 0L,
		       actionCollection(), "media_menu");
  statedep->insert(media);
  connect ( media, SIGNAL (activated (int)),
	    this, SLOT (slotMedia(int)) );
  
  
  
  zoomInAct =
    KStdAction::zoomIn (w,  SLOT(zoomIn() ),
			actionCollection(), "zoomIn");
  statedep->insert(zoomInAct);

  zoomOutAct =
    KStdAction::zoomOut (w,  SLOT(zoomOut() ),
			 actionCollection(), "zoomOut");
  statedep->insert(zoomOutAct);
  
  
  backAct =
    KStdAction::prior (w, SLOT(prevPage() ),
		       actionCollection(), "prevPage");
  statedep->insert(backAct);
  
  forwardAct =
    KStdAction::next (w, SLOT(nextPage() ),
		      actionCollection(), "nextPage");
  statedep->insert(forwardAct);
  
  startAct =
    new KAction(i18n("Go to start"),
		QIconSet(BarIcon("start", KGVFactory::instance())),
		CTRL+Key_Home, w, SLOT(goToStart() ), actionCollection(), "goToStart");
  statedep->insert(startAct);

  endAct =
    new KAction(i18n("Go to end"),
		QIconSet(BarIcon("finish", KGVFactory::instance())),
		CTRL+Key_End, w, SLOT(goToEnd() ), actionCollection(), "goToEnd");
  statedep->insert(endAct);

  endDocAct =
    new KAction(i18n("Read down document"),
		QIconSet(BarIcon("next", KGVFactory::instance())),
		Key_Space, w, SLOT(readDown() ),
		actionCollection(), "readDown");
  statedep->insert(endDocAct);

  markAct =
    new KAction(i18n("Toggle this page mark"),
		QIconSet(BarIcon("flag", KGVFactory::instance())),
		Key_Enter, w, SLOT(markPage() ), actionCollection(), "markPage");
  statedep->insert(markAct);

  gotoAct =
    KStdAction::gotoPage (w, SLOT(goToPage() ),
			  actionCollection(), "goToPage");
  statedep->insert(gotoAct);

  //Pagemarks
  statedep->insert( new KAction (i18n("Mark &Current Page"), 0, miniWidget(),
				 SLOT (markPage()),
				 actionCollection(), "mark_current") );
  statedep->insert
    (  new KAction (i18n("Mark &All Pages"), 0, miniWidget()->markList(),
		    SLOT (markAll()),
		    actionCollection(), "mark_all") );
  statedep->insert
    ( new KAction (i18n("Mark &Even Pages"), 0, miniWidget()->markList(),
		   SLOT (markEven()),
		   actionCollection(), "mark_even") );
  statedep->insert
    ( new KAction (i18n("Mark &Odd Pages"), 0, miniWidget()->markList(),
		   SLOT (markOdd()),
		   actionCollection(), "mark_odd") );

  statedep->insert
    ( new KAction (i18n("&Toggle Page Marks"), 0, miniWidget()->markList(),
		   SLOT (toggleMarks()),
		   actionCollection(), "toggle") );
  statedep->insert
    ( new KAction (i18n("&Remove Page Marks"), 0, miniWidget()->markList(),
		   SLOT (removeMarks()),
		   actionCollection(), "remove") );
      
  
  //TODO -- disable entry if there aren't any page names
  //Settings menu

  watchaction =
    new KToggleAction (i18n("&Watch File"), 0, this, SLOT (slotWatchFile()),
		       actionCollection(), "watch_file");
  statedep->insert(watchaction);

  showmarklist =
    new KToggleAction (i18n("Show &Page List"), 0, this,
		       SLOT (slotShowMarkList()),
		       actionCollection(), "show_page_list");
  statedep->insert(showmarklist);
  connect ( miniWidget(), SIGNAL (markListShown (bool)),
	    this, SLOT (slotMarkListShown(bool)) );
  fancyAct =
    new KToggleAction (i18n("Show Page Names"),
		       0, this, SLOT(slotFancy()), actionCollection(),
		       "fancy_page_labels");
  fancyAct->setChecked (w->areFancyPageLabelsEnabled ());
  statedep->insert(fancyAct);


  new KAction (i18n("Configure &Interpreter"), 0, miniWidget(),
	       SLOT (configureGhostscript()),
	       actionCollection(), "config_interpreter");

  
  //TODO
  // progress bars for conversions (see loadingProgress())
  
  new KAction (i18n("Configure &Interpreter"), 0, miniWidget(),
	       SLOT (configureGhostscript()),
	       actionCollection(), "config_interpreter");
  
  watch = 0L;
  
  m_extension = new KGVBrowserExtension( this );
  
  setXMLFile( "kgv_part.rc" );
  connect ( miniWidget(), SIGNAL (newPageShown()),
	    this, SLOT (slotNewPage()) );

  readSettings();

  enableStateDepActions( false ); // no document to apply the actions to, yet
}

KGVPart::~KGVPart()
{
  writeSettings();
}

void
KGVPart::writeSettings()
{
  KConfig *config = KGVFactory::instance()->config();

  config->setGroup( "General" );
  config->writeEntry ("WatchFile", watchaction->isChecked());
  config->writeEntry ("ShowPageList", showmarklist->isChecked());
  config->writeEntry ("ShowPageNames", fancyAct->isChecked());

  miniWidget()->writeSettings();

  config->sync();
}

void
KGVPart::readSettings()
{
  KConfig *config = KGVFactory::instance()->config();

  config->setGroup( "General" );

  watchaction->setChecked (config->readBoolEntry ("WatchFile", false));
  showmarklist->setChecked (config->readBoolEntry ("ShowPageList", true));
  fancyAct->setChecked (config->readBoolEntry ("ShowPageNames", false));

  slotWatchFile();
}

void
KGVPart::startWatching()
{
  //NOTE: isFileOpen() will (appropriately) return false if the viewer is
  // viewing data from stdin
  if (!miniWidget()->isFileOpen())
    return;

  if (watch!=0L)
    delete watch;

  watch = new QTimer;

  connect ( watch, SIGNAL (timeout()),
            miniWidget(),
	    SLOT (redisplayChanged ()) );
  connect ( miniWidget(), SIGNAL (redisplayChangedFailed()),
            this,
	    SLOT (slotCancelWatch()) );

  watch->start (500);
}

void
KGVPart::slotCancelWatch()
{
  //Every time kgv_miniwidget gets a signal from the watch QTimer
  // it tries to open a file.  If it files it posts a non-modal error
  // message, and then another, and another, and ... unless we
  // do this:
  printf ("SCW\n");
  stopWatching();
  watchaction->setChecked (false);
}
  
void
KGVPart::stopWatching()
{
  printf ("sw\n");
  if (watch!=0L)
    {
      delete watch;
      watch=0L;
    }
}

void
KGVPart::slotWatchFile()
{
  if (watchaction->isChecked())
    startWatching();
  else
    stopWatching();
}


void
KGVPart::enableStateDepActions( bool enable )
{
  int count = statedep->count();
  for ( int i = 0; i < count; i++ )
    statedep->action( i )->setEnabled( enable );

  if (true)
    media->setItems (miniWidget()->sizeList());
}

bool
KGVPart::openStdin()
{
  QString s ("-");
  bool b = w->openFile(s);
  if (b)
    enableStateDepActions( true );  //do we want eAA (b) ?
  return b;
}

bool
KGVPart::openFile()
{
  bool b = w->openFile(m_file);
  if (b)
    {
      enableStateDepActions( true );
      slotWatchFile();
    }

  return b;
}

void
KGVPart::slotFancy()
{
  w->enableFancyPageLabels
    (fancyAct->isChecked());
}

void
KGVPart::slotNewPage()
{
  media->setCurrentItem (miniWidget()->getSize()-1);
  orientation->setCurrentItem (miniWidget()->getOrientation()-1);
  //TODO -- zoom
}
void
KGVPart::slotOrientation(int id)
{
  miniWidget()->setOrientation (id+1);
}

void
KGVPart::slotMedia(int id)
{
  miniWidget()->setSize (id+1);
}

void
KGVPart::slotShowMarkList()
{
  miniWidget()->showMarkList (showmarklist->isChecked());
}

void
KGVPart::slotMarkListShown(bool yes)
{
  showmarklist->setChecked (yes);
}








KGVBrowserExtension::KGVBrowserExtension( KGVPart *parent ) :
  KParts::BrowserExtension( parent, "KGVBrowserExtension" )
{
}

void
KGVBrowserExtension::print()
{
  ((KGVPart *)parent())->miniWidget()->print();
}



#include "kgv_view.moc"

