#include "main.h"
#include "main.moc"
#include <stdio.h>
#include <math.h>
#include <qslider.h>
#include <kaction.h>
#include <kmessagebox.h>
#include <audiomanager.h>
#include <iostream>

using namespace Arts;

class VScale : public QFrame {
protected:
	KLed *leds[12];

public:
	VScale(QWidget *parent, bool blueState = false) : QFrame(parent) {
		setBackgroundColor(black);
//		show();
		QBoxLayout * l = new QVBoxLayout( this );
		l->setAutoAdd(TRUE);
		for(int i=0;i<12;i++)
		{
			QColor c = red;
			if(i>=2) c = yellow;
			if(i>=5) c = green;
			if(blueState) c = blue;
			{
				// put each led in its own frame, since it seems to be broken
				QFrame *lframe = new QFrame(this);
				QBoxLayout *lfl = new QVBoxLayout( lframe );
				lfl->setAutoAdd(TRUE);
				leds[i] =
					new KLed(c,KLed::Off, KLed::Sunken, KLed::Circular,lframe);
				//lfl->activate();
			}
		}
	//	l->activate();
	}

	void setValue(float f)
	{
		//printf("value %f\n",f);
		for(int i=11;i>=0;i--)
		{
			if(f > 0.06) leds[i]->setState(KLed::On);
			else		 leds[i]->setState(KLed::Off);
			f /= 1.25;
		}
	}
};

void FFTScopeView::updateScopeData()
{
	if(scopeData) delete scopeData;
	scopeData = scopefx.scope();
}

void FFTScopeView::updateScope()
{
	updateScopeData();

	for(unsigned int i=0;i<scopeData->size();i++)
	{
		scopeDraw[i] /= 1.25;
		if((*scopeData)[i] > scopeDraw[i]) scopeDraw[i] = (*scopeData)[i];
		scopeScales[i]->setValue(scopeDraw[i]);
	}
}

void FFTScopeView::closeEvent(QCloseEvent *e)
{
	e->accept();
	cout << "close()" << endl;
	emit closed();
}

FFTScopeView::FFTScopeView(SimpleSoundServer server)
	: server(server), scopeData(0)
{
	/*
	   create a stereo fft scope on the server and push it into the
	   effect chain
	*/
	{
		scopefx = DynamicCast(server.createObject("Arts::StereoFFTScope"));
		assert(!scopefx.isNull());
		scopefx.start();

		// put it into the effect chain
		effectID = server.outstack().insertBottom(scopefx,"FFT Scope");
	}

	updateScopeData();
	QBoxLayout * l = new QHBoxLayout( this );
	l->setAutoAdd(TRUE);

	for(unsigned int i=0;i<scopeData->size();i++)
	{
		scopeScales.push_back(new VScale(this,true));
		scopeDraw.push_back(0.0);
	}

	l->activate();
	show();
	setFixedSize(300,200);

	QTimer *updatetimer = new QTimer(this);
	updatetimer->start(100);
	connect(updatetimer,SIGNAL(timeout()),this,SLOT(updateScope()));
}

FFTScopeView::~FFTScopeView()
{
	// remove effect
	server.outstack().remove(effectID);
}

VControl::VControl(QWidget *parent) : QFrame(parent), scopeView(0),
	audioManager(0), server(Reference("global:Arts_SimpleSoundServer"))
{
	setMinimumSize(80,300);
   	if(server.isNull())
   	{
		KMessageBox::error( 0, i18n("Connection to the soundserver failed - make sure that artsd is really running."));
		exit(1);
   	}
	show();
	QBoxLayout * l = new QHBoxLayout( this );
	//l->setAutoAdd(TRUE);
	left = new VScale(this);
	l->addWidget(left);
	right = new VScale(this);
	l->addWidget(right);

	// 0   => 4.0
	// 200 => 2.0
	// 400 => 1.0
	// 600 => 0.5
	// 800 => 0.25
	// 1000 => 0.125
	// 1200 => 0.0 (forced)

	QSlider *slider;
	slider = new QSlider(0,1200,100,400,QSlider::Vertical,this,"volumeslider");
	connect(slider, SIGNAL(valueChanged(int)), this, SLOT(setVolume(int)));
	slider->setTickmarks( QSlider::Right );
	l->addWidget(slider);
	l->activate();
	/*
	   create a stereo volume control and push it into the
	   effect chain
	*/
	{
		// create a new stereo volume control object on the server
		svc = DynamicCast(server.createObject("Arts::StereoVolumeControl"));
		assert(!svc.isNull());
		svc.start();

		// put it into the effect chain
		effectID = server.outstack().insertBottom(svc,"Volume Control");
	}

	QTimer *updatescalestimer = new QTimer(this);
	updatescalestimer->start(100);
	connect(updatescalestimer,SIGNAL(timeout()),this,SLOT(updateScales()));
}

VControl::~VControl()
{
	removeScopeView();
	// remove effect from effect chain
	server.outstack().remove(effectID);
}

void VControl::updateScales()
{
	left->setValue(svc.currentVolumeLeft());
	right->setValue(svc.currentVolumeRight());
}

void VControl::setVolume(int vol)
{
	float volume = pow(2.0,(400.0 - vol)/200.0);
	if(vol == 1200) volume = 0.0;

	svc.scaleFactor(volume);
}

void VControl::showScopeView()
{
	if(!scopeView)
	{
		scopeView = new FFTScopeView(server);
		connect(scopeView,SIGNAL(closed()),this,SLOT(removeScopeView()));
	}
}

void VControl::showAudioManager()
{
	if(!audioManager)
	{
		cout << "new GUI_AUDIO_MANAGER" << endl;
		audioManager = new Gui_AUDIO_MANAGER();
		connect(audioManager,SIGNAL(closed()),this,SLOT(removeAudioManager()));
	}
}

void VControl::removeAudioManager()
{
	if(audioManager) delete audioManager;
	audioManager = 0;
}

void VControl::removeScopeView()
{
	if(scopeView) delete scopeView;
	scopeView = 0;
}

class MainWindow : public KMainWindow {
protected:
	VControl *vc;

	void closeEvent(QCloseEvent *)
	{
		// this is to prevent some collision between Qt and the MCOP I/O
		// system ... needs more investigations
		delete vc;
		cout << "hack: exit" << endl;
		exit(1);
	}
public:
	MainWindow() : KMainWindow(0)
	{
		vc = new VControl(this);
		setCentralWidget(vc);
    	(void)new KAction(i18n("View &FFT Scope"), 0, vc, SLOT(showScopeView()),
					actionCollection(), "view_fft_scope");
    	(void)new KAction(i18n("View &Audio Manager"), 0, vc,
			SLOT(showAudioManager()), actionCollection(), "view_audio_manager");
		createGUI("artscontrol.rc");
		resize(20,300);
		show();
	}
};

int main(int argc, char **argv)
{
	KAboutData aboutData( "artscontrol", I18N_NOOP("aRts control"),
		"0.1", "Control tool for the aRts server",
			KAboutData::License_GPL, "(c) 2000 Stefan Westerfeld");

  	KCmdLineArgs::init( argc, argv, &aboutData );

	// setup mcop communication
	//QIOManager qiomanager;
	Dispatcher dispatcher; //(&qiomanager);

	KApplication app(argc, argv);
	new MainWindow;
	return app.exec();
}
