/*
   Copyright (c) 2000 Stefan Schimanski <1Stein@gmx.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <string.h>
#include <qfile.h>
#include <kdebug.h>
#include "id3.h"


#define TIMER_DELAY 100
#define ITEMS_PER_TICK 10


void stripSpaces( char *s, int len )
{
    int p;
    for ( p=len-2; p>=0 && s[p]==' '; p-- );
    s[p+1] = 0;
}


bool ID3Tag::open( const KURL &url )
{
    if ( !url.isLocalFile() ) return false;

    QFile file( url.directory(false, false)+url.fileName() );
    if ( !file.open(IO_ReadOnly) ) {
        return false;
    }

    file.at( file.size()-128 );
    file.readBlock( (char*)&_id3tag, sizeof(_id3tag) );

    if ( strncmp(_id3tag.magic, "TAG", 3)!=0 ) {
        memset( &_id3tag, 0, sizeof(_id3tag) );
        return false;
    }

    stripSpaces( _id3tag.songname, 30 );
    stripSpaces( _id3tag.artist, 30 );
    stripSpaces( _id3tag.album, 30 );
    stripSpaces( _id3tag.note, 30 );

    return true;
}


/*****************************************************************/


ID3Reader::ID3Reader( QObject *parent, const char *name )
    : QObject( parent, name )
{
    connect( &_timer, SIGNAL(timeout()), this, SLOT(timeout()) );
    _queue.setAutoDelete( TRUE );
}


ID3Reader::~ID3Reader()
{
}


void ID3Reader::queue( const KURL &url, void *user )
{
    ID3Job *job = new ID3Job;
    job->url = url;
    job->user = user;
    _queue.append( job );

    if ( _queue.count()==1 ) {
        _timer.start( TIMER_DELAY, TRUE );
    }
}


void ID3Reader::dequeue( void *user )
{
    for ( ID3Job *job=_queue.first(); job!=0; job=_queue.next() ) {
        if ( job->user==user )
            _queue.remove( job );
    }
}


void ID3Reader::stop()
{
    _queue.clear();
    _timer.stop();
}


void ID3Reader::timeout()
{
    QList<ID3Tag> tags;
    tags.setAutoDelete( true );

    int i = ITEMS_PER_TICK;
    ID3Job *job = _queue.first();
    for ( ; i>0 && job!=0; i-- ) {
        ID3Tag *tag = new ID3Tag( job->user );
        if ( tag->open(job->url) )
            tags.append( tag );
        else
            delete tag;

        _queue.remove( job );
        job = _queue.next();
    }

    if ( tags.count()>0 )
        emit readID3( tags );


    if ( _queue.count()>0 )
        _timer.start( TIMER_DELAY, TRUE );
}

#include "id3.moc"
