/*
   Copyright (c) 2000 Stefan Schimanski <1Stein@gmx.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <qlayout.h>
#include <klocale.h>
#include <kdebug.h>
#include <qstrlist.h>
#include <qdragobject.h>
#include <kapp.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kpopupmenu.h>
#include <qnamespace.h>
#include <kfiledialog.h>
#include <kmimemagic.h>
#include <qtextstream.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include <kio/jobclasses.h>
#include <kio/global.h>
#include <kdirlister.h>
#include <qlist.h>
#include <kfileitem.h>
#include <qpalette.h>
#include <soundserver.h>
#include <kstddirs.h>

#include "playlistwin.h"
#include "id3.h"


/*************************************************************/

QString MediaFileItem::text( int col ) const
{
    switch ( col ) {
    case 0: return title();
        /*case 1: {
        int l = length();
        if ( l>0 )
            return QString("%1:%2").arg( l/60, l%60 );
        else
            return QString();
            }*/

    case 1: return url().prettyURL();
    default: return QString();
    }
}


void MediaFileItem::paintCell( QPainter *p, const QColorGroup & cg,
                               int column, int width, int alignment )
{
    QColorGroup mycg( cg );
    if ( _highlighted ) {
        QColor back(mycg.text());
        QColor fore(mycg.background());
        mycg.setColor( QColorGroup::Base, back );
        mycg.setColor( QColorGroup::Text, fore );
    }

    QListViewItem::paintCell( p, mycg, column, width, alignment );
}


/*************************************************************/

MediaListView::MediaListView(QWidget *parent, const char *name)
    : KListView( parent, name ), _active(0)
{
    addColumn( i18n("Title") );
// addColumn( i18n("Length") );
    addColumn( i18n("Filename") );

    setItemsMovable( true );
    setDragEnabled( true );
    setDropVisualizer( true );
    setSelectionModeExt( KListView::Extended );
    setAllColumnsShowFocus( true );
    setRootIsDecorated( false );
    setAcceptDrops(true);
    setSorting(-1);

    connect( this, SIGNAL(dropped(QDropEvent*,QListViewItem*,QListViewItem*)),
             SLOT(droppedFile(QDropEvent*,QListViewItem*,QListViewItem*)) );
    connect( this, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(playItem(QListViewItem*)) );
    connect( this, SIGNAL(contextMenu(KListView*,QListViewItem*,const QPoint&)),
             SLOT(menu(KListView*,QListViewItem*,const QPoint&)) );
    connect( &_ID3Reader, SIGNAL(readID3(const QList<ID3Tag> &)),
             this, SLOT(updateID3Tag(const QList<ID3Tag> &)) );
}


void MediaListView::droppedFile(QDropEvent *ev, QListViewItem * /*parent*/,
                                QListViewItem *after)
{
    QStrList urls;
    if ( !QUriDrag::decode(ev, urls) ) return;

    for ( char *url=urls.first(); url!=0; url=urls.next() )
       after=addNewItem( url, after );

    triggerUpdate();
}


void MediaListView::menu(KListView* /*l*/, QListViewItem* i, const QPoint& p)
{
    if ( i ) {
        KPopupMenu menu( this );
        menu.insertItem( i18n("Play"), this, SLOT(playSelected()) ) ;
        menu.insertItem( i18n("Remove"), this, SLOT(removeSelected()) ) ;
        menu.exec( p );
    }
}

bool MediaListView::acceptDrag(QDropEvent *event) const
{
    return QUriDrag::canDecode(event) || KListView::acceptDrag(event);
}

void MediaListView::playItem( QListViewItem *item )
{
    setActive( item );

    triggerUpdate();

    if ( active() )
        emit play( active() );
}

void MediaListView::playSelected()
{
    setActive( currentItem() );

    triggerUpdate();

    if ( active() )
        emit play( active() );
}

void MediaListView::keyPressEvent ( QKeyEvent *ev )
{
    if ( ev->key()==Key_Delete )
        removeSelected();
    KListView::keyPressEvent( ev );
}

void MediaListView::removeSelected()
{
    QList<QListViewItem> trash;
    trash.setAutoDelete( true );

    bool activeDeleted = false;

    for ( QListViewItem *i=firstChild(); i!=0; i=i->nextSibling() ) {
        if ( isSelected(i) ) {
            trash.append( i );
            _ID3Reader.dequeue( i );

            if ( _active==i )
                activeDeleted = true;
        }
    }

    trash.clear();

    if ( activeDeleted )
    {
        _active = 0;
        emit stop();
    }

    triggerUpdate();
}


MediaFileItem *MediaListView::next( bool reverse, bool loop )
{
    if ( active() ) {
        if ( reverse )
            setActive( active()->itemAbove() );
        else
            setActive( active()->itemBelow() );

        if ( loop && !active() )
            start( reverse );
    }

    triggerUpdate();

    return active();
}


MediaFileItem *MediaListView::start( bool reverse )
{
    if ( reverse )
        setActive( lastChild() );
    else
        setActive( firstChild() );

    triggerUpdate();

    return active();
}


void MediaListView::reset()
{
    setActive( 0 );
    triggerUpdate();
}


MediaFileItem *MediaListView::shuffle()
{
    if ( active() ) {
        int num = rand()%childCount();

        QListViewItem *cur = firstChild();
        for ( ; cur!=0 && num>0 ; num-- )
            cur = cur->nextSibling();

        setActive( cur );
    }

    triggerUpdate();

    return active();
}


MediaFileItem * MediaListView::addNewItem( const KURL &url, QListViewItem *after )
{
    MediaFileItem *item;
    if ( after )
        item = new MediaFileItem( this, after );
    else
        item = new MediaFileItem( this );

    item->setURL( url );

    if ( !url.isMalformed() && url.isLocalFile() )
        _ID3Reader.queue( url, item );

    return item;
}


void MediaListView::updateID3Tag( const QList<ID3Tag> &tags )
{
    QListIterator<ID3Tag> it( tags );
    for ( ; it.current()!=0; ++it ) {
        MediaFileItem *item = static_cast<MediaFileItem*>(it.current()->user());
        if ( item )
            item->setTitle( it.current()->artist() + " - " + it.current()->songName() );
    }

    triggerUpdate();
}


void MediaListView::setActive( QListViewItem *item )
{
    if ( item!=_active ) {
        // dehighlight old actuve item
        if ( _active ) static_cast<MediaFileItem*>(_active)->highlight( false );
        // set new active item
        _active = item;
        if ( _active ) {
            // highlight active item
            static_cast<MediaFileItem*>(_active)->highlight( true );

            // update position information
            int p=1;
            QListViewItem *i;
            for ( i=firstChild(); i!=0 && i!=_active; i=i->itemBelow() )
                p++;

            if (i==_active)
                static_cast<MediaFileItem*>(_active)->setPos( p );
            else
                static_cast<MediaFileItem*>(_active)->setPos( 0 );

            // scroll to new active item
            ensureItemVisible( _active );
        }
    }
}


void MediaListView::stopLoading()
{
    _ID3Reader.stop();
}


/*********************************************************************************/


MediaManager::MediaManager( const QString &title, const char *name )
    : MediaManagerBase( name ), _dirLister( 0 ), _title( title )
{
    setCaption( _title );
    initFileExtensions();

    // create directory lister
    _dirLister = new KDirLister;
    connect( _dirLister, SIGNAL(completed()), this, SLOT(dirListerCompleted()) );
    connect( _dirLister, SIGNAL(newItems(const KFileItemList&)),
             this, SLOT(dirListerNewItems(const KFileItemList&)) );
    _dirQueue.setAutoDelete( true );

    // create list view
    _list = new MediaListView( this, "MediaListView" );
    setCentralWidget( _list );

    connect( _list, SIGNAL(stop()), this, SIGNAL(stop()) );
    connect( _list, SIGNAL(play(MediaFile*)), this, SIGNAL(play(MediaFile*)) );

    // init config
    KConfig *config = kapp->config();
    config->setGroup(0);

    _baseURL = KURL(config->readEntry( "BaseURL" ));

    // create actions
    KStdAction::openNew( this, SLOT(openNew()), actionCollection() );
    KStdAction::open( this, SLOT(open()), actionCollection() );
    (void)new KAction( i18n("Remove"), "editdelete", 0, _list, SLOT(removeSelected()),
                       actionCollection(), "remove" );
    KStdAction::save( this, SLOT(save()), actionCollection() );
    KStdAction::saveAs( this, SLOT(saveAs()), actionCollection() );
    (void)new KAction( i18n("Add &Files"), "queue", Key_Insert, this, SLOT(addFiles()),
                       actionCollection(), "add_files" );
    (void)new KAction( i18n("Add &Directory"), "folder", 0, this, SLOT(addDir()),
                       actionCollection(), "add_dir" );
    KAction *stop = new KAction( i18n( "&Stop Loading" ), "stop", Key_Escape, this, SLOT(stopLoading()),
                                 actionCollection(), "stop" );
    stop->setEnabled( false );

    createGUI( "kaimanui.rc" );
}


MediaManager::~MediaManager()
{
    // save config
    if ( fileName().isMalformed() )
        savePlaylist( kapp->dirs()->saveLocation("data", "kaiman/")
                      + "default.m3u" );
    else
        savePlaylist( fileName() );

    KConfig *config = kapp->config();
    config->setGroup(0);
    config->writeEntry( "BaseURL", _baseURL.url() );
    config->sync();

    // delete running directory lister
    if ( _dirLister ) delete _dirLister;
}


void MediaManager::initFileExtensions()
{
    Arts::TraderQuery query;
    vector<Arts::TraderOffer> *results = query.query();

    for ( vector<Arts::TraderOffer>::iterator it=results->begin();
          it!=results->end(); it++ ) {

        vector<string> *exts = (*it).getProperty( "Extension" );

        for ( vector<string>::iterator ext=exts->begin(); ext!=exts->end(); ext++ )
            _fileExtensions.append( (*ext).c_str() );

        delete exts;
    }

    delete results;
}


MediaFile *MediaManager::next( bool reverse, bool loop )
{
    return _list->next( reverse, loop );
}


MediaFile *MediaManager::start( bool reverse )
{
    return _list->start( reverse );
}


void MediaManager::reset()
{
    _list->reset();
}


MediaFile *MediaManager::active()
{
    return _list->active();
}


MediaFile *MediaManager::shuffle()
{
    return _list->shuffle();
}


int MediaManager::count()
{
    return _list->childCount();
}


int MediaManager::position()
{
    MediaFileItem *media = _list->active();
    if ( media ) {
        return media->pos();
    } else
        return 0;
}

void MediaManager::clear()
{
    stopLoading();
    _list->stopLoading();
    _list->clear();
    _fileName = QString();
    setCaption( _title );
    emit stop();
}


void MediaManager::addPlaylist( const KURL &url, bool update )
{
    QString local;
    int errors = 0;
    if( KIO::NetAccess::download( url, local ) ) {
        QFile reader( local );
        if ( reader.open( IO_ReadOnly ) ) {
            QTextStream stream( &reader );

            QListViewItem *last = _list->lastItem();
            while ( !stream.eof() && errors<16 ) {
                QString file = stream.readLine();
                if ( !file.isNull() ) {
                    KURL url( file );
                    if ( !url.isMalformed() ) {
                        MediaFileItem *item;
                        last = item = _list->addNewItem( url, last );
                    } else
                        errors++;
                }
            }
        }

        KIO::NetAccess::removeTempFile( local );
        if ( update ) {
            _list->triggerUpdate();
            if ( errors>=16 )
                KMessageBox::error( this, i18n("Invalid filename: %1").arg(url.url()) );
        }
    }
}


void MediaManager::loadPlaylist( const KURL &url )
{
    clear();
    addPlaylist( url );
    _fileName = url;
    setCaption( fileName().fileName() );
}


QString MediaManager::allFilesFilter()
{
    // create file filter
    QString allFiles;
    for ( unsigned idx=0; idx<_fileExtensions.count(); idx++ ) {
        if ( allFiles.length()>0 )
            allFiles += " ";

        allFiles += "*." + _fileExtensions[idx];
    }

    return allFiles;
}


void MediaManager::addFiles()
{
    // create file filter
    QString filter;
    filter += allFilesFilter() + "|" + i18n("All Media Files");
    filter += "\n*|" + i18n("All Files");
    filter += "\n*.m3u|" + i18n("m3u Playlists");

    for ( unsigned idx=0; idx<_fileExtensions.count(); idx++ ) {
        QString fileType = "*." + _fileExtensions[idx] + "|" +
                           i18n("%1 Files").arg(_fileExtensions[idx]);
        if ( filter.length()>0 ) filter += "\n";
        filter += fileType;
    }

    // open file dialog
    KURL::List urls = KFileDialog::getOpenURLs( _baseURL.directory(), filter,
                                                this, i18n("Add Files") );

    // add selected files
    if ( urls.count()>0 ) {
        _baseURL = urls.first().path();
        QListViewItem *last = _list->lastChild();

        for ( KURL::List::Iterator url=urls.begin(); url!=urls.end(); ++url ) {

            if ( !(*url).isMalformed() ) {
                if ( (*url).fileName().right(4).lower()==".m3u" )
                    addPlaylist( *url );
                else
                    last = _list->addNewItem( *url, last );
            }
        }
    }
}


void MediaManager::addDirectory( const KURL &url )
{
    if ( !url.isMalformed() ) {
        if ( _addedDirs.contains( url.url() )==0 ) {
            _addedDirs += url.url();

            if ( _dirLister->isFinished() ) {
                _dirLister->setNameFilter( allFilesFilter() );
                _dirLister->openURL( url, false );
            } else
                _dirQueue.enqueue( new KURL(url) );

            KAction *stop = actionCollection()->action( "stop" );
            if ( stop ) stop->setEnabled( true );
        }
    }
}


void MediaManager::addDir()
{
    QString dir = KFileDialog::getExistingDirectory( _baseURL.directory(),
                                                     this, i18n("Add Directory") );
    if ( dir.length()>0 ) {
        addDirectory( KURL(dir) );
    }
}


void MediaManager::dirListerNewItems( const KFileItemList & items )
{
    KFileItemListIterator it( items );
    for (; it.current(); ++it ) {
        if ( it.current()->isDir() ) {
            if ( it.current()->isLink() )
                addDirectory( it.current()->linkDest() );
            else
                addDirectory( it.current()->url() );
        } else {
            addURL( it.current()->url(), false );
            _list->triggerUpdate();
        }
    }
}


void MediaManager::dirListerCompleted()
{
    _dirLister->stop();

    if ( _dirQueue.isEmpty() ) {
        KAction *stop = actionCollection()->action( "stop" );
        if ( stop ) stop->setEnabled( false );
        _addedDirs.clear();
    } else {
        KURL url( *_dirQueue.head() );
        _dirLister->setNameFilter( allFilesFilter() );
        _dirLister->openURL( url, false );
        _dirQueue.dequeue();
    }
}


bool MediaManager::savePlaylist( const KURL &url )
{
    QString local( "/tmp/kaimanplaylist-" + url.fileName() );
    QFile saver( local );
    if ( !saver.open( IO_WriteOnly ) )
        return false;

    QTextStream stream( &saver );

    for ( MediaFileItem *item=static_cast<MediaFileItem*>(_list->firstChild());
          item!=0;
          item=static_cast<MediaFileItem*>(item->itemBelow()) ) {
        if ( item->url().isLocalFile() ) {
            stream << item->url().directory( false, true )
                   << item->url().fileName( true )
                   << endl;
        } else
            stream << item->url().prettyURL() << '\n';
    }

    saver.close();

    if ( !KIO::NetAccess::upload( local, url ) ) {
        saver.remove();
        return false;
    } else {
        saver.remove();
        return true;
    }
}


void MediaManager::addURL( const KURL &url, bool update )
{
    if ( url.fileName().right(4).lower()==".m3u" )
        addPlaylist( url, update );
    else {
        if ( !url.isMalformed() ) {
            _list->addNewItem( url, _list->lastChild() );
            if ( update ) _list->triggerUpdate();
        } else
            if ( url.url().length()!=0 && update )
                KMessageBox::error( this, i18n("Invalid filename: %1").arg(url.url()) );
    }
}


void MediaManager::open()
{
    KURL url = KFileDialog::getOpenURL( _baseURL.directory(),
                                        "*.m3u *.M3U|" + i18n("m3u Playlists") +
                                        "\n*|" + i18n("All Files"),
                                        this, i18n("Open Playlist") );
    if ( !url.isMalformed() ) {
        loadPlaylist( url );
    } else
        if ( url.url().length()!=0 )
            KMessageBox::error( this, i18n("Invalid filename: %1").arg(url.url()) );
}


void MediaManager::save()
{
    if ( _fileName.isMalformed() ) {
        saveAs();
    } else {
         if ( !savePlaylist( _fileName ) )
            KMessageBox::error( this, i18n("Can't save playlist.") );
    }
}


void MediaManager::saveAs()
{
    KURL url = KFileDialog::getSaveURL( _baseURL.directory(),
                                        "*.m3u *.M3U|" + i18n("m3u Playlists") +
                                        "\n*|" + i18n("All Files"),
                                        this, i18n("Save Playlist") ) ;
    if ( !url.isMalformed() ) {
        if ( savePlaylist( url ) ) {
            _fileName = url;
            setCaption( fileName().fileName() );
        } else
            KMessageBox::error( this, i18n("Can't save playlist.") );
    } else
        if ( url.url().length()!=0 )
            KMessageBox::error( this, i18n("Invalid filename: %1").arg(url.url()) );
}


void MediaManager::openNew()
{
    clear();
    _fileName = QString();
    setCaption( _title );
}


void MediaManager::stopLoading()
{
    _dirLister->stop();
    _dirQueue.clear();
    _addedDirs.clear();

    KAction *stop = actionCollection()->action( "stop" );
    if ( stop ) stop->setEnabled( false );
}

#include "playlistwin.moc"
