/*
  surface wrapper for X11 Window
  Copyright (C) 2000  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "x11Surface.h"


const char *ERR_XI_STR[] = {
  "X initialisation OK!",
  "No Shared Memory available", 
  "cannot open Display", 
  "bad color depth",
  "can't create Window", 
  "can't alloc memory for virtual screen",
  "cannot create XImage",
  "can't alloc memory for Shared memory segment info",
  "cannot create Shared Memory XImage",
  "Shared memory segment info error",
  "Shared memory virtual screen allocation failed",
  "cannot attach Shared Memory segment to display"
};


#ifndef KDE_USE_FINAL
static int dummy(Display* , XErrorEvent*) {
  cout << "received x11 error!"<<endl;
  return true;
}
#endif

X11Surface::X11Surface() {

  //XInitThreads();

  xWindow=(XWindow *)malloc(sizeof(XWindow));
  xWindow->lOpen=false;

  imageMode=_IMAGE_NONE;
  imageSize=_SIZE_NONE;

  imageCurrent=NULL;
  xWindow->lOpen=false;


  xWindow->display=XOpenDisplay(NULL);
  XFlush(xWindow->display);
  xWindow->redMask=0;
  xWindow->greenMask=0;
  xWindow->blueMask=0;
  lXVAllow=true;
  
}


X11Surface::~X11Surface() {
  close();
  XCloseDisplay(xWindow->display);
  delete xWindow;
}


int X11Surface::getHeight() {
  return xWindow->height;
}


int X11Surface::getWidth() {
  return xWindow->width;
}

int X11Surface::isOpen() {
  return xWindow->lOpen;
}


int X11Surface::open(int width, int height,const char *title) {

  close();
  xWindow->width=width;
  xWindow->height=height;


  if(!xWindow->display) {
    printf("\nX initialisation error:\n *** %s\n",ERR_XI_STR[ERR_XI_DISPLAY]);
    printf("check ipcs and delete resources with ipcrm\n");
    exit(0);
  }

  xWindow->screennum=DefaultScreen(xWindow->display);
  xWindow->screenptr=DefaultScreenOfDisplay(xWindow->display);
  xWindow->visual=DefaultVisualOfScreen(xWindow->screenptr);
  xWindow->depth=DefaultDepth(xWindow->display,xWindow->screennum);

  switch(xWindow->depth) {
    case 8:
      xWindow->pixelsize=1;
      break;
    case 16:
      xWindow->pixelsize=2;
      break;
    case 24:
      xWindow->pixelsize=4;
      break;
    case 32:
      xWindow->pixelsize=4;
      break;
   default:
     cout << "unknown pixelsize for depth:"<<xWindow->depth<<endl;
     exit(0);
  }
  XSetWindowAttributes attributes;
  attributes.backing_store=NotUseful;

  xWindow->window=XCreateWindow(xWindow->display,
				RootWindowOfScreen(xWindow->screenptr),
				0,0,
				xWindow->width, 
				xWindow->height,0,
				xWindow->depth,
				InputOutput, xWindow->visual,
				CWBackingStore ,
				&attributes);
  
  if(!xWindow->window) {
    printf("\nX initialisation error:\n *** %s\n",ERR_XI_STR[ERR_XI_WINDOW]);
    printf("check ipcs and delete resources with ipcrm\n"); 
    return false;
  }

  WM_DELETE_WINDOW = XInternAtom(xWindow->display, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(xWindow->display, xWindow->window, &WM_DELETE_WINDOW, 1);

  XSetErrorHandler(dummy);

  XStoreName(xWindow->display,xWindow->window,title);
  XSelectInput(xWindow->display,xWindow->window,
	       ExposureMask|KeyPressMask|KeyReleaseMask|ButtonPressMask);
  xWindow->gc=XCreateGC(xWindow->display,xWindow->window,0,NULL);
  XMapRaised(xWindow->display,xWindow->window);
  XFlush(xWindow->display);

  if (xWindow->depth >= 16) {
    initColorDisplay(xWindow);

  } else {
    // depth is <= 8
    // allocate memory for dithertables
    // gets the rgb masks
    initColorDisplay(xWindow);
    // create 8 bit dithertables 
    // create private colormap
    initSimpleDisplay(xWindow);
   
  }

  xWindow->palette=NULL;
  xWindow->screensize=xWindow->height*xWindow->width*xWindow->pixelsize;
  xWindow->lOpen=true;

  imageDeskX11Out=new ImageDeskX11(xWindow);
  imageFullDGA=new ImageDGAFull(xWindow);
  imageDeskXV=new ImageXVDesk(xWindow);


  return true;
}


int X11Surface::close() {
  if (isOpen()==false) {
    return true;
  }
  closeImage();
  delete imageDeskX11Out;
  delete imageFullDGA;
  delete imageDeskXV;

  XFreeGC(xWindow->display,xWindow->gc);
  XDestroyWindow(xWindow->display,xWindow->window);

  xWindow->lOpen=false;


  return true;
}



int X11Surface::openImage(int imageSize,int imageMode) {
  if (this->imageMode != _IMAGE_NONE) {
    cout << "bad open error X11Surface::openImage"<<endl;
    exit(0);
  }
  this->imageMode=imageMode;
  imageCurrent=NULL;
  this->imageSize=imageSize;
  if (imageMode == _IMAGE_DESK) {
    if (imageDeskX11Out->support()) {
      imageCurrent=imageDeskX11Out;
    }
    // xv is better
    if (imageDeskXV->support()) {
      if (lXVAllow) {
	imageCurrent=imageDeskXV;
      }
    }
  }
  if (imageMode == _IMAGE_FULL) {
    if (imageFullDGA->support()) {
      imageCurrent=imageFullDGA;
    }
  }
  if (imageCurrent != NULL) {
    imageCurrent->openImage(imageSize);
  }
  return (imageCurrent != NULL);
}


int X11Surface::closeImage() {
  if (imageMode != _IMAGE_NONE) {
    imageCurrent=NULL;
    imageMode=_IMAGE_NONE;
    return true;
  }
  return true;
 
}


int X11Surface::dither(YUVPicture* pic) {
  if (imageCurrent != NULL) {
    imageCurrent->ditherImage(pic);
  }
  return true;
}


int X11Surface::putImage(YUVPicture* ) {
  if (imageCurrent != NULL) {
    imageCurrent->putImage();
  }
  return true; 
}


int X11Surface::getDepth() {
  return xWindow->depth;
}

int X11Surface::checkEvent(int* newMode,int* newSize) {
  XEvent event;

  if (isOpen()==false) {
    return false;
  }
  // check if we forward the call to the FULLSCREEN mode
  if (imageCurrent == imageFullDGA) {
    if (imageFullDGA->active() == false) {
      *newMode=_IMAGE_DESK;
      return true;
    }
    // we are still running DGA
    return true;
  }
  // normal X11 images use the X11 event queue
  
  if (XCheckTypedWindowEvent(xWindow->display,
			     xWindow->window,ButtonPress,&event)) {
    if (event.xbutton.button == Button1) {
      if (imageSize==_SIZE_NORMAL) {
	*newSize=_SIZE_DOUBLE;
      } else {
	*newSize=_SIZE_NORMAL;
      }
	
    }
    if (event.xbutton.button == Button3) {
      if (imageMode== _IMAGE_DESK) {
	if (imageFullDGA->support()==false) {
	  return false;
	}
	*newMode=_IMAGE_FULL;
      } else {
	*newMode=_IMAGE_DESK;
      }
    }
    return true;
  }
  return false;


}



void X11Surface::config(const char* key,
			const char* value, void* ) {
  if (strcmp(key,"xvAllow")==0) {
    lXVAllow=atoi(value);
  } 
}
 
