/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apsjob.C
 *
 * Description: Defines the ApsJobObject class methods.
 *
 */                   

#include "apscpp.h"
 
/*---------------------------------------------------------------------------
 * ApsJobObject()
 *
 * This constructor takes Aps_JobHandle type as argument.
 *
 * Parameters:
 *     jobHandle : Job handle.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsJobObject::ApsJobObject(Aps_JobHandle jobHandle)
                         : ApsObject((Aps_Handle)jobHandle)
{
}
 
/*---------------------------------------------------------------------------
 * ApsJobObject()
 *
 * Copy constructor.
 *
 * Parameters:
 *     apsJob : Reference to ApsFilterObject object.
 *
 *     Return: ApsResultObject.
 *
 */               
ApsJobObject::ApsJobObject(const ApsJobObject &apsJob)
{
    Aps_Handle apsHandle;
 
    apsHandle = apsJob.GetHandle();
    SetHandle(apsHandle);
}             
/*---------------------------------------------------------------------------
 * WriteJob()
 *
 * Writes data to be printed as part of this job.
 * This method internally calls Aps_JobWrite Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *   data : A pointer to the buffer containing data to be printed.
 *   size : The number of bytes to be printed from the buffer pointed to by
 *          data.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::WriteJob(void *data, int size)
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobWrite(jobHandle, data, size)));
}

/*---------------------------------------------------------------------------
 * WriteBlock()
 *
 * Writes a printer-specific block of data, including printer-specific 
 * commands to triggerthe settings that have been selected via the job 
 * attributes mechanism.status.
 * This method internally calls Aps_JobWriteBlock Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     blockType : An Aps_BlockType representing the type of block to be 
 *                 written.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::WriteBlock(Aps_BlockType blockType)
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobWriteBlock(jobHandle, blockType)));
}

/*---------------------------------------------------------------------------
 * GetFileDescriptor()
 *
 * Obtains a file descriptor to allow the application to write directly to 
 * this job without using WriteJob function.
 * This method internally calls Aps_JobGetFileDescriptor Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *     fileDescriptor : The address of an int to receive the file descriptor 
 *                      for writing to this job.
 * 
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::GetFileDescriptor(int *fileDescriptor)
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobGetFileDescriptor(jobHandle,
                                                          fileDescriptor)));
}

/*---------------------------------------------------------------------------
 * End()
 *
 * Called by the application to indicate that all data for this job has now 
 * been sent by the application and that the job now be scheduled for printing.
 * This method internally calls Aps_JobEnd Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     None.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::End()
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobEnd(jobHandle)));
}

/*---------------------------------------------------------------------------
 * Abort()
 *
 * Called by the application to terminate the job.
 * This method internally calls Aps_JobAbort Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     None.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::Abort()
{
    Aps_JobHandle jobHandle;

    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobAbort(jobHandle)));
}

/*---------------------------------------------------------------------------
 * Update()
 *
 * Updates a job object to reflect its current status.
 * This method internally calls Aps_JobUpdate Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     None.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::Update()
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobUpdate(jobHandle)));
}

/*---------------------------------------------------------------------------
 * GetStatus()
 *
 * Obtains status info for a job,indicating whether it is currently being 
 * spooled, waiting to be printed, being printed, completed, on hold for any 
 * reason, or cancelled for any reason.
 * This method internally calls Aps_JobGetStatus Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     status : A pointer to an Aps_JobStatus to receive an identifier 
 *              describing the phase this the job is currently in.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::GetStatus(Aps_JobStatus *status)
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobGetStatus(jobHandle, status)));
}

/*---------------------------------------------------------------------------
 * IssueCommand()
 *
 * Performs a single specific operation on a job.
 * This method internally calls Aps_JobIssueCommand Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     operationCode : One of the defined APS_OP_Jxxxxx commands described 
 *                     below.
 *     reserved : Optional parameters. Not used at the moment; can safely be 
 *                set to NULL.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::IssueCommand(Aps_OperationID operationCode,
                                                  void *reserved)
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobIssueCommand(jobHandle, operationCode, 
                                                           reserved)));
}

/*---------------------------------------------------------------------------
 * MakeQuickJobInfo()
 *
 * Returns a pointer to an Aps_QuickInfo structure containing a snapshot 
 * survey of a single jobs  current status.
 * This method internally calls Aps_JobMakeQuickJobInfo Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *    jobInfo : Pointer to a Aps_QuickJobInfo structure containing a snapshot 
 *              of the last stored status.  On failure, will be filled with NULL.
 *
 *     Return: ApsResultObject.
 *
 */        
ApsResultObject ApsJobObject::MakeQuickJobInfo(Aps_QuickJobInfo **jobInfo)
{
    Aps_JobHandle jobHandle;
 
    jobHandle = (Aps_JobHandle)GetHandle();
    return (ApsResultObject(Aps_JobMakeQuickJobInfo(jobHandle, jobInfo)));
}
