/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: attrcommon.c
 *
 * Description: Quick access to common attributes.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
 
#include "aps.h"
#include "apsinternal.h"
#include "jobattributes.h"
#include "attrprovider.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(attr)

/* Private function prototypes. */
static void AttrGetResFromString(Aps_Resolution *resInfo, const char *string);
static Aps_Result AttrFillPageSizeDetails(Aps_JobAttrHandle jobAttributes,
                                          Aps_PageSize *pageSize);
static Aps_Result AttrGetOptionsStrArray(Aps_JobAttrHandle jobAttributes,
                                         const char *attributeID,
                                         char ***optionsArray,
                                         int *numOptions);

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetLanguageLevel()
 *
 * Obtains the level of PostScript supported by this device.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             languageLevel - Receives the PS level supported, if any.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetPostScriptLevel(
                                        Aps_JobAttrHandle jobAttributes,
                                        Aps_PostScriptSupport *languageLevel)
{
    char *mainData;
    Aps_Result result;

    /* Check for reasonable parameter values. */
    if (languageLevel == NULL) {
        return APS_PS_LEVEL1;
    }

    /* Initialize output parameters to default values in case of failure. */
    *languageLevel = TRUE;

    /* Query the main data for this attribute. */
    result = Aps_AttrGetMainData(jobAttributes, "*LanguageLevel", &mainData);
    if (result == APS_NOT_FOUND) {
        /* If this attribute was not found, then PostScript is not suported. */
        *languageLevel = APS_PS_NOT_SUPPORTED;
        return APS_SUCCESS;
    } else if (result != APS_SUCCESS) {
        return result;
    }
    
    /* If we get to this point, then PostScript support is available, so 
     * indicate the PostScript level supported to the caller.
     */
    switch(atoi(mainData)) {
        case 1:
            *languageLevel = APS_PS_LEVEL1;
            break;
        case 2:
            *languageLevel = APS_PS_LEVEL2;
            break;
        case 3:
            *languageLevel = APS_PS_LEVEL3;
            break;
    }
    
    /* Dispose of temporary string returned by Aps_AttrGetMainData(). */
    Aps_ReleaseBuffer(mainData);
    
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickIsColorDevice()
 *
 * Determines whether or not this device supports color output.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             colorDevice   - Set to TRUE if this is a color-capable device.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickIsColorDevice(Aps_JobAttrHandle jobAttributes,
                                      int *colorDevice)
{
    char *mainData;
    Aps_Result result;

    /* Check for reasonable parameter values. */
    if (colorDevice == NULL) {
        return APS_INVALID_PARAM;
    }

    /* Initialize output parameters to default values in case of failure. */
    *colorDevice = TRUE;

    /* Query the main data for this attribute. */
    result = Aps_AttrGetMainData(jobAttributes, "*ColorDevice", &mainData);
    if (result == APS_NOT_FOUND) {
        /* If none of the attribute providers can supply information on
         * whether or not this printer is a color device, we assume that
         * it _IS_ color capable.
         */
        *colorDevice = TRUE;
        return APS_SUCCESS;
    } else if (result != APS_SUCCESS) {
        return result;
    }
    
    /* If we get to this point, then PostScript support is available, so 
     * indicate the PostScript level supported to the caller.
     */
    if (stricmp(mainData, "False") == 0) {
        *colorDevice = FALSE;
    } else {
        ASSERT(stricmp(mainData, "True") == 0);
        *colorDevice = TRUE;
    }
    
    /* Dispose of temporary string returned by Aps_AttrGetMainData(). */
    Aps_ReleaseBuffer(mainData);
    
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrGetResFromString()
 *
 * Fills an Aps_Resolution structure with resolution information from a
 * string of the format "???x???dpi" or "???dpi". If the string doesn't
 * follow this format, horizontal and vertical resolution values are set to
 * 0.
 *
 * Parameters: jobAttributes  - A handle to a job attributes object.
 *
 *             resolutions    - Updated based on string contents
 *
 *             numResolutions - The number of elements in the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
static void AttrGetResFromString(Aps_Resolution *resInfo, const char *string)
{
    char *secondValue;
    
    ASSERT(resInfo != NULL);
    ASSERT(string != NULL);

    /* Initialize structure verison number to 0. */
    resInfo->version = 0;
    
    /* Obtain the first resolution number from the option string. */
    resInfo->horizontalRes = resInfo->verticalRes = atoi(string);
        
    /* If there is a second value, it is the vertical resolution, and must
     * immediately follow an 'x' character by the PPD specifications.
     */
    secondValue = strchr(string, 'x');
    if (secondValue != NULL) {
        resInfo->verticalRes = atoi(secondValue + 1);
    }
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetResOptions()
 *
 * Obtains a list of available resolutions for this device.
 *
 * Parameters: jobAttributes  - A handle to a job attributes object.
 *
 *             resolutions    - Receives an array of Aps_Resolution structs.
 *                              Free with Aps_ReleaseBuffer()
 *
 *             numResolutions - The number of elements in the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetResOptions(Aps_JobAttrHandle jobAttributes,
                                      Aps_Resolution ***resolutions,
                                      int *numResolutions)
{
    Aps_Result result = APS_GENERIC_FAILURE;
    int numOptions = 0;
    Aps_AttrOption **options = NULL;
    TrackArrayIndirect_Resolution outputArray = NULL;
    int i;
    char *optionID;
    Aps_Resolution resInfo;

    /* Check for reasonable parameter values. */
    if (resolutions == NULL || numResolutions == NULL)
        return APS_INVALID_PARAM;
    
    /* Initialize output parameters to default values in case of failure. */
    *resolutions = NULL;
    *numResolutions = 0;

    /* Query to obtain the list of available options for this attribute. */
    result = Aps_AttrGetOptions(jobAttributes, "*Resolution", &numOptions,
                                &options);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If there are no resolution options available, we return to the
     * application with success, but with an empty (NULL) array.
     */
    if (numOptions == 0) {
        result = APS_SUCCESS;
        goto cleanup;
    }
        
    /* Attempt to create an array to contain the information to pass
     * back to the caller.
     */
    outputArray = TrackArrayIndirectNew_Resolution(NULL, 0);
    if (! outputArray) { result = APS_OUT_OF_MEMORY; goto cleanup; }

    /* Add an entry in the buffer to be output to the caller for each
     * option.
     */
    for (i = 0; i < numOptions; ++i) {
        /* Obtain the next option string, if available. */
        optionID = options[i]->optionID;
        if (optionID == NULL) continue;

        /* Construct resolution information to pass back to the caller. */
        AttrGetResFromString(&resInfo, optionID);
        ASSERT(resInfo.horizontalRes > 0);
        ASSERT(resInfo.verticalRes > 0);
        
        /* Attempt to add this information to the output array. */
        if (NULL != TrackArrayIndirectAddLastByRef_Resolution(
            & outputArray, & resInfo)) {
            result = APS_OUT_OF_MEMORY;
            break;
        }
    }

    if (result != APS_SUCCESS)
        goto cleanup;
        
    /* If we reach this point, then the operation has succeeded. */
    *resolutions = outputArray;
    *numResolutions = TrackArrayIndirectGetSize_Resolution(outputArray);
    result = APS_SUCCESS;
    
cleanup:
    /* Release the options array if it has been created. */
    if (options != NULL) {
        Aps_ReleaseBuffer(options);
    }

    /* Release the buffer array to be returned to the caller, if needed. */
    if ((result != APS_SUCCESS) && (outputArray != NULL)) {
        TrackArrayIndirectDelete_Resolution(outputArray);
    }
    
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetRes()
 *
 * Obtains the currently selected resolution setting.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             resolution    - The resolution currently set.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetRes(Aps_JobAttrHandle jobAttributes,
                               Aps_Resolution *resolution)
{
    Aps_Result result;
    char *setting;

    /* Check for reasonable parameter values. */
    if (resolution == NULL)
        return APS_INVALID_PARAM;

    /* Pass query on to the generic attributes functions. */
    result = Aps_AttrGetSetting(jobAttributes, "*Resolution", &setting);
    if (result != APS_SUCCESS)
        return result;

    /* Place the information from this setting into the resolution struct. */
    AttrGetResFromString(resolution, setting);
    
    /* Dispose of the setting value string. */
    Aps_ReleaseBuffer(setting);
                
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetRes()
 *
 * Changes the current resolution setting.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             resolution    - The resolution to print at.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetRes(Aps_JobAttrHandle jobAttributes,
                               const Aps_Resolution *resolution)
{
    Aps_Result result;
    /* size is enough for 2 numbers, 'x', "dpi" and zero-terminator. */
    char valueString[APSCFG_MAX_UINT_STR_SIZE * 2 + 5];

    /* Check for reasonable parameter values. */
    if (resolution == NULL)
        return APS_INVALID_PARAM;

    /* The resolution setting can be represented as a string of one of two
     * formats; either "???dpi" or "???x???dpi". We need to check which one
     * to use if the horizontal and vertical resolutions differ. We first try
     * the "???x???dpi" format regardless of whether the horizontal and
     * vertical resolutions are the same or different. If that fails, and if
     * the resolutions are the same, then we'll attempt the "???dpi" format.
     */
     
    /* Attempt format 1: "???x???dpi". */
    sprintf(valueString, "%dx%ddpi", (int)resolution->horizontalRes,
            (int)resolution->verticalRes);
    result = Aps_AttrSetSetting(jobAttributes, "*Resolution", valueString);
    if (result == APS_SUCCESS)
        return APS_SUCCESS;
    
    /* Attempt format 2: "???dpi". */
    if (resolution->horizontalRes == resolution->verticalRes) {
        sprintf(valueString, "%ddpi", (int)resolution->horizontalRes);
        result = Aps_AttrSetSetting(jobAttributes, "*Resolution", valueString);
        if (result == APS_SUCCESS)
            return APS_SUCCESS;
    }

    /* We failed on both formats, so the resolution parameter was not a
     * valid resolution.
     */
    return APS_INVALID_PARAM;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetMaxCopies()
 *
 * Obtains the maximum number of copies of a job that can be printed.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             maxCopies     - Receives the maximum number of copies.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetMaxCopies(Aps_JobAttrHandle jobAttributes,
                                     int *maxCopies)
{
    double minValue;
    double maxValue;
    Aps_Result result;
    
    /* Check for reasonable parameter values. */
    if (maxCopies == NULL)
        return APS_INVALID_PARAM;

    /* Query the range for possible values for the "NumCopies" attribute. */
    result = Aps_AttrGetRange(jobAttributes, "NumCopies", &minValue,
                              &maxValue);
    if (result != APS_SUCCESS)
        return result;

    /* Convert the maximum value to an integer. */
    *maxCopies = (int)maxValue;

    /* We're assuming that the maximum value will be a positive integer. */
    ASSERT(((double)*maxCopies) == maxValue);
    ASSERT(*maxCopies >= 1);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetNumCopies()
 *
 * Obtains the number of copies of a job that are set to be printed.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             numCopies     - Receives the number of copies setting.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetNumCopies(Aps_JobAttrHandle jobAttributes,
                                     int *numCopies)
{
    Aps_Result result;
    char *setting;

    /* Check for reasonable parameter values. */
    if (numCopies == NULL)
        return APS_INVALID_PARAM;

    /* Pass query on to the generic attributes functions. */
    result = Aps_AttrGetSetting(jobAttributes, "NumCopies", &setting);
    if (result != APS_SUCCESS)
        return result;

    /* Place the information from this setting into the caller's variable. */
    *numCopies = atoi(setting);
    ASSERT(*numCopies >= 1);
    
    /* Dispose of the setting value string. */
    Aps_ReleaseBuffer(setting);
                
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetNumCopies()
 *
 * Sets the number of copies of a job that are to be printed.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             numCopies     - New value for the copies setting.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetNumCopies(Aps_JobAttrHandle jobAttributes,
                                     int numCopies)
{
    Aps_Result result;
    char newValue[APSCFG_MAX_UINT_STR_SIZE + 1];

    /* Check for reasonable parameter values. */
    if (numCopies <= 0)
        return APS_INVALID_PARAM;

    /* Create a string representation of this setting. */
    sprintf(newValue, "%d", numCopies);

    /* Attempt to change the value of this attribute. */
    result = Aps_AttrSetSetting(jobAttributes, "NumCopies", newValue);
    
    return result;
}

/* ---------------------------------------------------------------------------
 * AttrGetOptionsStrArray()
 *
 * Obtains a string array with the possible options for the specified
 * attribute.
 *
 * Parameters: jobAttributes    - A handle to a job attributes object.
 *
 *             attributeID      - The ID of the attribute to query.
 *
 *             optionsArray     - Receives an array with the option strings.
 *                                Free with Aps_ReleaseBuffer()
 *
 *             numOptions       - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
static Aps_Result AttrGetOptionsStrArray(Aps_JobAttrHandle jobAttributes,
                                         const char *attributeID,
                                         char ***optionsArray,
                                         int *numOptions)
{
    Aps_Result result = APS_GENERIC_FAILURE;
    int numOptionsRetrieved = 0;
    Aps_AttrOption **options = NULL;
    TrackArray_PtrChar outputArray = NULL;
    int i;
    char *optionID;

    /* Check for reasonable parameter values. */
    if (optionsArray == NULL || numOptions == NULL)
        return APS_INVALID_PARAM;

    ASSERT(attributeID != NULL);
    
    /* Initialize output parameters to default values in case of failure. */
    *optionsArray = NULL;
    *numOptions = 0;

    /* Query to obtain the list of available options for this attribute. */
    result = Aps_AttrGetOptions(jobAttributes, "*Resolution",
                                &numOptionsRetrieved, &options);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If there are no options available for this attribute, we return to the
     * application with success, but with an empty (NULL) array.
     */
    if (numOptionsRetrieved == 0) {
        result = APS_SUCCESS;
        goto cleanup;
    }
        
    /* Attempt to create an array to contain the information to pass
     * back to the caller.
     */
    outputArray = TrackArrayNew_PtrChar(NULL, 0);
    if (! outputArray) { result = APS_OUT_OF_MEMORY; goto cleanup; }

    /* Add an entry in the buffer to be output to the caller for each
     * option.
     */
    for (i = 0; i < numOptionsRetrieved; ++i) {
        char *newString;
        /* Obtain the next option string, if available. */
        optionID = options[i]->optionID;
        if (optionID == NULL) continue;

        /* Attempt to add this option to the output array. */
        newString = TrackMemDupString(outputArray, optionID, 0);
        if (! newString) { result = APS_OUT_OF_MEMORY; break; }
        if (NULL != TrackArrayAddLast_PtrChar(& outputArray, newString))
            { result = APS_OUT_OF_MEMORY; break; }
    }

    if (result != APS_SUCCESS)
        goto cleanup;
        
    /* If we reach this point, then the operation has succeeded. */
    *optionsArray = outputArray;
    *numOptions = TrackArrayGetSize_PtrChar(outputArray);
    result = APS_SUCCESS;
    
cleanup:
    /* Release the options array if it has been created. */
    if (options != NULL) {
        Aps_ReleaseBuffer(options);
    }

    /* Release the buffer array to be returned to the caller, if needed. */
    if ((result != APS_SUCCESS) && (outputArray != NULL)) {
        TrackArrayDelete_PtrChar(outputArray);
    }
    
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetCollationOptions()
 *
 * Obtains the possible copy collation settings.
 *
 * Parameters: jobAttributes    - A handle to a job attributes object.
 *
 *             collationOptions - Receives an array of collation setting IDs.
 *                                Free with Aps_ReleaseBuffer()
 *
 *             numOptions       - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetCollationOptions(Aps_JobAttrHandle jobAttributes,
                                            char ***collationOptions,
                                            int *numOptions)
{
    /* This function is a simple case that can be implemented in terms of
     * AttrGetOptionsStrArray(), which simply enumerates the IDs of all
     * options and packages them into a string array for the application.
     */
    return AttrGetOptionsStrArray(jobAttributes, "*Collate", collationOptions,
                                  numOptions);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetCollation()
 *
 * Obtains the current copy collation setting.
 *
 * Parameters: jobAttributes    - A handle to a job attributes object.
 *
 *             collationSetting - Receives the collation setting ID string.
 *                                Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetCollation(Aps_JobAttrHandle jobAttributes,
                                     char **collationSetting)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrGetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrGetSetting(jobAttributes, "*Collate", collationSetting);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetCollation()
 *
 * Changes the current copy collation setting.
 *
 * Parameters: jobAttributes    - A handle to a job attributes object.
 *
 *             collationSetting - ID of the new collation setting to use.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetCollation(Aps_JobAttrHandle jobAttributes,
                                     const char *collationSetting)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrGetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrSetSetting(jobAttributes, "*Collate", collationSetting);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetPageSizeOptions()
 *
 * Obtains a list of available predefined page sizes for this device.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             pageSizes     - Receives an array of Aps_PageSize structures.
 *                             Free with Aps_ReleaseBuffer()
 *
 *             numPageSizes  - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetPageSizeOptions(Aps_JobAttrHandle jobAttributes,
                                           Aps_PageSize ***pageSizes,
                                           int *numPageSizes)
{
    Aps_AttrOption **options = NULL;
    int numOptions;
    int i;
    Aps_Result result;
    TrackArrayIndirect_PageSize pageSizeArray = NULL;
    Aps_PageSize *curPageSize;

    /* Check for reasonable parameter values. */
    if (pageSizes == NULL || numPageSizes == NULL) {
        result = APS_INVALID_PARAM;
        goto cleanup;
    }

    /* Obtain a list of all known page sizes. */
    result = Aps_AttrGetOptions(jobAttributes, "*PageSize",
                                &numOptions, &options);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Create an array to use in building list of page sizes to pass
     * back to the caller.
     */
    pageSizeArray = TrackArrayIndirectNew_PageSize(NULL, 0);
    if (! pageSizeArray) { result = APS_OUT_OF_MEMORY; goto cleanup; }

    /* Iterate through the loop of known page sizes. */
    for (i = 0; i < numOptions; ++i) {
        /* Add a new page size to the array for this page size. */
        curPageSize = TrackArrayIndirectAddLastByRef_PageSize(
            & pageSizeArray, NULL);
        if (! curPageSize) { result = APS_OUT_OF_MEMORY; break; }

        /* Setup the ID field for this page size. */
        curPageSize->id = TrackMemDupString(curPageSize,
            options[i]->optionID, 0);
        if (! curPageSize->id) { result = APS_OUT_OF_MEMORY; break; }

        /* Fill the detail fields in this page size structure. */
        result = AttrFillPageSizeDetails(jobAttributes, curPageSize);
        if (result != APS_SUCCESS) break;

        /* If we have a translated name for this page size, provide that
         * to the caller too.
         */
        if (options[i]->translatedName != NULL) {
            curPageSize->translatedName = TrackMemDupString(curPageSize,
                options[i]->translatedName, 0);
            if (! curPageSize->translatedName) { result = APS_OUT_OF_MEMORY;
                break; }
        } else curPageSize->translatedName = NULL;
    }

    /* Success! */
    result = APS_SUCCESS;
    *pageSizes = pageSizeArray;
    *numPageSizes = TrackArrayIndirectGetSize_PageSize(pageSizeArray);

cleanup:
    /* Release any temporary resources used within this function. */
    if (options != NULL)
        Aps_ReleaseBuffer(options);

    if ((result != APS_SUCCESS) && (pageSizeArray != NULL)) {
        TrackArrayIndirectDelete_PageSize(pageSizeArray);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * AttrFillPageSizeDetails()
 *
 * Fills in an Aps_PageSize structure with the details on a predefined page
 * size that is listed in this job attributes object.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *              pageSize     - A pointer to the Aps_PageSize structure to be
 *                             filled. This structure must already contain the
 *                             name of this page size in the id member.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
static Aps_Result AttrFillPageSizeDetails(Aps_JobAttrHandle jobAttributes,
                                          Aps_PageSize *pageSize)
{
    Aps_AttrOption **options;
    int numOptions;
    int i;
    Aps_Result result;
    char *nextValue;

    /* Set version information in this structure. */
    pageSize->version = 0;

    /* We don't supply a translated name inside this function. If the caller
     * wants to include this information in the structure, then it is the
     * caller's responsibility to fill this field.
     */
    pageSize->translatedName = NULL;

    /* We need to obtain the page dimensions for this page. */
    result = Aps_AttrGetOptions(jobAttributes, "*PaperDimension",
                                &numOptions, &options);
    if (result != APS_SUCCESS)
        return result;

    /* Look for a matching page dimension option. */
    for (i = 0; i < numOptions; ++i) {
        if (strcmp(options[i]->optionID, pageSize->id) == 0
            && options[i]->value != NULL) {
            /* We've found an entry for this page size. */

            /* Parse out the width of this page. */
            pageSize->mediaHeight = pageSize->mediaWidth
                = atof(options[i]->value);
 
            /* Parse out the height of this page. */
            nextValue = strchr(options[i]->value, ' ');
            if (nextValue != NULL) {
                pageSize->mediaHeight = atof(nextValue + 1);
            }

            /* Stop searching for a matching page size. */
            break;
        }
    }

    Aps_ReleaseBuffer(options);

    /* If we didn't find this page size in the attributes, then fail. */
    if (i == numOptions)
        return APS_NOT_FOUND;

    /* TBD: Obtain actual imagable area information. */
    pageSize->imageableAreaLLx = 0;
    pageSize->imageableAreaLLy = 0;
    pageSize->imageableAreaURx = pageSize->mediaWidth;
    pageSize->imageableAreaURy = pageSize->mediaHeight;

    /* Initialize custom page size related fields. */
    pageSize->isCustom = FALSE;
    pageSize->widthOffset = 0;
    pageSize->heightOffset = 0;
    pageSize->orientation = APS_ROTATE_0;
    pageSize->useHWMargins = TRUE;
    pageSize->leadingEdge = APS_LEADING_SHORT;

    /* We've now successfully setup this Aps_PageSize structure. */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetCustomPageSizeLimits()
 *
 * Obtains information on the custom page sizes that can be supported by this
 * device, if any.
 *
 * Parameters: jobAttributes        - A handle to a job attributes object.
 *
 *             customPageSizeLimits - Receives an Aps_CustomPageSizeLimits
 *                                    structure.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetCustomPageSizeLimits(
                            Aps_JobAttrHandle jobAttributes,
                            Aps_CustomPageSizeLimits **customPageSizeLimits)
{
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetPageSize()
 *
 * Obtains information on the page size that is currently set.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             pageSize      - Receives the current page size information.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetPageSize(Aps_JobAttrHandle jobAttributes,
                                    Aps_PageSize **pageSize)
{
    char *pageSizeSetting;
    Aps_Result result;

    /* Check for reasonable parameter values. */
    if (pageSize == NULL)
        return APS_INVALID_PARAM;

    /* Obtain a string identifying the current page size setting. */
    result = Aps_AttrGetSetting(jobAttributes, "*PageSize", &pageSizeSetting);
    if (result != APS_SUCCESS)
        return result;

    /* Allocate memory to hold page size structure and page size name. */
    *pageSize = TrackMemAlloc(NULL,
        sizeof(Aps_PageSize) + strlen(pageSizeSetting) + 1, 0);
    if (*pageSize == NULL)
        return APS_OUT_OF_MEMORY;

    /* Setup the page ID in this structure, discarding our temporary copy
     * of the page size name.
     */
    (*pageSize)->id = ((char *)pageSize) + sizeof(Aps_PageSize);
    strcpy((*pageSize)->id, pageSizeSetting);
    Aps_ReleaseBuffer(pageSizeSetting);

    /* Attempt to fill out the remaining details in the Aps_PageSize
     * structure.
     */
    result = AttrFillPageSizeDetails(jobAttributes, *pageSize);
    if (result != APS_SUCCESS) {
        /* On failure, clean up and abort. */
        Aps_ReleaseBuffer(*pageSize);
        *pageSize = NULL;
        return result;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetPredefinedPageSize()
 *
 * Sets the current page size to one of the predefined sizes obtained from
 * Aps_AttrQuickGetPageSizeOptions().
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             pageSizeID    - The string naming the page size to use.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetPredefinedPageSize(Aps_JobAttrHandle jobAttributes,
                                              const char *pageSizeID)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrSetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrSetSetting(jobAttributes, "*PageSize", pageSizeID);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetCustomPageSize()
 *
 * Sets the current page size to a custom size defined using an Aps_PageSize
 * structure.
 *
 * Parameters: jobAttributes  - A handle to a job attributes object.
 *
 *             customPageSize - The dimensions and other parameters
 *                              describing the page size to use.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetCustomPageSize(Aps_JobAttrHandle jobAttributes,
                                          const Aps_PageSize *customPageSize)
{
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetInputSlotOptions()
 *
 * Obtains a list of available input slots.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             inputSlots    - Receives an array of input slot ID strings.
 *                             Free with Aps_ReleaseBuffer()
 *
 *             numInputSlots - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetInputSlotOptions(Aps_JobAttrHandle jobAttributes,
                                            char ***inputSlots,
                                            int *numInputSlots)
{
    /* TBD: Implement "*ManualFeed" support here. */
    return AttrGetOptionsStrArray(jobAttributes, "*InputSlot",
                                  inputSlots, numInputSlots);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetInputSlot()
 *
 * Determines which input slot is currently selected.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             inputSlot     - Receives the ID of the currently input slot.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetInputSlot(Aps_JobAttrHandle jobAttributes,
                                     char **inputSlot)
{
    /* TBD: Implement "*ManualFeed" support here. */
    return Aps_AttrGetSetting(jobAttributes, "*InputSlot", inputSlot);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetInputSlot()
 *
 * Selects a specific input slot.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             inputSlot     - The ID of the input slot to use.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetInputSlot(Aps_JobAttrHandle jobAttributes,
                                     const char *inputSlot)
{
    /* TBD: Implement "*ManualFeed" support here. */
    return Aps_AttrSetSetting(jobAttributes, "*InputSlot", inputSlot);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetOutputBinOptions()
 *
 * Obtains a list of available output bins.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             outputBins    - Receives an array of output bin ID strings.
 *                             Free with Aps_ReleaseBuffer()
 *
 *             numOutputBins - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetOutputBinOptions(Aps_JobAttrHandle jobAttributes,
                                            char ***outputBins,
                                            int *numOutputBins)
{
    /* This function is a simple case that can be implemented in terms of
     * AttrGetOptionsStrArray(), which simply enumerates the IDs of all
     * options and packages them into a string array for the application.
     */
    return AttrGetOptionsStrArray(jobAttributes, "*OutputBin",
                                  outputBins, numOutputBins);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetOutputBin()
 *
 * Determines which output bin is currently selected.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             outputBin     - Receives the ID of the currently selected bin.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetOutputBin(Aps_JobAttrHandle jobAttributes,
                                     char **outputBin)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrGetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrGetSetting(jobAttributes, "*OutputBin", outputBin);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetOutputBin()
 *
 * Changes which output bin is selected.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             outputBin     - ID of the output bin to select.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetOutputBin(Aps_JobAttrHandle jobAttributes,
                                     const char *outputBin)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrGetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrSetSetting(jobAttributes, "*OutputBin", outputBin);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetDuplexOptions()
 *
 * Obtains a list of available duplex printing options.
 *
 * Parameters: jobAttributes     - A handle to a job attributes object.
 *
 *             duplexSettings    - Receives an array of duplex setting IDs.
 *                                 Free with Aps_ReleaseBuffer()
 *
 *             numDuplexSettings - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetDuplexOptions(Aps_JobAttrHandle jobAttributes,
                                         char ***duplexSettings,
                                         int *numDuplexSettings)
{
    /* This function is a simple case that can be implemented in terms of
     * AttrGetOptionsStrArray(), which simply enumerates the IDs of all
     * options and packages them into a string array for the application.
     */
    return AttrGetOptionsStrArray(jobAttributes, "*Duplex",
                                  duplexSettings, numDuplexSettings);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetDuplex()
 *
 * Obtains the current duplex printing setting.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             duplexSetting - Receives the current duplex printing setting.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetDuplex(Aps_JobAttrHandle jobAttributes,
                                  char **duplexSetting)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrGetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrGetSetting(jobAttributes, "*Duplex", duplexSetting);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickSetDuplex()
 *
 * Changes the current duplex printing setting.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             duplexSetting - The ID of the duplex setting to use.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickSetDuplex(Aps_JobAttrHandle jobAttributes,
                                  const char *duplexSetting)
{
    /* This function is a simple case that can currently be implemented in
     * terms of Aps_AttrGetSetting(), giving it the appropriate hard-coded
     * attribute ID.
     */
    return Aps_AttrSetSetting(jobAttributes, "*Duplex", duplexSetting);
}

/* ---------------------------------------------------------------------------
 * Aps_AttrQuickGetFonts()
 *
 * Obtains a list of fonts that are resident on the device.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             fonts         - Receives an array of pointers to Aps_FontInfo
 *                             structures.
 *                             Free with Aps_ReleaseBuffer()
 *
 *             numFonts      - Receives the size of the array.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrQuickGetFonts(Aps_JobAttrHandle jobAttributes,
                                 Aps_FontInfo ***fonts,
                                 int *numFonts)
{
    Aps_Result result = APS_GENERIC_FAILURE;
    int numOptions = 0;
    Aps_AttrOption **options = NULL;
    TrackArrayIndirect_FontInfo outputArray = NULL;
    char *defaultFont = NULL;
    int i;
    char *fontName;

    /* Check for reasonable parameter values. */
    if (fonts == NULL || numFonts == NULL)
        return APS_INVALID_PARAM;
    
    /* Initialize output parameters to default values in case of failure. */
    *fonts = NULL;
    *numFonts = 0;

    /* Query for the name of the default font. */
    result = Aps_AttrGetMainData(jobAttributes, "*DefaultFont", &fontName);
    if (result != APS_SUCCESS)
        goto cleanup;
    
    /* Query to obtain the list of available options for this attribute. */
    result = Aps_AttrGetOptions(jobAttributes, "*Font", &numOptions,
                                &options);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If there are no resolution options available, we return to the
     * application with success, but with an empty (NULL) array.
     */
    if (numOptions == 0) {
        result = APS_SUCCESS;
        goto cleanup;
    }
        
    /* Attempt to create an array to contain the information to pass
     * back to the caller.
     */
    outputArray = TrackArrayIndirectNew_FontInfo(NULL, 0);
    if (! outputArray) { result = APS_OUT_OF_MEMORY; goto cleanup; }

    /* Add an entry in the buffer to be output to the caller for each
     * option.
     */
    for (i = 0; i < numOptions; ++i) {
        Aps_FontInfo *fontInfo;

        /* Obtain the name of the next font. */
        fontName = options[i]->optionID;
        
        /* If we don't have a name for this font, then skip it. */
        if (fontName == NULL) continue;

        /* Construct font information to pass back to the caller. */
        fontInfo = TrackArrayIndirectAddLastByRef_FontInfo(
            & outputArray, NULL);
        if (! fontInfo) { result = APS_OUT_OF_MEMORY; break; }

        /* Set version number of structure. */
        fontInfo->version = 0;
        
        /* Provide the caller with the name of this font. */
        fontInfo->name = TrackMemDupString(fontInfo, fontName, 0);
        if (! fontInfo->name) { result = APS_OUT_OF_MEMORY; break; }

        /* Indicate whether or not this is the default font. */
        fontInfo->isDefault = (strcmp(defaultFont, fontName) == 0);
    }

    if (result != APS_SUCCESS)
        goto cleanup;
        
    /* If we reach this point, then the operation has succeeded. */
    result = APS_SUCCESS;
    *fonts = outputArray;
    *numFonts = TrackArrayIndirectGetSize_FontInfo(outputArray);

cleanup:
    /* Release the options array if it has been created. */
    if (options != NULL) {
        Aps_ReleaseBuffer(options);
    }

    /* Release the buffer array to be returned to the caller, if needed. */
    if ((result != APS_SUCCESS) && (outputArray != NULL)) {
        TrackArrayIndirectDelete_FontInfo(outputArray);
    }

    /* If default font string exists, deallocate it.*/
    if (defaultFont != NULL) {
        Aps_ReleaseBuffer(defaultFont);
    }
    
    return result;
}

