/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: bufferarray.c
 *
 * Description: Facilities for constructing and managing arrays of strings
 *              or other data structures. BufferArrays are specifically
 *              designed to meet the needs of an interface that needs to
 *              pass an array and its contents in a single block of memory
 *              (i.e. a *Buffer*) from one component to another.
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "bufferarray.h"

typedef struct {
    int numElements;
    int pointerSpace;
    int elementSpace;
    int nextDataPos;
    char **data;
} BufferArrayObject;

#define INITIAL_NUM_POINTERS 8
#define INITIAL_ELEMENT_SPACE 64

/* remove when fixed */
#define BUFARRADDSECONDARYBLOCK_KLUDGE

static Aps_Result BufArrVerifySpace(BufferArrayObject * arrayObject,
                                    int numElements,
                                    int dataSize);

/* ---------------------------------------------------------------------------
 * BufArrCreate()
 *
 * Create a new array for storing multiple buffers of any length.
 *
 * Parameters: array - Receives a pointer to the new BufferArray.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrCreate(BufferArray * array)
{
    BufferArrayObject *arrayObject;

    /* Check for obviously incorrect parameter values. */
    ASSERT(array != NULL);

    /* Allocate memory for the array object structure. */
    arrayObject = malloc(sizeof(BufferArrayObject));
    if (arrayObject == NULL)
        return APS_OUT_OF_MEMORY;

    /* Initialize the object structure. */
    arrayObject->numElements = 0;
    arrayObject->pointerSpace = INITIAL_NUM_POINTERS * sizeof(char *);

    arrayObject->elementSpace = INITIAL_ELEMENT_SPACE;
    arrayObject->nextDataPos = arrayObject->pointerSpace;
    arrayObject->data =
        malloc(arrayObject->pointerSpace + arrayObject->elementSpace);

    /* Check that we were able to allocate the data buffer. */
    if (arrayObject->data == NULL) {
        free(arrayObject);
        return APS_OUT_OF_MEMORY;
    }
    /* Return to the caller a pointer to this array object. */
    *array = arrayObject;
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrCreateCopy()
 *
 * Creates a complete copy of a buffer array and its buffers.
 *
 * Parameters: newArray      - Receives a pointer to the new buffer array.
 *
 *             existingArray - An existing buffer array that will not be
 *                             modified.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrCreateCopy(BufferArray * newArray, BufferArray existingArray)
{
    BufferArrayObject *newArrayObject;
    BufferArrayObject *existingArrayObject = (BufferArrayObject *) existingArray;
    int offset;
    int i;

    ASSERT(newArray != NULL);
    ASSERT(existingArray != NULL);

    /* Allocate memory for the array object structure. */
    newArrayObject = malloc(sizeof(BufferArrayObject));
    if (newArrayObject == NULL)
        return (APS_OUT_OF_MEMORY);

    /* Initialize the object structure. */
    memcpy(newArrayObject, existingArrayObject, sizeof(BufferArrayObject));

    /* Allocate space for data buffer. */
    newArrayObject->data = malloc(existingArrayObject->pointerSpace
                                  + existingArrayObject->elementSpace);
    if (newArrayObject->data == NULL) {
        free(newArrayObject);
        return (APS_OUT_OF_MEMORY);
    }
    /* Copy data from old array object to new one. */
    memcpy(newArrayObject->data, existingArrayObject->data,
           existingArrayObject->pointerSpace
           + existingArrayObject->elementSpace);

    /* Update pointer values. */
    offset = ((char *)newArrayObject->data)
        - ((char *)existingArrayObject->data);
    for (i = 0; i < newArrayObject->numElements; ++i) {
        newArrayObject->data[i] += offset;
    }

    /* Return to the caller a pointer to this array object. */
    *newArray = newArrayObject;
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrDelete()
 *
 * Deallocates a buffer array and associated buffers.
 *
 * Parameters: array - The buffer array to be deallocated.
 *
 *     Return: void
 */
void BufArrDelete(BufferArray array)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) array;

    ASSERT(arrayObject != NULL);

    /* Deallocate data buffer. */
    free(arrayObject->data);

    /* Deallocate object structure. */
    free(arrayObject);
}

/* ---------------------------------------------------------------------------
 * BufArrChangeToSingleBuffer()
 *
 * Obtains a single block of memory containing both the array of buffer
 * pointer and all the buffers themselves, deleting the BufferArray object.
 * Note that the BufferArray is deleted regardless of whether or not this
 * function succeeds.
 *
 * Parameters: oldArray - An existing buffer array to be converted into
 *                        a single block of memory. After this call, the
 *                        BufferArray is deleted and can no longer be
 *                        used.
 *
 *             buffer   - The address of a void ** to receive the address
 *                        of the memory block.
 *
 *             numElems - The number of elements in the array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrChangeToSingleBuffer(BufferArray oldArray, void ***buffer,
                                      int *numElems)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) oldArray;

    ASSERT(arrayObject != NULL);

    /* Provide the caller with the number of elements in the array. */
    *numElems = arrayObject->numElements;

    /* Because the current implementation of the buffer array object stores */
    /* the array in a single block of memory, we just need to strip off the */
    /* buffer array object, and provide the caller with the pointer to our  */
    /* internal buffer.                                                     */
    *buffer = (void **)arrayObject->data;

    /* Deallocate object structure. */
    free(arrayObject);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrAddString()
 *
 * Adds a '\0'-terminated string to the end of a buffer array.
 *
 * Parameters: array  - An existing BufferArray.
 *
 *             string - A pointer to the string to be added.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrAddString(BufferArray array, const char *string)
{
    return BufArrAdd(array, string, strlen(string) + 1);
}

/* ---------------------------------------------------------------------------
 * BufArrAdd()
 *
 * Adds an arbitrary-length block of data as the last element in the array.
 *
 * Parameters: array   - An existing BufferArray object.
 *
 *             element - A pointer to the block of data to be copied into the
 *                       array.
 *
 *             size    - The size of element, in bytes.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrAdd(BufferArray array, const void *element, int size)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *)array;

    ASSERT(arrayObject != NULL);
    ASSERT(element != NULL);
    ASSERT(size >= 1);

    /* Call the insertion function to insert a new element at the end of the
     * array.
     */
    return BufArrInsertAt(array, arrayObject->numElements, element, size);
}

/* ---------------------------------------------------------------------------
 * BufArrAddUninitialized()
 *
 * Adds an arbitrary-length block of data as the last element in the array,
 * making space to store the element's data, but without initializing the
 * contents of that block.
 *
 * Parameters: array   - An existing BufferArray object.
 *
 *             size    - The size of element, in bytes.
 *
 *             element - The address of a pointer to receive the location
 *                       of the new element's data.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrAddUninitialized(BufferArray array, int size,
                                  void **element)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) array;

    ASSERT(arrayObject != NULL);
    ASSERT(size >= 1);
    ASSERT(element != NULL);

    /* Call the insertion function to insert a new element at the end of the
     * array.
     */
    return BufArrInsertAtUninitialized(array, arrayObject->numElements, size,
                                       element);
}

/* ---------------------------------------------------------------------------
 * BufArrAddSecondaryBlock()
 *
 * Adds a block of memory that will be managed by this buffer array, but which
 * is not directly an element in the array. This function is intended for use
 * in situations where elements in the array in turn contain their own
 * pointers to strings or other data structures which must also be
 * encapsulated inside the same overall buffer.
 *
 * Parameters: array   - An existing BufferArray object.
 *
 *             size    - The required size of the block, in bytes.
 *
 *             block   - The address of a pointer to receive the location of
 *                       the secondary block.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrAddSecondaryBlock(BufferArray array, int size,
                                   void **block)
{
#ifdef BUFARRADDSECONDARYBLOCK_KLUDGE
    /* This causes a memory leak but avoids serious memory
     * corruption in the process.  FIX ME
     */
    *block = malloc(size);
#else

    BufferArrayObject *arrayObject = (BufferArrayObject *) array;
    Aps_Result result;

    ASSERT(arrayObject != NULL);
    ASSERT(size >= 1);
    ASSERT(block != NULL);

    /* Ensure that there is enough space for this new block. */
    result = BufArrVerifySpace(arrayObject, 0, size);
    if (result != APS_SUCCESS)
        return (result);

    /* Provide the caller with the address of this data block. */
    *block = (char *)arrayObject->data + arrayObject->nextDataPos;

    /* Update index of next available data space. */
    arrayObject->nextDataPos += size;
#endif

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrVerifySpace()
 *
 * Ensures that there is enough space in the data buffer to accomodate the
 * addition of the specified number of elements of the specified total size.
 * Grows the data buffer if needed.
 *
 * Parameters: arrayObject     - A pointer to the BufferArrayObject structure.
 *
 *             numElements     - The number of new elements required, if any.
 *
 *             dataSize        - The total number of bytes required to
 *                               store the contents of the elements.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
static Aps_Result BufArrVerifySpace(BufferArrayObject * arrayObject,
                                    int numElements,
                                    int dataSize)
{
    int newPointerSpace = arrayObject->pointerSpace;
    int newElementSpace = arrayObject->elementSpace;
    int newTotalSize;
    int offset;
    char **newData;
    int i;

    ASSERT(arrayObject != NULL);
    ASSERT(numElements >= 0);
    ASSERT(dataSize >= 0);

    /* Determine desired pointer space. */
    if (newPointerSpace <
        (arrayObject->numElements + numElements) * (int)sizeof(void *)) {
        /* We don't have enough pointer space. */
        if (newPointerSpace == 0) {
            /* No pointer space is currently allocated, so start with */
            /* the standard amount of initial pointer space.          */
            newPointerSpace = numElements * sizeof(void *);
        } else {
            /* Each time we run out of pointer space, we double the */
            /* amount of pointer space.                             */
            newPointerSpace = (newPointerSpace + numElements) * 2;
        }
    }
    /* Determine desired element space. */
    if (newElementSpace < arrayObject->nextDataPos - arrayObject->pointerSpace
        + dataSize) {
        /* If we don't already have enough space, then we'll grow the  */
        /* element space to twice the current space + the space needed */
        /* for the new element.                                        */
        newElementSpace = (newElementSpace + dataSize) * 2;
    }
    /* Determine whether a reallocation is required. */
    if (newPointerSpace == arrayObject->pointerSpace
        && newElementSpace == arrayObject->elementSpace) {
        return APS_SUCCESS;
    }
    newTotalSize = newPointerSpace + newElementSpace;

    /* Reallocate space for new buffer. */
    newData = realloc(arrayObject->data, newTotalSize);
    if (newData == NULL) {
        return APS_OUT_OF_MEMORY;
    }
    /* Move data buffers  */
    if (newPointerSpace != arrayObject->pointerSpace) {
        memmove(((char *)newData) + newPointerSpace,
                ((char *)newData) + arrayObject->pointerSpace,
                arrayObject->nextDataPos - arrayObject->pointerSpace);
    }
    /* Adjust data pointers. */
    offset = (newPointerSpace - arrayObject->pointerSpace)
        + ((int)newData - (int)arrayObject->data);
    if (offset != 0) {
        for (i = 0; i < arrayObject->numElements; ++i) {
            newData[i] += offset;
        }
    }
    /* Update array object members. */
    arrayObject->data = newData;
    arrayObject->nextDataPos +=
        (newPointerSpace - arrayObject->pointerSpace);
    arrayObject->pointerSpace = newPointerSpace;
    arrayObject->elementSpace = newElementSpace;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrGetSize()
 *
 * Obtains the number of elements in a buffer array.
 *
 * Parameters: array - An existing BufferArray.
 *
 *     Return: The number of elements in the BufferArray.
 */
int BufArrGetSize(BufferArray array)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) array;

    ASSERT(arrayObject != NULL);

    return (arrayObject->numElements);
}

/* ---------------------------------------------------------------------------
 * BufArrGetAt()
 *
 * Obtains the element at the specified index in an array.
 *
 * Parameters: array - An existing BufferArray.
 *
 *             index - The index of the element to retrieve.
 *
 *     Return: A void * to the array element at the specified index. Note that
 *             this is a pointer into the buffer array's internal data, and so
 *             its contents should not be modified.
 */
void *BufArrGetAt(BufferArray array, int index)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) array;

    ASSERT(arrayObject != NULL);
    ASSERT(index >= 0);
    ASSERT(index < arrayObject->numElements);

    return (arrayObject->data[index]);
}

/* ---------------------------------------------------------------------------
 * BufArrSetStringAt()
 *
 * Changes the contents of the string at the specified location in the array.
 *
 * Parameters: array  - An existing BufferArray.
 *
 *             index  - The index of the element to modify. This must be
 *                      an existing element; this function cannot be used to
 *                      add new elements to the array.
 *
 *             string - A '\0'-terminated string to set at this location
 *                      in the array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrSetStringAt(BufferArray array, int index,
                             const char *string)
{
    ASSERT(array != NULL);
    ASSERT(index >= 0);
    ASSERT(index < BufArrGetSize(array));
    ASSERT(string != NULL);

    /* We use the more generic BufArrSetAt() to implement this operation. */
    return BufArrSetAt(array, index, string, strlen(string) + 1);
}

/* ---------------------------------------------------------------------------
 * BufArrSetAt()
 *
 * Changes the contents of the element at the specified location in the array.
 *
 * Parameters: array   - An existing BufferArray.
 *
 *             index   - The index of the element to modify. This must be
 *                       an existing element; this function cannot be used to
 *                       add new elements to the array.
 *
 *             element - The new contents for this element.
 *
 *             size    - The size of the block pointed to by element.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrSetAt(BufferArray array, int index, const void *element,
                       int size)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) array;
    char *destElement;
    Aps_Result result;

    ASSERT(arrayObject != NULL);
    ASSERT(index >= 0);
    ASSERT(index <= arrayObject->numElements);
    ASSERT(element != NULL);
    ASSERT(size >= 1);

    /* Note: The implementation of this function could be made much      */
    /*       smarter by changing the way a BufferArray is stored         */
    /*       internally. At the momement, this function simply           */
    /*       allocates space for the new data, leaving the old data      */
    /*       as an unreferenced block of memory that will only be        */
    /*       released when the entire buffer is deallocated. A           */
    /*       slightly smarter implementation would check whether the     */
    /*       new element size would fit into the old element size,       */
    /*       and if so reuse that space. An even smarter implementation  */
    /*       could keep track of unused element space, and resuse and/or */
    /*       compact the space when appropriate. We haven't worried      */
    /*       about these details at this time, as BufferArrays are       */
    /*       primarily intended for building static arrays to be passed  */
    /*       around, and so we anticipate this function being used only  */
    /*       rarely.                                                     */

    /* Ensure that there is enough space for this new value. */
    result = BufArrVerifySpace(arrayObject, 0, size);
    if (result != APS_SUCCESS)
        return (result);

    /* Add the new element to the array of element pointers. */
    destElement = (char *)arrayObject->data + arrayObject->nextDataPos;
    arrayObject->data[index] = destElement;

    /* Copy the data into this element. */
    memcpy(destElement, element, size);

    /* Update indicies. */
    arrayObject->nextDataPos += size;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrInsertStringAt()
 *
 * Inserts a new string in the middle of the array, at the specified location.
 *
 * Parameters: array  - An existing BufferArray.
 *
 *             index  - The index where the new element should be inserted.
 *
 *             string - A '\0'-terminated string to store at this location
 *                      in the array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrInsertStringAt(BufferArray array, int index,
                                const char *string)
{
    /* Use the generic element insertion method to implement string
     * insertion.
     */
    return BufArrInsertAt(array, index, string, strlen(string) + 1);
}

/* ---------------------------------------------------------------------------
 * BufArrInsertAt()
 *
 * Inserts a new element in the middle of the array, at the specified
 * location.
 *
 * Parameters: array   - An existing BufferArray.
 *
 *             index   - The index where the new element should be inserted.
 *
 *             element - A pointer to the data to be inserted.
 *
 *             size    - The number of bytes to copy from element, into the
 *                       array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrInsertAt(BufferArray array, int index, const void *element,
                          int size)
{
    Aps_Result result;
    void *newElement;
    
    /* Call the function to insert an element pointer and reserve space for
     * that pointer.
     */
    result = BufArrInsertAtUninitialized(array, index, size, &newElement);
    if (result != APS_SUCCESS)
        return result;

    /* Copy the caller's data into the space that has now been reserved. */
    memcpy(newElement, element, size);
        
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * BufArrInsertAt()
 *
 * Inserts a new element in the middle of the array, at the specified
 * location, reserving space for that element but not copying any data into
 * the reserved space.
 *
 * Parameters: array   - An existing BufferArray.
 *
 *             index   - The index where the new element should be inserted.
 *
 *             size    - The amount of space to reserve for this element.
 *
 *             element - Receives the reserved space for this element.
 *
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result BufArrInsertAtUninitialized(BufferArray array, int index,
                                       int size, void **element)
{
    BufferArrayObject *arrayObject = (BufferArrayObject *) array;
    Aps_Result result;

    ASSERT(arrayObject != NULL);
    ASSERT(size >= 1);
    ASSERT(element != NULL);
    ASSERT(index >= 0);
    ASSERT(index <= arrayObject->numElements);

    /* Ensure that there is enough space for this new element. */
    result = BufArrVerifySpace(arrayObject, 1, size);
    if (result != APS_SUCCESS)
        return result;

    /* Obtain the address of the space being reserved for the new elemtn. */
    *element = (char *)arrayObject->data + arrayObject->nextDataPos;

    /* Update the next data position to reflect the amount of space that has
     * been reserved for this element.
     */
    arrayObject->nextDataPos += size;

    /* Move the element pointers, if we are inserting in the middle of the
     * array.
     */    
    if (index < arrayObject->numElements) {
        memmove(&(arrayObject->data[index + 1]),
                &(arrayObject->data[index]),
                sizeof(void *) * (arrayObject->numElements - index));
    }

    /* Update the number of element pointers. */
    arrayObject->numElements++;

    /* Set this element's pointer to point to the new element. */
    arrayObject->data[index] = *element;

    return APS_SUCCESS;
}

