/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: job.c
 *
 * Description: Functionality specific to manipulating a job object.
 *              This is the lower layer of operations -- ie. any and
 *              all interaction with the transport object occurs here.
 *
 *              [Higher-level functions are expected to only operate
 *               on the data they have available stored in the
 *               object.]
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>

#include "aps.h"
#include "apsinternal.h"
#include "job.h"
#include "jobattributes.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(job)

/* ---------------------------------------------------------------------------
 * JobGetPtrFromHandle()
 *
 * Obtains a pointer to a Job object, given an Aps_JobHandle.
 *
 * Parameters: handle - The handle for which the corresponding Job object
 *                      should be obtained.
 *
 *     Return: A pointer to a Job object, or NULL on failure.
 */
Job *JobGetPtrFromHandle(Aps_JobHandle handle)
{
    Job *job;

    if (handle == NULL)
        return (NULL);

    job = (Job *) handle;

    /* Check that this is actually a job object, and not something else. */
    if (job->baseClass.identifier != JOB_HANDLE)
        return (NULL);

    return (job);
}

/* ---------------------------------------------------------------------------
 * JobGetHandleFromPtr()
 *
 * Obtain the handle for a specified Job object.
 *
 * Parameters: job - The Job for which the corresponding handle shoudl be
 *                   obtained.
 *
 *     Return: An Aps_JobHandle.
 */
Aps_JobHandle JobGetHandleFromPtr(Job * job)
{
    ASSERT(job != NULL);

    return ((Aps_JobHandle) job);
}

/* ---------------------------------------------------------------------------
 * QuickJobInfoInit()
 *
 * Initialize a QuickJobInfo to defaults. This function always succeeds.
 *
 * Parameters: info    - pointer to info structure
 *             version - version # to initialize
 *
 * Return    : A standard APS result code. Always APS_SUCCESS unless
 *             there's an invalid parameter.
 *
 * NOTE: DO NOT CHANGE THESE DEFAULTS WITHOUT CONSIDERING THE CONSEQUENCES!!
 */
static const Aps_QuickJobInfo defaultsQuickJobInfo_v0 = {
    /** VERSION 0 **/
    /* version         */ 0,
    /* jobHandle       */ NULL,
    /* jobStatus       */ APS_JOB_UNKNOWN,
    /* jobHost         */ "",
    /* jobName         */ "",
    /* jobFilename     */ "",
    /* jobID           */ -1,
    /* jobSize         */ 0,
    /* jobCreationTime */ 0,
    /* jobFormat       */ "",
    /* jobOrder        */ -1,
    /* jobPriority     */ 0,
    /* printerHandle   */ NULL,
    /* printerStatus   */ APS_PRINTER_UNKNOWN,
    /* printerName     */ "",
    /* docTitle        */ "",
    /* docRevision     */ "",
    /* docComments     */ "",
    /* docAuthor       */ "",
    /* docType         */ "",
    /* docCreator      */ "",
    /* ownerName       */ "",
    /* ownerID         */ -1,
    /* localHost       */ "",
    /* localFile       */ "",
    /* spoolHost       */ "",
    /* spoolFile       */ "",
    /* jobAttr         */ NULL
};

Aps_Result QuickJobInfoInit(Aps_QuickJobInfo *info, int version)
{
    /* check params */
    if (! info) return APS_INVALID_PARAM;

    /* init */
    switch (version) {
        /* version 0 */
        case 0:
            memcpy(info, & defaultsQuickJobInfo_v0, sizeof(Aps_QuickJobInfo));
            break;
        /* unknown version */
        default:
            return APS_INVALID_PARAM;
    }
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Uninitialize a QuickJobInfo.  Free all resources associated with
 * the structure EXCEPT any data which may only be valid when associated
 * with a job, eg. locks / objects.
 * DOES NOT reset to defaults.
 *
 * Parameters: info    - pointer to info structure
 *
 * Return    : A standard APS result code. Always APS_SUCCESS unless
 *             there's an invalid parameter.
 */
Aps_Result QuickJobInfoCleanup(Aps_QuickJobInfo *info)
{
    /* check params */
    if (! info) return APS_INVALID_PARAM;

    /* version 0 */
    strupdate(& info->jobHost, NULL);
    strupdate(& info->jobName, NULL);
    strupdate(& info->jobFilename, NULL);
    strupdate(& info->jobFormat, NULL);
    strupdate(& info->printerName, NULL);
    strupdate(& info->docTitle, NULL);
    strupdate(& info->docRevision, NULL);
    strupdate(& info->docComments, NULL);
    strupdate(& info->docAuthor, NULL);
    strupdate(& info->docType, NULL);
    strupdate(& info->docCreator, NULL);
    strupdate(& info->ownerName, NULL);
    strupdate(& info->localHost, NULL);
    strupdate(& info->localFile, NULL);
    strupdate(& info->spoolHost, NULL);
    strupdate(& info->spoolFile, NULL);
    /* version 1, here */
    if (info->version >= 1) return APS_INVALID_PARAM;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Recycle a QuickJobInfo.  Frees all resources and resets to defaults
 * everything EXCEPT:
 *    printerHandle
 *    printerName
 *    printerStatus
 *
 * Parameters: info    - pointer to info structure
 *
 * Return    : A standard APS result code. Always APS_SUCCESS unless
 *             there's an invalid parameter.
 */
Aps_Result QuickJobInfoRecycle(Aps_QuickJobInfo *info)
{
    Aps_Result result;

    char *oldPrinterName;
    Aps_PrinterStatus oldPrinterStatus;
    Aps_PrinterHandle oldPrinterHandle;

    /* check params */
    if (! info) return APS_INVALID_PARAM;

    /* retain old data */
    oldPrinterName = info->printerName;
    info->printerName = NULL;
    oldPrinterStatus = info->printerStatus;
    oldPrinterHandle = info->printerHandle;

    /* reinit */
    result = QuickJobInfoCleanup(info);
    if (result != APS_SUCCESS) return result;
    result = QuickJobInfoInit(info, info->version);
    if (result != APS_SUCCESS) return result;

    /* restore old data */
    info->printerName = oldPrinterName; /* safe, default is "" constant */
    info->printerStatus = oldPrinterStatus;
    info->printerHandle = oldPrinterHandle;
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * QuickJobInfoCreatePackage()
 *
 * Create a cloned package of an existing Aps_QuickJobInfo structure
 * using a contiguous block of memory for storage. The resulting object
 * can then be freed using Aps_ReleaseBuffer().
 *
 * The resulting object can no longer be modified in the standard fashion.
 *
 * Parameters: src        - existing Aps_QuickJobInfo
 *
 * Return    : (in param) - new structure pointer returned
 *                          Free with Aps_ReleaseBuffer()
 *
 *             result     - code indicating failure
 */
Aps_Result QuickJobInfoCreatePackage(Aps_QuickJobInfo **dest,
    const Aps_QuickJobInfo *src)
{
    int extraStorage;

    /* Sanity check */
    ASSERT(src && dest);

    /* Compute size of strings */
    extraStorage = QuickJobInfoComputePackageSize(src);

    /*** Allocate storage space ***/
    *dest = (Aps_QuickJobInfo *)
        TrackMemAlloc(NULL, sizeof(Aps_QuickJobInfo) + extraStorage, 0);
    if (*dest) {
        QuickJobInfoFillInPackage(*dest,
            ((char *)*dest) + sizeof(Aps_QuickJobInfo), src);
        return APS_SUCCESS;
    }
    return APS_OUT_OF_MEMORY;
}

/* ---------------------------------------------------------------------------
 * QuickJobInfoComputePackageSize()
 *
 * Compute the amount of memory a QuickJobInfo structure would consume as
 * a contiguous block of memory.  For use with QuickJobInfoFillInPackage().
 *
 * Parameters: src        - existing Aps_QuickJobInfo
 *
 * Return    : result     - integer representing size of structure
 */
int QuickJobInfoComputePackageSize(const Aps_QuickJobInfo *src)
{
    int extraStorage = 0;

    /*** Add up the sizes of all strings... (unoptimized) ***/
    /** Version 0 **/
    extraStorage += strtotallen(src->jobHost);
    extraStorage += strtotallen(src->jobName);
    extraStorage += strtotallen(src->jobFilename);
    extraStorage += strtotallen(src->jobFormat);
    extraStorage += strtotallen(src->printerName);
    extraStorage += strtotallen(src->docTitle);
    extraStorage += strtotallen(src->docRevision);
    extraStorage += strtotallen(src->docComments);
    extraStorage += strtotallen(src->docAuthor);
    extraStorage += strtotallen(src->docType);
    extraStorage += strtotallen(src->docCreator);
    extraStorage += strtotallen(src->ownerName);
    extraStorage += strtotallen(src->localHost);
    extraStorage += strtotallen(src->localFile);
    extraStorage += strtotallen(src->spoolHost);
    extraStorage += strtotallen(src->spoolFile);
    /** Version 1, here **/
    if (src->version >= 1) { }

    return extraStorage;
}

/* ---------------------------------------------------------------------------
 * QuickJobInfoFillInPackage()
 *
 * Copy a source QuickJobInfo into destination of appropriate size.
 * The size must first have been computed using
 * QuickJobInfoComputePackageSize(). This function will trash memory if
 * this requirement is not met!
 *
 * The resulting object can no longer be modified in the standard fashion.
 *
 * Parameters: src        - existing Aps_QuickJobInfo
 *             dest       - pointer to storage
 */
void QuickJobInfoFillInPackage(Aps_QuickJobInfo *dest, void *storage,
    const Aps_QuickJobInfo *src)
{
    /*** Copy data ***/
    dest->version = src->version;
    /** Version 0 **/
    dest->jobHandle = src->jobHandle;
    dest->jobStatus = src->jobStatus;
    dest->jobHost = strcpyupdate((char **)& storage, src->jobHost);
    dest->jobName = strcpyupdate((char **)& storage, src->jobName);
    dest->jobFilename = strcpyupdate((char **)& storage, src->jobFilename);
    dest->jobID = src->jobID;
    dest->jobSize = src->jobSize;
    dest->jobCreationTime = src->jobCreationTime;
    dest->jobFormat = strcpyupdate((char **)& storage, src->jobFormat);
    dest->jobOrder = src->jobOrder;
    dest->jobPriority = src->jobPriority;
    dest->printerHandle = src->printerHandle;
    dest->printerStatus = src->printerStatus;
    dest->printerName = strcpyupdate((char **)& storage, src->printerName);
    dest->docTitle = strcpyupdate((char **)& storage, src->docTitle);
    dest->docRevision = strcpyupdate((char **)& storage, src->docRevision);
    dest->docComments = strcpyupdate((char **)& storage, src->docComments);
    dest->docAuthor = strcpyupdate((char **)& storage, src->docAuthor);
    dest->docType = strcpyupdate((char **)& storage, src->docType);
    dest->docCreator = strcpyupdate((char **)& storage, src->docCreator);
    dest->ownerName = strcpyupdate((char **)& storage, src->ownerName);
    dest->ownerID = src->ownerID;
    dest->localHost = strcpyupdate((char **)& storage, src->localHost);
    dest->localFile = strcpyupdate((char **)& storage, src->localFile);
    dest->spoolHost = strcpyupdate((char **)& storage, src->spoolHost);
    dest->spoolFile = strcpyupdate((char **)& storage, src->spoolFile);
    dest->jobAttr = src->jobAttr;
    /** Version 1, here **/
    if (src->version >= 1) { }
}

/* ---------------------------------------------------------------------------
 * JobCreate()
 *
 * Creates a new Job object associated with a specific printer. It is
 * fully-initialized to defaults and almost ready for use.
 *
 * Job attributes are set to NULL and must be setup if this job is to
 * be dispatched.
 *
 * Parameters: printerHandle - printer handle
 *
 * Return    : (in param) handle of new object
 *             (result)   code indicating failure
 *
 * Note: Job object is create in a locked state.
 */

Aps_Result JobCreate(Aps_PrinterHandle printerHandle,
    Aps_JobHandle *jobHandle)
{
    Aps_Result       result;
    Aps_QuickJobInfo *info;
    Job              *job;
    Printer          *printer;

    /* Check params */
    ASSERT(jobHandle);
    *jobHandle = NULL;

    /* Get a lock on the printer */
    if (printerHandle) {
        result = Aps_AddRef(printerHandle);
        if (result != APS_SUCCESS) return APS_INVALID_HANDLE;
    /* Get default printer */
    } else {
        result = Aps_OpenDefaultPrinter(& printerHandle);
        if (result != APS_SUCCESS) return APS_NOT_FOUND;
    }

    /* Locate printer control structure */
    printer = PrinterGetPtrFromHandle(printerHandle);
    ASSERT(printer);

    /* Allocate and initialize QuickJobInfo structure */
    info = malloc(sizeof(Aps_QuickJobInfo));
    if (info) {
        QuickJobInfoInit(info, 0);

        /* Allocate Job structure */
        job = malloc(sizeof(Job));
        if (job) {
            /* Initialize internal fields */
            job->transportDefaultData = NULL;
            job->transportData = NULL;
            job->info = info;
            job->printer = printer;
            job->transport = job->printer->transport;
            ASSERT(job->transport);

            /* Setup the baseclass */
            result = ObjectInitialize(& job->baseClass, JOB_HANDLE);
            if (result == APS_SUCCESS) {
                /* Setup the handle */
                *jobHandle = JobGetHandleFromPtr(job);
                ASSERT(jobHandle);

                /* Setup a few things in the info structure */
                job->info->jobHandle = *jobHandle;
                job->info->printerHandle = printerHandle;
                if (strupdate(& job->info->printerName,
                    (printer->name) ? printer->name : "")) {

                    /* Tell the transport about our brand new object */
                    result = job->transport->vtbl->JobInit(job->transport, job);
                    if (result == APS_SUCCESS) {
                        return result;
                    }
                    strupdate(& job->info->printerName, NULL);
                } else result = APS_OUT_OF_MEMORY;
                Aps_ReleaseHandle(*jobHandle);
                *jobHandle = NULL;
            }
            free(job);
        } else result = APS_OUT_OF_MEMORY;
        free(info);
    } else result = APS_OUT_OF_MEMORY;
    Aps_ReleaseHandle(printerHandle);
    return result;
}

/* ---------------------------------------------------------------------------
 * JobDelete()
 *
 * Deallocates a job instance.  Expected to be called by Aps_ReleaseHandle().
 *
 * Parameters: job - A pointer to the Job object that is no longer needed.
 *
 *     Return: None.
 */
void JobDelete(Job * job)
{
    ASSERT(job);

    /* Free the Transport's interests in this object */
    if (job->transport) {
        job->transport->vtbl->JobCleanup(job->transport, job);
    }

    /* Free the attributes and printer */
    if (job->info) {
        if (job->info->jobAttr) Aps_ReleaseHandle(job->info->jobAttr);
        if (job->info->printerHandle) Aps_ReleaseHandle(job->info->printerHandle);

    /* Free any remaining strings */
        QuickJobInfoCleanup(job->info);

    /* Free the storage space */
        free(job->info);
    }

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    job->baseClass.identifier = INVALID_HANDLE;
    free(job);
}

/* ---------------------------------------------------------------------------
 * JobSetAttributes()
 *
 * Set the printing attributes for an active job.
 *
 * Parameters: job            - pointer to job to be affected.
 *             attrHandle     - job attributes handle, if NULL uses defaults.
 *
 * Return    : A standard APS result code.
 *
 * Old attributes are unlocked and removed. New attributes are locked and
 * added.
 * Job should be in correct sequence for this.
 */
Aps_Result JobSetAttributes(Job *job, Aps_JobAttrHandle attrHandle)
{
    Aps_Result result;
    ASSERT(job && job->info);

    /* Get defaults if we need them, else lock the ones we got */
    if (! attrHandle) {
        result = Aps_PrinterGetDefAttr(job->info->printerHandle, & attrHandle);
    } else {
        result = Aps_AddRef(attrHandle);
    }
    if (result != APS_SUCCESS) return result;

    /* Remove old attributes */
    if (job->info->jobAttr) Aps_ReleaseHandle(job->info->jobAttr);
    job->info->jobAttr = attrHandle;
    return result;
}

/* ---------------------------------------------------------------------------
 * JobSetup()
 *
 * Prepare a job for JobStart() or JobDispatch(). Sets up a few defaults
 * on behalf of the transport.
 *
 * Parameters: job    - pointer to job to operate on
 *             format - format string / identifier
 *             name   - name of job
 *
 * Return : A standard APS result code.
 */
Aps_Result JobSetup(Job *job, const char *format, const char *name,
    const char *filename)
{
    struct passwd *userinfo;

    ASSERT(job && job->info && name);

    /* Prepare job */
    job->info->jobStatus = APS_JOB_SETUP;
    if (! strupdatehostname(& job->info->jobHost)) goto nomem;
    if (! strupdate(& job->info->jobName,
        (name) ? name : "")) goto nomem;
    if (! strupdate(& job->info->jobFilename,
        (filename) ? filename : "")) goto nomem;
    if (! strupdate(& job->info->jobFormat,
        (format) ? format : "")) goto nomem;
    job->info->jobCreationTime = time(NULL);

    /* Get current user id and name */
    userinfo = getpwuid(getuid());
    job->info->ownerID = userinfo->pw_uid;
    strupdate(& job->info->ownerName, userinfo->pw_name);
    return APS_SUCCESS;

nomem:
    return APS_OUT_OF_MEMORY;
}

/* ---------------------------------------------------------------------------
 * JobUpdate()
 *
 * Update a job object. This function may leave some things in the
 * transport's reserved storage space.
 *
 * Parameters: job - Pointer to the Job object to be updated.
 *
 * Return: codes as spec'd for Aps_JobUpdate() high-level
 *             APS_NOT_FOUND if gone
 *             APS_NO_CHANGE if no change
 *             APS_IGNORED   if unable to update but probably still
 *                             exists
 *             other errors...
 */
Aps_Result JobUpdate(Job *job)
{
    ASSERT(job && job->transport);

    /* Update the job information */
    return job->transport->vtbl->JobUpdate(job->transport, job);
}

/* ---------------------------------------------------------------------------
 * JobIsOperationAvailable()
 *
 * Determines whether or not the specified operation can currently be
 * performed on a particular APS object.
 *
 * Parameters: job               - A pointer to a Job object
 *
 *             operation         - The operation that the application is
 *                                 interested in.
 *
 *             anticipatedResult - APS_OPERATION_AVAILABLE if this operation
 *                                 is currently available, or the anticipated
 *                                 reason for failure if it is known that
 *                                 this operation cannot be performed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *             APS_SUCCESS       - test is accurate
 *             APS_IGNORED       - test cannot be performed
 *             APS_NOT_FOUND     - operation ID not recognized
 */
Aps_Result JobIsOperationAvailable(Job *job,
    Aps_OperationID operation, Aps_Result *anticipatedResult)
{
    ASSERT(job && job->transport);

    /* Ask the transport what we can do about this... */
    return job->transport->vtbl->JobIsOperationAvailable(job->transport, job,
        operation, anticipatedResult);
}
