/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: jobaccess.c
 *
 * Description: APIs for job status inqueries.
 *              Covers jobs in any phase.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "job.h"

DEBUG_CHANNEL_DEFAULT(job)

/* ---------------------------------------------------------------------------
 * Aps_JobGetAttributes()
 *
 * Obtains the job settings object associated with a job.
 *
 * Parameters: job        - An Aps_JobHandle identifying the job to be
 *                          queried.
 *
 *             attributes - A pointer to an Aps_JobAttrHandle to receive a
 *                          handle to the job's attributes object.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobGetAttributes(Aps_JobHandle jobHandle,
                                Aps_JobAttrHandle *attributes)
{
    Job *job;

    /* Obtain a pointer to the Job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Provide the caller with information on the current state of the job. */
    *attributes = job->info->jobAttr;
    return (*attributes) ? APS_SUCCESS : APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_JobSetAttributes()
 *
 * Updates a job with a new attributes object.
 *
 * Parameters: job        - An Aps_JobHandle identifying the job to be
 *                          modified.
 *
 *             attributes - The new job attributes to use for this job. The
 *                          original job attributes handle remains valid after
 *                          this call, and must still be released when no
 *                          longer needed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobSetAttributes(Aps_JobHandle jobHandle,
                                Aps_JobAttrHandle attributes)
{
    Job *job;
    JobAttributes *callersAttributes;
    JobAttributes *newAttributes;
    
    /* Obtain a pointer to the Job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (job == NULL)
        return APS_INVALID_HANDLE;

    /* Obtain the JobAttributes instance corresponding to the provided
     * handle.
     */
    callersAttributes = JobAttrGetPtrFromHandle(attributes);
    if (callersAttributes == NULL)
        return APS_INVALID_HANDLE;

    /* Attempt to create a copy of the caller's job attributes. */
    newAttributes = JobAttrCreateCopy(callersAttributes);
    if (newAttributes == NULL)
        return APS_OUT_OF_MEMORY;
        
    /* Replace the current job attrbutes for this job with a copy of those
     * provided by the caller.
     */
    Aps_ReleaseHandle(job->info->jobAttr);
    job->info->jobAttr = JobAttrGetHandleFromPtr(newAttributes);
    
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_JobGetStatus()
 *
 * Obtains the current status and phase of a job.
 *
 * Parameters: jobHandle - An Aps_JobHandle identifying the job to query.
 *
 *             status    - A pointer to an Aps_JobStatus to receive the
 *                         current state information regarding this job.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobGetStatus(Aps_JobHandle jobHandle, Aps_JobStatus * status)
{
    Job *job;

    /* Check for reasonable parameter values. */
    if (status == NULL)
        return (APS_INVALID_PARAM);

    /* Obtain a pointer to the Job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (job == NULL)
        return (APS_INVALID_HANDLE);

    /* Provide the caller with information on the current state of the job. */
    *status = job->info->jobStatus;

    return (APS_SUCCESS);
}

/* ---------------------------------------------------------------------------
 * Aps_JobUpdate()
 *
 * Updates the given job object to reflect its current state in the queue.
 *
 * Parameters: jobHandle - An Aps_JobHandle identifying the job to update.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *
 * May return APS_NO_CHANGE
 */

Aps_Result Aps_JobUpdate(Aps_JobHandle jobHandle)
{
    Job *job;

    /* Check parameters and get job structure */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return APS_INVALID_HANDLE;

    return JobUpdate(job); /* submit to lower layer */
}

/* ---------------------------------------------------------------------------
 * Aps_JobMakeQuickJobInfo()
 *
 * Create an information structure for the application reflecting the last
 * stored status of a job. Aps_JobUpdate() should usually be called first.
 *
 * Parameters: jobHandle - An Aps_JobHandle identifying the job to query.
 *
 *             jobInfo   - Pointer to Aps_QuickJobInfo pointer to return
 *                         data to the user.
 *                         Free with Aps_ReleaseBuffer()
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */

Aps_Result Aps_JobMakeQuickJobInfo(Aps_JobHandle jobHandle,
    Aps_QuickJobInfo **jobInfo)
{
    Job *job;

    /* Check parameters and get job structure */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return APS_INVALID_HANDLE;
    if (! jobInfo) return APS_INVALID_PARAM;

    /* For the purposes of handing the structure back to the application
     * we must use QuickJobInfoCreatePackage().
     * The function will create a contiguous buffer containing all
     * required fields, including strings.
     */
    return QuickJobInfoCreatePackage(jobInfo, job->info);
}
