/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: metaconfig.c
 *
 * Description: Manages additional information that APS tracks for each
 *              printer.
 */

#include <string.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <pwd.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "metaconfig.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(cfg)

/* Prototype for functions private to this module. */
static Aps_Result MetaOpenForRead(FILE **stream, const char *configEntity);
static Aps_Result MetaOpenForWrite(FILE **stream, const char *configEntity);
static char *MetaGetFilenameInDir(const char *dir, const char *configEntity);

/* ---------------------------------------------------------------------------
 * MetaOpenForRead()
 *
 * Opens the meta configuration file for the specified printer / other entity
 * for read access.
 *
 * Parameters: stream       - Receives a pointer to the opened file on
 *                            success.
 *
 *             configEntity - A pointer to a string containing the name of
 *                            the printer or other configurable entity.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result MetaOpenForRead(FILE **stream, const char *configEntity)
{
    char *userMetaDir = NULL;
    char *metaFilename = NULL;
    Aps_Result result;

    /* Start with stream pointer as NULL. */
    *stream = NULL;

    /* First, look for an existing meta configuration file for this printer /
     * other configurable entity in the user's home directory.
     */
    userMetaDir = MetaGetUserDir();
    if (userMetaDir == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    metaFilename = MetaGetFilenameInDir(userMetaDir, configEntity);
    if (metaFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Attempt to open this printer/entity's meta configuration file
     * in this user's directory.
     */
    *stream = fopen(metaFilename, "r");
    if (*stream != NULL) {
        result = APS_SUCCESS;
        goto cleanup;
    }

    /* If we haven't been able to open the file in this user's directory,
     * attempt to open the system wide meta-configuration file for this
     * printer/entity.
     */
    free(metaFilename);
    metaFilename = MetaGetFilenameInDir(APSCFG_CONFIG_PATH_META_SYS_DIR,
        configEntity);
    if (metaFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    *stream = fopen(metaFilename, "r");
    if (*stream != NULL) {
        result = APS_SUCCESS;
        goto cleanup;
    }

    /* If we reach this point, no meta configuration file was found for the
     * specified printer / other configurable entity.
     */
    result = APS_NOT_FOUND;

cleanup:
    /* Free any temporarily allocated resources. */
    if (userMetaDir) free(userMetaDir);
    if (metaFilename) free(metaFilename);

    return result;
}

/* ---------------------------------------------------------------------------
 * MetaOpenForWrite()
 *
 * Opens the meta configuration file for the specified printer / other entity
 * for write access, creating it if necessary.
 *
 * Parameters: stream       - Receives a pointer to the opened file on
 *                            success.
 *
 *             configEntity - A pointer to a string containing the name of
 *                            the printer or other configurable entity.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result MetaOpenForWrite(FILE **stream, const char *configEntity)
{
    char *userMetaDir = NULL;
    char *userMetaFilename = NULL;
    char *systemMetaFilename = NULL;
    Aps_Result result;
    struct stat fileInfo;
    FILE *sourceStream;
    char copyBuffer[APSCFG_COPY_BUFFER_SIZE];

    /* Start with stream pointer as NULL. */
    *stream = NULL;

    /* First, look for an existing meta configuration file for this printer /
     * other configurable entity in the user's home directory.
     */
    userMetaDir = MetaGetUserDir();
    if (userMetaDir == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    userMetaFilename = MetaGetFilenameInDir(userMetaDir, configEntity);
    if (userMetaFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Check whether this file exists. */
    if (stat(userMetaFilename, &fileInfo) == 0) {
        /* Attempt to open this printer/entity's meta configuration file
         * in this user's directory.
         */
        *stream = fopen(userMetaFilename, "r+");
        if (*stream != NULL) {
            result = APS_SUCCESS;
            goto cleanup;
        } else {
            /* A configuration file exists for this user, but it couldn't
             * be opened for some reason.
             */
            result = GetResultFromErrno();            
            goto cleanup;
        }
    }

    /* If we haven't been able to open the file in this user's directory,
     * attempt to open the system wide meta-configuration file for this
     * printer/entity.
     */
    systemMetaFilename = MetaGetFilenameInDir(APSCFG_CONFIG_PATH_META_SYS_DIR,
        configEntity);
    if (systemMetaFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    if (stat(systemMetaFilename, &fileInfo) == 0) {
       /* If the file already exists, open it for reading and writing. */
       *stream = fopen(systemMetaFilename, "r+");
    } else {
       /* If the file doesn't already exist, then create it. */
       *stream = fopen(systemMetaFilename, "w+");
    }
    if (*stream != NULL) {
        result = APS_SUCCESS;
        goto cleanup;
    }

    /* Ensure that ther user's meta configuration directory exists. */
    mkdir(userMetaDir, 0755);

    /* If we get to this point, then no user-specific configuration exists
     * for this user, and we were unable to write to or create the system-
     * wide configuration file, so we create a new user-specific meta
     * configuration file.
     */
    *stream = fopen(userMetaFilename, "w+");
    if (*stream == NULL) {
        result = GetResultFromErrno();            
        goto cleanup;
    }

    /* If a system-wide configuration file exists, then copy it's contents
     * to the new configuration file in the user's directory.
     */
    sourceStream = fopen(systemMetaFilename, "r");
    if (sourceStream != NULL) {
        while (fgets(copyBuffer, sizeof(copyBuffer), sourceStream)) {
            fputs(copyBuffer, *stream);
        }
        fclose(sourceStream);
    }

    /* If we reach this point, we have successfully create a new meta
     * configuration file for this user.
     */
    result = APS_SUCCESS;

cleanup:
    /* On success, we position the stream pointer to the start of the file. */
    if (result == APS_SUCCESS) {
        fseek(*stream, 0, SEEK_SET);
    }

    /* Free any temporarily allocated resources. */
    if (userMetaDir) free(userMetaDir);
    if (userMetaFilename) free(userMetaFilename);
    if (systemMetaFilename) free(systemMetaFilename);

    return result;
}

/* ---------------------------------------------------------------------------
 * MetaGetUserDir()
 *
 * Obtains the name of the directory in which meta configuration information
 * local to the current user is stored.
 *
 * Parameters: None.
 *
 *     Return: A pointer to a newly allocated string with this user's
 *             meta configuration directory, or NULL on failure.
 *             Free using free().
 */
char *MetaGetUserDir(void)
{
    struct passwd *userInfo;
    char *homeDir;
    char *userMetaDir;

    /* Obtain the current user's home directory. */
    userInfo=getpwuid(getuid());
    if (userInfo == NULL)
        return (NULL);
    homeDir = userInfo->pw_dir;

    /* Allocate memory to hold the user's meta congifuration directory. */
    userMetaDir = malloc(strlen(homeDir) +
        strlen(APSCFG_CONFIG_PATH_META_USER_RELDIR) + 2);
    if (userMetaDir == NULL)
        return NULL;

    /* Generate the path to the directory where the file will be stored. */
    strcpy(userMetaDir, homeDir);
    if (userMetaDir[strlen(userMetaDir) - 1] != '/')
        strcat(userMetaDir, "/");
    strcat(userMetaDir, APSCFG_CONFIG_PATH_META_USER_RELDIR);

    return userMetaDir;
}

/* ---------------------------------------------------------------------------
 * MetaGetFilenameInDir()
 *
 * Generates the full path and filename for a meta configuration file for
 * the specified entity and directory.
 *
 * Parameters: dir          - The name of the directory in which the file
 *                            should reside.
 *
 *             configEntity - The name of the configurable entity for which
 *                            a filename should be generated.
 *
 *     Return: A pointer to a newly allocated string with the full path and
 *             filename for the configuration file, or NULL on failure.
 *             Free using free().
 */
static char *MetaGetFilenameInDir(const char *dir, const char *configEntity)
{
    char *fullPath;

    /* Allocate memory for the filename buffer. */
    fullPath = malloc(strlen(dir) + strlen(configEntity)
                      + strlen(APSCFG_CONFIG_PATH_META_SUFFIX) + 1);
    if (fullPath == NULL)
        return (NULL);

    /* Generate the path to the directory where the file will be stored. */
    strcpy(fullPath, dir);

    /* Append meta configuration filename for the specified printer/entity. */
    strcat(fullPath, configEntity);
    strcat(fullPath, APSCFG_CONFIG_PATH_META_SUFFIX);

    return (fullPath);
}

/* ---------------------------------------------------------------------------
 * MetaRead()
 *
 * Reads the specified meta information associated with the specified printer
 * or other configurable entity.
 *
 * Parameters: value        - Receives the value of the specified key.
 *
 *             configEntity - A pointer to a string containing the name of
 *                            the printer or other configurable entity.
 *
 *             key          - A pointer to a string identifying which piece
 *                            of meta information should be retrieved.
 *
 *     Return: A pointer to a newly allocated string containing the value for
 *             the specified key, or NULL if the key couldn't be read. It
 *             is the callers responsibility to deallocate this string when
 *             it is finished with it.
 *             Free using free().
 */
Aps_Result MetaRead(char **value, const char *configEntity, const char *key)
{
    Aps_Result result;
    FILE *stream = NULL;
    char *fileBuffer = NULL;
    int fileBufferSize = APSCFG_FILE_BUFFER_SIZE;
    char *newBuffer;
    char *delimiter;

    /* Open the appropriate meta configuration file. */
    result = MetaOpenForRead(&stream, configEntity);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Allocate memory for file buffer. */
    fileBuffer = malloc(fileBufferSize);
    if (fileBuffer == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Look for the specified key in the file. */

    /* Start with an emtpy file buffer string. */
    fileBuffer[0] = '\0';

    while (!feof(stream)) {
        /* Ensure that the buffer is large enough to hold the next portion of
         * the line currently being read.
         */
        if (strlen(fileBuffer) + APSCFG_FILE_BUFFER_SIZE < fileBufferSize) {
            fileBufferSize += APSCFG_FILE_BUFFER_SIZE;
            newBuffer = realloc(fileBuffer, fileBufferSize);
            if (newBuffer == NULL) {
                result = APS_OUT_OF_MEMORY;
                goto cleanup;
            }
            fileBuffer = newBuffer;
        }

        /* Attempt to read the next line from the file. */
        if (fgets(fileBuffer + strlen(fileBuffer), APSCFG_FILE_BUFFER_SIZE, stream)
            == NULL)
            break;
        /* If we've now read the end of the line into the buffer, check whether
         * this is the key that the caller is querying.
         */
        if (fileBuffer[strlen(fileBuffer) - 1] == '\n') {
            /* Search for the first '=' character, which is the key name /
             * value delimiters. '=' characters are permitted in value
             * strings, but not in key strings.
             */
            delimiter = strchr(fileBuffer, '=');
            if (delimiter != NULL) {
                *delimiter = '\0';

                /* Compare the first part of the file buffer string, which
                 * now consists of only the key name, with the key requested
                 * by the caller.
                 */
                if (strcmp(fileBuffer, key) == 0) {
                    /* We've found the entry that the caller has queried. */

                    /* Move the value to the beginning of the string, so that
                     * we can pass this string back to the caller, and it
                     * can be deallocated by the caller.
                     */
                    memmove(fileBuffer, delimiter + 1,
                            strlen(delimiter + 1) + 1);
                    
                    /* Remove line feed character from end of string. */
                    fileBuffer[strlen(fileBuffer) - 1] = '\0';

                    /* Provide the caller with the pointer to this string. */
                    *value = fileBuffer;

                    /* Set local fileBuffer pointer to NULL, indicating that
                     * there is no longer a temporary file buffer owned by
                     * this function to be deleted before returning.
                     */
                    fileBuffer = NULL;

                    /* We've now successfully obtained the requested value
                     * for the caller.
                     */
                    result = APS_SUCCESS;

                    goto cleanup;
                }
            }

            /* Otherwise, continue by searching the next line in the file.
             * We empty the current file buffer in prepartion for receiving
             * the next line of the file.
             */
            fileBuffer[0] = '\0';
        }
    }

    /* If we reach this point, then the specified key was not found in the
     * configuration file.
     */
    result = APS_NOT_FOUND;

cleanup:
    /* Close the configuration file if it was ever opened. */
    if (stream != NULL) {
        fclose(stream);
    }

    /* Release the memory used by the file buffer. */
    if (fileBuffer) free(fileBuffer);

    return result;
}

/* ---------------------------------------------------------------------------
 * MetaWrite()
 *
 * Writes the specified meta information associated with the specified
 * printer or other configurable entity.
 *
 * Parameters: configEntity - A string containing the name of the printer.
 *
 *             key          - A string identifying which piece of information
 *                            should be written.
 *
 *             value        - A string containing the new value for this
 *                            setting.
 *
 *     Return: A standard Aps_Result code.
 */
Aps_Result MetaWrite(const char *configEntity, const char *key,
                     const char *value)
{
    Aps_Result result;
    FILE *stream = NULL;
    char fileBuffer[APSCFG_FILE_BUFFER_SIZE];
    char *delimiter;
    int startOfKey;
    char *tempFilename = NULL;
    FILE *tempStream = NULL;

    /* Open the appropriate meta configuration file. */
    result = MetaOpenForWrite(&stream, configEntity);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If the specified key already exists in the file, then remove it. */
    while (!feof(stream)) {
        /* Record where this key starts. */
        startOfKey = ftell(stream);

        /* Attempt to read the next line from the file. */
        if (fgets(fileBuffer, sizeof(fileBuffer), stream) == NULL)
            break;

        /* Search for the first '=' character. */
        delimiter = strchr(fileBuffer, '=');
        if (delimiter != NULL) {
            *delimiter = '\0';
            delimiter++;

            /* Compare the first part of the file buffer string, which
             * now consists of only the key name, with the key provided
             * by the caller.
             */
            if (strcmp(fileBuffer, key) == 0) {
                /* We've found this key already present in the file, so we
                 * will remove it.
                 */
                
                /* Find the start of the next key in the file. */
                if (delimiter[strlen(delimiter) - 1] != '\n') {
                    while (fileBuffer[strlen(fileBuffer) - 1] != '\n'
                        && !feof(stream)) {
                        if (fgets(fileBuffer, sizeof(fileBuffer), stream)
                            == NULL)
                           break;
                    }
                }

                /* Create a temporary file to hold the remainder of the
                 * configuration file while we insert the new value at
                 * this position in the file.
                 */
                tempFilename = tempnam(NULL, "APS__");
                if (tempFilename == NULL) {
                    goto cleanup;
                }
                tempStream = fopen(tempFilename, "w+");
                if (tempStream == NULL) {
                    result = GetResultFromErrno();
                    goto cleanup;
                }
                
                while(!feof(stream)) {
                    if (fgets(fileBuffer, sizeof(fileBuffer), stream)
                        == NULL)
                        break;
                    fputs(fileBuffer, tempStream);
                }

                /* Seek back to the position where the original key was found.
                 */
                fseek(stream, startOfKey, SEEK_SET);

                /* Write the new value of the key. */
                fprintf(stream, "%s=%s\n", key, value);

                /* Write the remainder of the original file after this key. */
                rewind(tempStream);
                while(!feof(tempStream)) {
                    if (fgets(fileBuffer, sizeof(fileBuffer), tempStream)
                        == NULL)
                        break;
                    fputs(fileBuffer, stream);
                }

                /* Truncate the file at this position. */
                ftruncate(fileno(stream), ftell(stream));

                /* We've now succeeded in writing the new value to the file. */
                result = APS_SUCCESS;
                goto cleanup;
            }
        }
    }

    /* A copy of the specified key was not found in this meta configuration
     * file, so we write the value at the end of the file.
     */
    fprintf(stream, "%s=%s\n", key, value);

    /* We've now succeeded in writing the new value to the file. */
    result = APS_SUCCESS;

cleanup:
    /* Close the configuration file if it was ever opened. */
    if (stream != NULL) {
        fclose(stream);
    }

    /* Close and delete the temporary file if it was ever opened. */
    if (tempStream != NULL) {
        fclose(tempStream);
        ASSERT(tempFilename != NULL);
        remove(tempFilename);
    }

    /* Release temporary memory allocated within this function. */
    if (tempFilename) free(tempFilename);

    return result;
}

/* ---------------------------------------------------------------------------
 * MetaRename()
 *
 * Changes the name of a configuration entity, maintaining any information
 * configured for that entity.
 *
 * Parameters: existingEntity - The name of an existing meta configuration
 *                              entity.
 *
 *             newEntity      - The new name to give this entity.
 *
 *     Return: A standard Aps_Result code.
 */
Aps_Result MetaRename(const char *existingEntity, const char *newEntity)
{
    char *sourceSysFilename = NULL;
    char *destSysFilename = NULL;
    char *sourceUserFilename = NULL;
    char *destUserFilename = NULL;
    char *userMetaDir = NULL;
    Aps_Result result;

    /* Attempt to rename any user-specific meta configuration for this
     * entity, for the current user.
     */
    userMetaDir = MetaGetUserDir();
    if (userMetaDir == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    sourceUserFilename = MetaGetFilenameInDir(userMetaDir, existingEntity);
    if (sourceUserFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    destUserFilename = MetaGetFilenameInDir(userMetaDir, newEntity);
    if (destUserFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    rename(sourceUserFilename, destUserFilename);

    /* Likewise, attempt to rename any system-specific meta configuration for
     * this entity.
     */
    sourceSysFilename = MetaGetFilenameInDir(APSCFG_CONFIG_PATH_META_SYS_DIR,
        existingEntity);
    if (sourceSysFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    destSysFilename = MetaGetFilenameInDir(APSCFG_CONFIG_PATH_META_SYS_DIR,
        newEntity);
    if (destSysFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    rename(sourceSysFilename, destSysFilename);

    /* We've completed renaming any renamable meta configuration file
     * associated with this entity.
     */
    result = APS_SUCCESS;
    
cleanup:
    /* Release any temporary resources allocated by this function. */
    if (sourceSysFilename) free(sourceSysFilename);
    if (destSysFilename) free(destSysFilename);
    if (sourceUserFilename) free(sourceUserFilename);
    if (destUserFilename) free(destUserFilename);
    if (userMetaDir) free(userMetaDir);

    return result;
}

/* ---------------------------------------------------------------------------
 * MetaRemove()
 *
 * Removes the specified key from the specified configuration entity,
 * if it exists. If the key does not exist, this function will still report
 * success.
 *
 * Parameters: configEntity - A string containing the name of the printer.
 *
 *             key          - A string identifying which piece of information
 *                            should be written.
 *
 *     Return: A standard Aps_Result code.
 */
Aps_Result MetaRemove(const char *configEntity, const char *key)
{
    Aps_Result result;
    FILE *stream = NULL;
    char fileBuffer[APSCFG_FILE_BUFFER_SIZE];
    char *delimiter;
    int startOfKey;
    char *tempFilename = NULL;
    FILE *tempStream = NULL;

    /* Open the appropriate meta configuration file. */
    result = MetaOpenForWrite(&stream, configEntity);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If the specified key already exists in the file, then remove it. */
    while (!feof(stream)) {
        /* Record where this key starts. */
        startOfKey = ftell(stream);

        /* Attempt to read the next line from the file. */
        if (fgets(fileBuffer, sizeof(fileBuffer), stream) == NULL)
            break;

        /* Search for the first '=' character. */
        delimiter = strchr(fileBuffer, '=');
        if (delimiter != NULL) {
            *delimiter = '\0';
            delimiter++;

            /* Compare the first part of the file buffer string, which
             * now consists of only the key name, with the key provided
             * by the caller.
             */
            if (strcmp(fileBuffer, key) == 0) {
                /* We've found this key already present in the file, so we
                 * will remove it.
                 */
                
                /* Find the start of the next key in the file. */
                if (delimiter[strlen(delimiter) - 1] != '\n') {
                    while (fileBuffer[strlen(fileBuffer) - 1] != '\n'
                        && !feof(stream)) {
                        if (fgets(fileBuffer, sizeof(fileBuffer), stream)
                            == NULL)
                           break;
                    }
                }

                /* Create a temporary file to hold the remainder of the
                 * configuration file.
                 */
                tempFilename = tempnam(NULL, "APS__");
                if (tempFilename == NULL) {
                    goto cleanup;
                }
                tempStream = fopen(tempFilename, "w+");
                if (tempStream == NULL) {
                    result = GetResultFromErrno();
                    goto cleanup;
                }
                
                while(!feof(stream)) {
                    if (fgets(fileBuffer, sizeof(fileBuffer), stream)
                        == NULL)
                        break;
                    fputs(fileBuffer, tempStream);
                }

                /* Seek back to the position where the original key was found.
                 */
                fseek(stream, startOfKey, SEEK_SET);

                /* Write the remainder of the original file after the key
                 * to be removed.
                 */
                rewind(tempStream);
                while(!feof(tempStream)) {
                    if (fgets(fileBuffer, sizeof(fileBuffer), tempStream)
                        == NULL)
                        break;
                    fputs(fileBuffer, stream);
                }

                /* Truncate the file at this position. */
                ftruncate(fileno(stream), ftell(stream));

                /* We've now succeeded in removing the key. */
                result = APS_SUCCESS;
                goto cleanup;
            }
        }
    }

    /* The specified key was not found in the file, so there was nothing
     * for us to do.
     */
    result = APS_SUCCESS;

cleanup:
    /* Close the configuration file if it was ever opened. */
    if (stream != NULL) {
        fclose(stream);
    }

    /* Close and delete the temporary file if it was ever opened. */
    if (tempStream != NULL) {
        fclose(tempStream);
        ASSERT(tempFilename != NULL);
        remove(tempFilename);
    }

    /* Release temporary memory allocated within this function. */
    if (tempFilename) free(tempFilename);

    return result;
}

/* ---------------------------------------------------------------------------
 * MetaRemoveAll()
 *
 * Removes any meta-configuration information associated with a particular
 * configurable entity.
 *
 * Parameters: configEntity - The name of an entity with associated meta-
 *                            configuration information (such as a printer)
 *                            to be removed.
 *
 *     Return: A standard Aps_Result code.
 */
Aps_Result MetaRemoveAll(const char *configEntity)
{
    char *userMetaDir = NULL;
    char *userMetaFilename = NULL;
    char *systemMetaFilename = NULL;
    Aps_Result result;

    /* Remove user-specific meta configuration file, if it exists. */
    userMetaDir = MetaGetUserDir();
    if (userMetaDir == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    userMetaFilename = MetaGetFilenameInDir(userMetaDir, configEntity);
    if (userMetaFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    remove(userMetaFilename);

    /* Remove system-wide meta configuraiton file. */
    systemMetaFilename = MetaGetFilenameInDir(APSCFG_CONFIG_PATH_META_SYS_DIR,
        configEntity);
    if (systemMetaFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    remove(systemMetaFilename);

    /* If we reach this point, we have successfully create a new meta
     * configuration file for this user.
     */
    result = APS_SUCCESS;

cleanup:
    /* Free any temporarily allocated resources. */
    if (userMetaDir) free(userMetaDir);
    if (userMetaFilename) free(userMetaFilename);
    if (systemMetaFilename) free(systemMetaFilename);

    return result;
}
