/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: model.c
 *
 * Description: Functionality for working with printer model objects.
 *
 */

#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "object.h"
#include "model.h"

DEBUG_CHANNEL_DEFAULT(model)

/* Definition of the Model object structure. */
typedef struct Model_ {
    ApsObject baseClass;
} Model;

/* Functions that are private to this model. */
static Model *ModelGetPtrFromHandle(Aps_ModelHandle handle);
static Aps_ModelHandle ModelGetHandleFromPtr(Model * model);

/* ---------------------------------------------------------------------------
 * ModelGetPtrFromHandle()
 *
 * Obtains a the pointer to a Model object given its handle.
 *
 * Parameters: handle - A handle to a currently open model.
 *
 *     Return: A pointer to a Model object, or NULL if this isn't a
 *             valid model handle.
 */
static Model *ModelGetPtrFromHandle(Aps_ModelHandle handle)
{
    Model *model;

    if (handle == NULL)
        return NULL;

    model = (Model *) handle;

    /* Check to see that this looks like a valid Printer object. */
    if (model->baseClass.identifier != MODEL_HANDLE)
        return NULL;

    return model;
}

/* ---------------------------------------------------------------------------
 * ModelGetHandleFromPtr()
 *
 * Obtains a handle to a given Model object.
 *
 * Parameters: model - A pointer to a Model object.
 *
 *     Return: A handle to the model object.
 */
static Aps_ModelHandle ModelGetHandleFromPtr(Model * model)
{
    return (Aps_ModelHandle)model;
}

/* ---------------------------------------------------------------------------
 * ModelCreate()
 *
 * Creates a new model object.
 *
 * Parameters: modelHandle - Receives a handle to a new model object.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result ModelCreate(Aps_ModelHandle *modelHandle)
{
    Model *model;
    Aps_Result result;

    /* Attempt to allocate space for the new NotificationSub. */
    model = (Model *)malloc(sizeof(Model));
    if (model == NULL)
        return APS_OUT_OF_MEMORY;

    /* Attempt to initialize the base information common to all publically-
     * visible APS objects.
     */
    result = ObjectInitialize(&model->baseClass, MODEL_HANDLE);
    if(result != APS_SUCCESS) {
        free(model);
        return result;
    }

    /* Provide the caller with a handle to this model object. */
    *modelHandle = ModelGetHandleFromPtr(model);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ModelDelete()
 *
 * Deletes a model object, given its handle.
 *
 * Parameters: modelHandle - A handle to a currently open model.
 *
 *     Return: None.
 */
void ModelDelete(Aps_ModelHandle modelHandle)
{
    Model *model = ModelGetPtrFromHandle(modelHandle);
    ASSERT(model != NULL);

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    model->baseClass.identifier = INVALID_HANDLE;

    /* Deallocate the memory used by the Model structure. */
    free(model);
}
