/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printerconfig.c
 *
 * Description: APIs for administrative and tuning configuration of a
 *              printer.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetMaxJobSize()
 *
 * Obtains the maximum size of job, in kilobytes, that this printer will
 * accept.
 *
 * Parameters: printerHandle - A handle to the printer in question.
 *
 *             maxSize - A pointer to an int to receive the maximum job size.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterGetMaxJobSize(Aps_PrinterHandle printerHandle, int *maxSize)
{
    Printer *printer;

    /* Check for reasonable parameter values. */
    if (printerHandle == NULL)
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Dispatch to the transport for processing */
    ASSERT(printer->transport);
    return printer->transport->vtbl->PrinterGetMaxJobSize(printer->transport,
        printerHandle, maxSize);
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterSetMaxJobSize()
 *
 * Changes the maximum size of job, in kilobytes, that a printer will accept.
 *
 * Parameters: printerHandle - A handle to the printer to modify.
 *
 *             maxSize - The maximum size of job to accept, or
 *                       APS_NO_MAX_SIZE to permit arbitrarily large jobs.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterSetMaxJobSize(Aps_PrinterHandle printerHandle, 
                                                             int maxSize)
{
    Printer *printer;

    /* Check for reasonable parameter values. */
    if ((printerHandle == NULL) || (maxSize < 0 ))
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Dispatch to the transport for processing */
    ASSERT(printer->transport);
    return printer->transport->vtbl->PrinterSetMaxJobSize(printer->transport,
        printerHandle, maxSize);
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetConfigFlags()
 *
 * Obtains the set of printer configuration flags that currently apply to a
 * particular printer.
 *
 * Parameters: printerHandle     - A handle to the printer to query.
 *
 *             configFlags - A pointer to the int to receive the printer
 *                           configuration flags that are currently on.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterGetConfigFlags(Aps_PrinterHandle printerHandle,
                                     long int *configFlags)
{
    Printer *printer;

    /* Check for reasonable parameter values. */
    if (printerHandle == NULL)
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Dispatch to the transport for processing */
    ASSERT(printer->transport);
    return printer->transport->vtbl->PrinterGetConfigFlags(printer->transport,
        printer, configFlags);
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterSetConfigFlags()
 *
 * Changes the set of printer configuration flags to apply to this printer.
 *
 * Parameters: printerHandle -The printer whose configuration flags are to be
 *                            modified.
 *
 *             flagsToSet   - The set of flags to turn on, or 0 for none.
 *
 *             flagsToReset - The set of flags to turn off, or 0 for none.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterSetConfigFlags(Aps_PrinterHandle printerHandle,
                                     long int flagsToSet,
                                     long int flagsToReset)
{
    Printer *printer;
    long int flagMask = (flagsToSet | flagsToReset);
    long int flagSet = flagsToSet;
    long int allFlagsValue; 


    allFlagsValue = (APS_CONFIG_EOF_AT_END |
                     APS_CONFIG_ADD_CR |
                     APS_CONFIG_TEXT_AS_TEXT|
                     APS_CONFIG_HEADER_PAGE);
    /* If no flags to be Set/Reset, return */
    if (flagMask == 0)
        return APS_NO_CHANGE;

    /* Check for reasonable parameter values. */
    if ((printerHandle == NULL) || (flagsToSet  & flagsToReset))
        return APS_INVALID_PARAM;

    if ((flagMask  & (~allFlagsValue)) || (flagSet & (~allFlagsValue)))
        return APS_INVALID_PARAM;



    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Dispatch to the transport for processing */
    ASSERT(printer->transport);
    return printer->transport->vtbl->PrinterSetConfigFlags(printer->transport,
        printer, flagMask, flagSet);
}
