/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printerdiag.c
 *
 * Description: Printer diagnosis APIs.
 *
 */

#include <stdio.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <time.h>
 
#include "aps.h"
#include "apsinternal.h"
#include "printer.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* Prototypes for private helper functions. */
static Aps_Result DiagSendPSTestPage(Aps_PrinterHandle printerHandle,
                                     Aps_JobHandle job);
static Aps_Result DiagDrawPSText(Aps_JobHandle job, const char *text,
                                 int x, int y);
static Aps_Result DiagMoveTo(Aps_JobHandle job, int x, int y);
static Aps_Result DiagDrawLabelledRectangle(Aps_JobHandle job,
                                            int left,
                                            int top,
                                            int right,
                                            int bottom,
                                            const char *text);

/* Macro to simplify code for writing data to a job within this module. */
#define WRITESTR(job, str) result = Aps_JobWriteString(job, str); \
                           if (result != APS_SUCCESS) goto cleanup

#define DISPLAY_STR_SIZE 100
                                                          
/* ---------------------------------------------------------------------------
 * Aps_PrinterSendTestPage()
 *
 * Sends a standard test page to the printer in question.
 *
 * Parameters: printer   - A handle to the printer to print a test page on.
 *
 *             jobOutput - An optional Aps_JobHandle to receive a handle to
 *                         the test page job to allow future monitoring of
 *                         this job, or NULL if not requested.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterSendTestPage(Aps_PrinterHandle printer,
                                   Aps_JobHandle *jobOutput)
{
    Aps_Result result;
    Aps_JobHandle job;

    /* Attempt to start a new job on this printer. */
    result = Aps_PrinterStartJob(printer, APS_FORMAT_POSTSCRIPT, &job);
    if (result != APS_SUCCESS)
        return result;

    /* Send the a PostScript-format test page. */
    result = DiagSendPSTestPage(printer, job);
    if (result != APS_SUCCESS) {
        Aps_JobAbort(job);
        return result;
    }

    /* Provide the caller with a handle to this job to permit further
     * monitoring of job, if they wish to receive it.
     */
    if (jobOutput != NULL) {
        *jobOutput = job;
        
        /* Record that the application has a reference to this job, so that
         * it will be deleted only when this last reference is released.
         */
        Aps_AddRef(job);
    }

    /* Indidate that we've finished generating this job, so that it will
     * be scheduled for printing if it hasn't already been. This function
     * also implicity releases our reference to the job handle.
     */
    Aps_JobEnd(job);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * DiagSendPSTestPage()
 *
 * Sends PostScript code to a job to print a test page.
 *
 * Parameters: printerHandle - A handle to the printer to which the test page
 *                             is being sent.
 *
 *             job           - A handle to the job to receive PostScript code
 *                             to render the test page.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result DiagSendPSTestPage(Aps_PrinterHandle printerHandle,
                                     Aps_JobHandle job)
{
    Aps_Result result;
    Printer *printer = PrinterGetPtrFromHandle(printerHandle);
    int leftMargin = 90;
    int vertPos;
    char *tempStr1;
    char *tempStr2;
    Aps_JobAttrHandle jobAttributes;
    Aps_AttrOption **options;
    int numOptions;
    int i;
    Aps_Resolution res;
    Aps_PageSize *pageSize = NULL;
    char displayStr[DISPLAY_STR_SIZE];
    Aps_ConnectionType connection;
    struct passwd *userInfo;
    int left, top, right, bottom;
    time_t currentTime;
        
    /* Check for reasonable parameter values. */
    if (printer == NULL)
        return APS_INVALID_HANDLE;

    /* Obtain a handle to the job attributes object for this job. This
     * will be identical to the printer's default job attributes. Since
     * this isn't a copy of the job attributes, it doesn't need to be
     * explicitly released.
     */
    if ((result = Aps_JobGetAttributes(job, &jobAttributes)) != APS_SUCCESS)
        goto cleanup;
        
    /* Write file header to switch printer into PostScript mode. */
    if ((result = Aps_JobWriteBlock(job, APS_BLOCK_FILE_HEADER)) != APS_SUCCESS)
        goto cleanup;

    /* Obtain the default page size information, if available. */
    Aps_AttrQuickGetPageSize(jobAttributes, &pageSize);
            
    /* Begin PostScript portion of the job, writing DSC-compliant header
     * section.
     */
    WRITESTR(job, "%!PS-Adobe-3.0\n");
    WRITESTR(job, "%%Title: Printer Test Page\n");
    WRITESTR(job, "%%Creator: Application Print Services Library\n");
    WRITESTR(job, "%%LanguageLevel: 1\n");
    WRITESTR(job, "%%Pages: 1\n");
    WRITESTR(job, "%%DocumentNeededFonts: Helvetica\n");
    WRITESTR(job, "%%BoundingBox: 0 0 612 792\n");
    WRITESTR(job, "%%EndComments\n");

    /* Write firmware patch to the device, if we have one. */
    if ((result = Aps_JobWriteBlock(job, APS_BLOCK_PS_PATCH)) != APS_SUCCESS)
        goto cleanup;
    
    /* Write procedure definition section. */
    WRITESTR(job, "%%BeginProlog\n");
    WRITESTR(job, "/BeginEPSF {\n"
                  "  /b4_Inc_state save def\n"
                  "  /dict_count countdictstack def\n"
                  "  /op_count count 1 sub def\n"
                  "  userdict begin\n"
                  "  /showpage {} def\n"
                  "} bind def\n");
    WRITESTR(job, "/EndEPSF {\n"
                  "  count op_count sub {pop} repeat\n"
                  "  countdictstack dict_count sub {end} repeat\n"
                  "  b4_Inc_state restore\n"
                  "  } bind def\n");
    WRITESTR(job, "%%EndProlog\n");

    /* Write document setup section. */
    WRITESTR(job, "%%BeginSetup\n");
    if ((result = Aps_JobWriteBlock(job, APS_BLOCK_PS_DEV_SETUP)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "%%EndSetup\n");

    /* Write the page itself. */

    /* Write page header and setup comments. */
    WRITESTR(job, "%%Page: 1 1\n");
    WRITESTR(job, "%%BeginPageSetup\n");
    WRITESTR(job, "%%EndPageSetup\n");

    /* Select the font to be used for page rectangle captions. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "8 scalefont setfont\n\n");

    /* Draw rectangles showing the edge of the page. */
    
    /* Determine where the edge of the page lies. */
    left = 0;
    bottom = 0;
    if (pageSize != NULL) {
        top = pageSize->mediaHeight;
        right = pageSize->mediaWidth;
    } else {
        /* Assume 8.5" x 11" page if this information has not been configured. */
        top = 792;
        right = 612;
    }

    /* Draw these rectangles and their text in light gray. */
    WRITESTR(job, ".75 setgray\n\n");
    
    /* Draw rectangle at very edge of page. */
    if ((result = DiagDrawLabelledRectangle(job, left, top, right, bottom,
                                            "Edge of page")) != APS_SUCCESS)
        goto cleanup;
        
    /* Draw rectangle .25" from the edge of the page. */
    left += 18;
    bottom += 18;
    right -= 18;
    top -= 18;
    if ((result = DiagDrawLabelledRectangle(job, left, top, right, bottom,
                                            "1/4\" / 6.35mm from edge of page"))
                                            != APS_SUCCESS)
        goto cleanup;

    /* Draw rectangle .5" from the edge of the page. */
    left += 18;
    bottom += 18;
    right -= 18;
    top -= 18;
    if ((result = DiagDrawLabelledRectangle(job, left, top, right, bottom,
                                            "1/2\" / 12.7mm from edge of page"))
                                            != APS_SUCCESS)
        goto cleanup;

    /* Draw rectangle 1" from the edge of the page. */
    left += 36;
    bottom += 36;
    right -= 36;
    top -= 36;
    if ((result = DiagDrawLabelledRectangle(job, left, top, right, bottom,
                                            "1\" / 25.4mm from edge of page"))
                                            != APS_SUCCESS)
        goto cleanup;

    WRITESTR(job, "0 setgray\n\n");
        
    /* Draw imagable area rectangle, if known. */
    if (pageSize != NULL) {
        if ((result = DiagDrawLabelledRectangle(job, left, top, right, bottom,
                                            "Edge of imageable area"))
                                            != APS_SUCCESS)
            goto cleanup;
        
    }
        
    /* Determine vertical position to start drawing text at. */
    if (pageSize != NULL) {
        vertPos = pageSize->mediaHeight - 90; /* One inch from the top. */
    } else {
        vertPos = 702; /* One inch from the top of an 8.5" x 11" page. */
    }
        
    /* Select the font to be used for writing the title text. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "48 scalefont setfont\n\n");
    
    /* Write the title text. */
    vertPos -= 48;
    if ((result = DiagDrawPSText(job, "Printer Test Page",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;

    /* Leave 1/2 an inch of whitespace. */
    vertPos -= 36;

    /* Select the font to be used for the explanatory text and printer
     * configuration title.
     */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "10 scalefont setfont\n\n");

    /* Write the explanatory text. */
    vertPos -= 10;
    if ((result = DiagDrawPSText(job,
        "This test page confirms that your printer has been successfully setup "
        "and is ready for use.",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
    vertPos -= 10;
    if ((result = DiagDrawPSText(job,
        "The information below can be used by support staff or system "
        "administrators to diagnose problems.",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;

    /* Leave 1/2 an inch of whitespace. */
    vertPos -= 36;

    /* Select the font to be used for writing key printer information. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "24 scalefont setfont\n\n");
    
    /* Write the printer's name. */
    vertPos -= 24;
    if ((result = DiagDrawPSText(job, "Printer name:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagDrawPSText(job, printer->name,
                                 leftMargin + 180, vertPos)) != APS_SUCCESS)
        goto cleanup;

    /* Write the user's name. */
    userInfo = getpwuid(getuid());
    vertPos -= 24;
    if ((result = DiagDrawPSText(job, "Page sent by:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagDrawPSText(job, userInfo->pw_name,
                                 leftMargin + 180, vertPos)) != APS_SUCCESS)
        goto cleanup;

    /* Leave 1/2 an inch of whitespace. */
    vertPos -= 36;

    /* Select the font to be used for the configuration information title. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "12 scalefont setfont\n\n");

    /* Write the printer configuration title. */
    vertPos -= 12;
    if ((result = DiagDrawPSText(job, "Host Configuration Information",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;

    /* Select the font to be used for writing configuration information. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "8 scalefont setfont\n\n");
    
    /* Write the manufacturer and model name, if known */
    if (Aps_PrinterGetModel(printerHandle, &tempStr1, &tempStr2)
        == APS_SUCCESS) {
        /* Write the manufacturer's name. */
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Manufacturer:",
                                     leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, tempStr1,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
    
        /* Write the model name. */
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Model:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, tempStr2,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
        
        /* Release temporary strings. */
        Aps_ReleaseBuffer(tempStr1);
        Aps_ReleaseBuffer(tempStr2);
    }

    /* Write the printer's location information, if available. */
    if (Aps_PrinterGetConnectInfo(printerHandle, &connection, &tempStr1)
        == APS_SUCCESS) {
        /* Generate connection information string. */
        switch(connection) {
            case APS_CONNECT_LOCAL:
                sprintf(displayStr, "Local device (%50s)", tempStr1);
                break;
            case APS_CONNECT_NETWORK_LPD:
                sprintf(displayStr, "Unix network (%50s)", tempStr1);
                break;
            case APS_CONNECT_NETWORK_SMB:
                sprintf(displayStr, "Windows network (%50s)", tempStr1);
                break;
            default:
                sprintf(displayStr, "%50s", tempStr1);
        }
        Aps_ReleaseBuffer(tempStr1);

        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Printer connection:",
                                     leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, displayStr,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
    }
    
    /* Write the PPD file name, if applicable. */
    if (Aps_PrinterGetPPDFileName(printerHandle, &tempStr1) == APS_SUCCESS) {
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "PPD file:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, tempStr1,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
        
        Aps_ReleaseBuffer(tempStr1);
    }

    /* Write the GhostScript driver name, if applicable. */
    if (Aps_AttrGetMainData(jobAttributes, "gsdevice", &tempStr1)
        == APS_SUCCESS) {
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "GhostScript driver:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, tempStr1,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
        
        Aps_ReleaseBuffer(tempStr1);
    }

    /* Write the default page size and imagable area information, if
     * available.
     */
    if (pageSize != NULL) {
        /* Generate page size string to print. */
        if (pageSize->translatedName != NULL) {
            sprintf(displayStr, "%s (%.2f\" x %.2f\")",
                    pageSize->translatedName,
                    pageSize->mediaWidth / 72,
                    pageSize->mediaHeight / 72);
        } else if (pageSize->id != NULL) {
            sprintf(displayStr, "%s (%.2f\" x %.2f\")",
                    pageSize->id,
                    pageSize->mediaWidth / 72,
                    pageSize->mediaHeight / 72);
        } else {
            sprintf(displayStr, "%.2f\" x %.2f\"",
                    pageSize->mediaWidth / 72,
                    pageSize->mediaHeight / 72);
        }

        /* Print page size information. */
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Default page size:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, displayStr,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;

        /* Write the default imageable area information. */
        sprintf(displayStr, "t: %.2f\", b: %.2f\", l: %.2f\", r: %.2f\"",
                (pageSize->mediaHeight - pageSize->imageableAreaURy) / 72,
                pageSize->imageableAreaLLy,
                pageSize->imageableAreaLLx,
                (pageSize->mediaWidth - pageSize->imageableAreaURx) / 72);
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Default imageable area:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, displayStr,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
    }

    /* Write the default resolution information, if available. */
    res.version = 0;
    if (Aps_AttrQuickGetRes(jobAttributes, &res) == APS_SUCCESS) {
        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Default resolution:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;

        if (res.horizontalRes == res.verticalRes) {
            sprintf(displayStr, "%d dpi", (int)res.horizontalRes);
        } else {
            sprintf(displayStr, "%d x %d dpi", (int)res.horizontalRes,
                    (int)res.verticalRes);
        }

        if ((result = DiagDrawPSText(job, displayStr,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
    }

    /* Write the default color rendering information, if available. */
    if (Aps_AttrGetSetting(jobAttributes, "colorrendering", &tempStr1)
        == APS_SUCCESS) {
        /* Look for translated name, if available. Otherwise, use non-
         * translated name.
         */
        displayStr[0] = '\0';
        if (Aps_AttrGetOptions(jobAttributes, "colorrendering", &numOptions,
                               &options) == APS_SUCCESS) {
            for (i = 0; i < numOptions; ++i) {
                if (strcmp(options[i]->optionID, tempStr1) == 0) {
                    if (options[i]->translatedName != NULL)
                        strcpy(displayStr, options[i]->translatedName);
                }
            }
            Aps_ReleaseBuffer(options);
        }

        /* If we didn't find a translated name, use the non-translated
         * name.
         */
        if (strlen(displayStr) == 0)
            strcpy(displayStr, tempStr1);

        vertPos -= 8;
        if ((result = DiagDrawPSText(job, "Default color rendering:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;
        if ((result = DiagDrawPSText(job, displayStr,
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
            goto cleanup;
        
        Aps_ReleaseBuffer(tempStr1);
    }

    /* Write information on the time at which the test page was printed. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Page printed on:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    currentTime = time(NULL);
    if ((result = DiagDrawPSText(job, asctime(localtime(&currentTime)),
                                 leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;

    /* Leave 1/2 an inch of whitespace. */
    vertPos -= 36;

    /* Select the font to be used for the PS interpreter title. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "12 scalefont setfont\n\n");

    /* Write the PS interpreter title. */
    vertPos -= 12;
    if ((result = DiagDrawPSText(job, "PostScript Interpreter Information",
                                 leftMargin, vertPos)) != APS_SUCCESS)
            goto cleanup;

    /* Select the font to be used for writing configuration information. */
    WRITESTR(job, "/Helvetica findfont\n");
    WRITESTR(job, "8 scalefont setfont\n\n");

    /* Define temporary string variable to use when retrieving data from
     * interpreter.
     */
    WRITESTR(job, "/tempstr 100 string def\n");
    
    /* Write product name information. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Product:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagMoveTo(job, leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "systemdict /product known {product} {(-unknown-)} "
                  "ifelse show\n\n");

    /* Write product revision number information. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Revision:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagMoveTo(job, leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "systemdict /revision known {revision tempstr cvs}"
                  " {(-unknown-)} ifelse show\n\n");

    /* Write language level information. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Language level:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagMoveTo(job, leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "systemdict /languagelevel known {languagelevel tempstr cvs}"
                  " {(-unknown-)} ifelse show\n\n");

    /* Write available virtual memory information. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Available VM:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagMoveTo(job, leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "vmstatus tempstr cvs show pop pop\n\n");

    /* Write page size information. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Current page size:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagMoveTo(job, leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "systemdict /currentpagedevice known {\n"
                  "  currentpagedevice /PageSize known {\n"
                  "    currentpagedevice /PageSize get dup dup dup\n"
                  "    0 get 72 div tempstr cvs show (\" x ) show\n"
                  "    1 get 72 div tempstr cvs show (\" \\() show\n"
                  "    0 get 72 div 25.4 mul tempstr cvs show (mm x ) show\n"
                  "    1 get 72 div 25.4 mul tempstr cvs show (mm\\)) show\n"
                  "  } {\n"
                  "    (-unknown-) show\n"
                  "  } ifelse\n"
                  "} {\n"
                  "  (-unknown-) show\n"
                  "} ifelse\n\n");
    
    /* Write hardware resolution information. */
    vertPos -= 8;
    if ((result = DiagDrawPSText(job, "Hardware resolution:",
                                 leftMargin, vertPos)) != APS_SUCCESS)
        goto cleanup;
    if ((result = DiagMoveTo(job, leftMargin + 96, vertPos)) != APS_SUCCESS)
        goto cleanup;
    WRITESTR(job, "systemdict /currentpagedevice known {\n"
                  "  currentpagedevice /HWResolution known {\n"
                  "    currentpagedevice /HWResolution get dup\n"
                  "    0 get tempstr cvs show ( x ) show\n"
                  "    1 get tempstr cvs show ( dpi) show\n"
                  "  } {\n"
                  "    (-unknown-) show\n"
                  "  } ifelse\n"
                  "} {\n"
                  "  (-unknown-) show\n"
                  "} ifelse\n\n");

    /* Display user-customizable test pattern file, if it exists. */
    if (access(APSCFG_CONFIG_PATH_TESTPATTERN_FILE, R_OK) == 0) {
        WRITESTR(job, "BeginEPSF\n");
        WRITESTR(job, "%%BeginDocument: testpat.eps\n");
        if ((result = Aps_JobWriteFile(job,
            APSCFG_CONFIG_PATH_TESTPATTERN_FILE)) != APS_SUCCESS) goto cleanup;
        WRITESTR(job, "%%EndDocument\n");
        WRITESTR(job, "EndEPSF\n\n");
    }
            
    /* End this page, causing it to be printed. */
    WRITESTR(job, "showpage\n");
    WRITESTR(job, "%%PageTrailer\n");
    
    /* Write the PostScript trailer. */
    WRITESTR(job, "%%Trailer\n");
    WRITESTR(job, "%%EOF\n");
        
    /* Write file footer to switch printer into wait-for-jobs mode. */
    if ((result = Aps_JobWriteBlock(job, APS_BLOCK_FILE_FOOTER)) != APS_SUCCESS)
        goto cleanup;

    result = APS_SUCCESS;
        
cleanup:
    /* Release any temporary resources allocated by this function. */
    if (pageSize != NULL)
        Aps_ReleaseBuffer(pageSize);
    
    return result;
}

/* ---------------------------------------------------------------------------
 * DiagDrawPSText()
 *
 * Outputs PostScript code to draw the specified text at the specified
 * location, using the currently selected font.
 *
 * Parameters: job  - A handle to the job to receive PostScript code to draw
 *                    this text.
 *
 *             text - The text to print.
 *
 *             x    - The horizontal position for the text, in points.
 *
 *             y    - The vertical position for the text, in points.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result DiagDrawPSText(Aps_JobHandle job, const char *text,
                                 int x, int y)
{
    Aps_Result result;

    /* Move to the position specified by the caller. */
    result = DiagMoveTo(job, x, y);
    if (result != APS_SUCCESS)
        return result;
        
    /* Output the string to be printed. */
    WRITESTR(job, "(");
    WRITESTR(job, text);
    WRITESTR(job, ") show\n\n");

    result = APS_SUCCESS;
    
cleanup:        
    return result;
}

/* ---------------------------------------------------------------------------
 * DiagMoveTo()
 *
 * Emits the necessary PostScript code into a job to change the current
 * position to the specified coordinates.
 *
 * Parameters: job  - A handle to the job to receive PostScript code.
 *
 *             x    - The horizontal position for the text, in points.
 *
 *             y    - The vertical position for the text, in points.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result DiagMoveTo(Aps_JobHandle job, int x, int y)
{
    char number[20];
    Aps_Result result;

    /* Write horizontal position to draw text at. */
    sprintf(number, "%d", x);
    WRITESTR(job, number);

    /* We need a space between the two numbers. */
    WRITESTR(job, " ");
    
    /* Write vertical position to draw text at. */
    sprintf(number, "%d", y);
    WRITESTR(job, number);

    /* Make this the current position in the PS graphics state. */
    WRITESTR(job, " moveto\n");

    result = APS_SUCCESS;
    
cleanup:        
    return result;
}

/* ---------------------------------------------------------------------------
 * DiagDrawLabelledRectangle()
 *
 * Emits the necessary PostScript code into a job to draw a rectangle and
 * label it with the specified text.
 *
 * Parameters: job      A handle to the job to receive PostScript code.
 *
 *             left   - The location of the left edge of the rectangle.
 *
 *             top    - The location of the top edge of the rectangle.
 *
 *             right  - The location of the right edge of the rectangle.
 *
 *             bottom - The location of the bottom edge of the rectangle.
 *
 *             text   - The text to label the rectangle with.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result DiagDrawLabelledRectangle(Aps_JobHandle job,
                                            int left,
                                            int top,
                                            int right,
                                            int bottom,
                                            const char *text)
{
    Aps_Result result;
    char rectangleCmd[100];

    /* Draw the rectangle. */
    sprintf(rectangleCmd, "%d %d %d %d rectstroke\n",
            left,
            bottom,
            right - left,
            top - bottom);
    WRITESTR(job, rectangleCmd);
    
    /* Draw the caption text. */
    result = DiagDrawPSText(job, text, left + 2, bottom + 2);
    
    cleanup:
        return result;
}
