/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printerlocation.c
 *
 * Description: APIs for accessing printer location information.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_GetRemotePrinters()
 *
 * Obtains a list of available network printers.
 *
 * Parameters: connectionType - An Aps_ConnectionType identifying the type of
 *                              connection to look for remote printers on.
 *
 *             subLocation    - An optional string identifying the server or
 *                              other grouping of printers within this
 *                              connection type.
 *
 *             remotePrinters - An Aps_RemotePrinter ** to receive an array
 *                              of structure pointers with information on
 *                              the available network printers.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_GetRemotePrinters(Aps_ConnectionType connectionType,
                                 const char *subLocation,
                                 Aps_RemotePrinter ***remotePrinters)
{
    /* Check for reasonable parameter values. */
    if (remotePrinters == NULL) {
        return APS_INVALID_PARAM;
    }
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetConnectInfo()
 *
 * Obtains information on how a connection is established with a particular
 * printer. This includes the connection method, such as a printer connected
 * to a local device, or a network printer. It also includes a string
 * identifying the specific device or network name of the printer.
 *
 * Parameters: printer        - A handle to the printer in question.
 *
 *             connectionType - The address of an Aps_ConnectionType to
 *                              receive information on which type of
 *                              connection is being used.
 *
 *             location       - The address of a char * to receive a string
 *                              with the device/network name that this
 *                              printer is attached to. If this function
 *                              succeeds, it is the caller's responsibility
 *                              to dispose of this string by passing it to
 *                              Aps_ReleaseBuffer().
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterGetConnectInfo(Aps_PrinterHandle printerHandle,
                                     Aps_ConnectionType *connectionType,
                                     char **location)
{
    Printer *printer;

    /* Check for reasonable parameter values. */
    if (connectionType == NULL || location == NULL || 
                         printerHandle == NULL)
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Dispatch to the transport for processing */
    ASSERT(printer->transport);
    return printer->transport->vtbl->PrinterGetConnectInfo(printer->transport,
        printer, connectionType, location);
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterSetConnectInfo()
 *
 * Changes the method and/or location used to connect to a particular printer.
 * The method can be via a local device, or via a network. In the case of a
 * local printer, the location is the name of the device that the printer is
 * connected to. In the case of a network printer, the location is the
 * network path to the printer.
 *
 * Parameters: printer        - A handle to the printer in question.
 *
 *             connectionType - An Aps_ConnectionType identifying the
 *                              mechanism (local / remote) used to
 *                              communicate with this printer.
 *
 *             location       - A string containing the device name or
 *                              network path for this printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterSetConnectInfo(Aps_PrinterHandle printerHandle,
                                     Aps_ConnectionType connectionType,
                                     const char *location)
{
    Printer *printer;

    /* Check for reasonable parameter values. */
    if ( (connectionType != APS_CONNECT_LOCAL) && 
         (connectionType != APS_CONNECT_NETWORK_LPD)&&
         (connectionType != APS_CONNECT_NETWORK_SMB)&&
         (connectionType != APS_CONNECT_ALL))
        return APS_INVALID_PARAM;

    if (location == NULL || printerHandle == NULL)
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Dispatch to the transport for processing */
    ASSERT(printer->transport);
    return printer->transport->vtbl->PrinterSetConnectInfo(printer->transport,
        printer, connectionType, location);
}
