/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: trackarray.c
 *
 * Description: Builds on memory handling schemes implemented by
 *              TrackMem to construct compact dynamically-
 *              resized arrays of elements of an arbitrary size.
 *              Of course the contained elements must not lose
 *              coherence on shallow copy operations.
 */
#include <stdlib.h>
#include <string.h>
#include "trackmem.h"
#include "trackarray.h"

#define APS_ENVIRONMENT 1
#if APS_ENVIRONMENT
#include "apsinternal.h"
DEBUG_CHANNEL_DEFAULT(mem)
#endif

/* define ASSERT */
#ifndef ASSERT
#include <assert.h>
#define ASSERT assert(x)
#endif

/* define TINLINE and TINLINE_ON */
#if APS_ENVIRONMENT
#if APSCFG_INLINE
#define TINLINE INLINE
#define TINLINE_ON
#else
#define TINLINE
#endif
#else
#ifndef INLINE
#define TINLINE
#else
#define TINLINE INLINE
#define TINLINE_ON
#endif
#endif

/*
 * Private structures and data
 */
static const int TrackArray_Magic = 0x64417272; /* 'dArr' */

typedef struct TrackArray_Control_ {
    int    magic;
    size_t elemSize;
    int    elemCount;
    int    realElemCount; /* actual number of elements allocated */
} TrackArray_Control;

/*
 * Inline operations
 */
/*** Get the control structure associated with an array ***/
/* Note, in the absence of C++ function overriding, this function
 * takes a "const" but discards the qualifier upon return. */
#ifdef TINLINE_ON
static TINLINE TrackArray_Control *TrackArray_GetControl(
    const TrackArray *array) {
    TrackArray_Control *control;
    ASSERT(array);
    control = TrackMemTagFind(array, TRACKMEM_TAG_TRACKARRAY, NULL);
    ASSERT(control && (control->magic == TrackArray_Magic));
    return control;
}
#else
#define TrackArray_GetControl(array) \
    ((TrackArray_Control *)TrackMemTagFind(array, TRACKMEM_TAG_TRACKARRAY, NULL))
#endif

/* ---------------------------------------------------------------------------
 * TrackArrayNew()
 *
 * Create a new array control structure for elements of a specific
 * size.  The array is created blank initially.
 *
 * It is legal to cast the returned pointer directly to a native C/C++
 * array base pointer type of the appropriate size and manipulate the
 * elements directly.  Of course, no type checking will be performed on
 * such a pointer but in return you can expect a very good performance
 * increase. *8)  [So it'll crash faster if you're not careful!]
 *
 * Note: The address of the array may change if it is resized.
 *       This is an unavoidable side-effect of doing things in C.
 *       To get around this, keep only one copy of the address or
 *       use a double-indirection scheme.
 *
 * Parameters:  memparent - pointer to memory block to use as parent
 *              size      - size of a native storage element of this
 *                          array (all elements are equally-sized)
 *              initial   - initial number of elements to allocate
 *                          (must be >= 0)
 *
 * Returns a pointer to a TrackMem object containing the array.  This
 * pointer may be directly used to manipulate array contents.
 * (ie. You can use this as a drop-in malloc() replacement assuming you
 *      replace calls to free() with appropriate TrackArrayDelete() calls)
 */
TrackArray *TrackArrayNew(TMEM void *memparent, size_t size, int initial)
{
    TrackArray *array;

    ASSERT(initial >= 0);
    /* create array storage */
    array = (TrackArray *)TrackMemAlloc(memparent, size * initial,
        sizeof(TrackArray_Control) + TRACKMEM_TAG_RESERVESIZE);
    if (array) {
        TrackArray_Control *control;
        /* get storage for control tag */
        control = (TrackArray_Control *)TrackMemTagAdd(& array,
            TRACKMEM_TAG_TRACKARRAY, sizeof(TrackArray_Control));
        if (control) {
            control->magic = TrackArray_Magic;
            control->elemSize = size;
            control->realElemCount = control->elemCount = initial;
            return array;
        }
    }
    TrackMemFree(array);
    return NULL;
}

/* ---------------------------------------------------------------------------
 * TrackArrayDelete()
 *
 * Disposes of an existing array and any of the dependancies it might have
 * had.  Follows the same rules as TrackMemFree() regarding destructor call
 * ordering, etc...
 *
 * Parameters:  array     - pointer to array base pointer to free
 *
 * Provided for completeness.  Arrays may be deleted simply by
 * calling TrackMemFree() on the base pointer.  Of course, using this is
 * more explicit and generally a Good Idea.
 */
void TrackArrayDelete(TrackArray *array)
{
    ASSERT(array);
    TrackMemFree(array);
}

/* ---------------------------------------------------------------------------
 * TrackArrayClone()
 *
 * Clones an existing array excluding any dependancies it may have.  Though
 * TrackMemClone() is sufficient for this operation, it is recommended to
 * use this one instead as it will also compact the array and purge any
 * unnecessary administrative data (assuming there was any).
 *
 * Note: To clone a C-style array you should do the following instead:
 *       (note, appropriate casts may be necessary)
 *
 *       Elem *array, *oldArray;
 *       array = TrackArrayNew(parent, sizeof(Elem), numElems);
 *       if (array) memcpy(array, oldArray, sizeof(Elem) * numElems);
 *
 * Parameters:  memparent - pointer to memory block to use as parent
 *              array     - pointer to an existing array which will be
 *                          cloned
 *
 * Returns a pointer to a TrackMem object containing the array.  This
 * pointer may be directly used to manipulate array contents.
 * (ie. You can use this as a drop-in malloc() replacement assuming you
 *      replace calls to free() with appropriate TrackArrayDelete() calls)
 */
TrackArray *TrackArrayClone(TMEM void *memparent, const TrackArray *array)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    TrackArray *newArray;

    ASSERT(array);
    newArray =
        TrackArrayNew(memparent, control->elemSize, control->elemCount);
    if (newArray)
        memcpy(newArray, array, control->elemSize * control->elemCount);
    return newArray;
}

/* ---------------------------------------------------------------------------
 * TrackArrayUnlink()
 *
 * Disposes of all the memory tracking info that was tied with an array
 * and returns a plain-jane C-style array which may be freed using the
 * default destructor specified by the memFree() macro of the TrackMem
 * subsystem.  Follows the same rules as TrackMemUnlink().
 *
 * Avoid doing this on arrays with memory tracked dependancies as they
 * will be orphaned and may only be subsequently freed via explicit
 * specification of TrackMemFree().
 *
 * Parameters:  array     - pointer to an existing array which will be
 *                          unlinked
 *
 * Remember: Once unlinked you may no longer use any of the TrackArray...()
 *           functions on this array or any TrackMem...() primitives.
 *           The returned array may also have a different base address.
 *
 * Provided for completeness.  Arrays may be unlinked simply by calling
 * TrackMemUnlink() on the base pointer.
 */
void *TrackArrayUnlink(TrackArray *array)
{
    ASSERT(array);
    return TrackMemUnlink(array);
}

/* ---------------------------------------------------------------------------
 * TrackArrayResize()
 *
 * Adjusts the size of an array increasing or decreasing the bounds (and
 * amount of allocated storage) as necessary.  As usual, if space for 'n'
 * elements has been allocated, then the largest available index is 'n-1'.
 * If 'n' == 0, then the array will be considered empty.
 *
 * Let x = old number of elements
 * Let y = new number of elements
 * if (y == x) then {
 *    no change
 * } else if (y > x) then {
 *    array elements 0 through x-1 will be preserved
 *    array elements x through y-1 will be uninitialized (assume garbage!)
 * } else if (y != 0) then {
 *    array elements 0 through y-1 will be preserved
 *    array elements y through x-1 will be discarded (assume inaccessible)
 *    --> note no external resources used by the elements will be
 *        released; you must explicitely free any such resources
 * } else {
 *    all array elements will be discarded (as above)
 *    Beware! Most access (get) operations will throw exceptions.
 * }
 *
 * Note, though some implementations may choose to preallocate storage
 * for more elements than necessary (eg. classic doubling scheme), this
 * should not be assumed.  Moreover, one should assume that Resize()
 * performs the most efficient rellocation scheme possible for a given
 * implementation so please don't try to second-guess the algorithm
 * -- ie. Please don't re-implement a doubling scheme as it may make
 *        the algo less efficient.
 *
 * Parameters:  array     - pointer to pointer to an existing array to be
 *                          resized
 *              numElems  - new number of elements to allocate
 *
 * Returns TRUE on success, FALSE otherwise.
 *
 * On some implementations, the result of a failed resize will be the
 * destruction of the array.  In these cases *array will be set to NULL
 * whenever possible. (check the results and cross your fingers)
 */
int TrackArrayResize(TrackArray **array, int numElems)
{
    TrackArray_Control *control;
    ASSERT(array && (numElems >= 0));
    control = TrackArray_GetControl(*array);
    if (numElems > control->realElemCount) {
        int     newNumElems;
        TrackArray *newArray;
        /* rule #1: array extension
         *  - if array has no storage, then new array length is
         *    number of requested elements
         *  - otherwise, if the new size is less than twice the
         *    current size, double the current size
         *  - otherwise, if the new size is the requested number of
         *    elements
         */
        newNumElems = control->realElemCount << 1;
        if (numElems > newNumElems) newNumElems = numElems;

        newArray = TrackMemRealloc(*array, control->elemSize * newNumElems);
        if (! newArray) return FALSE;
        *array = newArray;
        control = TrackArray_GetControl(newArray);
        control->realElemCount = newNumElems;
    } else {
        int     newNumElems;
        /* rule #2: array reduction
         *  - if array has no storage, do nothing
         *  - otherwise, if the new size is more than half the
         *    current size, leave it alone
         *  - otherwise, if the new size if less than half but
         *    more than a quarter of the current size, halve it
         *  - otherwise, the new size is the requested number of
         *    elements
         */
        newNumElems = control->realElemCount >> 1;
        if (numElems < newNumElems) {
            TrackArray *newArray;
            if (numElems < (newNumElems >> 1)) newNumElems = numElems;

            newArray = TrackMemRealloc(*array,
                control->elemSize * newNumElems);
            /* ignore if resize failed while shrinking (strange!) */
            if (newArray) {
                *array = newArray;
                control = TrackArray_GetControl(newArray);
                control->realElemCount = newNumElems;
            }
        }
    }
    control->elemCount = numElems;
    return TRUE;
}

/* ---------------------------------------------------------------------------
 * TrackArrayEnsureStorageForElem()
 *
 * Ensure that there is enough storage for the specified element.  This may
 * be useful in cases where you are not sure if storage was allocated for
 * a specific element.  Additionally this may provide useful hints to a
 * derived implementation supporting partial sparse arrays.
 *
 * if (elem >= TrackArrayGetSize(array)) TrackArrayResize(array, elem + 1);
 *
 * Parameters:  array     - pointer to pointer to an existing array to be
 *                          verified / resized
 *              elem      - element number to ensure storage for
 *
 * Returns TRUE on success, FALSE otherwise.
 */
int TrackArrayEnsureStorageForElem(TrackArray **array, int elem)
{
    TrackArray_Control *control;
    ASSERT(array);
    control = TrackArray_GetControl(*array);
    ASSERT(elem >= 0);
    if (elem >= control->elemCount) {
        return TrackArrayResize(array, elem + 1);
    }
    return TRUE;
}

/* ---------------------------------------------------------------------------
 * TrackArrayGetSize()
 *
 * Get the size of the array in terms of how many elements it contains.  This
 * function may very well return 0 to indicate that the array has no storage
 * capacity.
 *
 * Parameters:  array     - pointer to an array to examine
 *
 * Returns the size of the array (in #'s of elements).
 */
int TrackArrayGetSize(const TrackArray *array)
{
    const TrackArray_Control *control= TrackArray_GetControl(array);
    return control->elemCount;
}

/* ---------------------------------------------------------------------------
 * TrackArrayGetElemSize()
 *
 * Get the size of each elements in multiples of sizeof(char *) (usually
 * bytes).  This function may return 0 to indicate that the elements are
 * sizeless.  Generally that is a rather useless "feature".
 *
 * Parameters:  array     - pointer to an array to examine
 *
 * Returns the size of each element of the array (in multiples of
 * sizeof(char *).)
 */
size_t TrackArrayGetElemSize(const TrackArray *array)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    return control->elemSize;
}

/* ---------------------------------------------------------------------------
 * TrackArrayGetIndexByRef()
 *
 * Get the index of an element given a reference to its storage location in
 * the array.  Please use this whenever it is necessary to convert a pointer
 * to an address.
 *
 * Note: Checks bounds and asserts in case of misaligned pointers.
 *
 * Parameters:  array     - pointer to an array to examine
 *              elem      - pointer to element's storage
 *
 * Returns an integer representing this element's index in the array.
 * The following assertions are valid:
 * ASSERT(TrackArrayGetIndexByRef(array, TrackArrayElemAt(array, num)) \
 *     == num);
 * ASSERT(TrackArrayGetIndexByRef(array, & array[num]) == num);
 */
int TrackArrayGetIndexByRef(const TrackArray *array, const void *elem)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    if (control->elemSize) {
        div_t xdiv = div((char *)elem - (char *)array, control->elemSize);
        ASSERT(xdiv.rem == 0);
        return xdiv.quot;
    }
    return 0;
}

/* ---------------------------------------------------------------------------
 * TrackArrayAddLast()
 *
 * Add an element to the end of an array.  Increases the effective user-
 * addressable size of the array by 1 element and extends the physical
 * storage if necessary so that the new element will fit.
 *
 * If data is NOT NULL, copies source data into the new slot, else
 *   creates space but does not copy.
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *              data      - pointer to a buffer containing initialization
 *                          data for the element (ignored if NULL)
 *
 * Returns pointer to new element's storage.
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 */
void *TrackArrayAddLast(TrackArray **array, const void *data)
{
    TrackArray_Control *control;
    void *storage;
    ASSERT(array);
    control = TrackArray_GetControl(*array);
    if (TrackArrayResize(array, control->elemCount + 1)) {
        control = TrackArray_GetControl(*array);
        storage = (void *)(((char *)*array) +
            ((control->elemCount - 1) * control->elemSize));
        if (data) memcpy(storage, data, control->elemSize);
    } else storage = NULL;
    return storage;
}

/* ---------------------------------------------------------------------------
 * TrackArrayRemoveLast()
 *
 * Remove an element from the end of the array.  Decreases the effective user-
 * addressable size of the array by 1 element and shrinks the physical
 * storage if possible and desirable [implementation-defined].
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 *
 * ASSERT's on out of bounds. (array must not be empty)
 */
void TrackArrayRemoveLast(TrackArray **array)
{
    TrackArray_Control *control;
    int               lastElem;
    ASSERT(array);
    control = TrackArray_GetControl(*array);
    lastElem = control->elemCount - 1;
    ASSERT(lastElem >= 0);
    TrackArrayResize(array, lastElem);
}

/* ---------------------------------------------------------------------------
 * TrackArrayInsertAt()
 *
 * Insert an element at position elem.  Increases the effective user-
 * addressable size of the array by 1 element and extends the physical
 * storage if necessary so that the new element will fit.
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *              elem      - index before which to insert the new element
 *                          (existing elements are pushed down such that
 *                           array[elem] contains the new element)
 *              data      - pointer to a buffer containing initialization
 *                          data for the element which is copied to the
 *                          new slot unless NULL)
 *
 * Returns pointer to new element's storage.
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 *
 * ASSERT's on out of bounds. (0 <= x <= length)
 */
void *TrackArrayInsertAt(TrackArray **array, int elem, const void *data)
{
    TrackArray_Control *control;
    void *storage;
    ASSERT((array) && (elem >= 0));
    control = TrackArray_GetControl(*array);
    ASSERT(elem <= control->elemCount);
    if (TrackArrayResize(array, control->elemCount + 1)) {
        int   lastElem;
        control = TrackArray_GetControl(*array);
        lastElem = control->elemCount - 1;
        storage = (void *)(((char *)*array) + (elem * control->elemSize));
        if (elem != lastElem) {
            memmove((char *)storage + control->elemSize, storage,
                (lastElem - elem) * control->elemSize);
        }
        if (data) memcpy(storage, data, control->elemSize);
    } else storage = NULL;
    return storage;
}

/* ---------------------------------------------------------------------------
 * TrackArrayRemoveAt()
 *
 * Remove an element at position elem.  Decreases the effective user-
 * addressable size of the array by 1 element and shrinks the physical
 * storage if possible and desirable [implementation-defined].
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *              elem      - index of element to be removed (existing elements
 *                          (existing elements are moved up such that
 *                           array[elem] is removed)
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 *
 * ASSERT's on out of bounds. (0 <= x < length)
 */
void TrackArrayRemoveAt(TrackArray **array, int elem)
{
    TrackArray_Control *control;
    int               lastElem;
    ASSERT((array) && (elem >= 0));
    control = TrackArray_GetControl(*array);
    lastElem = control->elemCount - 1;
    ASSERT(elem <= lastElem);
    if (elem != lastElem) {
        void *storage = (void *)(((char *)*array) + (elem * control->elemSize));
        memmove(storage, (char *)storage + control->elemSize,
            (lastElem - elem) * control->elemSize);
    }
    TrackArrayResize(array, lastElem);
}

/* ---------------------------------------------------------------------------
 * TrackArrayInsertManyAt()
 *
 * Insert many elements such that the first new elements is given the
 * specified index and that the old element from that index is pushed
 * down beyond the space consumed by the new entries.  Increases the
 * effective user-addressable size of the array by the total count of
 * elements inserted.
 *
 * Functionally, this is equivalent to the iterative application of
 * TrackArrayInsertAt() but far more efficient.
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *              elem      - index before which to insert the first new
 *                          element (existing elements are pushed down)
 *              count     - number of elements to insert
 *              data      - pointer to a buffer containing initialization
 *                          data for the element which is copied to the
 *                          new slot unless NULL)
 *
 * Returns pointer to first new element's storage.
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 *
 * ASSERT's on out of bounds. (0 <= x <= length)
 */
void *TrackArrayInsertManyAt(TrackArray **array, int elem,
    int count, const void *data)
{
    TrackArray_Control *control;
    void *storage;
    if (! count) return TrackArrayElemAt(array, elem);
    ASSERT((array) && (elem >= 0) && (count > 0));
    control = TrackArray_GetControl(*array);
    ASSERT(elem <= control->elemCount);
    if (TrackArrayResize(array, control->elemCount + count)) {
        int    lastElem;
        size_t newElemSpace;
        control = TrackArray_GetControl(*array);
        newElemSpace = count * control->elemSize;
        lastElem = control->elemCount - 1;
        storage = (void *)(((char *)*array) + (elem * control->elemSize));
        if (elem != lastElem) {
            memmove((char *)storage + newElemSpace, storage,
                (lastElem - elem) * control->elemSize);
        }
        if (data) memcpy(storage, data, newElemSpace);
    } else storage = NULL;
    return storage;
}

/* ---------------------------------------------------------------------------
 * TrackArrayAppendMany()
 *
 * Appends many elements to the tail of an array such that the last
 * element of the array becomes the last element of the source data.
 * Increases the effective user-addressable size of the array by the
 * total count of elements appended.
 *
 * Functionally, this is equivalent to the iterative application of
 * TrackArrayAddLast() but far more efficient.
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *              elem      - index before which to insert the first new
 *                          element (existing elements are pushed down)
 *              count     - number of elements to append
 *              data      - pointer to a buffer containing initialization
 *                          data for the element which is copied to the
 *                          new slot unless NULL)
 *
 * Returns pointer to first new element's storage.
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 *
 * ASSERT's on out of bounds. (0 <= x <= length)
 */
void *TrackArrayAppendMany(TrackArray **array, int count, const void *data)
{
    TrackArray_Control *control;
    void *storage;
    size_t offset;
    ASSERT((array) && (count >= 0));
    control = TrackArray_GetControl(*array);
    offset = control->elemSize * (control->elemCount - 1);
    storage = TrackArrayElemLast(array);
    if (! count) return (void *)(((char *)*array) + offset);
    if (TrackArrayResize(array, control->elemCount + count)) {
        control = TrackArray_GetControl(*array);
        storage = (void *)(((char *)*array) + offset);
        if (data) memcpy(storage, data, count * control->elemSize);
    } else storage = NULL;
    return storage;
}

/* ---------------------------------------------------------------------------
 * TrackArrayRemoveManyAt()
 *
 * Remove many elements from the middle / end of an array such that
 * all elements in the range [elem]..[elem+count-1] are removed.
 * Decreases the effective user-addressable size of the array by the
 * total count of elements removed and shrinks the physical storage
 * if possible and desirable [implementation-defined].
 *
 * Functionally, this is equivalent to the iterative application of
 * TrackArrayRemoveAt() / TrackArrayRemoveLast() but far more efficient.
 *
 * Parameters:  array     - pointer to pointer to an array to be modified
 *              elem      - index before which to insert the first new
 *                          element (existing elements are pushed down)
 *              count     - number of elements to be removed
 *
 * Note: This function modified the array's base pointer if it was necessary
 *       to reallocate the storage, and is subject to the same rules as
 *       TrackArrayResize().
 *
 * ASSERT's on out of bounds. (0 <= x <= length)
 */
void TrackArrayRemoveManyAt(TrackArray **array, int elem, int count)
{
    TrackArray_Control *control;
    int                 lastElem;
    ASSERT((array) && (elem >= 0));
    control = TrackArray_GetControl(*array);
    lastElem = control->elemCount - count;
    ASSERT(elem <= lastElem);
    if (elem != lastElem) {
        void *storage = (void *)(((char *)*array) + (elem * control->elemSize));
        memmove(storage, (char *)storage + (count * control->elemSize),
            (lastElem - elem) * control->elemSize);
    }
    TrackArrayResize(array, lastElem);
}

/* ---------------------------------------------------------------------------
 * TrackArrayGetManyAt()
 *
 * Extracts a number of elements from an array and stores them in the user-
 * provided buffer.
 *
 * Functionally, this is equivalent to performing a mass memmove() on a
 * segment of the array, but also incorporates bounds checking.
 *
 * Parameters:  array     - pointer to an array to be examined
 *              elem      - index at which to start extracting elements
 *              count     - number of elements to get
 *              buffer    - user buffer where the data is to be stored
 *
 * ASSERT's on out of bounds. (0 <= x <= length)
 */
void TrackArrayGetManyAt(const TrackArray *array, int elem, int count,
    void *buffer)
{
    const TrackArray_Control *control;
    control = TrackArray_GetControl(array);
    ASSERT((buffer) && (elem >= 0) && (count >= 0) &&
        (elem + count <= control->elemCount));
    if (count) {
        memmove(buffer, (const char *)array + (elem * control->elemSize),
            count * control->elemSize);
    }
}

/* ---------------------------------------------------------------------------
 * TrackArraySetManyAt()
 *
 * Sets a number of an array's elements based on the contents of the user-
 * provided buffer.
 *
 * Functionally, this is equivalent to performing a mass memmove() on a
 * segment of the array, but also incorporates bounds checking.
 *
 * Parameters:  array     - pointer to an array to be modified
 *              elem      - index at which to start setting elements
 *              count     - number of elements to set
 *              buffer    - user buffer from which the data is to be fetched
 *
 * ASSERT's on out of bounds. (0 <= x <= length)
 */
void TrackArraySetManyAt(TrackArray *array, int elem, int count,
    const void *buffer)
{
    TrackArray_Control *control;
    control = TrackArray_GetControl(array);
    ASSERT((buffer) && (elem >= 0) && (count >= 0) &&
        (elem + count <= control->elemCount));
    if (count) {
        memmove((char *)array + (elem * control->elemSize), buffer,
            count * control->elemSize);
    }
}

/* ---------------------------------------------------------------------------
 * TrackArrayElemAt()
 *
 * Get the address of the storage space for the specified element.
 *
 * Parameters:  array     - pointer to array to be examined
 *              elem      - index of element to seek to (must be in
 *                          addressable bounds of array)
 *
 * Returns a pointer to the element's storage in the array.
 *
 * ASSERT's on out of bounds. (0 <= x < length)
 *
 * This function is not required to access members as it is quite legal
 * for one to simply cast the array pointer to a more useful pointer
 * type and use normal indexing procedures.
 *
 */
void *TrackArrayElemAt(const TrackArray *array, int elem)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    ASSERT((elem >= 0) && (elem <= control->elemCount));
    return (void *)(((char *)array) + (elem * control->elemSize));
}

/* ---------------------------------------------------------------------------
 * TrackArrayElemFirst()
 *
 * Get the address of the storage space for the first element of
 * the array (zero-th index).
 *
 * Parameters:  array     - pointer to array to be examined
 *
 * Returns a pointer to the element's storage in the array.
 *
 * ASSERT's if array length is 0.
 * See TrackArrayElemAt() for indexing information.
 */
void *TrackArrayElemFirst(const TrackArray *array)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    ASSERT(control->elemCount > 0);
    return (void *)array;
}

/* ---------------------------------------------------------------------------
 * TrackArrayElemLast()
 *
 * Get the address of the storage space for the last element of
 * the array ((bound-1)-th index).
 *
 * Parameters:  array     - pointer to array to be examined
 *
 * Returns a pointer to the element's storage in the array.
 *
 * ASSERT's if array length is 0.
 * See TrackArrayElemAt() for indexing information.
 */
void *TrackArrayElemLast(const TrackArray *array)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    ASSERT(control->elemCount > 0);
    return (void *)(((char *)array) +
        ((control->elemCount - 1) * control->elemSize));
}

/* ---------------------------------------------------------------------------
 * TrackArrayElemEndMark()
 *
 * Get the address of the storage space for the element one beyond
 * the end of the array.
 *
 * Parameters:  array     - pointer to array to be examined
 *
 * Returns a pointer to the element's storage in the array.
 *
 * This function always succeeds (assuming the array is valid)
 * See TrackArrayElemAt() for indexing information.
 */
void *TrackArrayElemEndMark(const TrackArray *array)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    return (void *)(((char *)array) +
        (control->elemCount * control->elemSize));
}

/* ---------------------------------------------------------------------------
 * TrackArrayElemNext()
 *
 * Given the address of an array element, get the address of the next
 * element of the same array.  If ptrelem = NULL, returns first element
 * of the array (assuming it has any).
 *
 * Parameters:  array     - pointer to array to be examined
 *              ptrelem   - pointer to element's storage -- used as start
 *                          point for locating the next one
 *
 * Returns a pointer to the element's storage in the array.
 *
 * This function tests bounds and will cause an ASSERT() if these
 * bounds are exceeded, but allows one to step to the "end marker"
 * positions (one past the end). (ie. stepped two past the end)
 *
 * Note: Checks for misaligned source elements.
 */
void *TrackArrayElemNext(const TrackArray *array, const void *ptrelem)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    if (ptrelem) {
        if (control->elemSize) {
            /* division multiple may be a little slower than alternatives
             * but allows us to test for misaligned and it therefore
             * inherently safer. */
            int index;
            div_t xdiv = div((char *)ptrelem - (char *)array,
                control->elemSize);
            ASSERT(xdiv.rem == 0);
            index = xdiv.quot + 1;
            ASSERT((index >= 0) && (index <= control->elemCount));
            return (void *)(((char *)ptrelem) + control->elemSize);
        }
    } else {
        ptrelem = (void *)array;
    }
    return (void *)ptrelem;
}

/* ---------------------------------------------------------------------------
 * TrackArrayElemSkip()
 *
 * Given the address of an array element, get the address of an
 * element indexed 'num' elements away (signed value).  If ptrelem = NULL,
 * indexes from beginning of array.
 *
 * Parameters:  array     - pointer to array to be examined
 *              ptrelem   - pointer to element's storage -- used as start
 *                          point for skipping past many
 *              num       - number of elements to skip
 *
 * Returns a pointer to the element's storage in the array.
 *
 * This function tests bounds and will cause an ASSERT() if these
 * bounds are exceeded, but allows one to step to the "end marker"
 * positions (one past the end). (ie. stepped two past the end)
 *
 * Note: Checks for misaligned source elements.
 */
void *TrackArrayElemSkip(const TrackArray *array, const void *ptrelem,
    int num)
{
    const TrackArray_Control *control = TrackArray_GetControl(array);
    if (ptrelem) {
        if (control->elemSize) {
            /* division multiple may be a little slower than alternatives
             * but allows us to test for misaligned and it therefore
             * inherently safer. */
            int index;
            div_t xdiv = div((char *)ptrelem - (char *)array,
                control->elemSize);
            ASSERT(xdiv.rem == 0);
            index = xdiv.quot + num;
            ASSERT((index >= 0) && (index <= control->elemCount));
            return (void *)(((char *)ptrelem) + control->elemSize);
        }
    } else {
        ptrelem = (void *)array;
    }
    return (void *)ptrelem;
}

/* ---------------------------------------------------------------------------
 * TrackArrayGetElemByRef()
 *
 * Fills in the supplied buffer with the contents of the element referenced
 * by the given pointer.  This function essentially performs a memmove() on
 * the data using the full element width as assigned by the array.  A net
 * advantage of this call is that it disambiguates operations which are
 * designed to manipulate array contents.
 *
 * Performs the equivalent of:
 *   memmove(data, elem, TrackArrayGetElemSize(array));
 *
 * Parameters:  array     - pointer to an array to be examined
 *              elem      - reference to element to fetch
 *              data      - user-supplied buffer where data should be
 *                          stored
 */
void TrackArrayGetElemByRef(const TrackArray *array, const void *elem,
    void *data)
{
    memmove(data, elem, TrackArrayGetElemSize(array));
}

/* ---------------------------------------------------------------------------
 * TrackArraySetElemByRef()
 *
 * Stores the contents of the supplied buffer into the array element
 * referenced by the given pointer.  This function essentially performs a
 * memmove() on the data using the full element width as assigned by the array.
 * A net advantage of this call is that it disambiguates operations which are
 * designed to manipulate array contents.
 *
 * Performs the equivalent of:
 *   memmove(elem, data, TrackArrayGetElemSize(array));
 *
 * Parameters:  array     - pointer to an array to be modified
 *              elem      - reference to element to modify
 *              data      - user-supplied buffer from which data should be
 *                          fetched
 */
void TrackArraySetElemByRef(TrackArray *array, void *elem, const void *data)
{
    memmove(elem, data, TrackArrayGetElemSize(array));
}
