/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: trackmem.h
 *
 * Description: Simple local implementation of memory tracking.
 *              Provides a primitive tracking mechanism for linked
 *              blocks of data.
 */

#ifndef TRACK_MEM_H
#define TRACK_MEM_H

#include <stdlib.h>
#include <malloc.h>

/*** Data ***/
/* Tag types -- don't change sequence */
typedef enum {
    TRACKMEM_TAG_BASE_PRIVATE = 0x00,
    TRACKMEM_TAG_BASE_OBJECTS = 0x40,
    TRACKMEM_TAG_TRACKARRAY   = TRACKMEM_TAG_BASE_OBJECTS + 0,
    TRACKMEM_TAG_TRACKSTRING  = TRACKMEM_TAG_BASE_OBJECTS + 1,
    TRACKMEM_TAG_BASE_USER    = 0x80
} TrackMem_TagType;

/* Destructor types -- don't change sequence */
typedef enum {
    TRACKMEM_DESTRUCTOR_DEFAULT = 0, /* uses default */
    TRACKMEM_DESTRUCTOR_NONE    = 1, /* none */
    TRACKMEM_DESTRUCTOR_SIMPLE  = 2, /* void (*func)(void *link) */
    TRACKMEM_DESTRUCTOR_SPECIAL = 3  /* void (*func)(void *link, void *memref) */
} TrackMem_DestructorType;

/* Add this to the size of EACH of your tag structures to compute the total
 * space requirements for TrackMemAlloc(). */
#define TRACKMEM_TAG_RESERVESIZE \
    (sizeof(char) + sizeof(char) + sizeof(short int))

/* Maximum total size of all tags */
/* (leaves room for admin info) */
#define TRACKMEM_TAG_MAXTOTALSIZE ((size_t)65500)

/* To make this more readable, we tag all trackmem allocations with TMEM
 * to distinguish a TMEM void * from a normal void *.  Of course, there
 * is no functional difference. */
#define TMEM

/*** Types ***/
typedef void (*TrackMem_LinkDestructorFunc)(void *link);
typedef void (*TrackMem_LinkSpecialDestructorFunc)(void *link,
    TMEM void *ref);

/*** Protos ***/
/* allocation */
TMEM void *TrackMemAlloc(TMEM void *memparent, size_t size,
    size_t sizeTags);
TMEM void *TrackMemClone(TMEM void *memparent, const TMEM void *memsource);
TMEM void *TrackMemDupMemory(TMEM void *memparent, const void *memsource,
    size_t size, size_t sizeTags);
char *TrackMemDupString(TMEM void *memparent, const char *strsource,
    size_t sizeTags);
TMEM void *TrackMemLink(TMEM void *memparent, const void *target,
    size_t size, size_t sizeTags);
TMEM void *TrackMemRealloc(TMEM void *mem, size_t size);
TMEM void *TrackMemReallocTagSpace(TMEM void *mem, size_t size);
/* deallocation */
void  TrackMemFree(TMEM void *mem);
void  TrackMemFreeChildren(TMEM void *mem);
/* reassignment */
void *TrackMemUnlink(TMEM void *mem);
void  TrackMemReassign(TMEM void *memparent, TMEM void *mem);
void  TrackMemReassignChildren(TMEM void *memparent, TMEM void *mem);
/* information */
void *TrackMemGetData(const TMEM void *mem);
TMEM void *TrackMemGetParent(const TMEM void *mem);
size_t  TrackMemGetSize(const TMEM void *mem);
size_t  TrackMemGetTagSpaceSize(const TMEM void *mem);
/* tagging */
void *TrackMemTagAdd(TMEM void **mem, TrackMem_TagType tagType,
    size_t tagSize);
void *TrackMemTagSet(TMEM void *mem, TrackMem_TagType tagType,
    size_t tagSize);
void *TrackMemTagFind(const TMEM void *mem, TrackMem_TagType tagType,
    size_t *tagSize);
int   TrackMemTagSetLinkDestructor(TMEM void *mem,
    TrackMem_DestructorType dType, void *dFunc);
void  TrackMemTagUnset(TMEM void *mem, TrackMem_TagType tagType);

#endif /* (! TRACK_MEM_H) */
