/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 * The Initial Developer of the Original Code is Corel Corporation.
 * Portions created by Corel are Copyright (C) 2000 Corel Coroporation.
 * All Rights Reserved.
 *
 * Contributor(s):_______________________________________________.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * the GNU Lesser General Public License Version 2.1 or higher (the "LGPL
 * License"), in which case the provisions of the LGPL License are applicable
 * instead of those above. If you wish to allow use of your version of this
 * file only under the terms of the LGPL License and not to allow others to
 * use your version of this file under the CPL, indicate your decision by
 * deleting the provisions above and replace them with the notice and other
 * provisions required by the LGPL License. If you do not delete the
 * provisions above, a recipient may use your version of this file under
 * either the CPL or the LGPL License.
 *
 *
 *        File: Transport.h
 *
 * Description: Basic implementation of the transport abstraction
 *              layer's interface.
 *
 */

#ifndef TRANSPORT_H
#define TRANSPORT_H

typedef struct Transport_ Transport;
typedef struct TransportVtbl_ TransportVtbl;

#include "printer.h"
#include "job.h"

struct TransportVtbl_
{
    /* Is printer known by this transport? */
    int (*IsPrinterKnown) (Transport * this, const char *name);

    /* Create a printer object */
    Printer *(*CreatePrinterInstance) (Transport * this, const char *name);

    /* Called before a job is used to allow the transport to
     * set up any information it needs.  job->printer and
     * job->transport are valid, and job->info has been set
     * mostly to defaults.
     *
     * Should create a structure to hold whatever temporary data it
     * requires in job->transportData.  Should not depend on
     * information in job->info EXCEPT the printerHandle.
     */
    Aps_Result(*JobInit) (Transport * this, Job *job);

    /* Called before a job is about to be deleted to allow the
     * transport to cleanup anything it has left in its reserved
     * storage area as the result of a queue query.
     *
     * Should simply release job->transportData, but may still read from
     * the job->info member.
     */
    Aps_Result(*JobCleanup) (Transport * this, Job *job);

    /* Called before any writes are performed on a job
     *
     * This operation can expect the following fields to be
     * correct:
     *    job->transportData = NULL
     *    job->printer       = pointer to printer
     *    info->jobHandle
     *    info->jobHost      = hostname of source
     *    info->jobName      = application provided
     *    info->jobFilename  = ""
     *    info->jobFormat
     *    info->printerHandle
     *    info->printerName
     *    info->docXXXX
     *    info->ownerName       (usually from current user)
     *    info->ownerID         (usually from current user)
     *
     * High-level operations set defaults for the following fields
     * which may need to be updated in some cases.
     *    info->jobCreationTime (default: current time)
     *                          -- if high-latency)
     *
     * Must set the following data immediately:
     *    info->jobSize         (set to 0)
     *    info->localHost       (hostname of THIS system)
     *    info->localFile       (if spooling to a temp file, else "")
     *
     * Should set this data as soon as it is becomes using the
     * same conventions as JobUpdate():
     *    info->jobID           (if known, try hard!)
     *    info->printerStatus   (if known)
     *    info->ownerName       (if different, eg. due to redirection /
     *    info->ownerID          special configuration)
     *    info->spoolHost       (if known)
     *    info->spoolFile       (if known)
     *
     * Sets APS_JOB_FAILED on failure else does not modify status.
     */
    Aps_Result(*JobStart) (Transport * this, Job *job);

    /* Write data to a job
     *
     * This operation can expect the same fields to be correct
     * as with JobStart().
     *
     * Must set the following data unless already correct:
     *    info->jobSize         (update incrementally)
     *    info->localHost       (hostname of THIS system)
     *    info->localFile       (if spooling to a temp file, else "")
     *
     * Should set this data as soon as it is becomes using the
     * same conventions as JobUpdate():
     *    info->jobID           (if known, try hard!)
     *    info->printerStatus   (if known)
     *    info->ownerName       (if different, eg. due to redirection /
     *    info->ownerID          special configuration)
     *    info->spoolHost       (if known)
     *    info->spoolFile       (if known)
     *
     * Sets APS_JOB_FAILED on catastrophic failure else does not
     *   modify status even if the parameters are bogus
     *   (eg. set APS_JOB_FAILED only if the printer 'vanishes')
     */
    Aps_Result(*JobWrite) (Transport * this, Job *job,
                           const void *data, size_t size);

    /* Called anytime between JobStart and JobEnd to discard the
     * job.  JobEnd MUST NOT be called after this.
     * Sets APS_JOB_ABORTED if successfully aborted, else
     *   proceeds as if JobEnd had been called.
     *
     * This operation can expect the same fields to be correct
     * as with JobStart().
     *
     * Must set the following data unless already correct:
     *    info->jobSize         (total size to date)
     *    info->localHost       (hostname of THIS system)
     *    info->localFile       (if spool file not removed, else "")
     *
     * To assist with future monitoring, this operation should
     * update the following fields using the same conventions as
     * JobUpdate():
     *    info->printerStatus   (if known)
     *
     * If the abort fails, these should also be filled in.
     *    info->jobID           (try hard!)
     *    info->ownerName       (if different, eg. due to redirection /
     *    info->ownerID          special configuration)
     *    info->spoolHost       (if known)
     *    info->spoolFile       (if known)
     *    info->jobStatus       set APS_JOB_QUEUEING/APS_JOB_PENDING/
     *                              APS_JOB_SENDING/PRINTING/PAUSED/
     *                              SUSPENDED/FAILED/DISCARDED, etc...
     */
    Aps_Result(*JobAbort) (Transport * this, Job *job);

    /* Called after all writes have been performed
     *
     * This operation can expect the same fields to be correct
     * as with JobStart().
     *
     * Must set the following data unless already correct:
     *    info->jobSize         (total size to date)
     *    info->localHost       (hostname of THIS system)
     *    info->localFile       (if spool file not removed, else "")
     *
     * To assist with future monitoring, this operation should
     * update the following fields using the same conventions as
     * JobUpdate():
     *    info->jobID           (try hard!)
     *    info->printerStatus   (if known)
     *    info->ownerName       (if different, eg. due to redirection /
     *    info->ownerID          special configuration)
     *    info->spoolHost       (try hard!)
     *    info->spoolFile       (try hard!)
     *
     * The operation MUST update the following fields:
     *    info->jobStatus       set APS_JOB_QUEUEING/APS_JOB_PENDING/
     *                              APS_JOB_SENDING/PRINTING/PAUSED/
     *                              SUSPENDED/FAILED/DISCARDED, etc...
     */
    Aps_Result(*JobEnd) (Transport * this, Job *job);

    /* Called to get a file descriptor for writing to a job.
     * This function should try to emulate file access
     * if not supported.  Otherwise it should return a file descriptor
     * of -1 and an appropriate error code.
     */
    Aps_Result(*JobGetFileDescriptor) (Transport * this, Job * job,
        int *fd);

    /* Called to dispatch a queueing operation atomically.
     *
     * This operation can expect the same fields to be correct
     * as with JobStart().
     *
     * Must set the following data unless already correct:
     *    info->jobSize         (total size to date)
     *    info->localHost       (hostname of THIS system)
     *    info->localFile       (name of file we just got)
     *
     * To assist with future monitoring, this operation should
     * UPDATE FIELDS as with JobEnd().
     *
     * The operation MUST update the following fields:
     *    info->jobStatus       set APS_JOB_SENDING/PRINTING/PAUSED/
     *                              SUSPENDED/FAILED/DISCARDED, etc...
     */
    Aps_Result(*JobDispatch) (Transport * this, Job * job,
        const char *filename);

    /* Update a Job given its current entry.
     * The transport is expected to ONLY update fields of QuickJobInfo
     * for which it can provide accurate or truly useful information.
     * (eg. it should not set docName to "" simply because it is
     *      unable to determine otherwise)
     *
     *
     * This operation must not affect the status of existing jobs
     * during the queue reading operation.
     *
     * String fields MUST be updated using the strupdate() function of
     * APS utils.c.
     *
     * Normally a transport will have very little information to rely
     * on for finding a match in the queue for a specific Job. The
     * following procedure is recommended for locating jobs:
     *
     * 1. If job->transportData contains any useful internal
     *    information, use that to find the job. Unless the data
     *    is unreliable (why keep it then?), it is safe to
     *    return failure for the Update operation if the job could
     *    not be located with stored information.
     *
     * 2a. Use job->jobID if not '-1' and reliable.  If it doesn't
     *     match, bail out.
     * 2b. Use job->localHost and job->localFile if not NULL/""
     *     and reliable.  If it doesn't match, bail out.
     * 2c. Use any other reliable (stored) fields which can be
     *     located easily and verify uniqueness if unsure.
     *
     * 3. Give up! Return APS_IGNORED unless previous checks have
     *    _convinced_ you that the job does not exist anymore, in
     *    which case APS_NOT_FOUND is the way to go.
     *
     * The transport will always know the following fields when
     * an update is performed.
     *    info->printerName     printer's name
     *    info->printerHandle   printer handle
     *    job->printer          set accordingly
     *
     * The transport should update the following fields if possible:
     *    info->jobStatus
     *    info->printerStatus
     *
     * The transport should _make an effort_ toward filling in these
     * fields if they are currently unknown:
     *    info->jobHost         (if "")
     *    info->jobName         (if "")
     *    info->jobFilename     (if "")
     *    info->jobID           (if -1)
     *    info->jobSize         (if 0)
     *    info->jobCreationTime (if 0)
     *    info->docTitle        (if NULL)
     *    info->docRevision     (if NULL)
     *    info->docComments     (if NULL)
     *    info->docAuthor       (if NULL)
     *    info->docType         (if NULL)
     *    info->docCreator      (if NULL)
     *    info->ownerName       (if NULL)
     *    info->ownerID         (if -1)
     *    info->localHost       (if NULL)
     *    info->localFile       (if NULL)
     *    info->spoolHost       (if NULL)
     *    info->spoolFile       (if NULL)
     *
     * The transport should probably ignore the following fields unless
     * it's VERY smart, and only if they are unknown:
     *    info->jobFormat
     *    info->jobAttr
     *
     * The transport must not modify the following fields:
     *    info->version
     *    info->jobHandle
     *    info->printerHandle
     *    info->printerName
     *
     * Returns APS_NOT_FOUND if not found,
     *         APS_NO_CHANGE if there was no change of any kind,
     *         APS_IGNORED if it couldn't be updated,
     *         APS_SUCCESS on success,
     *         other appropriate errors...
     */
    Aps_Result(*JobUpdate) (Transport * this, Job *job);

    /* Given a job handle, returns the next job handle in the
     * list.  If the handle is NULL, the transport may choose
     * to reload its internal caches.
     *
     * The sequence of returned jobs is undefined but must not
     * be interrupted or repeated.  If new jobs are added to
     * the queue while scanning, they MUST eventually be
     * listed by this iterator.  Similarly, jobs which have already
     * been returned must not again be returned unless the
     * iterator is reset by starting with NULL.
     *
     * This operation may automatically update jobs as they
     * are returned.
     *
     * This operation may affect the status of existing jobs
     * during the queue reading operation.
     *
     * Return *job=NULL when done and APS_NO_MORE_DATA.
     *        else APS_SUCCESS.
     *
     * Recommended implementation:
     *   - Maintain a doubly-linked list of jobs in the transport's
     *     private data
     *   - On JobInit() add a job to the tail of the list
     *   - In JobCleanup() unlink a job from the list (this job is
     *     guaranteed never to be reused).
     *   - If a job is present in the stored queue but not in the
     *     cache, check if it was created by the application using
     *     JobStart() / JobDispatch(). If not, then assume the job
     *     no longer exists and set its status to APS_JOB_COMPLETED
     *     if it was not in APS_JOB_PHASE_ENDED at the time.
     *   - Retain a lock on "discovered" jobs while stored in the cache
     *     unless absolutely necessary as this will help prevent
     *     excessive cleanup operations. They should be unlocked when
     *     they cease to exist, physically.
     *   - If it is necessary to create a job object, call the
     *     TransJobCreate(printer, info, & job)
     *   - Before returning an object, lock it.  If an object is passed
     *     as source, free it.
     *   - When called with *job=NULL, return the job from the
     *     head of the list
     *   - Else return the next job
     */
    Aps_Result(*JobIterate) (Transport *this, Printer *printer,
        Job **job);

    /* Rebuild the transport's internal representation of the queue
     * Equivalent to applying JobUpdate() on all active jobs.
     *
     * The transport must update the status of all of its jobs
     * at once. To do so, it must have retained a cache of all
     * jobs in its possession using JobInit and JobCleanup.
     * The procedures for both JobUpdate() and JobIterate() apply
     * here.
     *
     * Always return APS_SUCCESS unless catastrophic failure.
     */
    Aps_Result(*RebuildQueue) (Transport *this, Printer *printer);

    /* Perform the operation specified by 'op' on the job.
     * May alter status. (See Aps_OperationID enum)
     *
     * Used for both queries and commands, by Aps_JobIssueQuery()
     * and Aps_JobIssueCommand()
     *
     * Must implement:
     *  - APS_OP_JDELETE                : purge individual job
     *  - APS_OP_JCANCEL                : (as above if no distinction)
     * Should implement: (return APS_NOT_SUPPORTED if not)
     *  - APS_OP_JPAUSE                 : pause individual job
     *  - APS_OP_JRESUME                : resume individual job
     * May implement: (return APS_NOT_SUPPORTED if not)
     *  - APS_OP_JSTORENOW              : store job for later
     *  - APS_OP_JSTOREWHENCOMPLETE     : store job when complete
     */
    Aps_Result(*JobControl) (Transport * this, Job *job,
        Aps_OperationID op, void *argin, void *argout);

    /* Perform the operation specified by 'op' on the queue.
     * May alter status of some or all jobs. (See Aps_
     *   OperationID enum)
     *
     * Used for both queries and commands, by Aps_QueueIssueQuery()
     * and Aps_QueueIssueCommand()
     *
     * Must implement:
     *  - APS_OP_QPURGE                 : purge all accessible
     *  - APS_OP_QSUSPEND               : suspend printing
     *  - APS_OP_QRESUME                : resume printing
     * Should implement: (return APS_NOT_SUPPORTED if not)
     *  - APS_IS_QREADY                 : is queue ready?
     *  - APS_IS_QSUSPENDED             : is queue suspended?
     * May implement: (return APS_NOT_SUPPORTED if not)
     *  - APS_OP_QPAUSEALLJOBS          : pause all
     *  - APS_OP_QRESUMEALLJOBS         : resume all
     *  - APS_OP_QAUTOPAUSENEWJOBS      : auto-pause
     *  - APS_OP_QNOAUTOPAUSENEWJOBS
     *  - APS_IS_QAUTOPAUSENEWJOBS      : auto-pause enabled?
     *  - APS_OP_QAUTODISCARDNEWJOBS    : auto-discard
     *  - APS_OP_QNOAUTODISCARDNEWJOBS
     *  - APS_IS_QAUTODISCARDNEWJOBS    : auto-discard enabled?
     *  - APS_OP_QAUTODENYNEWJOBS       : auto-deny
     *  - APS_OP_QNOAUTODENYNEWJOBS
     *  - APS_IS_QAUTODENYNEWJOBS       : auto-deny enabled?
     */
    Aps_Result(*QueueControl) (Transport * this, Printer *printer,
        Aps_OperationID op, void *argin, void *argout);

    /* Query if a specific operation is supported for a given job or queue.
     * This must not alter the status of a job, nor should it perform
     * expensive tests (especially of a speculative nature) to determine
     * if an operation is supported explicitely for the specified object.
     *
     * If an appropriate test was performed or if we have no doubt about
     *   its outcome, return APS_SUCCESS.
     * If the test could not be performed, return APS_IGNORED.
     * If the operation was not recognized, return APS_NOT_FOUND.
     *
     * Set *anticipatedResult to the anticipated result code of the
     * operation, were it attempted.  For instance,
     *   APS_OPERATION_AVAILABLE  - operation is available
     *   APS_NOT_SUPPORTED        - operation not supported
     *   APS_NOT_FOUND            - physical device couldn't be located
     *   <other conditions....>
     */
    Aps_Result(*JobIsOperationAvailable) (Transport * this, Job *job,
        Aps_OperationID op, Aps_Result *anticipatedResult);
    Aps_Result(*QueueIsOperationAvailable) (Transport * this,
        Printer *printer, Aps_OperationID op, Aps_Result *anticipatedResult);

    /* Get a list of printers */
    Aps_Result(*GetPrinters) (Transport * this, TrackArray_PtrChar *names);

    /* Get default printer's name */
    char *(*GetDefaultPrinterName) (Transport * this);

    /* Get/Set PPD filename */
    char *(*GetPPDFileName) (Transport * this, Printer * printer);
    Aps_Result(*SetPPDFileName) (Transport * this, Printer * printer,
                                 const char *name);

    /* Create a new printer entry with the specified name. */
    Aps_Result (*AddPrinter) (Transport * this, const char *name, 
                                          Aps_PrinterHandle *handle);

    /* Deletes the specified printer, releasing its handle. */
    Aps_Result (*PrinterRemove) (Transport * this,
                                          Aps_PrinterHandle handle);

    /* Sets the default printer, given a transport pointer. */
    Aps_Result (*PrinterSetAsDefault) (Transport * this,
                                          Aps_PrinterHandle handle);
    /* Checks if the printer ifor which handle passed is default . */
    Aps_Result (*PrinterIsDefault)(Transport * this,
                                  Aps_PrinterHandle handle, int *isDefault);
    /* Change the name of the printer for which handle is passed. */
    Aps_Result (*PrinterRename)(Transport * this,
                        Aps_PrinterHandle handle, const char *newName);
    /* Set the max Job size for which handle is passed. */
    Aps_Result (*PrinterSetMaxJobSize)(Transport * this,
                        Aps_PrinterHandle handle, int  newSize);
    /* Get the max Job size for which handle is passed. */
    Aps_Result (*PrinterGetMaxJobSize)(Transport * this,
                        Aps_PrinterHandle handle, int  *newSize);
    /* Sets the standard configuration flags for a particular printer */
    Aps_Result (*PrinterSetConfigFlags)(Transport * this,
        Printer *printer, long int flagMask, long int flagSet);
    /* Obtains the standard configuration flags that are set for a printer */
    Aps_Result (*PrinterGetConfigFlags)(Transport * this,
        Printer *printer, long int  *configFlags);
    /* Get the printer connection info */
    Aps_Result (*PrinterGetConnectInfo)(Transport * this,
        Printer *printer, Aps_ConnectionType *connectionType,
        char **location);
    /* Set the Printer Connection Info */
    Aps_Result (*PrinterSetConnectInfo)(Transport *this, 
        Printer *printer, Aps_ConnectionType connectionType,
        const char *location);

    /* Attempts to determine the model of a particular printer. Called if APS
     * isn't already configured with this information for this printer.
     */
    Aps_Result (*ProbeModelForPrinter)(Transport *this,
                                       Printer *printer,
                                       Aps_ModelHandle *model);

    /* Get default printer configuration attributes.
     * Should extract / guess as much meaningful information as possible
     * and assign to correct providers for processing.
     *
     * Call this function with a pointer to the set of job attributes to
     * be updated.  May be printer->defaultJobAttributes.
     *
     * To perform a consistency check, diff the returned attributes against
     * a reference base.
     */
    Aps_Result (*PrinterAttrGetDefaults)(Transport *this,
        Printer *printer, JobAttributes *attr);

    /* Set default printer configuration based on a set of attributes.
     * Should extract as much meaningful information from these attribs
     * (using the various providers) and update the cached context or
     * other configs based on that.
     *
     * Call this function with a pointer to the set of job attributes to
     * be saved.  Usually printer->defaultJobAttributes.
     */
    Aps_Result (*PrinterAttrSaveDefaults)(Transport *this,
        Printer *printer, JobAttributes *attr);
};

/* This object defines a data transport through which all print
 * operations will occur.
 */
struct Transport_
{
    TransportVtbl *vtbl;  /* vector table for object polymorphism */
};

/* Lookup functions */
Transport **TransGetAllTransports(int *count);
Transport *TransGetFromPrinterName(const char *name);
Aps_Result TransGetPrinter(const char *name, Printer ** printer);

/* support functions */
Aps_Result TransJobCreate(Printer *printer, Aps_QuickJobInfo *info, Job **job);

/* default entry points for transports only supporting BULK access */
Aps_Result DefaultBULK_JobStart(Transport *thisBase, Job *job);
Aps_Result DefaultBULK_JobWrite(Transport *thisBase, Job *job,
    const void *data, size_t size);
Aps_Result DefaultBULK_JobGetFileDescriptor(Transport *thisBase, Job *job,
    int *fd);
Aps_Result DefaultBULK_JobEnd(Transport *thisBase, Job *job);
Aps_Result DefaultBULK_JobAbort(Transport *thisBase, Job *job);

/* default entry points for transports only supporting PROGRESSIVE access */
Aps_Result DefaultPROG_JobDispatch(Transport *thisBase,
    Job *job, const char *filename);
Aps_Result DefaultPROG_JobGetFileDescriptor(Transport *thisBase, Job *job,
    int *fd);

#endif /* !TRANSPORT_H */
