/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apsqueue.c
 *
 * Description: Defines the ApsQueueObject class method.
 *
 */              

#include "apscpp.h"

/*---------------------------------------------------------------------------
 * ApsQueueObject ()
 *
 *
 * This constructor takes queue handle as argument.
 *
 * Parameters:
 *    queueHandle : handle to some queue.
 *
 *     Return: ApsResultObject.
 *
 */                                        
ApsQueueObject::ApsQueueObject(Aps_QueueHandle queueHandle)
                         : ApsObject((Aps_Handle)queueHandle)
{
}

/*---------------------------------------------------------------------------
 * ApsQueueObject ()
 *
 * Copy constructor.
 *
 * Parameters:
 *    apsQueue : reference to ApsQueueObject object.
 *
 *
 *     Return: ApsResultObject.
 *
 */              
ApsQueueObject::ApsQueueObject(const ApsQueueObject &apsQueue)
{
    Aps_Handle apsHandle;

    apsHandle = apsQueue.GetHandle();
    SetHandle(apsHandle);
}        
/*---------------------------------------------------------------------------
 * OpenGlobalQueue()
 *
 * Obtains a handle to the specified printers  queue.
 * This method internally calls Aps_JobWrite Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    handle : Pointer to an Aps_QueueHandle that will receive the new handle.
 *             On failure, will be filled with NULL.
 *
 *     Return: ApsResultObject.
 *
 */                   
ApsResultObject ApsQueueObject::OpenGlobalQueue()
{
    Aps_Result resultCode;
    Aps_QueueHandle queueHandle;
 
    resultCode =  Aps_OpenGlobalQueue(&queueHandle);
    if (Aps_Succeeded(resultCode))
        SetHandle(queueHandle);
    return (ApsResultObject(resultCode));
}

/*---------------------------------------------------------------------------
 * IssueCommand()
 *
 * Performs a single specific verification request.
 * This method internally calls Aps_JobWrite Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     operationCode : One of the defined APS_OP_Qxxxxx commands described below.
 *     reserved : Optional parameters. 
 *
 *     Return: ApsResultObject.
 *
 */                   
ApsResultObject ApsQueueObject::IssueCommand(Aps_OperationID operationCode,
                                                    void *reserved)
{
    Aps_QueueHandle handle;

    handle = (Aps_QueueHandle)GetHandle(); 
    return (ApsResultObject(Aps_QueueIssueCommand(handle, operationCode, 
                                                           reserved)));
}

/*---------------------------------------------------------------------------
 * IssueQuery()
 *
 * Performs a single specific verification request from a queue.
 * This method internally calls Aps_QueueIssueQuery Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     operationCode : One of the defined APS_IS_Qxxxxx commands.
 *     output : A query-specific type to receive the output of this query. 
 *
 *     Return: ApsResultObject.
 *
 */                   
ApsResultObject ApsQueueObject::IssueQuery(Aps_OperationID operationCode, 
                                                         void *reserved)
{
    Aps_QueueHandle handle;

    handle = (Aps_QueueHandle)GetHandle(); 
    return (ApsResultObject(Aps_QueueIssueQuery(handle, operationCode, 
                                                                reserved)));
}

/*---------------------------------------------------------------------------
 * GetNumberOfJobs()
 *
 * Gets the number of jobs matching the filter in the queue.
 * This method internally calls Aps_QueueGetNumberOfJobs Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *    numberOfJobs : Receives the total number of jobs matching the specified 
 *                   filter.
 *
 *     Return: ApsResultObject.
 *
 */                   
ApsResultObject ApsQueueObject::GetNumberOfJobs(int *numberOfJobs)
{
    Aps_QueueHandle handle;

    handle = (Aps_QueueHandle)GetHandle(); 
    return (ApsResultObject(Aps_QueueGetNumberOfJobs(handle, numberOfJobs)));
}

/*---------------------------------------------------------------------------
 * MakeQuickPrinterQInfoArray()
 *
 * Writes data to be printed as part of this job.
 * This method internally calls Aps_QueueMakeQuickPrinterInfoArray Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *     printerQueueInfo : Pointer to an array of pointers to 
 *                        Aps_QuickPrinterQInfo structures.
 *
 *     Return: ApsResultObject.
 *
 */                   
ApsResultObject ApsQueueObject::MakeQuickPrinterQInfoArray(
                                    Aps_QuickPrinterQInfo ***printerQueueInfo,
				    int *numEntries)
{
    Aps_QueueHandle handle;

    handle = (Aps_QueueHandle)GetHandle(); 
    return (ApsResultObject(Aps_QueueMakeQuickPrinterQInfoArray(handle, 
                                            printerQueueInfo, numEntries)));
}  
