/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: aps.h
 *
 * Description: Definitions of public API functions and types provided
 *              by the APS library. For information on the use of the APS
 *              library and these functions, refer to doc/api.html. To make
 *              it easy to correspond the functions and types in this header
 *              file with the documentation, both are organized in the same
 *              manner.
 */

#ifndef APS_H
#define APS_H

#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

/* ---------------------------------------------------------------------------
 * 3.1 LIBRARY-WIDE ELEMENTS
 */

/* Boolean constants are defined here, if not already present. */
#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE (!FALSE)
#endif

/* 3.1.1 LIBRARY-WIDE ELEMENTS - Result Code Support Functionality */

/* APS standard result codes. */
/* These codes must NOT be shuffled around!! */
typedef enum {
    APS_SUCCESS = 0, /* let's make this more explicit */
    APS_NOT_IMPLEMENTED,
    APS_NOT_SUPPORTED,
    APS_INVALID_PARAM,
    APS_MORE_DATA,
    APS_OUT_OF_MEMORY,
    APS_NOT_FOUND,
    APS_ACCESS_DENIED,
    APS_INVALID_HANDLE,
    APS_GENERIC_FAILURE,
    APS_DISK_FULL,
    APS_INVALID_PWD,
    APS_OUT_OF_SEQUENCE,
    APS_VIOLATES_CONSTRAINTS,
    APS_INVALID_PPD,
    APS_WRONG_TYPE,
    APS_ALREADY_EXISTS,
    APS_OPERATION_AVAILABLE,
    APS_NO_CHANGE,
    APS_IGNORED,
    APS_PARTIAL_SUCCESS,
    APS_OPERATION_TIMEOUT,
    APS_HAS_EXTENDED_SUPPORT,
    APS_IO_ERROR,
    APS_SUBPROGRAM_FAILED,
    APS_FILTER_NOT_SUPPORTED,
    APS_NO_MORE_DATA,
    APS_MODEL_UNKNOWN,

    APS_NUMRESULTS
} Aps_Result;

/* Result code support functions */
Aps_Result Aps_GetResultText(Aps_Result result, char *text, int size);
int Aps_Succeeded(Aps_Result result);


/* 3.1.2 LIBRARY-WIDE ELEMENTS - Operations That Can be Performed on Any Type
 *                               Of Handle
 */

/* APS handle types. */
typedef void * Aps_Handle;
typedef Aps_Handle Aps_PrinterHandle;
typedef Aps_Handle Aps_JobHandle;
typedef Aps_Handle Aps_JobAttrHandle;
typedef Aps_Handle Aps_ModelHandle;
typedef Aps_Handle Aps_NotificationHandle;
typedef Aps_Handle Aps_FilterHandle;
typedef Aps_Handle Aps_QueueHandle;

/* Events that the application can be notified of. */
typedef enum {
    APS_EVENT_O_BASE = 0x8000,
    APS_EVENT_Q_BASE = 0x9000,
    APS_EVENT_QCHANGED,
    APS_EVENT_J_BASE = 0xA000,
    APS_EVENT_P_BASE = 0xB000,
    APS_EVENT_A_BASE = 0xC000
} Aps_Event;

/* Event notification callback function prototype. */
typedef void (*Aps_NotificationCallback)(
    Aps_Handle objectSendingNotification,
    Aps_Event eventThatHasOccurred,
    void *eventData,
    void *appSpecifiedData);
    
/* Operations that the application can perform on objects. */
typedef enum {
    /* OPERATIONS FOR ALL OBJECTS */

    /* Basic operations */
    APS_OP_O_BASE = 0x0000,
    APS_OP_OINFO,
    APS_OP_ODO,
    
    /* OPERATIONS FOR OBJECTS WITH SET/GET/QUERY */
    APS_OP_O_BASE2 = 0x0040,
    APS_OP_OSET,
    APS_OP_OGET,
    APS_OP_OQUERY,
    
    /* OPERATIONS FOR OBJECTS WITH NOTIFICATION */
    APS_OP_O_BASE3 = 0x0080,
    APS_OP_ONOTIFYADD,
    APS_OP_ONOTIFYREMOVE,
    
    /* OPERATIONS FOR OBJECTS WITH FILTER SUPPORT */
    APS_OP_O_BASE4 = 0x00C0,
    APS_OP_OFILTERATTACH,
    APS_OP_OFILTERDETACH,

    /* QUERY OPERATIONS SUPPORTED BY ALL OBJECTS */
    APS_IS_O_BASE = 0x0100,
    APS_IS_OVALID,
    APS_IS_OQUEUE,
    APS_IS_OPRINTER,
    APS_IS_OATTRIBUTE,

    /* QUEUE OBJECT */

    /* Queue control operations */
    APS_OP_Q_BASE = 0x1000,
    APS_OP_QPURGE,
    APS_OP_QSUSPEND,
    APS_OP_QRESUME,
    APS_OP_QPAUSEALLJOBS,
    APS_OP_QRESUMEALLJOBS,
    APS_OP_QAUTOPAUSENEWJOBS,
    APS_OP_QNOAUTOPAUSENEWJOBS,
    APS_OP_QAUTODISCARDNEWJOBS,
    APS_OP_QNOAUTODISCARDNEWJOBS,
    APS_OP_QAUTODENYNEWJOBS,
    APS_OP_QNOAUTODENYNEWJOBS,

    /* Queue is/query operations */
    APS_IS_Q_BASE = 0x1100,
    APS_IS_QREADY,
    APS_IS_QSUSPENDED,
    APS_IS_QAUTOPAUSENEWJOBS,
    APS_IS_QAUTODENYNEWJOBS,
    APS_IS_QAUTODISCARDNEWJOBS,

    /* JOB OBJECT */

    /* Job control operations */
    APS_OP_J_BASE = 0x2000,
    APS_OP_JPAUSE,
    APS_OP_JRESUME,
    APS_OP_JSTORENOW,
    APS_OP_JSTOREWHENCOMPLETE,
    APS_OP_JDELETE,
    APS_OP_JCANCEL,

    /* Job is/query operations */
    APS_IS_J_BASE = 0x2100,

    /* PRINTER OBJECT */
    APS_OP_P_BASE = 0x3000,
    APS_IS_P_BASE = 0x3100,

    /* ATTRIBUTES OBJECT */
    APS_OP_A_BASE = 0x4000,
    APS_IS_A_BASE = 0x4100
} Aps_OperationID;

/* Generic object functions */
Aps_Result Aps_AddRef(Aps_Handle handle);
Aps_Result Aps_ReleaseHandle(Aps_Handle handle);
Aps_Result Aps_SubscribeToNotificationCB(Aps_Handle object,
                                         Aps_Event event,
                                         Aps_NotificationCallback
                                             callbackFunction,
                                         void *appData,
                                         Aps_NotificationHandle *
                                             notificationHandle);
Aps_Result Aps_IsOperationAvailable(Aps_Handle object,
                                    Aps_OperationID operation,
                                    Aps_Result *anticipatedResult);

                                             
/* 3.1.3 LIBRARY-WIDE ELEMENTS - Other General-Purpose Library Functions. */

Aps_Result Aps_ReleaseBuffer(const void *buffer);


/* ---------------------------------------------------------------------------
 * 3.2 PRINTER ENUMERATION AND INSTATIATION
 */

/* Connection type used for identifying protocol used to connect to a
 * remote (network) printer.
 */
typedef enum {
    APS_CONNECT_LOCAL,
    APS_CONNECT_NETWORK_LPD,
    APS_CONNECT_NETWORK_SMB,
    APS_CONNECT_ALL
} Aps_ConnectionType;

/* Information on the name and location of an available remote printer. */
typedef struct Aps_RemotePrinter_ {
    char *name;
    Aps_ConnectionType connectionType;
    char *location;
} Aps_RemotePrinter;

Aps_Result Aps_OpenDefaultPrinter(Aps_PrinterHandle *printer);
Aps_Result Aps_GetPrinters(char ***names, int *count);
Aps_Result Aps_OpenPrinter(const char *name, Aps_PrinterHandle *printer);
Aps_Result Aps_OpenRemotePrinter(Aps_ConnectionType connectionType,
                                 const char *location,
                                 Aps_PrinterHandle *printer);


/* ---------------------------------------------------------------------------
 * 3.3 PRINTER INSTALLATION AND REMOVAL
 */

/* Printer installation and removal functions (printeraddremove.c). */
Aps_Result Aps_AddPrinter(const char *name, Aps_PrinterHandle *printer);
Aps_Result Aps_PrinterRemove(Aps_PrinterHandle printer);
Aps_Result Aps_PrinterInstallLocalCopy(Aps_PrinterHandle remotePrinter,
                                       Aps_PrinterHandle *newLocalCopy);


/* ---------------------------------------------------------------------------
 * 3.4 PRINTER CONFIGURATION
 */

/* 3.4.1 PRINTER CONFIGURATION - Printer Identification */

Aps_Result Aps_PrinterGetName(Aps_PrinterHandle printerHandle, char **name);
Aps_Result Aps_PrinterRename(Aps_PrinterHandle printer, const char *newName);


/* 3.4.2 PRINTER CONFIGURATION - Default Printer Tagging */

Aps_Result Aps_PrinterIsDefault(Aps_PrinterHandle printer, int *isDefault);
Aps_Result Aps_PrinterSetAsDefault(Aps_PrinterHandle printer);


/* 3.4.3 PRINTER CONFIGURATION - Administrative and Tuning Configuration */

/* Administrative and tuning configuration. */
/* Maximum job size magic values. */
#define APS_NO_MAX_SIZE 0

/* Configuration flags. */
#define APS_CONFIG_EOF_AT_END    0x1
#define APS_CONFIG_ADD_CR        0x2
#define APS_CONFIG_TEXT_AS_TEXT  0x4
#define APS_CONFIG_HEADER_PAGE   0x8

/* Administrative and tuning configuration functions. */
Aps_Result Aps_PrinterGetMaxJobSize(Aps_PrinterHandle printer, int *maxSize);
Aps_Result Aps_PrinterSetMaxJobSize(Aps_PrinterHandle printer, int maxSize);
Aps_Result Aps_PrinterGetConfigFlags(Aps_PrinterHandle printer,
                                     long int *configFlags);
Aps_Result Aps_PrinterSetConfigFlags(Aps_PrinterHandle printer,
                                     long int flagsToSet,
                                     long int flagsToReset);

                                     
/* 3.4.4 PRINTER CONFIGURATION - Printer Diagnostics */

Aps_Result Aps_PrinterSendTestPage(Aps_PrinterHandle printer,
                                   Aps_JobHandle *job);

                                   
/* ---------------------------------------------------------------------------
 * 3.5 PRINTER MANUFACTURER AND MODEL INFORMATION
 */

/* 3.5.1 PRINTER MANUFACTURER AND MODEL INFORMATION - Printer Model Database
 *                                                    Access
 */

Aps_Result Aps_GetKnownManufacturers(char ***manufacturerNames,
                                     int *numManufacturers);
Aps_Result Aps_GetKnownModels(const char *manufacturerName,
                              char ***modelNames,
                              int *numModels);
Aps_Result Aps_GetModel(const char *manufacturer,
                        const char *model,
                        Aps_ModelHandle *modelHandle);
Aps_Result Aps_AddModel(const char *manufacturer,
                        const char *model,
                        Aps_ModelHandle *modelHandle);
Aps_Result Aps_AddModelFromPPD(const char *pathToPPDFile,
                               Aps_ModelHandle *modelHandle);

                        
/* 3.5.2 PRINTER MANUFACTURER AND MODEL INFORMATION - Model Object
 *                                                    Manipulation
 */

Aps_Result Aps_GetPropertyString(Aps_Handle object, const char *propertyName,
                                 char **value);
Aps_Result Aps_SetPropertyString(Aps_Handle object, const char *propertyName,
                                 const char *value);
Aps_Result Aps_GetPropertyStrArray(Aps_Handle object, const char *propertyName,
                                  char ***value, int *numElements);
Aps_Result Aps_SetPropertyStrArray(Aps_Handle object, const char *propertyName,
                                   const char **value, int numElements);
Aps_Result Aps_ModelCommitToDatabase(Aps_ModelHandle model);
Aps_Result Aps_ModelRemove(Aps_ModelHandle model);


/* 3.5.3 PRINTER MANUFACTURER AND MODEL INFORMATION - Configuring the Model
 *                                                    Information Associated
 *                                                    With a Printer
 */

Aps_Result Aps_PrinterGetModel(Aps_PrinterHandle printer,
                               char **manufacturer,
                               char **model);
Aps_Result Aps_PrinterSetModel(Aps_PrinterHandle printer,
                               const char *manufacturer,
                               const char *model);
Aps_Result Aps_PrinterGetModelHandle(Aps_PrinterHandle printer,
                                     Aps_ModelHandle *model);
Aps_Result Aps_PrinterGetPPDFileName(Aps_PrinterHandle printer,
                                     char **filename);
Aps_Result Aps_PrinterSetPPDFileName(Aps_PrinterHandle printer,
                                     const char *filename);

                                     
/* ---------------------------------------------------------------------------
 * 3.6 PRINTER LOCATION INFORMATION
 */

Aps_Result Aps_GetRemotePrinters(Aps_ConnectionType connectionType,
                                 const char *subLocation,
                                 Aps_RemotePrinter ***remotePrinters);
Aps_Result Aps_PrinterGetConnectInfo(Aps_PrinterHandle printer,
                                     Aps_ConnectionType *connectionType,
                                     char **location);
Aps_Result Aps_PrinterSetConnectInfo(Aps_PrinterHandle printer,
                                     Aps_ConnectionType connectionType,
                                     const char *location);

                                     
/* ---------------------------------------------------------------------------
 * 3.7 PRINTER/JOB ATTRIBUTES
 */

/* 3.7.1 PRINTER/JOB ATTRIBUTES - Accessing a Printer's Default Attributes */

Aps_Result Aps_PrinterGetDefAttr(Aps_PrinterHandle printer,
                                 Aps_JobAttrHandle *jobAttributes);
Aps_Result Aps_PrinterSetDefAttr(Aps_PrinterHandle printer,
                                 Aps_JobAttrHandle jobAttributes);

                                 
/* 3.7.2 PRINTER/JOB ATTRIBUTES - Accessing a Job's Attributes */

Aps_Result Aps_JobGetAttributes(Aps_JobHandle job,
                                Aps_JobAttrHandle *attributes);
Aps_Result Aps_JobSetAttributes(Aps_JobHandle job,
                                Aps_JobAttrHandle attributes);

                                
/* 3.7.3 PRINTER/JOB ATTRIBUTES - Generic Job Attribute Querying and
 *                                Manipulation
 */

/* Identifying the type of an individual attribute. */
typedef enum {
    APS_ATTR_READ_ONLY_DATA,
    APS_ATTR_SETTING_INT,
    APS_ATTR_SETTING_FLOAT,
    APS_ATTR_SETTING_BOOL,
    APS_ATTR_SETTING_PICK_ONE,
    APS_ATTR_SETTING_PICK_MANY,
    APS_ATTR_SETTING_STRING
} Aps_AttrType;

/* Predefined attribute groups. */
#define APS_GROUP_ROOT              ((const char *)0x01)
#define APS_GROUP_ROOT_SETTINGS     ((const char *)0x02)
#define APS_GROUP_ALL_SETTINGS      ((const char *)0x03)
#define APS_GROUP_READ_ONLY_DATA    ((const char *)0x04)

/* Information on one possible state of a particular attribute. */
typedef struct {
    int version;
    char *optionID;
    char *translatedName;
    char *value;
} Aps_AttrOption;

/* Generic job attribute querying and manipulation. */
Aps_Result Aps_AttrCreateCopy(Aps_JobAttrHandle existingAttributes,
                              Aps_JobAttrHandle *newAttributes);
Aps_Result Aps_AttrGetList(Aps_JobAttrHandle jobAttributes,
                           const char *group,
                           int *numAttributes,
                           char ***attributeIDs);
Aps_Result Aps_AttrGetSubGroups(Aps_JobAttrHandle jobAttributes,
                                const char *group,
                                int *numSubGroups,
                                char ***subGroupNames);
Aps_Result Aps_AttrGetTranslatedName(Aps_JobAttrHandle jobAttributes,
                                     const char *attributeID,
                                     char **translatedName);
Aps_Result Aps_AttrGetMainData(Aps_JobAttrHandle jobAttributes,
                               const char *attributeID,
                               char **mainData);
Aps_Result Aps_AttrGetTranslatedData(Aps_JobAttrHandle jobAttributes,
                               const char *attributeID,
                               char **translatedData);
Aps_Result Aps_AttrGetType(Aps_JobAttrHandle jobAttributes,
                           const char *attributeID,
                           Aps_AttrType *attributeType);
Aps_Result Aps_AttrGetRange(Aps_JobAttrHandle jobAttributes,
                            const char *attributeID,
                            double *minSetting,
                            double *maxSetting);
Aps_Result Aps_AttrGetOptions(Aps_JobAttrHandle jobAttributes,
                              const char *attributeID,
                              int *numOptions,
                              Aps_AttrOption ***options);
Aps_Result Aps_AttrGetSetting(Aps_JobAttrHandle jobAttributes,
                              const char *attributeID,
                              char **setting);
Aps_Result Aps_AttrSetSetting(Aps_JobAttrHandle jobAttributes,
                              const char *attributeID,
                              const char *setting);
Aps_Result Aps_AttrCheckConstraints(Aps_JobAttrHandle jobAttributes,
                                    const char *attributeID,
                                    const char *setting,
                                    char **conflictingAttribute,
                                    char **conflictingSetting);


/* 3.7.4 PRINTER/JOB ATTRIBUTES - Quick Access to Common Attributes */

/* PostScript language level supported. */
typedef enum {
    APS_PS_NOT_SUPPORTED,
    APS_PS_LEVEL1,
    APS_PS_LEVEL2,
    APS_PS_LEVEL3
} Aps_PostScriptSupport;

/* Printer resolution information. */
typedef struct Aps_Resolution_ {
    int version;          /* 0 for now. */
    double horizontalRes; /* Horizontal resolution in dots per inch */
    double verticalRes;   /* Vertical resolution, also in DPI */
} Aps_Resolution;

/* Page size definitions. */
typedef enum {
    APS_ROTATE_0,
    APS_ROTATE_90,
    APS_ROTATE_180,
    APS_ROTATE_270
} Aps_Rotation;

typedef enum {
    APS_LEADING_SHORT,
    APS_LEADING_LONG,
    APS_LEADING_PREFERLONG,
    APS_LEADING_FORCED,
    APS_LEADING_UNKNOWN
} Aps_LeadingEdge;

typedef struct Aps_PageSize_ {
    int version;             /* 0 for now. */

    char *id;                /* A string uniquely identifying this page. */
    char *translatedName;    /* The page size name to display to the user. */
    double mediaWidth;       /* The width of the paper/media in points. */
    double mediaHeight;      /* The height in points. */
    double imageableAreaLLx; /* X-coordinate of lower left corner. */
    double imageableAreaLLy; /* Y-coordinate of lower left corner. */
    double imageableAreaURx; /* X-coordinate of upper right corner. */
    double imageableAreaURy; /* Y-coordinate of upper right corner. */

    /* Information used only for custom page sizes. */
    int isCustom;            /* TRUE if this is a custom page size. */
    double widthOffset;      /* The position of the page on the media. */
    double heightOffset;
    Aps_Rotation orientation;
    int useHWMargins;        /* Normally TRUE, FALSE to force roll-fed. */
    Aps_LeadingEdge leadingEdge;
} Aps_PageSize;

/* Information on constraints placed on custom page sizes. */
typedef struct Aps_CustomPageSizeLimits_ {
    int version;            /* 0 for now. */

    double maxMediaWidth;   /* Maximum width of the page. */
    double maxMediaHeight;  /* Maximum height of the page. */
    int centerRegistered;   /* TRUE if page must be centered on media. */
    double leftHWMargin;    /* Size of non-imageable area at left edge. */
    double bottomHWMargin;  /* Size of non-imageable area at bottom. */
    double rightHWMargin;   /* Size of non-imageable area at right edge. */
    double topHWMargin;     /* Size of non-imageable area at top. */
    int optionalHWMargins;  /* TRUE if HW margins can be turned off. */
    double minWidthOffset;  /* The minimum offset of page across media. */
    double maxWidthOffset;  /* The maximum offset of page across media. */
    double minHeightOffset; /* The minimum offset of page down media. */
    double maxHeightOffset; /* The maximum offset of page down media. */
    Aps_LeadingEdge leadingEdgeOptions;
} Aps_CustomPageSizeLimits;

/* Information on an available font. */
typedef struct Aps_FontInfo_ {
    int version;       /* 0 for now. */

    char *name;        /* The name of the font. */
    int isDefault;     /* TRUE if this is the device's default font. */
} Aps_FontInfo;

Aps_Result Aps_AttrQuickGetPostScriptLevel(
                                        Aps_JobAttrHandle jobAttributes,
                                        Aps_PostScriptSupport *languageLevel);
Aps_Result Aps_AttrQuickIsColorDevice(Aps_JobAttrHandle jobAttributes,
                                      int *colorDevice);
Aps_Result Aps_AttrQuickGetResOptions(Aps_JobAttrHandle jobAttributes,
                                      Aps_Resolution ***resolutions,
                                      int *numResolutions);
Aps_Result Aps_AttrQuickGetRes(Aps_JobAttrHandle jobAttributes,
                               Aps_Resolution *resolution);
Aps_Result Aps_AttrQuickSetRes(Aps_JobAttrHandle jobAttributes,
                               const Aps_Resolution *resolution);
Aps_Result Aps_AttrQuickGetMaxCopies(Aps_JobAttrHandle jobAttributes,
                                     int *maxCopies);
Aps_Result Aps_AttrQuickGetNumCopies(Aps_JobAttrHandle jobAttributes,
                                     int *numCopies);
Aps_Result Aps_AttrQuickSetNumCopies(Aps_JobAttrHandle jobAttributes,
                                     int numCopies);
Aps_Result Aps_AttrQuickGetCollationOptions(Aps_JobAttrHandle jobAttributes,
                                            char ***collationOptions,
                                            int *numOptions);
Aps_Result Aps_AttrQuickGetCollation(Aps_JobAttrHandle jobAttributes,
                                     char **collationSetting);
Aps_Result Aps_AttrQuickSetCollation(Aps_JobAttrHandle jobAttributes,
                                     const char *collationSetting);
Aps_Result Aps_AttrQuickGetPageSizeOptions(Aps_JobAttrHandle jobAttributes,
                                           Aps_PageSize ***pageSizes,
                                           int *numPageSizes);
Aps_Result Aps_AttrQuickGetCustomPageSizeLimits(
                            Aps_JobAttrHandle jobAttributes,
                            Aps_CustomPageSizeLimits **customPageSizeLimits);
Aps_Result Aps_AttrQuickGetPageSize(Aps_JobAttrHandle jobAttributes,
                                    Aps_PageSize **pageSize);
Aps_Result Aps_AttrQuickSetPredefinedPageSize(Aps_JobAttrHandle jobAttributes,
                                              const char *pageSizeID);
Aps_Result Aps_AttrQuickSetCustomPageSize(Aps_JobAttrHandle jobAttributes,
                                          const Aps_PageSize *customPageSize);
Aps_Result Aps_AttrQuickGetInputSlotOptions(Aps_JobAttrHandle jobAttributes,
                                            char ***inputSlots,
                                            int *numInputSlots);
Aps_Result Aps_AttrQuickGetInputSlot(Aps_JobAttrHandle jobAttributes,
                                     char **inputSlot);
Aps_Result Aps_AttrQuickSetInputSlot(Aps_JobAttrHandle jobAttributes,
                                     const char *inputSlot);
Aps_Result Aps_AttrQuickGetOutputBinOptions(Aps_JobAttrHandle jobAttributes,
                                            char ***outputBins,
                                            int *numOutputBins);
Aps_Result Aps_AttrQuickGetOutputBin(Aps_JobAttrHandle jobAttributes,
                                     char **outputBin);
Aps_Result Aps_AttrQuickSetOutputBin(Aps_JobAttrHandle jobAttributes,
                                     const char *outputBin);
Aps_Result Aps_AttrQuickGetDuplexOptions(Aps_JobAttrHandle jobAttributes,
                                         char ***duplexSettings,
                                         int *numDuplexSettings);
Aps_Result Aps_AttrQuickGetDuplex(Aps_JobAttrHandle jobAttributes,
                                  char **duplexSetting);
Aps_Result Aps_AttrQuickSetDuplex(Aps_JobAttrHandle jobAttributes,
                                  const char *duplexSetting);
Aps_Result Aps_AttrQuickGetFonts(Aps_JobAttrHandle jobAttributes,
                                 Aps_FontInfo ***fonts,
                                 int *numFonts);


/* ---------------------------------------------------------------------------
 * 3.8 JOB PRODUCTION AND DISPATCH
 */

/* Standard predefined job formats. */
#define APS_FORMAT_POSTSCRIPT "application/postscript" 
#define APS_FORMAT_TEXT       "text/plain"
#define APS_FORMAT_NATIVE     "application/octet-stream"
#define APS_FORMAT_PDF        "application/pdf"
#define APS_FORMAT_JPEG       "image/jpeg"
#define APS_FORMAT_HTML       "text/html"
#define APS_FORMAT_MULTIPART  "multipart/mixed"

/* Predefined standard blocks that Aps_JobWriteBlock() can generate. */
typedef enum {
    APS_BLOCK_FILE_HEADER,
    APS_BLOCK_PS_PATCH,
    APS_BLOCK_PS_DEV_SETUP,
    APS_BLOCK_FILE_FOOTER,
} Aps_BlockType;

/* Job production/dispatch functions. */
Aps_Result Aps_DispatchJob(Aps_PrinterHandle printer, const char *filename,
                           const char *format, Aps_JobAttrHandle jobAttributes,
                           Aps_JobHandle *job);
Aps_Result Aps_PrinterStartJob(Aps_PrinterHandle printer, const char *format,
                               Aps_JobHandle *job);
Aps_Result Aps_JobGetFileDescriptor(Aps_JobHandle job, int *fd);
Aps_Result Aps_JobWrite(Aps_JobHandle job, const void *data, size_t size);
Aps_Result Aps_JobWriteString(Aps_JobHandle jobHandle, const char *string);
Aps_Result Aps_JobWriteFile(Aps_JobHandle jobHandle, const char *filename);
Aps_Result Aps_JobWriteBlock(Aps_JobHandle job, Aps_BlockType blockType);
Aps_Result Aps_JobEnd(Aps_JobHandle job);
Aps_Result Aps_JobAbort(Aps_JobHandle job);


/* ---------------------------------------------------------------------------
 * 3.9 QUEUE MONITORING AND MANAGEMENT
 */

/* 3.9.1 QUEUE MONITORING AND MANAGEMENT - Accessing and Controlling Queues */

/* Printer status flags */
typedef enum {
    APS_PRINTER_UNKNOWN
        = 0x00000000, /* unknown / default status */
    /* printer status */
    APS_PRINTER_ERROR
        = 0x00000001, /* printer error, check extended codes */
    APS_PRINTER_READY
        = 0x00000002, /* printer is ready for work (online) */
    APS_PRINTER_SUSPENDED
        = 0x00000004, /* print queue is suspended / paused */
    /* states inherited from the days of parallel ports */
    APS_PRINTER_PAR_BUSY
        = 0x00000100, /* parallel port BUSY line set (can't send) */
    APS_PRINTER_PAR_PAPER_OUT
        = 0x00000200, /* parallel port PRINTER OUT line set */
    APS_PRINTER_PAR_SELECT
        = 0x00000400, /* parallel port SELECT line set (online) */
    APS_PRINTER_PAR_ERROR
        = 0x00000800, /* parallel port ERROR line set */
    /* indications of printer activity */
    APS_PRINTER_IDLE
        = 0x00010000, /* printer is not doing anything */
    APS_PRINTER_ACTIVE
        = 0x00020000, /* printer is doing something */
    APS_PRINTER_OCCUPIED
        = 0x00040000, /* printer is busy doing something not related to
                       * our requests */
    /* queue information */
    APS_PRINTER_DENY_NEW_JOBS
        = 0x01000000, /* denies new jobs */
    APS_PRINTER_PAUSE_NEW_JOBS
        = 0x02000000, /* pauses new jobs */
    APS_PRINTER_DISCARD_NEW_JOBS
        = 0x04000000, /* discards new jobs */
} Aps_PrinterStatus;

/* Quick queue information structure */
typedef struct Aps_QuickPrinterQInfo_ {
    int                version;         /* Version 0 for now */
    Aps_PrinterHandle  printerHandle;   /* v0 - handle of printer */
    Aps_PrinterStatus  printerStatus;   /* v0 - RESERVED */
    char              *printerName;     /* v0 - name of printer */
    /* (Total)    # Jobs visible (ignores filter) */
    int numJobsTotal;    /* v0 */
    /* (Match)    # Jobs matching filter */
    int numJobsMatch;    /* v0 */
    /* (Producing)# Jobs matching filter AND APS_JOB_PHASE_PRODUCTION */
    int numJobsProduction;
    /* (Pending)  # Jobs matching filter AND APS_JOB_PHASE_PENDING */
    int numJobsPending;  /* v0 */
    /* (Working)  # Jobs matching filter AND APS_JOB_PHASE_WORKING */
    int numJobsWorking;  /* v0 */
    /* (On Hold)  # Jobs matching filter AND APS_JOB_PHASE_ON_HOLD */
    int numJobsOnHold;  /* v0 */
    /* (Ended)    # Jobs matching filter AND APS_JOB_PHASE_ENDED */
    int numJobsEnded;   /* v0 */
} Aps_QuickPrinterQInfo;

/* Queue access and control functions. */
Aps_Result Aps_PrinterOpenQueue(Aps_PrinterHandle printerHandle,
                                Aps_QueueHandle *queueHandle);
Aps_Result Aps_OpenGlobalQueue(Aps_QueueHandle *queueHandle);
Aps_Result Aps_QueueIssueCommand(Aps_QueueHandle queueHandle,
                                 Aps_OperationID operationCode,
                                 void *reserved);
Aps_Result Aps_QueueIssueQuery(Aps_QueueHandle queueHandle,
                               Aps_OperationID queryCode,
                               void *output);
Aps_Result Aps_QueueGetNumberOfJobs(Aps_QueueHandle queueHandle,
                                    int *numberOfJobs);
Aps_Result Aps_QueueMakeQuickPrinterQInfoArray(Aps_QueueHandle queueHandle,
                                    Aps_QuickPrinterQInfo ***printerQueueInfo,
                                    int *numEntries);


/* 3.9.2 QUEUE MONITORING AND MANAGEMENT - Accessing and Controlling Jobs */

/* Job status flags. */
typedef enum {
    APS_JOB_UNKNOWN   = 0x00000000,
    APS_JOB_SETUP     = 0x00000001,
    APS_JOB_SPOOLING  = 0x00000002,
    APS_JOB_QUEUEING  = 0x00000004,
    APS_JOB_PENDING   = 0x00000008,
    APS_JOB_SENDING   = 0x00000010,
    APS_JOB_PRINTING  = 0x00000020,
    APS_JOB_PAUSED    = 0x00000040,
    APS_JOB_SUSPENDED = 0x00000080,
    APS_JOB_COMPLETED = 0x00000100,
    APS_JOB_STORED    = 0x00000200,
    APS_JOB_FAILED    = 0x00000400,
    APS_JOB_ABORTED   = 0x00000800,
    APS_JOB_CANCELLED = 0x00001000,
    APS_JOB_DISCARDED = 0x00002000,
    APS_JOB_INVALID   = 0x00004000,
    APS_JOB_PHASE_PRODUCTION = (APS_JOB_SETUP | APS_JOB_SPOOLING),
    APS_JOB_PHASE_PENDING    = (APS_JOB_PENDING),
    APS_JOB_PHASE_WORKING    = (APS_JOB_SENDING | APS_JOB_PRINTING),
    APS_JOB_PHASE_ON_HOLD    = (APS_JOB_PAUSED | APS_JOB_SUSPENDED),
    APS_JOB_PHASE_ENDED      = (APS_JOB_STORED | APS_JOB_FAILED |
                                APS_JOB_ABORTED | APS_JOB_CANCELLED |
                                APS_JOB_COMPLETED | APS_JOB_DISCARDED)
} Aps_JobStatus;

/* Job information structure. */
typedef struct Aps_QuickJobInfo_ {
    int               version;         /* Version 0 for now */

    /* job information */
    Aps_JobHandle jobHandle;    /* v0 - handle of job */
    Aps_JobStatus jobStatus;    /* v0 - status of job at last update */
    char      *jobHost;         /* v0 - hostname Job was created on or "" */
    char      *jobName;         /* v0 - name of job or ""
                                 *      may be same as filename */
    char      *jobFilename;     /* v0 - original filename of job or "" */
    int        jobID;           /* v0 - job ID (assigned by transport) */
    size_t     jobSize;         /* v0 - size of job in bytes */
    time_t     jobCreationTime; /* v0 - date and time of job creation */
    char      *jobFormat;       /* v0 - job format string or "" */
    int        jobOrder;        /* v0 - rank of job in the printer queue
                                 *      indicating the order in which the
                                 *      jobs will be printed
                                 *      <0 - unknown or complete
                                 *       0 - being serviced now or complete
                                 *      >0 - next job to be printed is #1,
                                 *           etc...
                                 *    [numbers not necessarily sequential] */
    int        jobPriority;     /* v0 - RESERVED */

    /* printer information */
    Aps_PrinterHandle printerHandle; /* v0 - handle of printer or NULL */
    Aps_PrinterStatus printerStatus; /* v0 - RESERVED */
    char             *printerName;   /* v0 - name of printer or "" */

    /* document information, fields are "" if unknown */
    char *docTitle;    /* v0 - title of document */
    char *docRevision; /* v0 - name / number of document revision */
    char *docComments; /* v0 - document comments (free form), may contain LF's
                        *      but not CR's */
    char *docAuthor;   /* v0 - name of author */
    char *docType;     /* v0 - document type (file extension) */
    char *docCreator;  /* v0 - name of app and version string as "name (vers)"
                        *      (with parenthesis), eg. "Corel Draw (9.01)" */

    /* owner information, fields are 0 or "" if unknown / remote */
    char      *ownerName; /* v0 - name of job owner */
    short int  ownerID;   /* v0 - id of owner */

    /* local routing information */
    char *localHost;   /* v0 - qualified hostname of local machine or ""*/
    char *localFile;   /* v0 - original path of file that was added to
                        *      the local print queue.  Differs from
                        *      jobFilename which tells us the original name
                        *      of the file printed, rather than the one
                        *      we are actually printing from local disk
                        *      space. "" if unknown or none. */

    /* spool routing information */
    char *spoolHost;  /* v0 - qualified hostname of spool machine
                       *      when job is sent, or "" if unknown */
    char *spoolFile;  /* v0 - path of spool file on host or "" if
                       *      synthesized / pipe / unknown */

    /* job attributes */
    Aps_JobAttrHandle jobAttr; /* v0 - handle of job attributes or NULL */
} Aps_QuickJobInfo;

/* Job access and control functions. */
Aps_Result Aps_QueueIterateJobs(Aps_QueueHandle queueHandle,
    Aps_JobHandle *jobHandle);
Aps_Result Aps_JobUpdate(Aps_JobHandle job);
Aps_Result Aps_JobGetStatus(Aps_JobHandle job, Aps_JobStatus *status);
Aps_Result Aps_JobIssueCommand(Aps_JobHandle job,
                               Aps_OperationID operationCode, void *reserved);
Aps_Result Aps_JobMakeQuickJobInfo(Aps_JobHandle job,
                                   Aps_QuickJobInfo **jobInfo);


/* 3.9.3 QUEUE MONITORING AND MANAGEMENT - Componsing Job and Queue Filters */

/* Options for use in filter attachment or checks */
typedef enum Aps_FilterOptions_ {
    APS_FILTER_INVERT    = 0x0001,   /* invert selection */
    APS_FILTER_FALLBACK  = 0x0002,   /* fallback on none */
} Aps_FilterOptions;

/* Max Printer Name length */
#define MAX_PRINTER_NAME 255
/* Modes for use in filter composition */
typedef enum {
    /* Mutually-exclusive modes */
    APS_FILTER_MODE_MASK = 0x000f,   /* mask */
    APS_FILTER_INCLUDE   = 0x0001,   /* OR */
    APS_FILTER_REQUIRE   = 0x0002,   /* AND */
    APS_FILTER_EXCLUDE   = 0x0003,   /* AND NOT */

    /* Option bits */
    APS_FILTER_NOCASE      = 0x0100,   /* strings: ignore case */
    APS_FILTER_WILDCARDS   = 0x0200,   /* strings: use '*' wildcard */
    APS_FILTER_GREATERTHAN = 0x0400,   /* strings / ints: if greater than */
    APS_FILTER_LESSTHAN    = 0x0800,   /* strings / ints: if less than */
    APS_FILTER_ALLBITS     = 0x1000    /* bitfields: require all bits */
} Aps_FilterMode;

/* Filter callback function type */
typedef int (*Aps_FilterFunc)(Aps_Handle object, void *arg);

/* Operations that can be performed on objects that support filtering */
Aps_Result Aps_AttachFilter(Aps_Handle objHandle,
    Aps_FilterHandle filterHandle, Aps_FilterOptions filterOptions);
Aps_Result Aps_DetachFilter(Aps_Handle objHandle,
    Aps_FilterHandle *oldFilterHandle, Aps_FilterOptions *oldFilterOptions);

/* Special general-purpose filter composition functions */
Aps_Result Aps_FilterWithFunction(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, Aps_FilterFunc matchFunction,
    void *userArg);
Aps_Result Aps_FilterAll(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode);
Aps_Result Aps_FilterMerge(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, Aps_FilterHandle srcHandle);
Aps_Result Aps_FilterClear(Aps_FilterHandle *filterHandle);

/* Check if an object matches a filter */
Aps_Result Aps_FilterCheck(Aps_FilterHandle filterHandle,
    Aps_FilterOptions filterOptions, Aps_Handle objHandle,
    int *matchesFilter);

/* Filter creation functions of the form:
 *   Aps_Result Aps_Filter[ObjectClass]By[FieldName](
 *     Aps_FilterHandle *filterHandle,
 *     Aps_FilterMode    filterMode,
 *     [FieldType]       matchField);
 */
Aps_Result Aps_FilterJobByStatus(Aps_FilterHandle, Aps_FilterMode,
    Aps_JobStatus);
Aps_Result Aps_FilterJobByHost(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByName(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByFilename(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByID(Aps_FilterHandle, Aps_FilterMode,
    int);
Aps_Result Aps_FilterJobBySize(Aps_FilterHandle, Aps_FilterMode,
    int);
Aps_Result Aps_FilterJobByCreationTime(Aps_FilterHandle, Aps_FilterMode,
    time_t);
Aps_Result Aps_FilterJobByFormat(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByOrder(Aps_FilterHandle, Aps_FilterMode,
    int);
Aps_Result Aps_FilterJobByPriority(Aps_FilterHandle, Aps_FilterMode,
    int);
Aps_Result Aps_FilterJobByPrinterStatus(Aps_FilterHandle, Aps_FilterMode,
    Aps_PrinterStatus);
Aps_Result Aps_FilterJobByPrinterName(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByOwnerName(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByOwnerID(Aps_FilterHandle, Aps_FilterMode,
    int);
Aps_Result Aps_FilterJobByLocalHost(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobByLocalFile(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobBySpoolHost(Aps_FilterHandle, Aps_FilterMode,
    const char *);
Aps_Result Aps_FilterJobBySpoolFile(Aps_FilterHandle, Aps_FilterMode,
    const char *);

#ifdef __cplusplus
}
#endif
 
#endif /* !APS_H */
