/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apsupdatedb.c
 *
 * Description: Program to update printer model database based on contents
 *              of the Linux printing HOWTO printer compatibility database
 *              and/or the contents of a directory containing PPD files to
 *              be linked to.
 *              
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include "../include/aps.h"

/* Types local to this program. */
typedef enum {
    SECTION_UNKNOWN,
    SECTION_PRINTERS,
    SECTION_DRIVERS
} DatabaseSection;

typedef struct {
    int printersUpdated;
} UserReport;

/* Helper function prototypes. */
static Aps_Result ProcessPPDsInDir(char *dirName, UserReport *userReport);
static Aps_Result ProcessDBFile(char *fileName, UserReport *userReport);
static Aps_Result ProcessPrinterLine(char *line, UserReport *userReport);
static Aps_Result ProcessDriverLine(char *line);

/* Configurable constants. */
#define INITIAL_LINE_LEN 2048
#define MAX_FORMATS 3

/* ---------------------------------------------------------------------------
 * main()
 *
 * Program entry point.
 *
 * Parameters: argc - The number of arguments passed to the program.
 *
 *             argv - An array of pointers to argument strings.
 *
 *     Return: 0 on success, 1 on failure.
 */
int main(int argc, char *argv[])
{
    int arg;
    struct stat fileStat;
    Aps_Result result;
    char resultText[160];
    UserReport userReport = {0};

    /* If no command-line arguments have been supplied, display standard
     * program information text and exit.
     */
    if (argc < 2) {
        printf("Application Print Services Library - "
               "Printer Model Database Update Program\n\n");
        printf("Updates model database (typically /etc/printers/models) from "
               "Linux printer\n");
        printf("compatibility database and/or a directory containing PPD "
               "files to be linked to.\n\n");
        printf("Usage: %s [compatibilty database]... "
               "[directory with PPD files]...\n", argv[0]);
        return 1;
    }

    /* Loop through arguments, processing each file or directory. */
    for (arg = 1; arg < argc; ++arg) {
        /* Look to see whether this is a valid file or directory. */
        if (stat(argv[arg], &fileStat)) continue;

        /* Process either an entire directory, or a single file. */
        if (S_ISDIR(fileStat.st_mode)) {
            result = ProcessPPDsInDir(argv[arg], &userReport);
        } else {
            result = ProcessDBFile(argv[arg], &userReport);
        }

        /* Report result information, as necessary. */
        if (result != APS_SUCCESS) {
            Aps_GetResultText(result, resultText, sizeof(resultText));
            printf("%s\n", resultText);
            return 1;
        }
    }

    /* Report information to user on what has been done. */
    printf("%d printers added or updated.\n", userReport.printersUpdated);
    
    return 0;
}

/* ---------------------------------------------------------------------------
 * ProcessPPDsInDir()
 *
 * Adds or updates links to all PPD files in the specified directory.
 *
 * Parameters: dirName    - Name of the directory to process.
 *
 *             userReport - A structure to receive informaiton to report
 *                          back to the user on what actions have been taken
 *                          by this program.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ProcessPPDsInDir(char *dirName, UserReport *userReport)
{
    DIR *directory;
    struct dirent *dirEntry;
    
    /* Open the directory. */
    directory = opendir(dirName);
    if (directory == NULL) {
        printf("Unable to open directory: %s\n", dirName);
        return APS_GENERIC_FAILURE;
    }
    
    /* Loop through the PPD files in this directory. */
    while ((dirEntry = readdir(directory)) != NULL) {
        /* Skip non-PPD files. */
        if (strlen(dirEntry->d_name) < 5) continue;
        if (strcmp(((char *)dirEntry->d_name) + strlen(dirEntry->d_name) - 4,
            ".ppd") != 0)
            continue;
            
        /* Update the printer database with this PPD file. */
        if (Aps_AddModelFromPPD(dirEntry->d_name, NULL) != APS_SUCCESS) {
            printf("Unable to add link to file: %s\n", dirEntry->d_name);
        } else {
            /* Record that we've updated another printer. */
            userReport->printersUpdated++;
        }
    }
    
    /* Release temporary resources. */
    closedir(directory);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ProcessDBFile()
 *
 * Updates the printer model database with information from the specified
 * Linux Printing HOWTO format printer compatibility database file.
 *
 * Parameters: fileName   - The name of the input database file to process.
 *
 *             userReport - A structure to receive informaiton to report
 *                          back to the user on what actions have been taken
 *                          by this program.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ProcessDBFile(char *fileName, UserReport *userReport)
{
    FILE *srcDB = NULL;
    char *line = NULL;
    int lineSize = INITIAL_LINE_LEN;
    Aps_Result result;
    DatabaseSection section = SECTION_UNKNOWN;

    /* Open the file to be processed. */
    srcDB = fopen(fileName, "r");
    if (srcDB == NULL) {
        result = APS_GENERIC_FAILURE;
        goto cleanup;
    }

    /* Allocate buffer to hold data read from file. */
    line = malloc(lineSize);
    if (line == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Loop until we've reached the end of the database. */
    while (!feof(srcDB)) {
        /* Obtain the next line from the source database. */
        if (fgets(line, lineSize, srcDB) == NULL)
            break;

        /* Remove trailing line feeds from data read from database. */
        if (line[strlen(line) - 1] == '\n')
            line[strlen(line) - 1] = '\0';

        /* Handle comment lines. */
        if (line[0] == '#') {
            /* The only comments we care about are those that delimit the
             * beginning of a new section in the database.
             */
            if (strcmp(line, "#section: printers") == 0) {
                section = SECTION_PRINTERS;
            } else if (strcmp(line, "#section: drivers") == 0) {
                section = SECTION_DRIVERS;
            }

            /* Proceed to the next line in the database. */
            continue;
        }

        /* Handle blank lines. */
        if (strlen(line) == 0) {
            /* Proceed to the next line in the database. */
            continue;
        }

        /* Handle lines of actual data, based on the current section we're
         * examining.
         */
        switch (section) {
            case SECTION_PRINTERS:
                result = ProcessPrinterLine(line, userReport);
                break;

            case SECTION_DRIVERS:
                result = ProcessDriverLine(line);
                break;

            default:
                /* Ignore any data in this section. */
                result = APS_SUCCESS;
                break;
        }

        if (result != APS_SUCCESS && result != APS_IGNORED)
            goto cleanup;
    }

    /* We've now successfully processed the source database. */
    result = APS_SUCCESS;

cleanup:
    /* Release temporary resources. */
    if (srcDB != NULL)
        fclose(srcDB);

    if (line != NULL)
        free(line);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ProcessPrinterLine()
 *
 * Process a line from the printer section of the HOWTO printer compatibility
 * database.
 *
 * Parameters: line       - The line of text to be processed.
 *
 *             userReport - A UserReport structure to be updated based on
 *                          any changes to the APS printer database made from
 *                          this line in the source file.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ProcessPrinterLine(char *line, UserReport *userReport)
{
    char *make, *model, *color, *func, *type_verbose, *gs_driver,
         *lang_verbose, *x, *y, *gs_driver_url, *ascii, *pdq_driver,
         *cups_driver, *recnum, *pnp_mfg, *pnp_mdl, *pnp_cmd, *pnp_des,
         *verified, *notes;
    Aps_ModelHandle modelObject = NULL;
    Aps_Result result;
    char *formats[MAX_FORMATS];
    int numFormats;

    /* Parse the fields out of this line. */
    make = strtok(line, "\t");
    model = strtok(NULL, "\t");
    color = strtok(NULL, "\t");
    func = strtok(NULL, "\t");
    type_verbose = strtok(NULL, "\t");
    gs_driver = strtok(NULL, "\t");
    lang_verbose = strtok(NULL, "\t");
    x = strtok(NULL, "\t");
    y = strtok(NULL, "\t");
    gs_driver_url = strtok(NULL, "\t");
    ascii = strtok(NULL, "\t");
    pdq_driver = strtok(NULL, "\t");
    cups_driver = strtok(NULL, "\t");
    recnum = strtok(NULL, "\t");
    pnp_mfg = strtok(NULL, "\t");
    pnp_mdl = strtok(NULL, "\t");
    pnp_cmd = strtok(NULL, "\t");
    pnp_des = strtok(NULL, "\t");
    verified = strtok(NULL, "\t");
    notes = strtok(NULL, "\t");

    /* Skip printers that we cannot support for one reason or another. */

    /* If there is no GhostScript driver for this printer, ignore it for now.
     */
    if (strcmp(gs_driver, "N/A") == 0) {
        result = APS_IGNORED;
        goto cleanup;
    }

    /* Look for an existing entry in the printer model database for this
     * printer.
     */
    result = Aps_GetModel(make, model, &modelObject);

    if (result == APS_NOT_FOUND) {
        /* No record was found for this model, so create a new one. */
        result = Aps_AddModel(make, model, &modelObject);
    }

    if (result != APS_SUCCESS)
        goto cleanup;

    /* Set GhostScript-specific information for this printer, if applicable. */
    if (strcmp(gs_driver, "N/A") != 0) {
        /* Set the GhostScript driver name for this device. */
        result = Aps_SetPropertyString(modelObject, "gsdevice", gs_driver);
        if (result != APS_SUCCESS)
            goto cleanup;
    }

    /* Set the data types directly suppported by this printer. */
    numFormats = 0;
    if (strcmp(gs_driver, "N/A") != 0)
        formats[numFormats++] = APS_FORMAT_NATIVE;
    if (strcmp(ascii, "T") == 0)
        formats[numFormats++] = APS_FORMAT_TEXT;
    result = Aps_SetPropertyStrArray(modelObject, "formats",
                                     (const char **)formats, numFormats);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Attempt to commit our changes to this model object into the printer
     * model database.
     */
    result = Aps_ModelCommitToDatabase(modelObject);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If we reach this point, we've successfully processed this line. */
    userReport->printersUpdated ++;

    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (modelObject != NULL)
        Aps_ReleaseHandle(modelObject);

    return result;
}

/* ---------------------------------------------------------------------------
 * ProcessDriverLine()
 *
 * Process a line from the driver section of the HOWTO printer compatibility
 * database.
 *
 * Parameters: line - The line of text to be processed.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ProcessDriverLine(char *line)
{
    char *make, *model, *resolution_x, *resolution_y, *color, *cmd_type_short,
         *cmd, *comment;
    Aps_ModelHandle modelObject = NULL;
    Aps_Result result;

    /* Parse the fields out of this line. */
    make = strtok(line, "\t");
    model = strtok(NULL, "\t");
    resolution_x = strtok(NULL, "\t");
    resolution_y = strtok(NULL, "\t");
    color = strtok(NULL, "\t");
    cmd_type_short = strtok(NULL, "\t");
    cmd = strtok(NULL, "\t");
    comment = strtok(NULL, "\t");

    /* Look for an existing entry in the printer model database for this
     * printer.
     */
    result = Aps_GetModel(make, model, &modelObject);
    if (result == APS_NOT_FOUND)
        result = APS_IGNORED;
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If we reach this point, we've successfully processed this line. */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (modelObject != NULL)
        Aps_ReleaseHandle(modelObject);

    return result;
}
