/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: attrgeneric.c
 *
 * Description: Generic job attribute querying and manipulation APIs.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "jobattributes.h"
#include "attrprovider.h"

DEBUG_CHANNEL_DEFAULT(attr)

/* ---------------------------------------------------------------------------
 * Aps_AttrGetList()
 *
 * Obtains a list of individual attributes available in a particular job
 * attributes object. This can be a list of all attributes, or only the
 * subset of attributes specified by the group parameter.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             group         - A string containing the name of the attributes
 *                             group to be retrieved, or APS_GROUP_??? to
 *                             retrieve a predefined subset of attributes.
 *
 *             numAttributes - A pointer to an integer to receive the number
 *                             of attributes that were found.
 *
 *             attributeIDs  - A char *** that receives a buffer containing
 *                             an array of pointers to strings composing the
 *                             attribute IDs. When finished with this buffer,
 *                             you must call Aps_ReleaseBuffer()
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrGetList(Aps_JobAttrHandle jobAttributes,
                           const char *group,
                           int *numAttributes,
                           char ***attributeIDs)
{
    JobAttributes *attributes;
    TrackArray_PtrChar attrIDArray = NULL;
    Aps_Result result = APS_SUCCESS;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (group == NULL || numAttributes == NULL || attributeIDs == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *numAttributes = 0;
    *attributeIDs = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Create an array to hold the identifier strings for all */
    /* available attributes.                                  */
    attrIDArray = TrackArrayNew_PtrChar(NULL, 0);
    if (! attrIDArray) return APS_OUT_OF_MEMORY;

    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Obtain a list of attribute IDs from this provider. */
        result = provider->vtbl->AttrGetList(provider, group, & attrIDArray);

        /* If the attribute provider encountered an error, then stop, */
        /* cleanup, and propogate the failure back to the client.     */
        if (result != APS_SUCCESS) break;
    }

    if (result != APS_SUCCESS) {
        TrackArrayDelete_PtrChar(attrIDArray);
        return result;
    }
    *attributeIDs = attrIDArray;
    *numAttributes = TrackArrayGetSize_PtrChar(attrIDArray);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetSubGroups()
 *
 * Obtains a list of attribute groups belonging to another attribute group.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             group         - A string containing the name of the attributes
 *                             group to be searched for subgroups,
 *                             APS_GROUP_ROOT to obtain groups at the root
 *                             level, or APS_GROUP_ROOT_SETTINGS to obtain
 *                             root level groups or user-controllable
 *                             settings.
 *
 *             numSubGroups  - A pointer to an integer to receive the number
 *                             of groups that were found.
 *
 *             subGroupNames - A char *** that receives a buffer containing
 *                             an array of pointers to strings composing the
 *                             list of group names. When finished with this
 *                             buffer, you must call Aps_ReleaseBuffer().
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result Aps_AttrGetSubGroups(Aps_JobAttrHandle jobAttributes,
                                const char *group,
                                int *numSubGroups,
                                char ***subGroupNames)
{
    JobAttributes *attributes;
    TrackArray_PtrChar groupArray = NULL;
    Aps_Result result = APS_SUCCESS;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (group == NULL || numSubGroups == NULL || subGroupNames == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *numSubGroups = 0;
    *subGroupNames = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Create an array to hold the identifier strings for all */
    /* available attributes.                                  */
    groupArray = TrackArrayNew_PtrChar(NULL, 0);
    if (! groupArray) return APS_OUT_OF_MEMORY;

    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Obtain a list of attribute IDs from this provider. */
        result = provider->vtbl->AttrGetSubGroups(provider, group,
                                                  & groupArray);

        /* If the attribute provider encountered an error, then stop, */
        /* cleanup, and propogate the failure back to the client.     */
        if (result != APS_SUCCESS) break;
    }

    if (result != APS_SUCCESS) {
        TrackArrayDelete_PtrChar(groupArray);
        return result;
    }
    *subGroupNames = groupArray;
    *numSubGroups = TrackArrayGetSize_PtrChar(groupArray);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetTranslatedName()
 *
 * Obtains the local translated name for the specified attribute. This is
 * the text that should be displayed to the user when presenting a list of
 * settings that can be altered.
 *
 * Parameters: jobAttributes  - A handle to a job attributes object.
 *
 *             atributeID     - A string identifying the attribute being
 *                              queried.
 *
 *             translatedName - A pointer to a string pointer that receives
 *                              a new buffer with the translated name for
 *                              this attribute. When you are finished with
 *                              this buffer, call Aps_ReleaseBuffer().
 *                              Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetTranslatedName(Aps_JobAttrHandle jobAttributes,
                                     const char *attributeID,
                                     char **translatedName)
{
    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || translatedName == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *translatedName = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetTranslatedName(provider,
                                                       attributeID,
                                                       translatedName);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetMainData()
 *
 * Obtains the main (read-only) data provided by this attribute. The meaning
 * and contents of the main data string is dependant on the particular
 * attribute being queried.
 *
 * Parameters: jobAttributes - A handle to the job attributes object.
 *
 *             attributeID   - A string uniquely identifying the attribute to
 *                             retrieve.
 *
 *             mainData      - A pointer to a char* to receive a new buffer
 *                             containing the main data stored in this
 *                             attribute.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetMainData(Aps_JobAttrHandle jobAttributes,
                               const char *attributeID,
                               char **mainData)
{
    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || mainData == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *mainData = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetMainData(provider,
                                                 attributeID,
                                                 mainData);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetTranslatedData()
 *
 * Obtains the translated data string for a particular attribute.
 *
 * Parameters: jobAttributes  - A handle to a job attributes object.
 *
 *             attributeID    - A string uniquely identifying the attribute.
 *
 *             translatedData - A string pointer to receive a new buffer with
 *                              the translated data. When finished with this
 *                              buffer, you must call Aps_ReleaseBuffer().
 *                              Free with Aps_ReleaseBuffer
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetTranslatedData(Aps_JobAttrHandle jobAttributes,
                                     const char *attributeID,
                                     char **translatedData)
{
    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || translatedData == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *translatedData = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetTranslatedData(provider,
                                                       attributeID,
                                                       translatedData);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetType()
 *
 * Obtains the type of a particular attribute. Attributes may be read-only
 * informational attributes, or may store one of a variety of types of user-
 * modifyable settings.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             attributeID   - A string uniquely identifying an attribute.
 *
 *             attributeType - A pointer to an Aps_AttrType to receive the
 *                             type of the attribute.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetType(Aps_JobAttrHandle jobAttributes,
                           const char *attributeID,
                           Aps_AttrType * attributeType)
{
    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || attributeType == NULL)
        return APS_INVALID_PARAM;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetType(provider,
                                             attributeID,
                                             attributeType);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetRange()
 *
 * Obtains the range of possible values for attributes with interger or
 * floating point user settings.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             attributeID   - A string uniquely identifying the attribute
 *                             being queried.
 *
 *             minSetting    - A double to receive the minimum value.
 *
 *             maxSetting    - A double to receive the maximum value.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetRange(Aps_JobAttrHandle jobAttributes,
                            const char *attributeID,
                            double *minSetting,
                            double *maxSetting)
{
    Aps_Result result;
    JobAttributes *attributes;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || minSetting == NULL || maxSetting == NULL)
        return APS_INVALID_PARAM;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetRange(provider,
                                              attributeID,
                                              minSetting,
                                              maxSetting);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetOptions()
 *
 * Obtains a list of options (e.g. possible values for the user-setting) of
 * the specified attribute.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             attributeID   - A string uniquely identifying the attribute
 *                             being queried.
 *
 *             numOptions    - The number of available options.
 *
 *             options       - An Aps_AttrOption pointer to receive a pointer
 *                             to a new array. When finished with this array,
 *                             you must pass it to Aps_ReleaseBuffer().
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetOptions(Aps_JobAttrHandle jobAttributes,
                              const char *attributeID,
                              int *numOptions,
                              Aps_AttrOption *** options)
{
    Aps_Result result;
    JobAttributes *attributes;
    TrackArrayIndirect_AttrOption optionArray;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || numOptions == NULL || options == NULL)
        return APS_INVALID_PARAM;

    /* Intialize the caller's variable to reasonable default values. */
    *numOptions = 0;
    *options = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Create an array to hold the options for this attribute. */
    optionArray = TrackArrayIndirectNew_AttrOption(NULL, 0);
    if (! optionArray) return APS_OUT_OF_MEMORY;

    /* Look for an attribute provider that recognizes this attribute ID. */
    result = APS_NOT_FOUND;
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetOptions(provider,
                                                attributeID,
                                                & optionArray);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result == APS_SUCCESS) {
            *options = optionArray;
            *numOptions = TrackArrayIndirectGetSize_AttrOption(optionArray);
            return APS_SUCCESS;
        } else if (result != APS_NOT_FOUND) break;
    }

    /* If we get to this point, then either no attribute providers know
     * about the attribute (result == APS_NOT_FOUND) or something
     * more catastrophic has occured. */
    TrackArrayIndirectDelete_AttrOption(optionArray);
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrGetSetting()
 *
 * Obtains the current value of the user-setting for a particular attribute.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             attributeID   - The ID of attribute to retrieve.
 *
 *             setting       - A pointer to a char * to receive a new buffer
 *                             containing the current user-setting for this
 *                             attribute. When finished with this buffer,
 *                             deallocate it by calling Aps_ReleaseBuffer().
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrGetSetting(Aps_JobAttrHandle jobAttributes,
                              const char *attributeID,
                              char **setting)
{

    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || setting == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *setting = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrGetSetting(provider,
                                                attributeID,
                                                setting);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrSetSetting()
 *
 * Changes the user-setting for a particular attribute.
 *
 * Parameters: jobAttributes - A handle to a job attributes object.
 *
 *             attributeID   - A string containing the ID of the attribute
 *                             to be changed.
 *
 *             setting       - The new value for the user-setting.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AttrSetSetting(Aps_JobAttrHandle jobAttributes,
                              const char *attributeID,
                              const char *setting)
{
    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || setting == NULL)
        return APS_INVALID_PARAM;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Look for an attribute provider that recognizes this attribute ID. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this query on to this attribute provider. */
        result = provider->vtbl->AttrSetSetting(provider,
                                                attributeID,
                                                setting);

        /* If this attribute provider didn't recognize this attribute, then */
        /* proceed to the next attribute provider. Otherwise, we pass the   */
        /* result from this attribute provider back to the client.          */
        if (result != APS_NOT_FOUND)
            return result;
    }

    /* If we get to this point, then no attribute providers know this */
    /* attribute ID.                                                  */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrCheckConstraints()
 *
 * Tests whether a proposed change to a particular attribute's user-setting
 * would violate any of the user-setting constraints.
 *
 * Parameters: jobAttributes        - A handle to a job attributes object.
 *
 *             attributeID          - A string uniquely identifying the
 *                                    attribute whose user setting is
 *                                    being proposed for modification.
 *
 *             setting              - A string containing the new user
 *                                    setting for this attribute.
 *
 *             conflictingAttribute - If a conflict is found, this
 *                                    will be given a string buffer
 *                                    containing the ID of the first
 *                                    attribute found to conflict with
 *                                    the proposed change. Use
 *                                    Aps_ReleaseBuffer() to deallocate
 *                                    this buffer when finished with it.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *             conflictingSetting   - If a conflict is found, this will
 *                                    contain conflicting user-setting.
 *                                    Use Aps_ReleaseBuffer() to
 *                                    deallocate this buffer.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *
 *     Return: APS_SUCCESS if no constraints would be violated by this
 *             change, APS_VIOLATES_CONSTRAINTS if there is is a conflict, or
 *             another Aps_Result code on failure.
 */
Aps_Result Aps_AttrCheckConstraints(Aps_JobAttrHandle jobAttributes,
                                    const char *attributeID,
                                    const char *setting,
                                    char **conflictingAttribute,
                                    char **conflictingSetting)
{
    JobAttributes *attributes;
    Aps_Result result;
    int i;

    /* Safeguard against obviously incorrect parameters. */
    if (attributeID == NULL || setting == NULL || conflictingAttribute == NULL
        || conflictingSetting == NULL)
        return APS_INVALID_PARAM;

    /* Set the caller's variables to empty, in case we fail. */
    *conflictingAttribute = NULL;
    *conflictingSetting = NULL;

    /* Obtain the JobAttributes structure from the handle. */
    attributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (attributes == NULL)
        return APS_INVALID_HANDLE;

    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Check whether this setting would violate any constraints for */
        /* this attribute provider.                                     */
        result = provider->vtbl->AttrCheckConstraints(provider,
                                                      attributeID,
                                                      setting,
                                                      conflictingAttribute,
                                                      conflictingSetting);

        /* If any attribute provider finds a conflict, or fails for any   */
        /* reason, then return that result directly to the client without */
        /* querying any other attribute providers.                        */
        if (result != APS_SUCCESS)
            return result;
    }

    return APS_SUCCESS;
}
