/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: attribute.c
 *
 * Description: Implementation of an attribute, which is a type of
 *              APS object property that includes information such as
 *              the range of possible values for the setting, translated
 *              text to be displayed to the user, etc. Attribute
 *              properties are the basis of a JobAttributes object,
 *              which stores properties pertaining to a current or
 *              future print jobs.
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "attribute.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(attr)

/* ---------------------------------------------------------------------------
 * AttrCreate()
 *
 * Creates a new, empty, Attribute instance.
 *
 * Parameters: attribute - The address of an Attribute pointer to receive
 *                         the new Attribute instance.
 *                         Free using AttrDelete() / free()
 *
 *             provider  - Pointer to the attribute provider to associate
 *                         with the new attribute.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocate failure.
 */
Aps_Result AttrCreate(Attribute **attribute, AttrProvider * provider)
{
    ASSERT(attribute != NULL);

    /* Allocate memory for the Attribute structure. */
    *attribute = calloc(1, sizeof(Attribute));
    if (*attribute == NULL)
        return APS_OUT_OF_MEMORY;

    /* Associate the new attribute with the specified attribute provider. */
    ASSERT(provider != NULL);
    (*attribute)->provider = provider;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrDelete()
 *
 * Deletes an attribute, releasing all associated memory.
 *
 * Parameters: attribute - A pointer to the Attribute to deallocate.
 *
 *     Return: None.
 */
void AttrDelete(Attribute *attribute)
{
    ASSERT(attribute != NULL);

    /* Delete the simple strings that can be stored as part of an
     * attribute.
     */
    if (attribute->translatedName != NULL)
        free(attribute->translatedName);
    if (attribute->mainData != NULL)
        free(attribute->mainData);
    if (attribute->translatedData != NULL)
        free(attribute->translatedData);
    if (attribute->setting != NULL)
        free(attribute->setting);

    /* Delete the options list, if present. */
    AttrRemoveAllOptions(attribute);

    /* Finally, delete the Attribute structure. */
    free(attribute);
}

/* ---------------------------------------------------------------------------
 * AttrCopyContents()
 *
 * Copies the contents from one existing attribute to another existing
 * attribute. Does not disturb the pointer to the associated attribute
 * provider.
 *
 * Parameters: dest - The destination attribute object to copy into.
 *
 *             src  - The source attribute object to copy from.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrCopyContents(Attribute *dest, const Attribute *src)
{
    Aps_Result result;
    int i;

    /* Copy the translated name, if needed. */
    result = StrDupWithNull(&dest->translatedName, src->translatedName);
    if (result != APS_SUCCESS)
        return result;

    /* Copy the main data, if needed. */
    result = StrDupWithNull(&dest->mainData, src->mainData);
    if (result != APS_SUCCESS)
        return result;

    /* Copy the translated data, if needed. */
    result = StrDupWithNull(&dest->translatedData, src->translatedData);
    if (result != APS_SUCCESS)
        return result;

    /* Copy attribute type field. */
    dest->type = src->type;
    
    /* Copy setting limits. */
    dest->minSetting = src->minSetting;
    dest->maxSetting = src->maxSetting;

    /* Copy the options list, if any. */
    AttrRemoveAllOptions(dest);
    for (i = 0; i < src->numOptions; ++i) {
        result = AttrAddOption(dest, src->options[i].optionID,
                                     src->options[i].translatedName,
                                     src->options[i].value);
        if (result != APS_SUCCESS)
            return result;
    }
    
    /* Copy the setting itself. */
    result = StrDupWithNull(&dest->setting, src->setting);
    if (result != APS_SUCCESS)
        return result;
        
    /* We've succesfully copies the contents of the source attribute. */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrRemoveAllOptions()
 *
 * Removes all options from the specified attribute.
 *
 * Parameters: attribute - A pointer to the Attribute to remove all options
 *                         from.
 *
 *     Return: None.
 */
void AttrRemoveAllOptions(Attribute *attribute)
{
    int i;

    if (attribute->options == NULL)
        return; /* There is nothing for us to do. */

    /* Delete all entries in the options array. */
    for (i = 0; i < attribute->numOptions; ++i) {
        if (attribute->options[i].optionID != NULL)
            free(attribute->options[i].optionID);
        if (attribute->options[i].translatedName != NULL)
            free(attribute->options[i].translatedName);
        if (attribute->options[i].value != NULL)
            free(attribute->options[i].value);
    }

    /* Delete the options array itself. */
    free(attribute->options);
    attribute->options = NULL;

    attribute->numOptions = 0;
}

/* ---------------------------------------------------------------------------
 * AttrAddOption()
 *
 * Adds another object to the list of available options for the spcified
 * attribute.
 *
 * Parameters: attribute      - A pointer to the Attribute to remove all
 *                              options from.
 *
 *             optionID       - A string uniquely identifying this option
 *                              (required).
 *
 *             translatedName - The translated name to display for this
 *                              option (optional).
 *
 *             value          - The associated value string (optional).
 *
 *     Return: None.
 */
Aps_Result AttrAddOption(Attribute *attribute, const char *optionID,
                         const char *translatedName, const char *value)
{
    Aps_AttrOption *newOptionArray;
    Aps_AttrOption *newOption;

    /* Make space for another option in the option array. */
    newOptionArray = (Aps_AttrOption *)realloc(attribute->options,
        (attribute->numOptions + 1) * sizeof(Aps_AttrOption));
    if (newOptionArray == NULL)
        return APS_OUT_OF_MEMORY;
    attribute->options = newOptionArray;

    /* Obtain a pointer to the new option being added to the array. */
    newOption = attribute->options + attribute->numOptions;

    /* Copy the option ID string. */
    ASSERT(optionID != NULL);
    newOption->optionID = malloc(strlen(optionID) + 1);
    if (newOption->optionID == NULL)
        return APS_OUT_OF_MEMORY;
    strcpy(newOption->optionID, optionID);

    /* Copy the translated name, if needed. */
    if (translatedName == NULL) {
        newOption->translatedName = NULL;
    } else {
        newOption->translatedName = malloc(strlen(translatedName) + 1);
        if (newOption->translatedName == NULL) {
            free(newOption->optionID);
            return APS_OUT_OF_MEMORY;
        }
        strcpy(newOption->translatedName, translatedName);
    }

    /* Copy the value string for this option, if needed. */
    if (value == NULL) {
        newOption->value = NULL;
    } else {
        newOption->value = malloc(strlen(value) + 1);
        if (newOption->value == NULL) {
            free(newOption->optionID);
            if (newOption->translatedName != NULL)
                free(newOption->translatedName);
            return APS_OUT_OF_MEMORY;
        }
        strcpy(newOption->value, value);
    }

    /* If we reach this point, then we've successfully added the new
     * option to this attribute, so updated the option count and
     * let the caller know that all is well.
     */
    attribute->numOptions++;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrSetSetting()
 *
 * Changes the setting assocaited with the specified option.
 *
 * Parameters: attribute - A pointer to the Attribute to modify.
 *
 *             setting   - The new value for this attribute's setting.
 *
 *     Return: None.
 */
Aps_Result AttrSetSetting(Attribute *attribute, const char *setting)
{
    char *newSetting;

    /* Attempt to allocate space in which to store the new value for this
     * setting.
     */
    newSetting = malloc(strlen(setting) + 1);
    if (newSetting == NULL)
        return APS_OUT_OF_MEMORY;

    /* Copy the new value for this setting into the newly allocated block
     * of memory.
     */
    strcpy(newSetting, setting);

    /* Deallocate the old setting string, if there is any. */
    if (attribute->setting != NULL) {
        free(attribute->setting);
    }

    /* Store the new setting string in this attribute. */
    attribute->setting = newSetting;

    return APS_SUCCESS;
}


