/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: attrprovider.c
 *
 * Description: Implementation of generic attribute provider.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "jobattributes.h"
#include "attrprovider.h"
#include "object.h"
#include "metaconfig.h"

DEBUG_CHANNEL_DEFAULT(attrprov)

/* ---------------------------------------------------------------------------
 * AttrProvInitialize()
 *
 * Called to Initialize a new generic attribute provider.
 *
 * Parameters: this             - A pointer to the base AttrProvider instance
 *                                for this Attribute Provider.
 *                                Must have been allocated using stdlib.
 *
 *             associatedObject - The APS object that this attribute provider
 *                                is providing attributes to.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvInitialize(AttrProvider * this,
                              Aps_Handle associatedObject)
{
    /* Store a pointer to the job attributes object that this attribute
     * provider is associated with.
     */
    this->associatedObject = associatedObject;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvDelete()
 *
 * Deletes an attribute provider, calling its destructor and deleting the
 * object itself.
 *
 * Parameters: this - A pointer to the attribute provider to be deleted.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
void AttrProvDelete(AttrProvider * this)
{
    /* Call the attribute provider's destructor to release any of its
     * members.
     */
    this->vtbl->Destructor(this);
      
    /* Release the memory used by the AttrProvider itself. */
    free(this);
}
 
/* ---------------------------------------------------------------------------
 * AttrProvDestructor()
 *
 * Called to delete an Attribute Provider instance.
 *
 * Parameters: this - A pointer the the base AttrProvider instance for
 *                    this Attribute Provider.
 *
 *     Return: void
 */
void AttrProvDestructor(AttrProvider * this)
{
    Aps_Result result;
    ApsObject *associatedObject;
    TrackArray_PtrChar propertyNames = NULL;
    int numProperties;
    int i;
    char *property;
    Attribute *attribute;

    ASSERT(this != NULL);

    /* Obtain a pointer to the APS object containing the attributes
     * implemented by this attribute provider.
     */
    associatedObject = ObjectGetPtrFromHandle(this->associatedObject);
    ASSERT(associatedObject != NULL);

    /* Obtain a list of all properties stored in the associated object. */
    propertyNames = TrackArrayNew_PtrChar(NULL, 0);
    if (! propertyNames) { result = APS_OUT_OF_MEMORY; goto cleanup; }
    result = ObjectGetProperties(associatedObject, & propertyNames);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Loop through all properties of the associated object, deleting any 
     * attributes owned by this provider.
     */
    numProperties = TrackArrayGetSize_PtrChar(propertyNames);
    for (i = 0; i < numProperties; ++i) {
        property = propertyNames[i];

        /* We are only interested in object properties that are full-fledged
         * attributes.
         */
        if (ObjectGetPropertyType(associatedObject, property)
            != PROPERTY_ATTRIBUTE)
            continue;

        /* Obtain an Attribute instance for this property. */
        result = ObjectGetPropertyAttribute(associatedObject,
                                            property,
                                            &attribute);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* We are only interested in those attributes provided by this
         * attribute provider.
         */
        if (attribute->provider != this)
            continue;
            
        /* Remove this property. */
        ObjectRemoveProperty(associatedObject, property);
    }

cleanup:
    /* Release any temporary storage allocated by this function. */
    if (propertyNames != NULL) {
        TrackArrayDelete_PtrChar(propertyNames);
    }

    /* TBD: Currently this function doesn't propogate any failures back to
     *      its caller.
     */
}

/* ---------------------------------------------------------------------------
 * AttrProvCreateCopy()
 *
 * Creates a deep copy of an existing attribute provider that is semantically
 * identical to the original.
 *
 * Parameters: this             - The existing attribute provider to copy.
 *
 *             newProvider      - A pointer to receive the new attribute
 *                                provider.
 *
 *             associatedObject - The associated job attributes or other
 *                                APS object for which this provider is
 *                                implementing 0 or more attributes.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvCreateCopy(AttrProvider * this,
                              AttrProvider ** newProvider,
                              Aps_Handle associatedObject)
{
    Aps_Result result;
    TrackArray_PtrChar propertyIDs = NULL;
    int i;
    int numProperties;
    char *property;
    Attribute *originalAttr;
    Attribute *newAttr;
    
    /* This default implementation only handles the standard AttrProvider
     * structure. If a subclass defines its own structure, it must look after
     * copying that data itself.
     */

    /* Allocate memory for the new attribute provider. */
    *newProvider = (AttrProvider *)calloc(1, sizeof(AttrProvider));
    if (*newProvider == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }
    
    /* Give the new attribute provider the same virtual function table as the
     * original.
     */
    (*newProvider)->vtbl = this->vtbl;
    
    /* Perform the standard initialization of the new attribute provider,
     * associating it with the appropriate APS object.
     */
    result = AttrProvInitialize(*newProvider, associatedObject);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Copy any attributes from the source attribute provider's associated
     * object to the new attribute provider's associated object.
     */
     
    /* Obtain a list of all properties from the original provider's associated
     * object.
     */
    propertyIDs = TrackArrayNew_PtrChar(NULL, 0);
    if (! propertyIDs) { result = APS_OUT_OF_MEMORY; goto cleanup; }
    result = ObjectGetProperties(this->associatedObject, & propertyIDs);
    if (result != APS_SUCCESS)
        goto cleanup;
        
    /* Loop through all property names, looking for those associated with the
     * original attribute provider.
     */
    numProperties = TrackArrayGetSize_PtrChar(propertyIDs);
    for (i = 0; i < numProperties; ++i) {
        property = propertyIDs[i];

        /* We are only interested in object properties that are full-fledged
         * attributes.
         */
        if (ObjectGetPropertyType(this->associatedObject, property)
            != PROPERTY_ATTRIBUTE)
            continue;

        /* Obtain an Attribute instance for this property. */
        result = ObjectGetPropertyAttribute(this->associatedObject,
                                            property,
                                            &originalAttr);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* We are only interested in those attributes provided by this
         * attribute provider.
         */
        if (originalAttr->provider != this)
            continue;
            
        /* Add a new attribute to the new provider's associated object. */
        result = ObjectGetOrAddPropertyAttribute(associatedObject, property,
                                                 &newAttr, *newProvider);
        if (result != APS_SUCCESS)
            goto cleanup;
            
        /* Copy information into the new attribute. */
        result = AttrCopyContents(newAttr, originalAttr);
        if (result != APS_SUCCESS)
            goto cleanup;
    }

    /* We've now successfully cloned ourself. */
    result = APS_SUCCESS;

cleanup:
    /* On failure, release the new attribute provider if it was ever
     * allocated.
     */
    if (result != APS_SUCCESS && *newProvider != NULL) {
        AttrProvDelete(*newProvider);
    }
    
    /* Release any temporary resource allocated by this function. */
    if (propertyIDs != NULL) {
        TrackArrayDelete_PtrChar(propertyIDs);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetList()
 *
 * Obtains a list of individual attributes available from this attribute
 * provider. This can be a list of all attributes, or only the subset of
 * attributes specified by the group parameter.
 *
 * Parameters: this          - A pointer the the base AttrProvider instance
 *                             for this Attribute Provider.
 *
 *             group         - A string containing the name of the attributes
 *                             group to be retrieved, or APS_GROUP_??? to
 *                             retrieve a predefined subset of attributes.
 *
 *             attributeIDs  - Pointer to an array that receives a list of
 *                             attribute IDs.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result AttrProvGetList(AttrProvider * this,
                           const char *group,
                           TrackArray_PtrChar *attributeIDs)
{
    Aps_Result result;

    /* Query for any Attribute properties in the associated job attributes
     * object.
     */
    result = ObjectGetProperties(ObjectGetPtrFromHandle(this->associatedObject),
                                 attributeIDs);
    if (result != APS_SUCCESS)
        return result;

    /* TBD: Filter out properties not provided by this attribute provider,
     *      non-Attribute properties, and properties not in the specified
     *      group.
     */

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetSubGroups()
 *
 * Obtains a list of attribute groups belonging to another attribute group.
 *
 * Parameters: this          - A pointer the the base AttrProvider instance
 *                             for this Attribute Provider.
 *
 *             group         - A string containing the name of the attributes
 *                             group to be searched for subgroups,
 *                             APS_GROUP_ROOT to obtain groups at the root
 *                             level, or APS_GROUP_ROOT_SETTINGS to obtain
 *                             root level groups or user-controllable
 *                             settings.
 *
 *             subGroupNames - Pointer to an array to receive a list of group
 *                             names.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
Aps_Result AttrProvGetSubGroups(AttrProvider * this,
                                const char *group,
                                TrackArray_PtrChar *subGroupNames)
{
    /* The generic attribute provider doesn't group items into groups. */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetTranslatedName()
 *
 * Obtains the local translated name for the specified attribute. This is
 * the text that should be displayed to the user when presenting a list of
 * settings that can be altered.
 *
 * Parameters: this           - A pointer the the base AttrProvider instance
 *                              for this Attribute Provider.
 *
 *             atributeID     - A string identifying the attribute being
 *                              queried.
 *
 *             translatedName - A pointer to a string pointer that receives
 *                              a new buffer with the translated name for
 *                              this attribute. When you are finished with
 *                              this buffer, call Aps_ReleaseBuffer().
 *                              Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetTranslatedName(AttrProvider * this,
                                     const char *attributeID,
                                     char **translatedName)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Check whether this attribute has a translated name. */
    if (attribute->translatedName == NULL)
        return APS_NOT_FOUND;

    /* Attempt to allocate a buffer in which to pass the translated name
     * back to the caller.
     */
    *translatedName = TrackMemDupString(NULL, attribute->translatedName, 0);
    if (*translatedName == NULL)
        return APS_OUT_OF_MEMORY;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetMainData()
 *
 * Obtains the main (read-only) data provided by this attribute. The meaning
 * and contents of the main data string is dependant on the particular
 * attribute being queried.
 *
 * Parameters: this        - A pointer the the base AttrProvider instance
 *                           for this Attribute Provider.
 *
 *             attributeID - A string uniquely identifying the attribute to
 *                           retrieve.
 *
 *             mainData    - A pointer to a char* to receive a new buffer
 *                           containing the main data stored in this
 *                           attribute.
 *                           Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetMainData(AttrProvider * this,
                               const char *attributeID,
                               char **mainData)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Check whether this attribute has a main data. */
    if (attribute->mainData == NULL)
        return APS_NOT_FOUND;

    /* Attempt to allocate a buffer in which to pass the main data
     * back to the caller.
     */
    *mainData = TrackMemDupString(NULL, attribute->mainData, 0);
    if (*mainData == NULL)
        return APS_OUT_OF_MEMORY;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetTranslatedData()
 *
 * Obtains the translated data string for a particular attribute.
 *
 * Parameters: this           - A pointer the the base AttrProvider instance
 *                              for this Attribute Provider.
 *
 *             attributeID    - A string uniquely identifying the attribute.
 *
 *             translatedData - A string pointer to receive a new buffer with
 *                              the translated data. When finished with this
 *                              buffer, you must call Aps_ReleaseBuffer().
 *                              Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetTranslatedData(AttrProvider * this,
                                     const char *attributeID,
                                     char **translatedData)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Check whether this attribute has a translated data. */
    if (attribute->translatedData == NULL)
        return APS_NOT_FOUND;

    /* Attempt to allocate a buffer in which to pass the translated data
     * back to the caller.
     */
    *translatedData = TrackMemDupString(NULL, attribute->translatedData, 0);
    if (*translatedData == NULL)
        return APS_OUT_OF_MEMORY;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetType()
 *
 * Obtains the type of a particular attribute. Attributes may be read-only
 * informational attributes, or may store one of a variety of types of user-
 * modifyable settings.
 *
 * Parameters: this          - A pointer the the base AttrProvider instance
 *                             for this Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying an attribute.
 *
 *             attributeType - A pointer to an Aps_AttrType to receive the
 *                             type of the attribute.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetType(AttrProvider * this,
                           const char *attributeID,
                           Aps_AttrType * attributeType)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Provide the caller with the type of this attribute. */
    *attributeType = attribute->type;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetRange()
 *
 * Obtains the range of possible values for attributes with interger or
 * floating point user settings.
 *
 * Parameters: this          - A pointer the the base AttrProvider instance
 *                             for this Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying the attribute
 *                             being queried.
 *
 *             minSetting    - A double to receive the minimum value.
 *
 *             maxSetting    - A double to receive the maximum value.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetRange(AttrProvider * this,
                            const char *attributeID,
                            double *minSetting,
                            double *maxSetting)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Ensure that this is the appropriate type of attribute to make this
     * query.
     */
    if (attribute->type != APS_ATTR_SETTING_INT
        && attribute->type != APS_ATTR_SETTING_FLOAT) {
        return APS_WRONG_TYPE;
    }

    /* Provide the caller with the requested range information. */
    if (minSetting != NULL) {
        *minSetting = attribute->minSetting;
    }

    if (maxSetting != NULL) {
        *maxSetting = attribute->maxSetting;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetOptions()
 *
 * Obtains a list of options (e.g. possible values for the user-setting) of
 * the specified attribute.
 *
 * Parameters: this          - A pointer the the base AttrProvider instance
 *                             for this Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying the attribute
 *                             being queried.
 *
 *             optionArray   - Pointer to an array of Aps_AttrOption instances
 *                             listing the available options.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetOptions(AttrProvider * this,
                              const char *attributeID,
                              TrackArrayIndirect_AttrOption *optionArray)
{
    Aps_Result result;
    Attribute *attribute;
    int i;
    Aps_AttrOption *option;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Check whether this attribute has any options listed. */
    if (attribute->options == NULL)
        return APS_NOT_FOUND;

    /* Loop through all listed options, add them to the array. */
    for (i = 0; i < attribute->numOptions; ++i) {
        /* Add another entry to the array of possible options. */
        option = TrackArrayIndirectAddLastByRef_AttrOption(optionArray, NULL);
        if (! option) return APS_OUT_OF_MEMORY;

        /* Fill in this entry in the option array. */
        if (attribute->options[i].optionID) {
            option->optionID = (char *)TrackMemDupString(option,
                attribute->options[i].optionID, 0);
            if (! option->optionID) return APS_OUT_OF_MEMORY;
        } else option->optionID = NULL;
        if (attribute->options[i].translatedName) {
            option->optionID = (char *)TrackMemDupString(option,
                attribute->options[i].translatedName, 0);
            if (! option->translatedName) return APS_OUT_OF_MEMORY;
        } else option->translatedName = NULL;
        if (attribute->options[i].value) {
            option->value = (char *)TrackMemDupString(option,
                attribute->options[i].value, 0);
            if (! option->value) return APS_OUT_OF_MEMORY;
        } else option->value = NULL;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvGetSetting()
 *
 * Obtains the current value of the user-setting for a particular attribute.
 *
 * Parameters: this        - A pointer the the base AttrProvider instance
 *                           for this Attribute Provider.
 *
 *             attributeID - The ID of attribute to retrieve.
 *
 *             setting     - A pointer to a char * to receive a new buffer
 *                           containing the current user-setting for this
 *                           attribute. When finished with this buffer,
 *                           deallocate it by calling Aps_ReleaseBuffer().
 *                           Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvGetSetting(AttrProvider * this,
                              const char *attributeID,
                              char **setting)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Check whether this attribute has a setting. */
    if (attribute->setting == NULL)
        return APS_NOT_FOUND;

    /* Attempt to allocate a buffer in which to pass the setting
     * back to the caller.
     */
    *setting = TrackMemDupString(NULL, attribute->setting, 0);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvSetSetting()
 *
 * Changes the user-setting for a particular attribute.
 *
 * Parameters: this        - A pointer the the base AttrProvider instance
 *                           for this Attribute Provider.
 *
 *             attributeID - A string containing the ID of the attribute
 *                           to be changed.
 *
 *             setting     - The new value for the user-setting.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvSetSetting(AttrProvider * this,
                              const char *attributeID,
                              const char *setting)
{
    Aps_Result result;
    Attribute *attribute;

    /* Look for the Attribute structure for this attribute ID. */
    result = ObjectGetPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        attributeID,
        &attribute);
    if (result != APS_SUCCESS)
        return result;

    /* Update the setting stored in this attribute. */
    return AttrSetSetting(attribute, setting);
}

/* ---------------------------------------------------------------------------
 * AttrProvCheckConstraints()
 *
 * Tests whether a proposed change to a particular attribute's user-setting
 * would violate any of the user-setting constraints.
 *
 * Parameters: this                 - A pointer the the base AttrProvider
 *                                    instance for this Attribute Provider.
 *
 *             attributeID          - A string uniquely identifying the
 *                                    attribute whose user setting is
 *                                    being proposed for modification.
 *
 *             setting              - A string containing the new user
 *                                    setting for this attribute.
 *
 *             conflictingAttribute - If a conflict is found, this
 *                                    will be given a string buffer
 *                                    containing the ID of the first
 *                                    attribute found to conflict with
 *                                    the proposed change. Use
 *                                    Aps_ReleaseBuffer() to deallocate
 *                                    this buffer when finished with it.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *             conflictingSetting   - If a conflict is found, this will
 *                                    contain conflicting user-setting.
 *                                    Use Aps_ReleaseBuffer() to
 *                                    deallocate this buffer.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS if no constraints would be violated by this
 *             change, APS_VIOLATES_CONSTRAINTS if there is is a conflict, or
 *             another Aps_Result code on failure.
 */
Aps_Result AttrProvCheckConstraints(AttrProvider * this,
                                    const char *attributeID,
                                    const char *setting,
                                    char **conflictingAttribute,
                                    char **conflictingSetting)
{
    /* The generic attributes provider doesn't have any constraints on
     * the possible combinations of settings, so always indicate that
     * any proposed new setting would be conflict-free.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvWriteBlock()
 *
 * Writes a standard block of data into a job, based on the current state
 * of the attributes supplied by this attribute provider.
 *
 * Parameters: this                 - A pointer the the base AttrProvider
 *                                    instance for this Attribute Provider.
 *
 *             job                  - A handle to the job to write into.
 *
 *             blockType            - The type of data to be written.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvWriteBlock(AttrProvider * this,
                              Aps_JobHandle job,
                              Aps_BlockType blockType)
{
    /* The generic attributes provider doesn't have any information to
     * write.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvResetToModelDefaults()
 *
 * Resets these job attributes to the defaults for the specified model of
 * printer, based on information retrieved from the model database.
 *
 * Parameters: this    - Pointer to the attributes provider to operate on.
 *
 *             printer - A pointer to the printer with which the default
 *                       attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvResetToModelDefaults(AttrProvider * this,
                                        Printer * printer)
{
    /* This attributes provider doesn't currently store any model-dependant
     * information.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvSaveAsPrinterDefaults()
 *
 * Saves the settings in these job attributes as the default for the specified
 * printer.
 *
 * Parameters: this    - Pointer to the attributes provider to operate on.
 *
 *             printer - A pointer to the printer with which the default
 *                       attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvSaveAsPrinterDefaults(AttrProvider * this,
                                         Printer * printer)
{
    Aps_Result result;
    ApsObject *associatedObject;
    TrackArray_PtrChar propertyNames = NULL;
    int numProperties;
    int i;
    char *property;
    Attribute *attribute;

    ASSERT(this != NULL);
    ASSERT(printer != NULL);

    /* Obtain a pointer to the APS object containing the attributes
     * implemented by this attribute provider.
     */
    associatedObject = ObjectGetPtrFromHandle(this->associatedObject);
    ASSERT(associatedObject != NULL);

    /* Create a buffer array to hold the names of properties of this
     * object.
     */
    propertyNames = TrackArrayNew_PtrChar(NULL, 0);
    if (! propertyNames) { result = APS_OUT_OF_MEMORY; goto cleanup; }
    result = ObjectGetProperties(associatedObject, & propertyNames);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Loop through all properties of the associated object. */
    numProperties = TrackArrayGetSize_PtrChar(propertyNames);
    for (i = 0; i < numProperties; ++i) {
        property = propertyNames[i];

        /* We are only interested in object properties that are full-fledged
         * attributes.
         */
        if (ObjectGetPropertyType(associatedObject, property)
            != PROPERTY_ATTRIBUTE)
            continue;

        /* Obtain an Attribute instance for this property. */
        result = ObjectGetPropertyAttribute(associatedObject,
                                            property,
                                            &attribute);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* We are only interested in those attributes provided by this
         * attribute provider.
         */
        if (attribute->provider != this)
            continue;

        /* Store the current setting of this property in the printer's
         * meta configuration.
         */
        if (attribute->setting != NULL) {
            result = MetaWrite(printer->name, property,
                               attribute->setting);
            if (result != APS_SUCCESS)
                goto cleanup;
        }
    }

    /* If we reach this point we've successfully saved these attributes
     * as the default for this printer.
     */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary storage allocated by this function. */
    if (propertyNames != NULL) {
        TrackArrayDelete_PtrChar(propertyNames);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * AttrProvSetToPrinterDefaults()
 *
 * Initializes these job attributes based on the defaults for the specified
 * printer.
 *
 * Parameters: this    - Pointer to the attributes provider to operate on.
 *
 *             printer - A pointer to the printer with which the default
 *                       attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvSetToPrinterDefaults(AttrProvider * this,
                                        Printer * printer)
{
    /* This attributes provider doesn't currently save its defaults with the
     * printer.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * AttrProvRemovePrinterDefaults()
 *
 * Removes any default settings associated with the specified printer that
 * are provided by this attribute provider. 
 *
 * Parameters: this    - Pointer to the attributes provider to operate on.
 *
 *             printer - A pointer to the printer in question.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result AttrProvRemovePrinterDefaults(AttrProvider * this,
                                         Printer * printer)
{
    Aps_Result result;
    ApsObject *associatedObject;
    TrackArray_PtrChar propertyNames = NULL;
    int numProperties;
    int i;
    char *property;
    Attribute *attribute;

    ASSERT(this != NULL);
    ASSERT(printer != NULL);

    /* Obtain a pointer to the APS object containing the attributes
     * implemented by this attribute provider.
     */
    associatedObject = ObjectGetPtrFromHandle(this->associatedObject);
    ASSERT(associatedObject != NULL);

    /* Obtain a list of all properties stored in the associated object. */
    propertyNames = TrackArrayNew_PtrChar(NULL, 0);
    if (! propertyNames) { result = APS_OUT_OF_MEMORY; goto cleanup; }
    result = ObjectGetProperties(associatedObject, & propertyNames);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Loop through all properties of the associated object. */
    numProperties = TrackArrayGetSize_PtrChar(propertyNames);
    for (i = 0; i < numProperties; ++i) {
        property = propertyNames[i];

        /* We are only interested in object properties that are full-fledged
         * attributes.
         */
        if (ObjectGetPropertyType(associatedObject, property)
            != PROPERTY_ATTRIBUTE)
            continue;

        /* Obtain an Attribute instance for this property. */
        result = ObjectGetPropertyAttribute(associatedObject,
                                            property,
                                            &attribute);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* We are only interested in those attributes provided by this
         * attribute provider.
         */
        if (attribute->provider != this)
            continue;

        /* Remove the current setting of this property in the printer's
         * meta configuration.
         */
        if (attribute->setting != NULL) {
            result = MetaRemove(printer->name, property);
            if (result != APS_SUCCESS)
                goto cleanup;
        }
    }

    /* If we reach this point we've successfully saved these attributes
     * as the default for this printer.
     */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary storage allocated by this function. */
    if (propertyNames != NULL) {
        TrackArrayDelete_PtrChar(propertyNames);
    }

    return result;
}
