/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: modeldb.c
 *
 * Description: Functions accessing the printer model database.
 *
 *              The printer model database is stored in a text file,
 *              with one record for each known printer model. Each
 *              record is structured as follows:
 *
 *                  # All comments will start with # at the begining 
 *                  # of the line
 *                  beginrecord
 *                  <option name>=<option value>
 *                  <option name>=[<option value 1>][<option value 2>]\
 *                                [<option value 3>]
 *                  endrecord
 *
 *              Standard option fields are as follows:
 *
 *                  manufacturer   - Name of manufacturer, or Generic
 *                  model          - Name of model, without manufacturer name
 *                  ppd            - Name of associated PPD file, if any
 *                  gsdevice       - Name of GhostScript device driver
 *                  resolutions    - A list of avilable resolutions
 *                  colorrendering - A list of color depth options
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "aps.h"
#include "apsinternal.h"
#include "model.h"
#include "modeldb.h"
#include "resultcode.h"
#include "ppdmanager.h"
#include "utils.h"
#include "object.h"

DEBUG_CHANNEL_DEFAULT(model)

/* Model database type. */
typedef FILE * ModelDb;

/* Functions private to this module. */
static Aps_Result ModelGetName(Aps_ModelHandle model, char **manName,
                               char **modName);
static Aps_Result ModelDbUpdateRecord(char *manName, char *modName,
                                      Aps_ModelHandle model);
static Aps_Result ModelDbWriteRecord(ModelDb database, Aps_ModelHandle model);
static Aps_Result ModelDbOpen(ModelDb *database, int writeAccess);
static void ModelDbClose(ModelDb database);
static Aps_Result ModelDbMoveToNextRecord(ModelDb database);
static Aps_Result ModelDbParseCurField(ModelDb database, char** buffer,
                                       int* size, char **name, char **value);
static Aps_Result ModelDbGetField(ModelDb database, char** buffer,
                                  int *size, const char* optionName,
                                  char **value);
static Aps_Result ModelDbGetModelForCurrentRecord(ModelDb database,
                                                  Aps_ModelHandle *model);
static int ModelDbGetNumberOfValues(char* buffer, int len);
static void ModelDbGetMultipleEntriesFromBuffer(char* buffer, char** saveIn,
                                                int number);
                                                
/* Recognized separators between fields names and values. */
#define NUM_FIELD_NAME_DELIMTERS 1
char fieldNameDelimiters[NUM_FIELD_NAME_DELIMTERS] = {'='};
                                                
/* ---------------------------------------------------------------------------
 * Aps_GetKnownManufacturers()
 *
 * Obtains a list of all printer manufacturers listed in the printer model
 * database.
 *
 * Parameters: manufacturerNames - A char *** to receive a buffer with an
 *                                 array of string pointers to printer
 *                                 manufacturer names. If this function
 *                                 succeeds, it is the caller's responsibility
 *                                 to dispose of this array using
 *                                 Aps_ReleaseBuffer().
 *                                 Free with Aps_ReleaseBuffer()
 *
 *             numManufacturers  - A pointer ot an int to receive the number
 *                                 of manufacturers in the array.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_GetKnownManufacturers(char ***manufacturerNames,
                                     int *numManufacturers)
{
    Aps_Result result;
    ModelDb database = NULL;
    char *fieldBuffer = NULL;
    int fieldBufferSize = APSCFG_FILE_BUFFER_SIZE;
    char *manufacturer;
    int index;
    int relativePos;
    int exists;
    char *arrayEntry;

    /* Check for reasonable parameter values. */
    if (manufacturerNames == NULL || numManufacturers == NULL) {
        return APS_INVALID_PARAM;
    }
    
    /* Initialize output parameters to default values in case of failure. */
    *manufacturerNames = NULL;
    *numManufacturers = 0;

    /* Attempt to open the printer model database for read-only access. */
    result = ModelDbOpen(&database, FALSE);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Attempt to allocate memory to hold the manufacturer name. */
    fieldBuffer = (char *)malloc(fieldBufferSize);
    if (fieldBuffer == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Create a buffer array to hold the names retrieved from the database. */
    *manufacturerNames = TrackArrayNew_PtrChar(NULL, 0);
    if (! *manufacturerNames) { result = APS_OUT_OF_MEMORY; goto cleanup; }

    /* Loop through all entries, building a list of manufacturer names in
     * more-or-less alphabetical order. That is, all true manufacturer names
     * will be listed in alphabetical order, but the "Generic" manufacturer
     * entry will be listed at the top.
     */
    for(;;) {
        /* Advance to the beginning of the next entry in the database. */
        result = ModelDbMoveToNextRecord(database);
        if (result == APS_NOT_FOUND) {
            /* We've reached the end of the database; no problem. */
            break;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }

        /* Obtain the manufacturer name of this database entry. */
        result = ModelDbGetField(database, &fieldBuffer, &fieldBufferSize,
                                 "manufacturer", &manufacturer);
        if (result == APS_NOT_FOUND) {
            /* This entry doesn't have a manufacturer name, so proceed to the
             * next entry.
             */
            continue;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }
                                               
        /* Look for this name in the current array. */
        exists = FALSE;
        
        for (index = 0;
             index < TrackArrayGetSize_PtrChar(*manufacturerNames);
             ++index) {
            /* Compare the current manufacturer name with the manufacturer at
             * the specified position in the array, always treating "Generic"
             * as coming before any other manufacturer names.
             */
            arrayEntry = (*manufacturerNames)[index];
            if (strcmp(manufacturer, "Generic") == 0 &&
                strcmp(arrayEntry, "Generic") == 0) {
                relativePos = 0;
            } else if (strcmp(manufacturer, "Generic") == 0) {
                relativePos = -1;
            } else if (strcmp(arrayEntry, "Generic") == 0) {
                relativePos = 1;
            } else {
                relativePos = strcmp(manufacturer, arrayEntry);
            }

            /* If we find this exact manufacturer name already in the array,
             * then we do not want to add it.
             */
            if (relativePos == 0) {
                exists = TRUE;
                break;
            }
            
            /* If this manufacturer name should appear before the current
             * position in the array, then stop searching.
             */
            if (relativePos < 0)
                break;
        }
        
        /* If this manufacturer name isn't already listed in the buffer array,
         * then it should be inserted at the current index.
         */
        if (!exists) {
            char *newString;
            newString = TrackMemDupString(*manufacturerNames, manufacturer, 0);
            if (! newString) { result = APS_OUT_OF_MEMORY; goto cleanup; }
            if (! TrackArrayInsertAt_PtrChar(manufacturerNames, index,
                newString)) { result = APS_OUT_OF_MEMORY; goto cleanup; }
        }
    }
    
    /* Well, it looks like we succeeded.  At this point we can just fall
     * through the cleanup code and exit. */
    result = APS_SUCCESS;
    *numManufacturers = TrackArrayGetSize_PtrChar(*manufacturerNames);

cleanup:
    /* Close the printer model database if it was ever opened. */
    if (database != NULL) {
        ModelDbClose(database);
    }

    /* Dispose of an incomplete buffer on failure */
    if ((result != APS_SUCCESS) && (*manufacturerNames)) {
        TrackArrayDelete_PtrChar(*manufacturerNames);
        *manufacturerNames = NULL;
    }

    /* Get rid of the temporary buffer holding manufacturer names. */
    if (fieldBuffer != NULL) {
        free(fieldBuffer);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_GetKnownModels()
 *
 * Obtains a list of all printer models for a given manufacturer in the
 * printer model database.
 *
 * Parameters: manufacturerName - The name of the manufacturer to look up.
 *
 *             modelNames       - The address of a char ** to receive an
 *                                array of string pointers to printer model
 *                                names. On success, this array must be
 *                                deleted when no longer needed using the
 *                                Aps_ReleaseBuffer() function.
 *                                Free with Aps_ReleaseBuffer()
 *
 *             numModels        - An int to receive the number of models in
 *                                the array.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_GetKnownModels(const char *manufacturerName,
                              char ***modelNames,
                              int *numModels)
{
    Aps_Result result;
    ModelDb database = NULL;
    char *fieldBuffer = NULL;
    int fieldBufferSize = APSCFG_FILE_BUFFER_SIZE;
    char *field;
    int index;

    /* Check for reasonable parameter values. */
    if (manufacturerName == NULL || strlen(manufacturerName) == 0
        || modelNames == NULL || numModels == NULL) {
        return APS_INVALID_PARAM;
    }
    
    /* Initialize output parameters to default values in case of failure. */
    *modelNames = NULL;
    *numModels = 0;

    /* Attempt to open the printer model database for read-only access. */
    result = ModelDbOpen(&database, FALSE);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Attempt to allocate memory for temporary storage of information from
     * database.
     */
    fieldBuffer = (char *)malloc(fieldBufferSize);
    if (fieldBuffer == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Create a buffer array to hold the names retrieved from the database. */
    *modelNames = TrackArrayNew_PtrChar(NULL, 0);
    if (! *modelNames) { result = APS_OUT_OF_MEMORY; goto cleanup; }

    /* Loop through all entries, building a list of any model names for this
     * manufacturer, maintaining the list of model names in alphabetical
     * order.
     */
    for(;;) {
        /* Advance to the beginning of the next entry in the database. */
        result = ModelDbMoveToNextRecord(database);
        if (result == APS_NOT_FOUND) {
            /* We've reached the end of the database; no problem. */
            break;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }

        /* Skip entries that don't match the manufacturer name requested by
         * the application.
         */
        result = ModelDbGetField(database, &fieldBuffer, &fieldBufferSize,
                                 "manufacturer", &field);
        if (result == APS_NOT_FOUND) {
            /* This entry doesn't have a manufacturer name, so only proceed
             * with this entry if the queried manufacturer name is an empty
             * string ("").
             */
            if (strlen(manufacturerName) != 0)
                continue;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }
        
        if (strcmp(field, manufacturerName) != 0)
            continue;

        /* Obtain the model name of this database entry. */
        result = ModelDbGetField(database, &fieldBuffer, &fieldBufferSize,
                                 "model", &field);
        if (result == APS_NOT_FOUND) {
            /* This entry doesn't have a manufacturer name, so proceed to the
             * next entry.
             */
            continue;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }

        /* Look for the position to insert this model name into the aray. */
        for (index = 0;
             index < TrackArrayGetSize_PtrChar(*modelNames); ++index) {
            /* Compare the current model name with the model at the current
             * position in the array.
             */
            if (strcmp(field, (*modelNames)[index]) < 0)
                break;
        }

        {
            char *newString;
            newString = TrackMemDupString(*modelNames, field, 0);
            if (! newString) { result = APS_OUT_OF_MEMORY; goto cleanup; }
            if (! TrackArrayInsertAt_PtrChar(modelNames, index, newString)) {
                result = APS_OUT_OF_MEMORY;
                goto cleanup;
            }
        }
    }
    
    /* Well, it looks like we succeeded.  At this point we can just fall
     * through the cleanup code and exit. */
    result = APS_SUCCESS;
    *numModels = TrackArrayGetSize_PtrChar(*modelNames);

cleanup:
    /* Close the printer model database if it was ever opened. */
    if (database != NULL) {
        ModelDbClose(database);
    }

    /* Dispose of an incomplete buffer on failure */
    if ((result != APS_SUCCESS) && (*modelNames)) {
        TrackArrayDelete_PtrChar(*modelNames);
        *modelNames = NULL;
    }

    /* Get rid of the temporary buffer holding a database field. */
    if (fieldBuffer != NULL) {
        free(fieldBuffer);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_GetModel()
 *
 * Obtains a model object with information on a particular entry from the
 * printer model database.
 *
 * Parameters: manufacturer - The name of the manufacturer of this printer.
 *
 *             modelName    - The manufacturer's name for this model.
 *
 *             modelHandle  - A pointer to an Aps_ModelHandle to
 *                            receive a handle to the model object.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_GetModel(const char *manufacturer,
                        const char *modelName,
                        Aps_ModelHandle *modelHandle)
{
    const char *fields[2];
    const char *values[2];

    /* Check for reasonable parameter values. */
    if (manufacturer == NULL || modelName == NULL || strlen(modelName) == 0
        || modelHandle == NULL) {
        return APS_INVALID_PARAM;
    }
    
    /* Perform a standard search of the printer database, retrieving a model
     * object for the first record that matches the model and manufacturer
     * fields.
     */
    fields[0] = "manufacturer";
    values[0] = manufacturer;
    fields[1] = "model";
    values[1] = modelName;
    
    return ModelDbSearch(modelHandle, fields, values, 2);
}

/* ---------------------------------------------------------------------------
 * Aps_AddModel()
 *
 * Creates an empty entry in the printer database for a new printer model.
 *
 * Parameters: manufacturer   - The name of the manufacturer of this printer.
 *
 *             model          - The manufacturer's name for this model.
 *
 *             modelHandleOut - An option pointer to an Aps_ModelHandle to
 *                              receive a handle to the new model object.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AddModel(const char *manufacturer,
                        const char *model,
                        Aps_ModelHandle *modelHandleOut)
{
    Aps_Result result;
    Aps_ModelHandle modelHandle = NULL;

    /* Check for reasonable parameter values. */
    if (manufacturer == NULL || model == NULL || strlen(model) == 0) {
        return APS_INVALID_PARAM;
    }

    /* Construct a new model object. */
    result = ModelCreate(&modelHandle);
    if (result != APS_SUCCESS) goto cleanup;

    /* Set the manufacturer and model name properites for this model. */
    result = Aps_SetPropertyString(modelHandle, "manufacturer",
                                   manufacturer);
    if (result != APS_SUCCESS) goto cleanup;
    
    result = Aps_SetPropertyString(modelHandle, "model",
                                   model);
    if (result != APS_SUCCESS) goto cleanup;

    /* If the caller wishes to receive a handle to the new model object,
     * pass it back.
     */
    if (modelHandleOut != NULL) {
        *modelHandleOut = modelHandle;
        
        /* Record the caller's reference to this model object. */
        Aps_AddRef(modelHandle);
    }

    /* If we reach this point, we've successfully created the new model. */
    result = APS_SUCCESS;
    
cleanup:
    if (modelHandle != NULL) {
        Aps_ReleaseHandle(modelHandle);
    }
    
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_AddModelFromPPD()
 *
 * Adds a new printer to the model database based on the information in a
 * PPD file. If the specified PPD file does not already exist in the PPD file
 * directory, a copy of the specified PPD file is created in that directory.
 * The newly added model record will refer to that PPD file.
 *
 * Parameters: pathToPPDFile  - The full path and filename of the PPD file
 *                              to use in creating the new entry in the
 *                              printer model database.
 *
 *             modelHandleOut - Optionally receives a handle to the newly
 *                              added printer model.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AddModelFromPPD(const char *pathToPPDFile,
                               Aps_ModelHandle *modelHandleOut)
{
    Aps_ModelHandle model = NULL;
    PPDHandle ppdManager = NULL;
    Aps_Result result;
    char *manufacturerName;
    char *modelName;
    char *manKey;
    char *modelKey;
    const char *basePPDFilename;
    char *newPPDFile = NULL;
    const char *linkedPPDFile = NULL;
    char copyBuffer[APSCFG_COPY_BUFFER_SIZE];
    FILE *source = NULL;
    FILE *dest = NULL;
    ModelDb database = NULL;
    char *firstSpace;
    char *formats[1] = {APS_FORMAT_POSTSCRIPT};
    
    /* Check for reasonable parameter values. */
    if (pathToPPDFile == NULL || strlen(pathToPPDFile) == 0) {
        return APS_INVALID_PARAM;
    }

    /* Attempt to setup a new PPD manager to read the specified PPD file. */
    ppdManager = PPDCreate();
    if (!PPDSetFilename(ppdManager, pathToPPDFile)) {
        result = APS_GENERIC_FAILURE;
        goto cleanup;
    }

    /* Attempt to obtain the manufacturer's name from this PPD file. */
    if (PPDGetKey(ppdManager, "*Manufacturer", &manKey, NULL) && manKey != NULL
     && PPDGetKey(ppdManager, "*Model", &modelKey, NULL) && modelKey != NULL) {
        manufacturerName = manKey;
        modelName = modelKey;

        /* Trim the manufacturer's name from the bare model name, if possible. */
        if (strncmp(modelName, manufacturerName, strlen(manufacturerName))
            == 0) {
            /* The model name starts with an exact duplicate of the manufacturer
             * name, so trim off that part of the name.
             */
            modelName += strlen(manufacturerName);
        
            /* Also trim off any whitespace between the manufacturer name and
             * model name.
             */
            while(*modelName != '\0' && *modelName != ' ') {
                ++modelName;
            }
        }
    } else if(PPDGetKey(ppdManager, "*ModelName", &modelKey, NULL)
              || PPDGetKey(ppdManager, "*NickName", &modelKey, NULL)) {
        firstSpace = strchr(modelKey, ' ');
        if (firstSpace == NULL) {
            manufacturerName = "Generic";
            modelName = modelKey;
        } else {
            manufacturerName = modelKey;
            *firstSpace = '\0';
            modelName = firstSpace + 1;
        }
    } else {
       /* We weren't able to get the manufacturer and model names from the
        * PPD file.
        */
       result = APS_INVALID_PPD;
       goto cleanup;
    }

    /* Attempt to create a new model with this name. */
    result = Aps_AddModel(manufacturerName, modelName, &model);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Attempt to copy the PPD file to the PPD directory, if it isn't already
     * there.
     */
    if (strncmp(pathToPPDFile, APSCFG_CONFIG_PATH_PPD_SYS_DIR,
        strlen(APSCFG_CONFIG_PATH_PPD_SYS_DIR)) != 0) {
        /* Find the base name of the PPD file. */
        basePPDFilename = strrchr(pathToPPDFile, '/');
        if (basePPDFilename == NULL) {
            basePPDFilename = pathToPPDFile;
        } else {
            basePPDFilename++;
        }
        
        /* Generate the name of the new PPD file. */
        newPPDFile = malloc(strlen(APSCFG_CONFIG_PATH_PPD_SYS_DIR)
            + strlen(basePPDFilename) + 1);
        if (newPPDFile == NULL) {
            result = APS_OUT_OF_MEMORY;
            goto cleanup;
        }
        
        strcpy(newPPDFile, APSCFG_CONFIG_PATH_PPD_SYS_DIR);
        strcat(newPPDFile, basePPDFilename);

        /* Copy the contents from the source file to the destination file. */
        source = fopen(pathToPPDFile, "r");
        if (source == NULL) {
            result = GetResultFromErrno();
            goto cleanup;
        }

        dest = fopen(newPPDFile, "w"); /* TBD: ensure filename is unique. */
        if (dest == NULL) {
            result = GetResultFromErrno();
            goto cleanup;
        }
        
        while (fgets(copyBuffer, sizeof(copyBuffer), source) != NULL) {
            fputs(copyBuffer, dest);
        }
        
        linkedPPDFile = newPPDFile;
        
        /* To avoid unnecessary complexity, files will be closed in cleanup
         * section of this function.
         */
    } else {
        /* The PPD file is already in the right place, so we'll directly link
         * to the file specified by the caller.
         */
        linkedPPDFile = pathToPPDFile;
    }

    /* Set the name of the PPD file to be linked to this model object. */
    result = Aps_SetPropertyString(model, "ppd", (char *)linkedPPDFile);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Set this printer as accepting data in PostScript format. */
    result = Aps_SetPropertyStrArray(model, "formats", (const char **)formats,
                                     1);

    /* Attempt to commit this model information into the database. */
    result = Aps_ModelCommitToDatabase(model);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Provide the caller with a handle to the new model object, if required. */
    if (modelHandleOut != NULL) {
        *modelHandleOut = model;
        
        /* Record this additional reference to this model object. */
        Aps_AddRef(model);
    }
    
    /* If we reach this point, we've successfully added the new model. */
    result = APS_SUCCESS;
    
cleanup:
    /* Release any resources that were allocated by this function. */
    if (ppdManager != NULL) {
        /* Deleting the PPD manager releases all strings that it allocated
         * for us.
         */
        PPDDestroy(ppdManager);
    }
    
    if (newPPDFile != NULL) {
        free(newPPDFile);
    }
    
    if (model != NULL) {
        Aps_ReleaseHandle(model);
    }

    if (source != NULL) {
        fclose(source);
    }
    
    if (dest != NULL) {
        fclose(dest);
    }

    if (database != NULL) {
        ModelDbClose(database);
    }
    
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_ModelCommitToDatabase()
 *
 * Commits the changes made to an in-memory model object back to persistent
 * storage.
 *
 * Parameters: model - A handle to a currently open model whose changes should
 *                     be committed.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_ModelCommitToDatabase(Aps_ModelHandle model)
{
    char *manName;
    char *modName;
    Aps_Result result;

    /* Obtain the name of the manufacturer and model to be written to or
     * updated in the database.
     */
    result = ModelGetName(model, &manName, &modName);
    if (result != APS_SUCCESS)
        return result;

    /* Write a record to the database for this model, replacing the existing
     * record, if any.
     */
    result = ModelDbUpdateRecord(manName, modName, model);

    Aps_ReleaseBuffer(manName);
    Aps_ReleaseBuffer(modName);

    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_ModelRemove()
 *
 * Deletes this model from the model database.
 *
 * Parameters: model - A handle to the model to be removed from the database.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_ModelRemove(Aps_ModelHandle model)
{
    char *manName;
    char *modName;
    Aps_Result result;

    /* Obtain the name of the manufacturer and model to be removed from the
     * database.
     */
    result = ModelGetName(model, &manName, &modName);
    if (result != APS_SUCCESS)
        return result;

    /* Remove this record from the database. */
    result = ModelDbUpdateRecord(manName, modName, NULL);

    Aps_ReleaseBuffer(manName);
    Aps_ReleaseBuffer(modName);

    /* On success, release this reference to this model object. */
    if (result != APS_SUCCESS)
        Aps_ReleaseHandle(model);

    return result;
}

/* ---------------------------------------------------------------------------
 * ModelGetName()
 *
 * Obtains the name of the manufacturer and model from a particular model
 * object in a single step.
 *
 * Parameters: model   - A handle to an existing model object.
 *
 *             manName - Receives the name of this printer's manufacturer.
 *
 *             modName - Receives the name of the model, without the
 *                       manufacturer name.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelGetName(Aps_ModelHandle model, char **manName,
                               char **modName)
{
    Aps_Result result;

    /* Attempt to obtain the manufacturer's name. */
    result = Aps_GetPropertyString(model, "manufacturer", manName);
    if (result != APS_SUCCESS)
        return result;

    /* Now, attempt to obtain the model name. */
    result = Aps_GetPropertyString(model, "model", modName);
    if (result != APS_SUCCESS) {
        Aps_ReleaseBuffer(manName);
        manName = NULL;
        return result;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ModelDbUpdateRecord()
 *
 * Updates the record in the database for the specified manufacturer and
 * model, removing existing contents for this record (if any), and optionally
 * writing new contents for the record.
 *
 * Parameters: manName - Receives the name of this printer's manufacturer.
 *
 *             modName - Receives the name of the model, without the
 *                       manufacturer name.
 *
 *             model   - A handle to an existing model object, or NULL to
 *                       remove a record from the database.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbUpdateRecord(char *manName, char *modName,
                                      Aps_ModelHandle model)
{
    Aps_Result result;
    ModelDb database = NULL;
    FILE *databaseFile;
    FILE *tempFile = NULL;
    char *tempFileName = NULL;
    char *fieldBuffer = NULL;
    int fieldBufferSize = APSCFG_FILE_BUFFER_SIZE;
    long recordPos;
    char *fieldValue;

    /* Attempt to open the printer model database for read/write access. */
    result = ModelDbOpen(&database, TRUE);
    if (result != APS_SUCCESS)
        goto cleanup;
    databaseFile = (FILE *)database;

    /* Create a temporary file to contain temporary data to be written back to
     * model database.
     */
    tempFileName = tempnam(NULL, "APS__");
    if (tempFileName == NULL)
        goto cleanup;
    tempFile = fopen(tempFileName, "w+");

    /* Attempt to allocate memory to hold a field from the database. */
    fieldBuffer = (char *)malloc(fieldBufferSize);
    if (fieldBuffer == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* STEP 1: Locate the record in the database if it already exists. When
     *         this step is completed, the database file pointer will be
     *         positioned where the record should be written, and the
     *         temporary file will contain any subsequent records to be
     *         written back to the database after this record.
     */
    for(;;) {
        /* Advance to the beginning of the next entry in the database. */
        result = ModelDbMoveToNextRecord(database);
        if (result == APS_NOT_FOUND) {
            /* We've reached the end of the database; no problem. */
            break;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }

        /* Remember where this record started in the file. */
        recordPos = ftell(databaseFile);

        /* Check whether this is the record we're looking for. */
        result = ModelDbGetField(database, &fieldBuffer, &fieldBufferSize,
                                 "manufacturer", &fieldValue);
        if (result != APS_SUCCESS)
            break;
        if (strcmp(fieldValue, manName) != 0) continue;

        result = ModelDbGetField(database, &fieldBuffer, &fieldBufferSize,
                                 "model", &fieldValue);
        if (result != APS_SUCCESS)
            break;
        if (strcmp(fieldValue, modName) != 0) continue;

        /* If we reach this point, we've found an existing record to be
         * updated.
         */

        /* Copy the contents of the database beginning with the next record,
         * into our temporary file to be written back to the database after
         * the record we're modifying / removing.
         */
        result = ModelDbMoveToNextRecord(database);
        if (result == APS_NOT_FOUND) {
            /* We've reached the end of the database; no problem. */
            break;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }

        result = CopyFileToEnd(tempFile, databaseFile);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* Move the file pointer back to the beginning of the record to be
         * modified, and then exit this loop to proceed with database
         * modification.
         */
        fseek(databaseFile, recordPos, SEEK_SET);

        break;
    }

    /* STEP 2: Write the new contents for this record at the current position,
     * if a model object was supplied by the caller.
     */
    if (model != NULL) {
        result = ModelDbWriteRecord(database, model);
        if (result != APS_SUCCESS)
            return result;
    }

    /* Write any information from the temporary file back into the main
     * database. If the caller provided new contents for this record, then
     * this data will be written after the updated record. On the other hand,
     * if the caller wanted to remove the record, then this information will
     * overwrite the record.
     */
    fseek(tempFile, 0, SEEK_SET);
    result = CopyFileToEnd(databaseFile, tempFile);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* All information to be retained in the database file is located prior to
     * the current position, so truncate the file at this point.
     */
    ftruncate(fileno(databaseFile), ftell(databaseFile));

    /* If we reach this point, then we've successfully updated the database. */
    result = APS_SUCCESS;

cleanup:
    /* Close the printer model database if it was ever opened. */
    if (database != NULL) {
        ModelDbClose(database);
    }

    if (tempFile != NULL) {
        fclose(tempFile);
        remove(tempFileName);
    }

    if (tempFileName != NULL) {
        free(tempFileName);
    }

    /* Dispose of other temporary resources. */
    if (fieldBuffer != NULL) {
        free(fieldBuffer);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * ModelDbWriteRecord()
 *
 * Writes the contents of the provided model object as a record at the current
 * location in the model database.
 *
 * Parameters: database - A currently open model database to write into.
 *
 *             model    - A handle to an existing model object, or NULL to
 *                        remove a record from the database.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbWriteRecord(ModelDb database, Aps_ModelHandle model)
{
    Aps_Result result;
    FILE *databaseFile = (FILE *)database;
    TrackArray_PtrChar properties = NULL;
    int propertyIndex;
    char *propertyName;
    char *valueStr;
    char **valueArray;
    int numValues;
    int valueIndex;
    int i;
    int numSpaces;

    /* Obtain a list of properties of this model object. */
    properties = TrackArrayNew_PtrChar(NULL, 0);
    if (properties == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    result = ObjectGetProperties(ObjectGetPtrFromHandle(model), &properties);
    if (result != APS_SUCCESS)
        goto cleanup;

    if (TrackArrayGetSize_PtrChar(properties) == 0) {
        /* No properties in this model object; it can't be written. */
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Begin a new record in the database. */
    fprintf(databaseFile, "beginrecord\n");

    /* Loop through all properties of this model object. */
    for (propertyIndex = 0;
         propertyIndex < TrackArrayGetSize_PtrChar(properties);
         ++propertyIndex) {
        propertyName = TrackArrayElemAt(properties, propertyIndex);
        switch (ObjectGetPropertyType(ObjectGetPtrFromHandle(model),
                                      propertyName)) {
            case PROPERTY_STRING:
                /* This property is a simple string. */

                /* Obtain the value to write to this field. */
                result = Aps_GetPropertyString(model, propertyName, &valueStr);
                if (result != APS_SUCCESS)
                    goto cleanup;

                /* Write the field. */
                fprintf(databaseFile, "%s=%s\n", propertyName, valueStr);

                /* Release temporary storage. */
                Aps_ReleaseBuffer(valueStr);

                break;

            case PROPERTY_STRING_ARRAY:
                /* This property is an array of strings. */

                /* Obtain the value to write to this field. */
                result = Aps_GetPropertyStrArray(model, propertyName,
                                                 &valueArray, &numValues);
                if (result != APS_SUCCESS)
                    goto cleanup;

                /* Gracefully skip empty string array properties. */
                if (numValues == 0) continue;

                /* Write the field. */
                fprintf(databaseFile, "%s=", propertyName);
                numSpaces = strlen(propertyName) + 1;
                for (valueIndex = 0; valueIndex < numValues; ++valueIndex) {
                    if (valueIndex > 0) {
                        fputs("\\\n", databaseFile);
                        for(i = 0; i < numSpaces; ++i)
                            fputc(' ', databaseFile);
                    }
                    fprintf(databaseFile, "[%s]", valueArray[valueIndex]);
                }
                fprintf(databaseFile, "\n");

                /* Release temporary storage. */
                Aps_ReleaseBuffer(valueArray);

                break;

            default:
                /* This model object contains a type of property that cannot be
                 * written to the model database.
                 */
                result = APS_WRONG_TYPE;
                goto cleanup;
        }
    }

    /* End this database record. */
    fprintf(databaseFile, "endrecord\n\n");

    /* If we reach this point, we've successfully written this model to the
     * database.
     */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (properties != NULL) {
        TrackArrayDelete_PtrChar(properties);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * ModelDbSearch()
 *
 * Searches the model database for a record (entry) that matches the specified
 * fields.
 *
 * Parameters: model     - Receives a handle to the matching printer model
 *                         on success.
 *
 *             fields    - An array of strings containing the field names to
 *                         match.
 *
 *             values    - A corresponding array of strings containing the
 *                         field values to match.
 *
 *             numFields - The number of elements in the fields and values
 *                         arrays.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result ModelDbSearch(Aps_ModelHandle *model, const char *fields[],
                         const char *values[], int numFields)
{
    Aps_Result result;
    ModelDb database = NULL;
    char *fieldBuffer = NULL;
    int fieldBufferSize = APSCFG_FILE_BUFFER_SIZE;
    char *value;
    int i;

    ASSERT(model != NULL);
    ASSERT(fields != NULL);
    ASSERT(values != NULL);
    ASSERT(numFields >= 1);

    /* Attempt to open the printer model database for read-only access. */
    result = ModelDbOpen(&database, FALSE);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Attempt to allocate memory to hold a field from the database. */
    fieldBuffer = (char *)malloc(fieldBufferSize);
    if (fieldBuffer == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }
        
    /* Loop through all entries, looking for one that matches this query.
     */
    for(;;) {
        /* Advance to the beginning of the next entry in the database. */
        result = ModelDbMoveToNextRecord(database);
        if (result == APS_NOT_FOUND) {
            /* We've reached the end of the database; no problem. */
            break;
        } else if (result != APS_SUCCESS) {
            /* Some more serious error has happened, abort this function. */
            goto cleanup;
        }

        /* Skip entries that don't match any one of the required fields.
         */
        for (i = 0; i < numFields; ++i) {
            /* Retrieve the value for this field from this record. */
            ASSERT(fields[i] && values[i]);
            result = ModelDbGetField(database, &fieldBuffer, &fieldBufferSize,
                                     fields[i], &value);
            if (result != APS_SUCCESS)
                break;
            
            /* Stop comparing fields if this field doesn't match the required
             * value.
             */
            if (strcmp(values[i], value) != 0)
                break;
        }

        /* If the number of matching fields is equal to the total number of
         * fields to compare, then this is the first record matching the
         * required criteria, so return it to the caller.
         */
        if (i == numFields) {
            result = ModelDbGetModelForCurrentRecord(database, model);

            /* Propogate the success/failure of creating the model object to the
             * caller.
             */        
            goto cleanup;
        }
    }
    
    /* If we reach this point, then we traversed the entire database without
     * finding a matching entry.
     */
    result = APS_NOT_FOUND;
    
cleanup:
    /* Close the printer model database if it was ever opened. */
    if (database != NULL) {
        ModelDbClose(database);
    }

    /* Get rid of the temporary buffer holding a database field. */
    if (fieldBuffer != NULL) {
        free(fieldBuffer);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * ModelDbOpen()
 *
 * Opens the printer model database for either read-only or read-write access.
 *
 * Parameters: database    - Receives a ModelDb to allow future access to 
 *                           this model database. The database should be
 *                           closed whenthe immediate operation(s) are
 *                           complete, to allow other processes to access it.
 *
 *             writeAccess - TRUE if write access to the database is
 *                           required, FALSE if only read access is neded.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbOpen(ModelDb *database, int writeAccess)
{
    FILE *fp;

    /* Attempt to open the model database file. */
    if (writeAccess) {
        fp = fopen(APSCFG_CONFIG_PATH_MODELDB_SYS_FILE, "r+");
    } else {
        fp = fopen(APSCFG_CONFIG_PATH_MODELDB_SYS_FILE, "r");
    }

    /* Check for failure. */
    if (fp == NULL) {
        /* On failure, attempt to map common errno values supported by this
         * C implementation to corresponding APS result codes. Otherwise,
         * return a generic failure result code.
         */
        return GetResultFromErrno();
    }

    *database = (ModelDb)fp;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ModelDbClose()
 *
 * Closes a currently open printer model database.
 *
 * Parameters: database - A reference to the opened model database.
 *
 *     Return: None.
 */
static void ModelDbClose(ModelDb database)
{
    FILE *fp = (FILE *)database;

    /* Close the file pointer used to access the model database. */
    fclose(fp);
}

/* ---------------------------------------------------------------------------
 * ModelDbMoveToNextRecord()
 *
 * Advances the current position in the model database to the beginning of the
 * current entry.
 *
 * Parameters: database - A reference to the opened model database.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbMoveToNextRecord(ModelDb database)
{
    FILE *fp = (FILE *)database;
    char  buffer[APSCFG_FILE_BUFFER_SIZE];
    long  oldPosition = 0L;

    /* Skip the current line, unless we're at the beginning of the file.
     * If we're not at the beginning of the file, then the file pointer
     * will already be positioned at the start of some entry.
     */
    if (ftell(fp) != 0) {
        if(fgets( buffer, APSCFG_FILE_BUFFER_SIZE, fp) == NULL )
            return APS_NOT_FOUND;
    }
    
    if(!fp)
        return APS_INVALID_PARAM;
    while(1)
    {
        /* Save the old file position. */ 
        oldPosition = ftell(fp);     
        if(fgets( buffer, APSCFG_FILE_BUFFER_SIZE, fp) == NULL )
            break;
        if(strncmp( buffer, "beginrecord", 11)== 0 )
        {
            /* Setting the file pointer to the start of the entry. */
            fseek( fp, oldPosition, SEEK_SET);
            return APS_SUCCESS;
        }
    }
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * ModelDbParseCurField()
 *
 * Obtains the value of the current field of the current record.
 * Assumes that supplied buffer is empty/filled with zeros.
 * When this fuction is called file position is always at the 
 * begining of a new line.
 *
 * Parameters: database - A reference to the opened model database.
 *
 *             buffer   - Buffer currently being used for holding
 *                        data read from file. This allows the same memory
 *                        to be reused, rather than allocating and deleting
 *                        memory for each line read.
 *
 *             size     - The current size of the buffer, which may be
 *                        adjusted by this function.
 *
 *             name     - Receives a pointer to the name of the current field.
 *                        This will be a pointer somewhere inside 'buffer',
 *                        and should not be explicitly deallocated.
 *                        Do not free
 *
 *             value    - Receives a pointer to the value of the current field.
 *                        This will also be a pointer into 'buffer', and so
 *                        also should not be explicitly freed.
 *                        Do not free
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbParseCurField(ModelDb database, char** buffer,
                                       int* size, char **name, char **value)
{
    FILE *fp = (FILE *)database;
    char tempBuffer[APSCFG_FILE_BUFFER_SIZE];
    int len;
    char *delimiter;
    char *charLoc;
    char *lastChar;
    int i;
    
    /* Start by initializing the buffer of the overall entry to an be an
     * empty string.
     */
    **buffer = '\0';
    if (name) *name = NULL;
    if (value) *value = NULL;
    
    if(!fp)
        return APS_INVALID_PARAM;
    while(1)
    {
        if(fgets( tempBuffer, APSCFG_FILE_BUFFER_SIZE, fp) == NULL)
            break;
        if( tempBuffer[0] != '#')
        {
            len = strlen(tempBuffer);

            /* This is not a comment. */
            if( strlen(*buffer) + len >= *size )
            {
                /* space is not enough we have to reallocate. */
                char *tempBuffer = (char*)realloc(*buffer,
                                         *size + len + 1);
                if(tempBuffer)
                {
                    /* *size has the total size of the buffer. */
                    (*size) += len + 1;
                    *buffer = tempBuffer;
                }
                else
                {
                    return APS_OUT_OF_MEMORY;
                }
            }
            if( tempBuffer[strlen(tempBuffer)-2] != '\\')
            {
                /* This option does not continue in the next line. */
                /* return here */
                strcat(*buffer, tempBuffer);
                
                /* Locate the name/value separator character. We do this by
                 * looping through all known delimiters, looking for the
                 * one that appears leftmost in the string.
                 */
                delimiter = NULL;
                for (i = 0; i < NUM_FIELD_NAME_DELIMTERS; ++i) {
                    charLoc = strchr(*buffer, fieldNameDelimiters[i]);
                    if (charLoc != NULL) {
                        /* This character was found in the string, so mark
                         * it as the delimiter unless we've already found
                         * one further to the left.
                         */
                        if (delimiter == NULL || charLoc < delimiter) {
                            delimiter = charLoc;
                        }
                    }
                }
                
                /* If no form of name/value delimiter was found, then fail.
                 */
                if (delimiter == NULL)
                    return APS_NOT_FOUND;

                /* Trim any trailing newline characters from buffer. */
                ASSERT(strlen(*buffer) > 0);
                lastChar = *buffer + strlen(*buffer) - 1;
                if (*lastChar == '\n') {
                    *lastChar = '\0';
                }
                    
                /* Provide the caller with separated pointers to the name
                 * and value strings.
                 */
                *delimiter = '\0';
                if (name != NULL)
                    *name = *buffer;
                if (value != NULL)
                    *value = delimiter + 1;

                return APS_SUCCESS;
            }
            strncat(*buffer, tempBuffer, len-2);
            len = strlen(*buffer);
        }
    }
    
    /* If we reach this point, then the last line of the database file
     * ended with a line continuation character!
     */
    return APS_GENERIC_FAILURE;
}

/* ---------------------------------------------------------------------------
 * ModelDbGetField()
 *
 * Obtains the value of the specified field from the current entry.
 * Assumes that supplied buffer is empty/filled with zeros.
 * When this fuction is called file position is always at the 
 * begining of a new entry.
 *
 * Parameters: database   - A reference to the opened model database.
 *
 *             buffer     - A buffer allocated outside of this function,
 *                          to be used for temporary storage.
 *
 *             size       - The size of buffer.
 *
 *             optionName - The name of the optoin field to retrieve.
 *
 *             value      - The value retrieved for this field.
 *                          Do not free
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbGetField(ModelDb database, char** buffer,
                                  int *size, const char* optionName,
                                  char **value)
{
    FILE *fp = (FILE *)database;
    char tempBuffer[APSCFG_FILE_BUFFER_SIZE];
    long entryPosition;
    long oldPosition;
    Aps_Result result = APS_GENERIC_FAILURE;

    /* Reset the contents of the caller's buffer to be empty. */
    memset(*buffer, 0, *size);
    if (value) *value = NULL;
        
    if(!fp)
        return APS_INVALID_PARAM;
    entryPosition = ftell(fp); /* saving start of an entry. */
    while(1)
    {
        oldPosition = ftell(fp); /* saving current option position...*/
        if(fgets( tempBuffer, APSCFG_FILE_BUFFER_SIZE, fp) == NULL)
            break;
        if( tempBuffer[0] !='#') /* We look only if not a comment*/ 
        {
            /* Check if this line contains the option we are 
             * looking for in this entry.
             */
            if(strncmp(tempBuffer, optionName, strlen(optionName)) == 0)
            {
                /* Found the option...
                 */
                /* Go to start of option.*/
                fseek( fp, oldPosition, SEEK_SET);
                
                /* Extract the option. */
                result = ModelDbParseCurField( fp, buffer, size, NULL, value);
                
                /* Setting file pointer to start of entry to look for
                 * other options for the same entry.
                 */
                fseek( fp, entryPosition, SEEK_SET);
                
                return result;
            }
            if( strncmp( tempBuffer, "endrecord", 9) == 0)
            {
                /* reached end of entry. Setting pointer back to start. */
                fseek( fp, entryPosition, SEEK_SET);
                return APS_NOT_FOUND;
            }
        }
    }
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * ModelDbGetModelForCurrentRecord()
 *
 * Creates a new model object with information from the entry currently
 * pointed to in the model database.
 *
 * Parameters: database - An open instance of the printer model database,
 *                        from which to read the model information.
 *
 *             model    - Receives a handle to a new Model object on
 *                        success.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result ModelDbGetModelForCurrentRecord(ModelDb database,
                                                  Aps_ModelHandle *model)
{
    FILE *fp = (FILE *)database;
    Aps_Result result;
    long entryPosition;
    long fieldPosition;
    char *fileBuffer = NULL;
    int fileBufferSize = APSCFG_FILE_BUFFER_SIZE;
    char *name;
    char *value;
    int numValues;
    char **separatedValues = NULL;
    char **newArray;

    ASSERT(fp != NULL);
    ASSERT(model != NULL);

    /* Save position of the start of this database entry. */
    entryPosition = ftell(fp);

    /* Allocate memory to hold temporary data read from file. */
    fileBuffer = malloc(fileBufferSize);
    if (fileBuffer == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }
    
    /* Create a new model object. */
    result = ModelCreate(model);
    if (result != APS_SUCCESS)
        goto cleanup;

    for(;;)
    {
        fieldPosition = ftell(fp); /* saving current option position...*/

        if(fgets(fileBuffer, fileBufferSize, fp) == NULL)
            break;

        if(fileBuffer[0] != '#') /* We look only if not a comment*/ 
        {
            /* Check whether we have reached the end of this entry. */
            if(strncmp(fileBuffer, "endrecord", 9) == 0)
                break;

            /* Go to start of option.*/
            fseek(fp, fieldPosition, SEEK_SET);

            result = ModelDbParseCurField(database, &fileBuffer,
                                          &fileBufferSize, &name, &value);
            if (result == APS_NOT_FOUND) {
                /* Skip to the next field if this one couldn't be parsed. */
                continue;
            } else if (result != APS_SUCCESS) {
                /* A more serious failure has occurred, so bail out. */
                goto cleanup;
            }

            /* Store this field as a property of the model object. */

            /* Determine whether this is a string or list field. */
            if (value[0] == '[') {
                /* This is a list field. */
                
                /* Determine the number of elements in the list. */
                numValues = ModelDbGetNumberOfValues(value, strlen(value));
                
                if (numValues > 0) {
                    /* Allocate space to hold pointers to the individual
                     * values.
                     */
                    newArray = (char **)realloc(separatedValues,
                        numValues * sizeof(char *));
                    if (newArray == NULL) {
                        result = APS_OUT_OF_MEMORY;
                        goto cleanup;
                    }
                    separatedValues = newArray;
                    
                    /* Parse the individual values out of the value string. */
                    ModelDbGetMultipleEntriesFromBuffer(value,
                                                        separatedValues,
                                                        numValues);
                                                        
                    /* Set this property in the model object. */
                    result = Aps_SetPropertyStrArray(*model, name,
                        (const char **)separatedValues, numValues);
                    if (result != APS_SUCCESS)
                        goto cleanup;
                }
            } else {
                /* This is a simple string field. */
                result = Aps_SetPropertyString(*model, name, value);
                if (result != APS_SUCCESS)
                    goto cleanup;
            }
        }
    }

    /* We have successfully setup a model object for the caller. */
    result = APS_SUCCESS;

cleanup:
    if (result != APS_SUCCESS && *model != NULL) {
        Aps_ReleaseHandle(*model);
        *model = NULL;
    }

    if (fileBuffer != NULL) {
        free(fileBuffer);
    }
    
    if (separatedValues != NULL) {
        free(separatedValues);
    }
    
    /* Before returning, we always reset the file pointer back to the beinning
     * of this entry.
     */
    fseek(fp, entryPosition, SEEK_SET);

    return result;
}

/* ---------------------------------------------------------------------------
 * ModelDbGetNumberOfValues()
 *
 * This function is to find out how many values are 
 * there in a buffer. Used only when there are multiple 
 * values for an option. Not to be used outside this file.
 *
 * Parameters: buffer - The buffer to examine.
 *
 *             len    - The number of characters in the buffer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static int ModelDbGetNumberOfValues(char* buffer, int len)
{
    int i;
    int left=0, right=0;
   
    for( i = 0; i < len; i++)
    {
        if( buffer[i] == '[')
            left++;
        if ( buffer[i] == ']')
            right++;
    }   
    if ( left == right)
       return left;
    return 0;
}

/* ---------------------------------------------------------------------------
 * ModelDbGetMultipleEntriesFromBuffer()
 *
 * Used to extract multiple values for an option.
 *
 * Parameters: buffer - The buffer to parse.
 *
 *             saveIn - The array of values.
 *
 *             number - The number of entries to copy.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static void ModelDbGetMultipleEntriesFromBuffer(char* buffer, char** saveIn,
                                                int number)
{
    int length = 0;  /* length of an entry. */
    int startAt = 0; /* Entry starts at. */
    int copied = 0;  /* Total entries extracted from the buffer. */
    int i = 0;       /* current index */
    
    while ( copied != number )
    {
        if( buffer[i] == '[')
        {
            startAt = i;
            length = 0;
        }
        else if ( buffer[i] == ']')
        {
            /* from 'startAt' to this position we have one value.
             * We have the length too.
             */
             saveIn[copied] = &buffer[startAt + 1];
             buffer[startAt + 1 + length] = '\0';
             startAt = i + 1;
             length = 0;
             copied++;
        }
        if( i > startAt)
        {
            length++;
        }
        i++; /* index for buffer. */
    } 
}
