/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: pointerarray.c
 *
 * Description: Manages an array of pointers to arbitrary objects.
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "pointerarray.h"

DEBUG_CHANNEL_DEFAULT(mem)

/* Information associated with a PointerArray. */
typedef struct {
    int numElements;
    void **elements;
} PointerArrayObject;

/* ---------------------------------------------------------------------------
 * PtrArrCreate()
 *
 * Creates a new pointer array.
 *
 * Parameters: array - The address of a PointerArray to receive a reference
 *                     to the new pointer array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result PtrArrCreate(PointerArray * array)
{
    PointerArrayObject *arrayObject;

    ASSERT(array != NULL);

    /* Initialize the PointerArray to NULL for more code predictibility on
     * failure.
     */
    *array = NULL;

    /* Allocate space for the pointer array object. */
    arrayObject = malloc(sizeof(PointerArrayObject));
    if (arrayObject == NULL)
        return APS_OUT_OF_MEMORY;

    /* Initialize the members of this object. */
    arrayObject->numElements = 0;
    arrayObject->elements = NULL;

    /* Provide the caller with a reference to this new pointer array. */
    *array = (PointerArray)arrayObject;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PtrArrDelete()
 *
 * Deallocates a PointerArray. Note that deleting a PointerArray does not
 * delete whatever the elements are pointing at.
 *
 * Parameters: array - The pointer array to delete.
 *
 *     Return: None.
 */
void PtrArrDelete(PointerArray array)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;

    ASSERT(arrayObject != NULL);

    /* Deallocate the array of pointers, if it was ever allocated. */
    if (arrayObject->elements != NULL)
        free(arrayObject->elements);

    /* Deallocate the memory used by the associated PointerArrayObject. */
    free(arrayObject);
}

/* ---------------------------------------------------------------------------
 * PtrArrAdd()
 *
 * Adds an element to the end of the pointer array.
 *
 * Parameters: array   - The pointer array to manipulate.
 *
 *             element - The pointer to add to the end of the array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result PtrArrAdd(PointerArray array, void *element)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;

    ASSERT(arrayObject != NULL);

    /* This can be implemented in terms of the more generic PtrArrInsertAt()
     * operation.
     */
    return PtrArrInsertAt(array, arrayObject->numElements, element);
}

/* ---------------------------------------------------------------------------
 * PtrArrGetSize()
 *
 * Obtains the number of elements in a PointerArray.
 *
 * Parameters: array - The pointer array to query.
 *
 *     Return: The number of elements in this array.
 */
int PtrArrGetSize(PointerArray array)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;

    ASSERT(arrayObject != NULL);

    return arrayObject->numElements;
}

/* ---------------------------------------------------------------------------
 * PtrArrGetAt()
 *
 * Obtains the pointer stored at the specified index in the array.
 *
 * Parameters: array - The pointer array to manipulate.
 *
 *             index - 0-based index of the element to retrieve.
 *
 *     Return: The value of the specified element.
 */
void *PtrArrGetAt(PointerArray array, int index)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;

    ASSERT(arrayObject != NULL);
    ASSERT(index >= 0);
    ASSERT(index < arrayObject->numElements);

    return arrayObject->elements[index];
}

/* ---------------------------------------------------------------------------
 * PtrArrSetAt()
 *
 * Changes the value of the specified elements in the array.
 *
 * Parameters: array   - The pointer array to manipulate.
 *
 *             index   - The 0-based index of the element to change.
 *
 *             element - The new value for this element.
 *
 *     Return: None.
 */
void PtrArrSetAt(PointerArray array, int index, void *element)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;

    ASSERT(arrayObject != NULL);
    ASSERT(index >= 0);
    ASSERT(index < arrayObject->numElements);

    arrayObject->elements[index] = element;
}

/* ---------------------------------------------------------------------------
 * PtrArrInsertAt()
 *
 * Inserts a new element at the specified position in the array, moving any
 * subsequent elements down one position in the array.
 *
 * Parameters: array   - The pointer array to manipulate.
 *
 *             index   - 0-based index of where the new element will be
 *                       inserted. This can range between the first element
 *                       of the array to the position immediately following
 *                       the last element.
 *
 *             element - The pointer to be inserted into the array.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result PtrArrInsertAt(PointerArray array, int index, void *element)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;
    void *newElementBlock;

    ASSERT(arrayObject != NULL);
    ASSERT(index >= 0);
    ASSERT(index <= arrayObject->numElements);

    /* Ensure that the array of element pointers is large enough, growing
     * it if needed.
     */
    newElementBlock = realloc(arrayObject->elements,
                              (arrayObject->numElements + 1) * sizeof(void *));
    if (newElementBlock == NULL)
        return APS_OUT_OF_MEMORY;
    arrayObject->elements = newElementBlock;

    /* Shift subsequent elements down by one, if we aren't merely appending
     * to the array.
     */
    if (index < arrayObject->numElements) {
        memmove(&(arrayObject->elements[index + 1]),
                &(arrayObject->elements[index]),
                (arrayObject->numElements - index) * sizeof(void *));
    }

    /* Store the caller's pointer at the specified position in the array. */
    arrayObject->elements[index] = element;

    /* Finally, record that we have one more element in the array. */
    arrayObject->numElements ++;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PtrArrRemoveAt()
 *
 * Removes one or more elements from this array.
 *
 * Parameters: array       - The pointer array to manipulate.
 *
 *             index       - The location of the first element to remove.
 *
 *             numElements - The number of elements to remove.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
void PtrArrRemoveAt(PointerArray array, int index, int numElements)
{
    PointerArrayObject *arrayObject = (PointerArrayObject *)array;

    ASSERT(arrayObject != NULL);
    ASSERT(index >= 0);
    ASSERT(index < arrayObject->numElements);
    ASSERT(numElements >= 1);

    /* We don't want to delete more elements than exist. */
    ASSERT(index + numElements <= arrayObject->numElements);

    /* Move remaining elements up in the array, if we aren't deleting all the
     * way to the end of the array.
     */
    if (index + numElements < arrayObject->numElements) {
        memmove(&(arrayObject->elements[index]),
                &(arrayObject->elements[index + numElements]),
                (arrayObject->numElements - index - numElements)
                * sizeof(void *));
    }

    /* Adjust the number of elements in the array to reflect. */
    arrayObject->numElements -= numElements;
}
