/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: queueaccess.c
 *
 * Description: Queue access APIs (for scanning active jobs and getting
 *              particulars)
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "job.h"
#include "queue.h"

DEBUG_CHANNEL_DEFAULT(queue)

/* Local protos */
static Aps_Result CallQueueControl(Aps_PrinterHandle printerHandle,
    Aps_OperationID operationCode, void *argin, void *argout);

/* --------------------------------------------------------------------
 * Aps_QueueIssueCommand()
 *
 * Performs a single specific operation on a queue.
 *
 * Parameters: queueHandle   - Handle of print queue on which to operate
 *
 *             operationCode - ID of operation to be performed
 *
 *             reserved      - RESERVED -- leave as NULL
 *
 *     Return: Detailed Aps_Result code for the application.
 */
Aps_Result Aps_QueueIssueCommand(
    Aps_QueueHandle queueHandle, Aps_OperationID operationCode,
    void *reserved)
{
    Aps_Result result = APS_SUCCESS;
    Aps_Result xresult;
    Queue *queue;

    /* Check parameters */
    queue = QueueGetPtrFromHandle(queueHandle);
    if (! queue) return APS_INVALID_HANDLE;
    if (reserved) return APS_INVALID_PARAM; /* for now */

    /* If a filter has been selected, we'll have to do things somewhat
     * differently.
     * Right only a few support filtering so we can handle them explicitely.
     */
    if (queue->baseClass.filterHandle) {
        if ((operationCode == APS_OP_QPURGE) ||
            (operationCode == APS_OP_QPAUSEALLJOBS) ||
            (operationCode == APS_OP_QRESUMEALLJOBS)) {
            Aps_JobHandle iterator = NULL;

            while (Aps_Succeeded(xresult =
                Aps_QueueIterateJobs(queueHandle, & iterator))) {
                switch (operationCode) {
                    case APS_OP_QPURGE:
                        xresult = Aps_JobIssueCommand(iterator,
                            APS_OP_JDELETE, NULL);
                        break;
                    case APS_OP_QPAUSEALLJOBS:
                        xresult = Aps_JobIssueCommand(iterator,
                            APS_OP_JPAUSE, NULL);
                        break;
                    case APS_OP_QRESUMEALLJOBS:
                        xresult = Aps_JobIssueCommand(iterator,
                            APS_OP_JRESUME, NULL);
                        break;
                    default:
                        ASSERT(FALSE); /* hunh?? */
                }
                if (xresult != APS_SUCCESS) result = xresult;
            }
            if (xresult != APS_SUCCESS) result = xresult;
            return result;
        }
        if (queue->baseClass.filterOpts & APS_FILTER_FALLBACK) {
            result = APS_PARTIAL_SUCCESS;
        } else return APS_FILTER_NOT_SUPPORTED;
    }

    /* If we're not the global queue, we can just sent any other
     * requests down the pipe.
     */
    if (queue->printerHandle) {
        xresult = CallQueueControl(queue->printerHandle,
            operationCode, NULL, NULL);
        if (xresult != APS_SUCCESS) result = xresult;
    /* If we're the global queue, we'll have to go through all attached
     * printers.  To do so, we cache the list of printers, open them
     * in turn, perform the desired operation, then release them.
     */
    } else {
        char **printerNames;
        int    numNames;
        if (Aps_GetPrinters(& printerNames, & numNames) == APS_SUCCESS) {
            int i;
            for (i = 0; i < numNames; i++) {
                Aps_PrinterHandle printerHandle;
                xresult = Aps_OpenPrinter(printerNames[i], & printerHandle);
                if (xresult == APS_SUCCESS) {
                    xresult = CallQueueControl(printerHandle,
                        operationCode, NULL, NULL);
                    if (xresult != APS_SUCCESS) result = xresult;
                    Aps_ReleaseHandle(printerHandle);
                }
            }
            Aps_ReleaseBuffer(printerNames);
        }
    }
    return result;
}

/* --------------------------------------------------------------------
 * Aps_QueueIssueQuery()
 *
 * Performs a single specific verification request from a queue.
 *
 * Parameters: queueHandle - Handle of print queue on which to operate
 *
 *             queryCode   - ID of query command to be performed
 *
 *             output      - Result of query.
 *
 *     Return: Detailed Aps_Result code for the application.
 */
Aps_Result Aps_QueueIssueQuery(
    Aps_QueueHandle queueHandle, Aps_OperationID queryCode,
    void *output)
{
    Queue *queue;

    /* Check parameters */
    queue = QueueGetPtrFromHandle(queueHandle);
    if (! queue) return APS_INVALID_HANDLE;
    if (! output) return APS_INVALID_PARAM;
    if (queue->baseClass.filterHandle) return APS_INVALID_PARAM;
    if (! queue->printerHandle) return APS_NOT_FOUND;

    /* Dispatch operation */
    return CallQueueControl(queue->printerHandle,
        queryCode, NULL, output);
}

/* ---------------------------------------------------------------------------
 * CallQueueControl()
 *
 * Low-level sub for Queue control / queries.  Passes the message
 * back to the transport for processing.
 *
 * Parameters : printerHandle  - handle of printer to access
 *              operationCode  - one of Aps_OperationID
 *              argin          - input arguments
 *              argout         - output arguments
 *
 * Returns    : A standard Aps_Result code indicating success / failure.
 */
static Aps_Result CallQueueControl(Aps_PrinterHandle printerHandle,
    Aps_OperationID operationCode, void *argin, void *argout)
{
    Printer   *printer;
    Transport *trans;
    printer = PrinterGetPtrFromHandle(printerHandle);
    if (! printer) return APS_INVALID_HANDLE;
    trans = printer->transport;
    ASSERT(trans);

    return trans->vtbl->QueueControl(trans, printer,
        operationCode, argin, argout);
}
