/* -*- C++ -*-
   This file implements kabs toplevel widget.

   the KDE addressbook

   $ Author: Mirko Boehm $
   $ Copyright: (C) 1996-2000, Mirko Boehm $
   $ Contact: mirko@kde.org
         http://www.kde.org $
   $ License: GPL with the following explicit clarification:
         This code may be linked against any version of the Qt toolkit
         from Troll Tech, Norway. $

   $Revision: 1.48 $	 
*/

#include "kab_topwidget.h"
#include "kab_kab1importer.h"
#include <kabapi.h>
#include <qconfigDB.h>
#include "look_basic.h"
#include "look_businesscard.h"
#include "look_edit.h"
#include "widget_datanavigator.h"
#include <kmenubar.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kcombobox.h>
#include <kapp.h>
#include <kstdaccel.h>
#include <kstddirs.h>
#include <kaboutdialog.h>
#include <qtimer.h>
#include <qframe.h>
#include <qstringlist.h>
#include <qlayout.h>
#include <stdio.h>
#include <kdebug.h>

TopLevelWidget::TopLevelWidget()
  : KMainWindow(0),
    viewFrame(0),
    view(0),
    currentView(NoView),
    current(0),
    hasEmailAddresses(false),
    hasURLs(false),
    timer(new QTimer(this)),
    messages(new QStringList),
    closingdown(false)
{
  kdDebug() << "TopLevelWidget ctor: called." << endl;
  // -----
  Section* section;
  KeyValueMap *keys;
  // -----
  api=new KabAPI(this);
  if(api==0)
    {
      KMessageBox::sorry
	(this, i18n("Out of memory."),
	 i18n("General failure"));
      ::exit(-1);
    }
  // -----
  connect(api, SIGNAL(setStatus(const QString&)),
	  SLOT(setStatus(const QString&)));
  connect(timer, SIGNAL(timeout()), SLOT(statusbarTimeOut()));
  // -----
  if(!initializeInterface())
    {
      KMessageBox::sorry
	(this, i18n("Could not connect to the address database."),
	 i18n("Database error"));
      ::exit(-1);
    }
  // -----
  makeMenu();
  makeToolbar();
  makeStatusbar();
  connect(this, SIGNAL(databaseChanged()),
	  api->addressbook(), SLOT(externalChange()));
  // -----
  loadConfiguration();
  entriesChanged();
  // ----- now read file configuration saved at end of last session:
  section=api->addressbook()->configurationSection();
  if(section==0)
    { // should not happen, but the user might have modified the file manually
      kdDebug() << "TopLevelWidget ctor: configuration section corrupted."
		<< endl;
    } else {
      QCString lastCurrentKey;
      long lastViewMode;
      int index;
      KabKey kabkey;
      // -----
      keys=section->getKeys();
      if(rememberLastEntry)
	{
	  if(keys->get("LastCurrentKey", lastCurrentKey))
	    {
	      QComboBox *combo=toolBar()->getCombo(Entries);
	      CHECK_PTR(combo);
	      kdDebug() << "TopLevelWidget ctor: found last current key "
			<< (const char*) lastCurrentKey << "." << endl;
	      kabkey.setKey(lastCurrentKey);
	      if(api->addressbook()->getIndex(kabkey, index)==AddressBook::NoError)
		{
		  kdDebug() << "TopLevelWidget ctor: entry still exists, index is "
			    << index << ", displaying..." << endl;
		  combo->setCurrentItem(index); // this does not emit a signal, thus...
		  entrySelected(index);
		}
	    }
	} else {
	  if(api->addressbook()->noOfEntries()>0)
	    {
	      QComboBox *combo=toolBar()->getCombo(Entries);
	      CHECK_PTR(combo);
	      combo->setCurrentItem(0); 
	      entrySelected(0);
	    }
	}
      if(rememberLastView)
	{
	  if(keys->get("LastViewMode", lastViewMode))
	    {
	      kdDebug() << "TopLevelWidget ctor: found last view mode "
			<< lastViewMode << "." << endl;
	      
	      if(lastViewMode<=NoView || lastViewMode>=NoOfViews)
		{
		  kdDebug() << "TopLevelWidget ctor: found last view mode "
			    << "is corrupt, ignoring." << endl;
		  selectViewEdit(); // this initializes the menu, too (default mode)
		} else {
		  switch(lastViewMode)
		    {
		    case BusinessCard:
		      selectViewBC();
		      break;
		    case Editing:
		      selectViewEdit();
		      break;
		    default:
		      selectViewEdit();
		    }
		}
	    }
	} else {
	  selectViewEdit();
	}
    }
  // -----
  setStatus(i18n("Welcome to kab 2.0"));
  kdDebug() << "TopLevelWidget ctor: done." << endl;
  modified=false;
}

TopLevelWidget::~TopLevelWidget()
{
  Section* section;
  KeyValueMap *keys;
  kdDebug() << "TopLevelWidget::destroyMe: called." << endl;
  // -----
  if(api!=0)
    {
      // ----- first save configuration settings for this file:
      section=api->addressbook()->configurationSection();
      if(section==0)
	{ // should not happen, but the user might have modified the file manually
	  kdDebug() << "TopLevelWidget::destroyMe: configuration section corrupted. "
		    << "Unable to save state." << endl;
	} else {
	  keys=section->getKeys();
	  if(keys->insert("LastCurrentKey", current->getKey(), true))
	    {
	      kdDebug() << "TopLevelWidget::destroyMe: saved last current key "
			<< (const char*) current->getKey() << "." << endl;
	    } else {
	      kdDebug() << "TopLevelWidget::destroyMe: unable to store last current key."
			<< endl;
	    } 
	  if(keys->insert("LastViewMode", (long int) currentView, true))
	    {
	      kdDebug() << "TopLevelWidget::destroyMe: stored last view mode "
			<< currentView << "." << endl;
	    } else {
	      kdDebug() << "TopLevelWidget::destroyMe: unable to store last view mode."
		    << endl;
	    }
	}
      // ----- save modifications:
      save();
      // ----- now shut down the KabAPI connection: 
      delete api;
      api=0;
    }
  if(current!=0)
    {
      delete current;
      current=0;
    }
  kdDebug() << "TopLevelWidget::destroyMe: done." << endl;
}

void TopLevelWidget::makeMenu()
{
  // ----- the file menu:
  file=new QPopupMenu;
  CHECK_PTR(file);
  // ----- the import (sub-) menu (more should follow): 
  import=new QPopupMenu;
  CHECK_PTR(import);
  import->insertItem(i18n("&KDE 1 address book"), this, 
		     SLOT(importKab1Addressbook()));
  file->insertItem(i18n("&Create new database..."), this,
		   SLOT(createNew()), KStdAccel::openNew());
  file->insertItem(i18n("&Open..."), this,
		   SLOT(loadDatabaseFile()), KStdAccel::open());
  file->insertItem(i18n("Open default &database"), this,
		   SLOT(loadDefaultDatabase()));
  idFileSave=file->insertItem(i18n("&Save"), this,
			      SLOT(save()), KStdAccel::save());
  file->insertSeparator();
  file->insertItem(i18n("&Import"), import);
  file->insertSeparator();
  file->insertItem(i18n("&Quit"), this, SLOT(quit()), KStdAccel::quit());
  // ----- the edit menu:
  edit=new QPopupMenu;
  idEditAdd=edit->insertItem(i18n("&Add entry..."), this, SLOT(add()), CTRL+Key_A);
  idEditEdit=edit->insertItem
    (i18n("&Edit entry..."), this, SLOT(editEntrySlot()), CTRL+Key_E);
  idEditRemove=edit->insertItem
    (i18n("&Remove entry"), this, SLOT(remove()), CTRL+Key_R);
  edit->insertSeparator();
  idEditMail=edit->insertItem
    (i18n("Send an e&mail"), this, SLOT(mail()), CTRL+Key_M);
  idEditBrowse=edit->insertItem
    (i18n("&Browse"), this, SLOT(browse()), CTRL+Key_B);
  edit->insertSeparator();
  edit->insertItem(i18n("Enable all &messages"), this,
		   SLOT(enableAllMessages()));
  edit->insertItem(i18n("&Configure this file..."), this,
		   SLOT(configureFile()));
  edit->insertSeparator();
  edit->insertItem(i18n("&Preferences for the KDE address book..."), this,
		   SLOT(configureKab()));
  // ----- the view menu:
  menuview=new QPopupMenu;
  idViewBC=menuview->insertItem(i18n("&Business card"), this,
				SLOT(selectViewBC()));
  idViewEdit=menuview->insertItem(i18n("&Editing"), this,
				  SLOT(selectViewEdit()));
  menuview->setCheckable(true);
  selectViewEdit(); // default look
  // -----
  menuBar()->insertItem(i18n("&File"), file);
  menuBar()->insertItem(i18n("&Edit"), edit);
  menuBar()->insertItem(i18n("&View"), menuview);
  menuBar()->insertItem
    (i18n("&Help"), helpMenu(QString::null, false));
}

void TopLevelWidget::makeToolbar()
{
  QStringList dummy;
  // -----
  toolBar()->insertButton(BarIcon("filenew"), NewEntry, true,
			  i18n("New entry"));
  toolBar()->insertButton(BarIcon("edit"), EditEntry, true,
			  i18n("Edit entry"));
  toolBar()->insertButton(BarIcon("editdelete"), DeleteEntry, true,
			  i18n("Delete entry"));
  toolBar()->insertLineSeparator();
  toolBar()->insertButton(BarIcon("start"), FirstEntry, true,
			  i18n("First entry"));
  toolBar()->insertButton(BarIcon("back"), PrevEntry, true,
			  i18n("Previous entry"));
  toolBar()->insertCombo(dummy, Entries, false,
			 SIGNAL(activated(int)),
			 this, SLOT(entrySelected(int)));
  toolBar()->setItemAutoSized(Entries);
  toolBar()->insertButton(BarIcon("forward"), NextEntry, true,
			  i18n("Next entry"));
  toolBar()->alignItemRight(NextEntry);
  toolBar()->insertButton(BarIcon("finish"), LastEntry, true,
			  i18n("Last entry"));
  toolBar()->alignItemRight(LastEntry);
  connect(toolBar(), SIGNAL(clicked(int)), SLOT(action(int)));
}

void TopLevelWidget::action(int i)
{
  QComboBox *combo=toolBar()->getCombo(Entries);
  CHECK_PTR(combo);
  int index;
  // -----
  switch(i)
    {
    case NewEntry: // add
      add();
      break;
    case EditEntry: // edit
      editEntrySlot();
      break;
    case DeleteEntry: // delete
      remove();
      break;
    case FirstEntry:
      index=0;
      combo->setCurrentItem(index);
      entrySelected(index);
      break;
    case PrevEntry:
      index=QMAX(0, combo->currentItem()-1);
      combo->setCurrentItem(index);
      entrySelected(index);
      break;
    case NextEntry:
      index=QMIN(combo->count()-1, combo->currentItem()+1);
      combo->setCurrentItem(index);
      entrySelected(index);
      break;
    case LastEntry:
      index=combo->count()==0 ? 0 : combo->count()-1;
      combo->setCurrentItem(index);
      entrySelected(index);
      break;
    default:
      kdDebug() << "TopLevelWidget::action: unknown action %i"
		<< i << endl;
    };
}


void TopLevelWidget::makeStatusbar()
{
  statusBar()->insertItem("20000/20000", Number);
  statusBar()->insertItem("", Text, 1000, true);
}

bool TopLevelWidget::initializeInterface()
{
  if(api->init()!=AddressBook::NoError)
    {
      KMessageBox::sorry
	(this, i18n("The database interface could not be created."),
	 i18n("Database error"));
      return false;
    } else {
      connect(api->addressbook(), SIGNAL(changed()),
	      SLOT(entriesChanged()));
      return true;
    }
}

void TopLevelWidget::entrySelected(int index)
{
  kdDebug() << "TopLevelWidget::entrySelected: called ( "
	    << index << ")" << endl;
  // -----
  AddressBook::Entry entry;
  KabKey key;
  int number;
  QString text;
  // ----- store user changes:
  if(closingdown) return;
  if(modified) // the current entry has been modified
    { // WORK_TO_DO: respect "Query save on change" configuration setting
      if(current!=0) // make sure non-debug versions also work stable
	{
	  if(KMessageBox::questionYesNo
	     (this,
	      i18n("You changed this entry.\nSave changes?\n"
		   "(Unsaved changes will be lost.)"),
	      i18n("Save changes?"))==KMessageBox::Yes)
	    {
	      save();
	    }
	}
    }
  // -----
  number=api->addressbook()->noOfEntries();
  kdDebug() << "TopLevelWidget::entrySelected: " << number 
	    <<" entries." << endl;
  if(number!=0)
    {
      if(api->addressbook()->getKey(index, key)!=AddressBook::NoError)
	{
	  kdDebug() << "TopLevelWidget::entrySelected: no such entry " 
		    << index << endl;
	}
      if(api->addressbook()->getEntry(key, entry)!=AddressBook::NoError)
	{
	  kdDebug() << "TopLevelWidget::entrySelected: cannot access"
		    << " existing entry." << endl;
	}
    } else {
      kdDebug() << "TopLevelWidget::entrySelected: no entries." << endl;
    }
  // -----
  if(number==0)
    {
      text=i18n("No entries.");
    } else {
      text=text.sprintf("%i/%i", index+1, number);
    }
  statusBar()->changeItem(text, Number);
  // -----
  view->setEntry(entry);
  hasEmailAddresses=entry.emails.count()>0;
  hasURLs=entry.URLs.count()>0;
  modified=false; // finally reset modification mark
  if(current!=0)
    { // make sure the old key is deleted:
      delete current; current=0;
    }
  current=new KabKey(key);
  // ----- enable/disable buttons:
  if(number<2)
    {
      toolBar()->setItemEnabled(FirstEntry, false);
      toolBar()->setItemEnabled(PrevEntry, false);
      toolBar()->setItemEnabled(LastEntry, false);
      toolBar()->setItemEnabled(NextEntry, false);
    } else {
      if(index==0)
	{
	  toolBar()->setItemEnabled(FirstEntry, false);
	  toolBar()->setItemEnabled(PrevEntry, false);
	} else {
	  toolBar()->setItemEnabled(FirstEntry, true);
	  toolBar()->setItemEnabled(PrevEntry, true);
	}
      if(index==number-1)
	{
	  toolBar()->setItemEnabled(LastEntry, false);
	  toolBar()->setItemEnabled(NextEntry, false);
	} else {
	  toolBar()->setItemEnabled(LastEntry, true);
	  toolBar()->setItemEnabled(NextEntry, true);
	}
    }
  enableButtons();
  if(showAssembledNameInWindowTitle)
    {
      bool initials=false; // WORK_TO_DO: read from prefs
      bool reverse=false; // WORK_TO_DO: read from prefs
      QString text;
      api->addressbook()->literalName(entry, text, reverse, initials);
      setCaption(text);
    } else {
      setCaption("");
    }
  // -----
  kdDebug() << "TopLevelWidget::entrySelected: done (" << index
	    << ")." << endl;
}

void TopLevelWidget::entriesChanged()
{ // this signal is emitted from AddressBook::updateEntriesMap
  kdDebug() << "TopLevelWidget::entriesChanged: called." << endl;
  // -----
  int current;
  QComboBox *combo=toolBar()->getCombo(Entries);
  QStringList headlines;
  // -----
  if(combo==0) return;
  if(api->addressbook()->getListOfNames(&headlines, true, false)
     !=AddressBook::NoError)
    {
      return;
    }
  current=combo->currentItem();
  if(current<0) current=0;
  combo->clear();
  combo->insertStringList(headlines);
  current=QMIN(combo->count()-1, current);
  combo->setCurrentItem(current);
  entrySelected(current);
  enableButtons();
  // -----
  kdDebug() << "TopLevelWidget::entriesChanged: done." << endl;  
}

void TopLevelWidget::importKab1Addressbook()
{ // WORK_TO_DO: Proof-read messages
  // ###########################################################################
  Kab1Importer importer(api, this);
  connect(&importer, SIGNAL(setStatus(const QString&)),
	  SLOT(setStatus(const QString&)));
  if(importer.exec())
    {
      entriesChanged();
    }
  // ###########################################################################
}

void TopLevelWidget::setStatus(const QString& text)
{
  // ###########################################################################
  messages->append(text);
  if(!timer->isActive())
    {
      timer->start(0);
    } else {
      timer->start(500);
    }
  // ###########################################################################
}

void TopLevelWidget::statusbarTimeOut()
{
  // ###########################################################################
  if(!messages->isEmpty())
    {
      statusBar()->changeItem(messages->first(), Text);
      messages->remove(messages->begin());
      if(messages->isEmpty())
	{
	  timer->start(5000, true);
	} else {
	  timer->start(1000, true);
	}
    } else {
      statusBar()->changeItem("", Text);
    }
  // ###########################################################################
}

void TopLevelWidget::loadDefaultDatabase()
{
  // ###########################################################################
  QString path;
  // -----
  path=api->addressbook()->getStandardFilename();
  if(!path.isEmpty())
    {
      if(api->addressbook()->load(path)!=AddressBook::NoError)
	{
	  KMessageBox::sorry
	    (this, i18n("The standard file could not be loaded."),
	     i18n("File error"));
	}
    }
  modified=false;
  enableButtons();
  // ###########################################################################
}

void TopLevelWidget::selectViewBC()
{
  menuview->setItemChecked(idViewBC, true);
  menuview->setItemChecked(idViewEdit, false);
  createView(BusinessCard);
  emit(setStatus(i18n("Business card view.")));
}

void TopLevelWidget::selectViewEdit()
{
  menuview->setItemChecked(idViewBC, false);
  menuview->setItemChecked(idViewEdit, true);
  createView(Editing);
  emit(setStatus(i18n("Editing view.")));
}

void TopLevelWidget::enableAllMessages()
{
  if(KMessageBox::questionYesNo
     (this,
      i18n("This will re-enable all messages you disabled before.\n"
	   "Continue?"),
      i18n("Enable messages?"))==KMessageBox::Yes)
    {
      KMessageBox::enableAllMessages();
    }
}

void TopLevelWidget::add()
{
  KabKey key;
  AddressBook::Entry entry; // WORK_TO_DO: possibly use templates or default entries?
  int index;
  // -----
  if(editEntry(entry))
    {
      switch(api->add(entry, key, true))
	{
	case AddressBook::NoError:
	  // ----- first save the database:
	  emit(setStatus(i18n("New entry added.")));
	  emit(setStatus(i18n("Saving new entry.")));
	  save();
	  emit(i18n("Done."));
	  emit(databaseChanged());
	  entriesChanged();
	  // ----- make the new entry the current (and visible):
	  if(api->addressbook()->getIndex(key, index)==AddressBook::NoError)
	    {
	      bool dummy;
	      QComboBox *combo;
	      combo=toolBar()->getCombo(Entries);
	      CHECK_PTR(combo);
	      combo->setCurrentItem(index); // this does not emit a signal, thus...
	      dummy=modified; modified=false;
	      entrySelected(index);
	      modified=dummy;
	    }
	  break;
	case AddressBook::PermDenied:
	  KMessageBox::information
	    (this,
	     i18n("Cannot add the new entry.\n"
		  "Permission denied.\n"
		  "You do not have writing permissions for this file."),
	     i18n("Error"));
	  return;
	default:
	  KMessageBox::information
	    (this,
	     i18n("Cannot add the new entry.\n"
		  "Unknown cause."),
	     i18n("Error"));
	  return;
	}
    } else {
      emit(setStatus(i18n("No entry added.")));
    }
}

void TopLevelWidget::remove()
{
  removeCurrentEntry();
}

bool TopLevelWidget::removeCurrentEntry(bool quiet)
{
  bool queryOnDelete=true; // WORK_TO_DO: respect configuration settings
  bool doIt=true;
  // -----
  if(queryOnDelete && !quiet)
    {
      doIt=KMessageBox::questionYesNo
	(this,
	 i18n("Really remove this entry?"),
	 i18n("Question"))==KMessageBox::Yes;
    }
  if(doIt)
    {
      if(api->remove(*current)==AddressBook::NoError)
	{
	  emit(setStatus(i18n("Entry deleted.")));
	  save();
	  emit(databaseChanged());
	  entriesChanged();
	  return true;
	} else {
	  KMessageBox::information
	    (this,
	     i18n("Could not delete the entry\n"
		  "(probably permission denied)."),
	     i18n("Error"));
	  emit(setStatus(i18n("Permission denied.")));
	  return false;
	}
    }
  return false;
}

void TopLevelWidget::createView(View v, bool recreate)
{
  bool modificationstate=modified;
  // ----- avoid deleting and creating:
  if(v==currentView && !recreate) return;
  // ----- create view frame if needed:
  //       The view frame serves as a container for the view widgets
  //       as KTMainWindow seems to be picky about changing the main
  //       widget.
  if(viewFrame==0)
    { // this should be called only once:
      viewFrame=new QFrame(this);
      QBoxLayout *viewLayout=new QBoxLayout
	(viewFrame, QBoxLayout::TopToBottom);
      viewLayout->setAutoAdd(true);
      setCentralWidget(viewFrame);
    }
  // -----
  AddressBook::Entry entry;
  // -----
  if(view!=0)
    {
      view->getEntry(entry);
      delete view;
      view=0;
    }
  // -----
  switch(v)
    {
    case BusinessCard:
      view=new KABBusinessCard(api, viewFrame);
      break;
    case Editing:
      view=new KABEditLook(api, viewFrame);
      connect(view, SIGNAL(entryChanged()), SLOT(entryChangedSlot()));
      break;
    default:
      kdDebug() << "TopLevelWidget::createView: unknown kind of view." << endl;
      view=new KABBusinessCard(api, viewFrame);
    }
  connect(view, SIGNAL(sendEmail(const QString&)), SLOT(mail(const QString&)));
  connect(view, SIGNAL(browse(const QString&)), SLOT(browse(const QString&)));
  connect(view, SIGNAL(saveMe()), SLOT(saveMeSlot()));
  view->setEntry(entry);
  viewFrame->setMinimumSize(320, 200);
  //  setCentralWidget(view);
  view->show();
  currentView=v;
  modified=modificationstate;
  enableButtons();
}

void TopLevelWidget::entryChangedSlot()
{
  modified=true;
}

void TopLevelWidget::saveMeSlot()
{
  save();
  enableButtons();
}

void TopLevelWidget::quit()
{
  register bool GUARD; GUARD=false;
  kdDebug() << "TopLevelWidget::quit: quitting." << endl;  
  closingdown=true; // important!
  kapp->quit();
}

void TopLevelWidget::showAboutApplication(void)
{
  QPixmap logo;
  QString path;
  KAboutDialog dialog(this);
  // -----
  path=locate("appdata", "pics/addressbook_logo.png");
  if(path.isEmpty())
    {
      kdDebug() << "TopLevelWidget::aboutKAB: cannot locate"
		<< " ressources." << endl; 
    } else {
      if(logo.load(path))
	{
	  dialog.setLogo(logo);
	}
    }
  dialog.setCaption("About KDE address book");
  dialog.setVersion("KDE address book 2.0alpha");
  dialog.setAuthor("Mirko Boehm", "mirko@kde.org", "", "Initial developer.");
  dialog.adjust();
  dialog.exec();
}

void TopLevelWidget::mail()
{
  mail(QString::null);
}

void TopLevelWidget::mail(const QString&)
{ // the given URL is ignored!
  AddressBook::Entry entry;
  int x;
  KDialogBase dialog
    (this, 0, true, i18n("Send an email"), KDialogBase::Ok|KDialogBase::Cancel,
     KDialogBase::Ok, true);
  QListBox lb(&dialog);
  dialog.setMainWidget(&lb);
  // ----- get the entry:
  if(api->addressbook()->getEntry(*current, entry)==AddressBook::NoError)
    {
      if(entry.emails.count()==0)
	{
	  emit(setStatus(i18n("No email addresses.")));
	  kapp->beep();
	  return;
	}
      if(entry.emails.count()>1)
	{
	  lb.insertStringList(entry.emails);
	  lb.setCurrentItem(0);
	  x=QMAX((int)(lb.maxItemWidth()*1.10), lb.sizeHint().width());
	  lb.setMinimumSize(x, QMAX(x/2, lb.sizeHint().height()));
	  if(dialog.exec())
	    { // ----- open the composer:
	      kapp->invokeMailer(lb.currentText(), QString::null);
	    } else {
	      emit(setStatus(i18n("Rejected.")));
	      kapp->beep();
	      return;
	    }
	} else {
	  kapp->invokeMailer(entry.emails.first(), QString::null);
	}
    } else {
      kdDebug() << "TopLevelWidget::mail: cannot find current entry." << endl;
    }
}

void TopLevelWidget::browse()
{
  browse(QString::null);
}

void TopLevelWidget::browse(const QString&)
{ // the given URL is ignored!
  AddressBook::Entry entry;
  int x;
  KDialogBase dialog
    (this, 0, true, i18n("Browse to ..."), KDialogBase::Ok|KDialogBase::Cancel,
     KDialogBase::Ok, true);
  QListBox lb(&dialog);
  dialog.setMainWidget(&lb);
  // ----- get the entry:
  if(api->addressbook()->getEntry(*current, entry)==AddressBook::NoError)
    {
      if(entry.URLs.count()==0)
	{
	  emit(setStatus(i18n("No URLs.")));
	  kapp->beep();
	  return;
	}
      if(entry.URLs.count()>1)
	{
	  lb.insertStringList(entry.URLs);
	  lb.setCurrentItem(0);
	  x=QMAX((int)(lb.maxItemWidth()*1.10), lb.sizeHint().width());
	  lb.setMinimumSize(x, QMAX(x/2, lb.sizeHint().height()));
	  if(dialog.exec())
	    { // ----- open the composer:
	      kapp->invokeBrowser(lb.currentText());
	    } else {
	      emit(setStatus(i18n("Rejected.")));
	      kapp->beep();
	      return;
	    }
	} else {
	  kapp->invokeBrowser(entry.URLs.first());
	}
    } else {
      kdDebug() << "TopLevelWidget::browse: cannot find current entry."
		<< endl;
    }
}

void TopLevelWidget::enableButtons()
{
  int entries=0;
  switch(api->addressbook()->getState())
    {
    case AddressBook::NoError: // a file has been loaded without errors
      // ----- menu items:
      entries=api->addressbook()->noOfEntries();
      edit->setItemEnabled(idEditAdd, true);
      edit->setItemEnabled
	(idEditEdit, currentView!=Editing && entries>0);
      edit->setItemEnabled(idEditRemove, entries>0);
      edit->setItemEnabled(idEditMail, hasEmailAddresses && entries>0);
      edit->setItemEnabled(idEditBrowse, hasURLs && entries>0);
      // ----- toolbar buttons:
      toolBar()->setItemEnabled(EditEntry, entries>0 && currentView!=Editing);
      toolBar()->setItemEnabled(DeleteEntry, entries>0);
      view->setEnabled(entries>0);
      break;
    case AddressBook::NoFile: // no file has been loaded
    case AddressBook::PermDenied: // error loading the file
      entries=0;
      edit->setItemEnabled(idEditAdd, false);
      edit->setItemEnabled(idEditEdit, false);
      edit->setItemEnabled(idEditRemove, false);
      edit->setItemEnabled(idEditMail, false);
      edit->setItemEnabled(idEditBrowse, false);
      // ----- toolbar buttons:
      toolBar()->setItemEnabled(NewEntry, false);
      toolBar()->setItemEnabled(EditEntry, false);
      toolBar()->setItemEnabled(DeleteEntry, false);
      view->setEnabled(false);
      break;
    default:
      kdDebug() << "TopLevelWidget::enableButtons: "
		<< "Unknown API state in switch." << endl;
    }
}

bool TopLevelWidget::queryClose()
{ // WORK_TO_DO: to query or not to query? There is no unsaved data, but we have to do some cleanup.
  // quit();
  if(modified)
    {
      kdDebug() << "TopLevelWidget::queryClose: saving modified database."
		<< endl;
      save();
    }
  return true;
}

void TopLevelWidget::loadConfiguration()
{
  Section* config;
  KeyValueMap *prefs;
  QConfigDB *db;
  // -----
  db=api->addressbook()->getConfig();
  connect(db, SIGNAL(fileChanged()), SLOT(loadConfiguration()));
  db->watch(true);
  if(db->get("config", config))
    {
      prefs=config->getKeys();
    } else { // set defaults
      showAssembledNameInWindowTitle=true;
      rememberLastView=true;
      rememberLastEntry=true;
      return;
    }
  if(prefs->get("RememberLastView", rememberLastView))
    {
      if(rememberLastView)
	{
	  emit(setStatus(i18n("Restoring last view.")));
	} else {
	  emit(setStatus(i18n("Not restoring last view.")));
	}
    } else {
      kdDebug() << "TopLevelWidget::loadConfiguration: no preference setting"
		<< " for RememberLastView" << endl;
      rememberLastView=true;
    }
  if(prefs->get("RememberLastEntry", rememberLastEntry))
    {
      if(rememberLastEntry)
	{
	  emit(setStatus(i18n("Remembering last entry.")));
	} else {
	  emit(setStatus(i18n("Not remembering last entry.")));
	}
    } else {
      kdDebug() << "TopLevelWidget::loadConfiguration: no preference setting"
		<< " for RememberLastEntry" << endl;
      rememberLastEntry=true;
    }
  if(prefs->get("ShowNameInWindowTitle", showAssembledNameInWindowTitle))
    {
      if(showAssembledNameInWindowTitle)
	{
	  emit(setStatus(i18n("Displaying assembled name in window title.")));
	} else {
	  emit(setStatus(i18n("Not displaying assembled name in window title.")));
	}
    } else {
      kdDebug() << "TopLevelWidget::loadConfiguration: no preference setting"
		<< " for ShowNameInWindowTitle" << endl;
      showAssembledNameInWindowTitle=true;
    }
  if(prefs->get("CreateBackupOnStartup", createBackupOnStartup))
    {
      if(createBackupOnStartup)
	{
	  emit(setStatus(i18n("Creating backup file on startup.")));
	} else {
	  emit(setStatus(i18n("Not creating a backup file on startup.")));
	}
    } else {
      kdDebug() << "TopLevelWidget::loadConfiguration: no preference setting"
		<< " for CreateBackupOnStartup" << endl;
      createBackupOnStartup=true;
    }
}

#include "kab_topwidget.moc"
