/*
 * daemondock.cpp
 * Copyright (C) 1999 Paul Campbell <paul@taniwha.com>
 *
 * This file contains the docked widget for the laptop battery monitor
 *
 * $Id: daemondock.cpp,v 1.9 2000/07/15 03:49:06 campbell Exp $
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <qtooltip.h>    
#include <kapp.h>
#include <stdio.h>
#include <stdlib.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kglobal.h>
#include <kpopupmenu.h>
#include <dcopclient.h>
#include "daemondock.h"
#include "portable.h"

laptop_dock::laptop_dock( laptop_daemon* parent)
  : KDockWindow()
{

    p = parent;
    current_code = -1;
      // popup menu for right mouse button
    QPopupMenu* popup = contextMenu();

    int can_standby = laptop_portable::has_standby();
    int can_suspend = laptop_portable::has_suspend();
    popup->insertItem(i18n("Setup..."), this, SLOT(invokeSetup()));
    if (can_standby && can_suspend) {
    	popup->insertSeparator();
    	if (can_standby) popup->insertItem(i18n("Standby..."), this, SLOT(invokeStandby()));
	if (can_suspend) popup->insertItem(i18n("&Lock && Suspend..."), this, SLOT(invokeLockSuspend()));
    	if (can_suspend) popup->insertItem(i18n("&Suspend..."), this, SLOT(invokeSuspend()));
    }
}


void laptop_dock::noop()
{
}
void laptop_dock::mousePressEvent( QMouseEvent *event )
{
 	if(event->button() == LeftButton) {
		QPopupMenu *popup = new QPopupMenu(0, "popup");

		if (!p->exists) {
			popup->insertItem(i18n("Power Manager Not Found"), this, SLOT(noop()));
		} else {
			QString tmp;

			if (p->left < 0) {	// buggy BIOS
				QString cp3 = i18n("charged");
				QString num2; num2.setNum(p->val);
				tmp = num2+"% "+cp3;
			} else {
				QString cp3 = i18n("minutes left");
				QString num2; num2.setNum(p->left/60);
				QString num3; num3.setNum(p->left%60);
				      	      num3.rightJustify(2, '0');
				tmp = num2+":"+num3+" "+cp3;
			}
			popup->insertItem(tmp, this, SLOT(noop()));
			popup->setItemEnabled(0, 0);
			popup->insertSeparator();
			if (p->powered) {
				popup->insertItem(i18n("Charging"), this, SLOT(noop()));
			} else {
				popup->insertItem(i18n("Not Charging"), this, SLOT(noop()));
			}
			popup->setItemEnabled(1, 0);
		}
	
		popup->popup(QCursor::pos());
	}
	
}
void laptop_dock::mouseReleaseEvent( QMouseEvent *e )
{
    if ( !rect().contains( e->pos() ) )
        return;
 
    switch ( e->button() ) {
    case LeftButton:
        break;
    case MidButton:
        // fall through
    case RightButton:
	{
		KPopupMenu *menu = contextMenu();
        	contextMenuAboutToShow( menu );
        	menu->popup( e->globalPos() );
	}
        break;
    default:
        // nothing
        break;
    }                   
}
void laptop_dock::showEvent( QShowEvent * )
{

}
void laptop_dock::invokeStandby()
{
	laptop_portable::invoke_standby();
}

void laptop_dock::invokeSuspend()
{
	laptop_portable::invoke_suspend();
}

void laptop_dock::invokeLockSuspend()
{
  DCOPClient* client = kapp->dcopClient();
  if (client)
    {
      client->attach();
      client->send("kdesktop", "KScreensaverIface", "lock()", "");
      client->detach();
    }
  laptop_portable::invoke_suspend();
}

void laptop_dock::invokeSetup()
{
	::system("kcmshell PowerControl/battery&");
}

laptop_dock::~laptop_dock() {
}
                                      

void laptop_dock::displayPixmap()
{
	int new_code;

	if (!p->exists)
		new_code = 1;
	else if (!p->powered)
		new_code = 2;
	else
		new_code = 3;

	if (current_code != new_code) {
		current_code = new_code;

		// we will try to deduce the pixmap (or gif) name now.  it will
		// vary depending on the dock and power
		QString pixmap_name, mini_pixmap_name;
	
		if (!p->exists)
			pixmap_name = p->noBatteryIcon;
		else if (!p->powered)
			pixmap_name = p->noChargeIcon;
		else
			pixmap_name = p->chargeIcon;

		pm = SmallIcon(pixmap_name);
	}

	// at this point, we have the file to display.  so display it

	QImage image = pm.convertToImage();

	int w = image.width();
	int h = image.height();
	int count = 0;
	QRgb rgb;
	int x, y;
	for (x = 0; x < w; x++)
	for (y = 0; y < h; y++) {
		rgb = image.pixel(x, y);
		if (qRed(rgb) == 0xff &&
		    qGreen(rgb) == 0xff &&
		    qBlue(rgb) == 0xff)
			count++;
	}
	int c = (count*p->val)/100;
	if (p->val == 100) {
		c = count;
	} else
	if (p->val != 100 && c == count)
		c = count-1;

	
	if (c) {
		uint ui;
		QRgb blue = qRgb(0x00,0x00,0xff);

		if (image.depth() <= 8) {
			ui = image.numColors();		// this fix thanks to Sven Krumpke
			image.setNumColors(ui+1);
			image.setColor(ui, blue);
		} else {
			ui = 0xff000000|blue;
		}
	
		for (y = h-1; y >= 0; y--) 
		for (x = 0; x < w; x++) {
			rgb = image.pixel(x, y);
			if (qRed(rgb) == 0xff &&
		    	    qGreen(rgb) == 0xff &&
		    	    qBlue(rgb) == 0xff) {
				image.setPixel(x, y, ui);
				c--;
				if (c <= 0)
					goto quit;
			}
		}
	}
quit:

	QPixmap q;
	q.convertFromImage(image);
	setPixmap(q);
	adjustSize();

	QString tmp;
	if (!p->exists) {
		tmp = i18n("Laptop Power Management not available");
	} else
	if (p->powered) {
		if (p->val == 100) {
			QString cp1 = i18n("Plugged in");
			QString cp2 = i18n("fully charged");
			tmp = cp1+" "+cp2;
		} else {
			QString cp1 = i18n("Plugged in");
			QString cp2 = i18n("charged");
			if (p->left >= 0) {
				QString cp3 = i18n("minutes");
				QString num1; num1.setNum(p->val);
				QString num2; num2.setNum(p->left/60);
				QString num3; num3.setNum(p->left%60);
					      num3.rightJustify(2, '0');
				tmp = cp1+" - "+num1+"% "+cp2+" ("+num2+":"+num3+" "+cp3+")";
			} else {
				QString num1; num1.setNum(p->val);
				tmp = cp1+" - "+num1+"% "+cp2;
			}
		}
	} else {
		QString cp1 =  i18n("Running on batteries");
		QString cp2 =  i18n("charged");
		if (p->left >= 0) {
			QString cp3 = i18n("minutes");
			QString num1; num1.setNum(p->val);
			QString num2; num2.setNum(p->left/60);
			QString num3; num3.setNum(p->left%60);
				      num3.rightJustify(2, '0');
			tmp = cp1+" - "+num1+"% "+cp2+" ("+num2+":"+num3+" "+cp3+")";
		} else {
			QString num1; num1.setNum(p->val);
			tmp = cp1+" - "+num1+"% "+cp2;
		}
	}
	QToolTip::add(this, tmp); 
}

#include "daemondock.moc"
