/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <ctype.h>
#include <stdio.h>

#include "dmaport.h"
#include "interrupt.h"
#include "ioaddress.h"
#include "memaddress.h"
#include "resourcespinbox.h"


ResourceSpinBox::ResourceSpinBox(ResourceList *reslist, int rescount, Resource *setting,
				 QWidget *parent, 
				 const char *name) : QSpinBox(0, rescount-1, 1, parent, name),
				 		     _reslist(reslist),
                                                     _setting(setting)
{
  setValue(_initial = (_reslist) ? _reslist->find(setting) : 0);
}


/*
 * Public methods
 *****************/
 
Resource  *ResourceSpinBox::resource(int index)	// I know, this has to be elsewhere...
{
  return (_reslist) ? _reslist->at(index) : _setting;
}


/*
 * Protected methods
 ********************/
 
int  ResourceSpinBox::mapTextToValue(bool *ok)
{
  QString   text   = currentValueText();
  ulong     fromval;
  ulong     range  = 1L;
  int       value  = -1;
  Resource  *first = (_reslist) ? _reslist->first() : _setting;
  Resource  *res   = 0L;
  
  switch (first->resourceType())
  {
    case Resource::ResDMAPort    : res = new DMAPort((DMAPort&) *first);
    				   *ok = isNumeric(text.ascii(), &fromval);
     				   break;    
    case Resource::ResInterrupt  : res = new Interrupt((Interrupt&) *first);
    				   *ok = isNumeric(text.ascii(), &fromval);
     				   break;
    case Resource::ResIOAddress  : res = new IOAddress((IOAddress&) *first);
    				   *ok = isHexRange(text.ascii(), &fromval, &range);                   
    				   break;    
    case Resource::ResMemAddress : res = new MemoryAddress((MemoryAddress&) *first);
    				   *ok = isHexRange(text.ascii(), &fromval, &range);                   
    				   break;
    default                      : *ok = false;
  }
  
  if (*ok)
  {
    if (_reslist)
    {
      res->setResourceValue(fromval);
      res->setResourceRange(range);
 
      *ok = ((value = _reslist->find(res)) >= 0);
    } else value = 0;
  }
  
  if (res) delete res;
  
  return value;
}
 
 
QString  ResourceSpinBox::mapValueToText(int val)
{
  QString   valtxt;
  Resource  *res  = (_reslist) ? _reslist->at(val) : _setting;
  ulong     value = res->resourceValue();
  ulong     range = res->resourceRange();
  
  switch (res->resourceType())
  {
    case Resource::ResDMAPort    : 
    case Resource::ResInterrupt  : valtxt.setNum(value);
    				   break;
    case Resource::ResIOAddress  : valtxt.sprintf("%04x-%04x", value, value+range-1);
    				   break;
    case Resource::ResMemAddress : valtxt.sprintf("%08x-%08x", value, value+range-1);                    
    				   break;
    default                      : valtxt = "";
  }
  
  return valtxt.upper();
}


/*
 * Private methods
 ******************/
 
bool  ResourceSpinBox::isHex(const char *text, ulong *value)
{
  while (*text)
    if (!isxdigit(*text++)) return false;
  
  return (sscanf(text, "%lx", value) == 1);
}


bool  ResourceSpinBox::isHexRange(const QString& text, ulong *fromval, ulong *range)
{
  QString  fromstr, tostr;
  ulong    toval;
  int      pos;
  
  if ((pos = text.find('-') < 0)) return false;
  if (text.find('-', pos+1) >= 0) return false;
  
  fromstr = text.left(pos);
  tostr   = text.right(text.length()-pos-1);
  
  if (!isHex(fromstr.ascii(), fromval) || !isHex(tostr.ascii(), &toval)) return false;
  if (toval < *fromval)                                                return false;
  
  *range = toval-*fromval;
  
  return true;
}


bool  ResourceSpinBox::isNumeric(const char *text, ulong *value)
{
  while (*text)
    if (!isdigit(*text++)) return false;
    
  return (sscanf(text, "%lu", value) == 1);
}






			
