// (c) 2000 Peter Putzer

#include <qframe.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qgrid.h>

#include <kdebug.h>
#include <klocale.h>
#include <klineedit.h>
#include <kiconloader.h>
#include <kdualcolorbtn.h>
#include <kapp.h>
#include <kfiledialog.h>
#include <kmessagebox.h>

#include "ksv_conf.h"
#include "ksv_core.h"

#include "PreferencesDialog.h"

#define MIN_SIZE(A) A->setMinimumSize (A->sizeHint())

KSVPreferences::KSVPreferences (QWidget* parent)
  :  KDialogBase (IconList, i18n("Preferences"), Help|Ok|Apply|Cancel, Ok,
				  parent, "KSysV Preferences", true, true),
  mConfig (KSVConfig::self())
{
  setMinimumSize (sizeHint ());
  
  /**
   * Colors Page
   */
  QFrame* colorPage = addPage (i18n("Colors"), QString::null, DesktopIcon
("colorize", 32));
  mNewColors = new KDualColorButton(black, white,
                                    colorPage, "NewColors Button", this);
  mChangedColors = new KDualColorButton(black, white,
                                        colorPage, "ChangedColors Button", this);

  QLabel* newColorsLabel = new QLabel(i18n("Colors for &new entries"), colorPage);
  QLabel* changedColorsLabel = new QLabel(i18n("Colors for c&hanged entries"), colorPage);
  newColorsLabel->setBuddy(mNewColors);
  changedColorsLabel->setBuddy(mChangedColors);
  MIN_SIZE(newColorsLabel);
  MIN_SIZE(changedColorsLabel);

  QBoxLayout* colorLayout = new QVBoxLayout(colorPage, 5);
  QBoxLayout* newColorsLayout = new QHBoxLayout();
  QBoxLayout* changedColorsLayout = new QHBoxLayout();
  colorLayout->addLayout(newColorsLayout, 5);
  colorLayout->addLayout(changedColorsLayout, 5);

  newColorsLayout->addWidget(newColorsLabel, 0, AlignLeft);
  newColorsLayout->addStretch(10);
  newColorsLayout->addWidget(mNewColors, 0, AlignRight);

  changedColorsLayout->addWidget(changedColorsLabel, 0, AlignLeft);
  changedColorsLayout->addStretch(10);
  changedColorsLayout->addWidget(mChangedColors, 0, AlignRight);

  colorLayout->addStretch(2*10);


  /**
   * Paths Page
   */
  QFrame* pathPage = addPage (i18n ("Paths"), QString::null, DesktopIcon ("kfm", 32));

  QBoxLayout* pathLayout = new QVBoxLayout(pathPage, 5);

  QLabel* prefix = new QLabel(i18n("Directory containing your &services (init-scripts)"), pathPage);
  MIN_SIZE(prefix);
  pathLayout->addWidget(prefix);

  QBoxLayout* scripts = new QHBoxLayout();
  pathLayout->addLayout(scripts, 1);
  pathLayout->addSpacing(10);

  // script path
  mScriptPath = new KLineEdit(pathPage);
  prefix->setBuddy(mScriptPath);
  MIN_SIZE(mScriptPath);

  mBrowseScripts = new QPushButton(i18n("Browse..."), pathPage);
  mBrowseScripts->setAutoDefault(FALSE);
  MIN_SIZE(mBrowseScripts);

  scripts->addWidget(mScriptPath, 10);
  scripts->addWidget(mBrowseScripts, 1);

  // Runlevel Path
  QLabel* runlevelsLabel = new QLabel(i18n("Directory containing your &runlevel-directories"), pathPage);
  MIN_SIZE(runlevelsLabel);
  pathLayout->addWidget(runlevelsLabel);

  QBoxLayout* Runlevels = new QHBoxLayout();
  pathLayout->addLayout(Runlevels, 1);
  pathLayout->addStretch( 2*10 );

  mRunlevelsPath = new KLineEdit(pathPage);
  runlevelsLabel->setBuddy(mRunlevelsPath);
  MIN_SIZE(mRunlevelsPath);

  mBrowseRunlevels = new QPushButton(i18n("Browse..."), pathPage);
  mBrowseRunlevels->setAutoDefault(FALSE);
  MIN_SIZE(mBrowseRunlevels);

  Runlevels->addWidget(mRunlevelsPath, 10);
  Runlevels->addWidget(mBrowseRunlevels, 1);

  // some connections
  connect(mBrowseScripts, SIGNAL(clicked()), this, SLOT(scriptDir()));
  connect(mBrowseRunlevels, SIGNAL(clicked()), this, SLOT(runlevelDir()));

  /**
   * Miscellaneous Page
   */
  QGrid* miscPage = addGridPage (2, QGrid::Vertical, i18n("Miscellaneous"),
                                 i18n("Settings Not Fitting Anywhere Else"),
                                 DesktopIcon ("misc", 32));
  
  QPushButton* showMessages = new QPushButton (i18n ("Show All Messages"), miscPage);
  connect (showMessages, SIGNAL (clicked()), this, SLOT (showAllMessages()));

  reReadConfig();
}

KSVPreferences::~KSVPreferences ()
{
}

void KSVPreferences::scriptDir()
{
  QString tmp = KFileDialog::getExistingDirectory(mScriptPath->text(), this);

  if (!tmp.isEmpty())
    mScriptPath->setText(tmp);
  
  mScriptPath->setFocus();
}

void KSVPreferences::runlevelDir()
{
  QString tmp = KFileDialog::getExistingDirectory(mRunlevelsPath->text(), this);

  if (!tmp.isEmpty())
    mRunlevelsPath->setText(tmp);

  mRunlevelsPath->setFocus();
}

void KSVPreferences::reReadConfig()
{
  mRunlevelsPath->setText(mConfig->runlevelPath());
  mScriptPath->setText(mConfig->scriptPath());

  mNewColors->setForeground (mConfig->newNormalColor());
  mNewColors->setBackground (mConfig->newSelectedColor());

  mChangedColors->setForeground (mConfig->changedNormalColor ());
  mChangedColors->setBackground (mConfig->changedSelectedColor ());
}

void KSVPreferences::slotCancel ()
{
  reject();
  reReadConfig();
}

void KSVPreferences::slotApply ()
{
  if (checkPaths())
    setConfig();
}

bool KSVPreferences::checkPaths ()
{
  bool result = true;

  if (!QDir(mScriptPath->text()).exists())
    result = (KMessageBox::warningContinueCancel(this,
                                                 i18n ("The service directory you specified does " \
                                                       "not exist.<br><br>" \
                                                       "You can continue if you want to, "\
                                                       "or you can select cancel " \
                                                       "to select a new directory."),
                                                 i18n ("Warning"),
                                                 i18n ("Continue"))
              != KMessageBox::Cancel);

  if (!QDir(mRunlevelsPath->text()).exists())
    result = result 
      && (KMessageBox::warningContinueCancel(this,
                                             i18n ("The runlevel directory you specified does " \
                                                   "not exist.<br><br>" \
                                                   "You can continue if you want to, "\
                                                   "or you can select cancel " \
                                                   "to select a new directory."),
                                             i18n ("Warning"),
                                             i18n ("Continue"))
          != KMessageBox::Cancel);
  
  return result;
}

void KSVPreferences::slotOk ()
{
  if (checkPaths())
    {
      accept();
      setConfig ();
    }
}

KSVPreferences* KSVPreferences::self ()
{
  KSVPreferences* prefs = new KSVPreferences (kapp->mainWidget());

  return prefs;
}

void KSVPreferences::setConfig ()
{
  // if necessary, update service path
  {
    QString sp = mScriptPath->text();
    if (sp != mConfig->scriptPath())
      {
        mConfig->setScriptPath (sp);
        
        emit updateServicesPath ();
      }
  }

  // if necesssary, update root path for runlevels
  {
    QString rp = mRunlevelsPath->text();
    if (rp != mConfig->runlevelPath ())
      {
        mConfig->setRunlevelPath (rp);
        
        emit updateRunlevelsPath ();
      }
  }
  
  // if necessary, update colors
  {
    QColor nn = newNormal();
    QColor ns = newSelected();
    QColor cn = changedNormal();
    QColor cs = changedSelected();
    
    if (nn != mConfig->newNormalColor() ||
        ns != mConfig->newSelectedColor() ||
        cn != mConfig->changedNormalColor() ||
        cs != mConfig->changedSelectedColor())
      {
        mConfig->setNewNormalColor (nn);
        mConfig->setNewSelectedColor (ns);
        mConfig->setChangedNormalColor (cn);
        mConfig->setChangedSelectedColor (cs);
        
        emit updateColors ();
      }
  }
}

void KSVPreferences::showAllMessages ()
{
  KMessageBox::enableAllMessages ();
}

#include "PreferencesDialog.moc"
