/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <kdebug.h>

#include "interrupt.h"
#include "ioaddress.h"
#include "memaddress.h"
#include "pciscanner.h"


PCIScanner::PCIScanner()
{
  _pcifhd = fopen("/proc/bus/pci/devices", "r");

  if (!_pcifhd) perror("fopen");
}


PCIScanner::~PCIScanner()
{
  if (_pcifhd) fclose(_pcifhd);
}


/*
 * Public methods
 *****************/

Device  *PCIScanner::firstDevice()
{
  if (_pcifhd)
  {
    fseek(_pcifhd, 0, SEEK_SET);       
    return device();
  }
  
  return 0L;
}


Device  *PCIScanner::nextDevice()
{
  if (_pcifhd) return device();
  
  return 0L;
}


/*
 * Private methods
 ******************/
 
PCIDevice  *PCIScanner::device()
{ 
  char                 buffer[256];
  uint                 devfn;
  ulong                device;
  uint                 irq;
  ulong                memadd[6], romadd;
  uint                 bus, slot, func;
  uint                 vendorid, deviceid, iorange;
  Device::baseclass    baseclass;
  PCIDevice::pciclass  pciclass;
  PCIDevice            *pcidev;
  int                  i;
 kdDebug() << "PCIScanner::device()" << endl; 
  if (fgets(buffer, 256, _pcifhd))
  {
    sscanf(buffer, "%x%lx%x%lx%lx%lx%lx%lx%lx%lx", &devfn,
  						   &device,
  						   &irq,
  						   &memadd[0],
  						   &memadd[1],
  						   &memadd[2],
  						   &memadd[3],
  						   &memadd[4],
  						   &memadd[5],
  						   &romadd);
 
    bus      = devfn >> 8U;
    slot     = (devfn >> 3) & 0x1f;
    func     = devfn & 0x07;
    vendorid = device >> 16;
    deviceid = device & 0xffff;
 
    if (readPCIHeader(bus, slot, func, buffer, 64))
    {
      baseclass = (Device::baseclass)   ((uchar) buffer[11]);
      pciclass  = (PCIDevice::pciclass) ((uchar) buffer[10]);
 
      pcidev    = new PCIDevice(bus, slot, func, vendorid, deviceid, baseclass, pciclass);
     
      if (irq) pcidev->addResource(new Interrupt(irq));
     
      for (i = 0; i < 6; i++)
      {
        if (memadd[i]) 
	{
	  if (memadd[i] <= 0xffff) 
	  {
	    memadd[i] &= 0xfffe;
	  
	    iorange = ioRange(memadd[i]);
	    pcidev->addResource(new IOAddress(memadd[i], iorange));
	  }
	  else pcidev->addResource(new MemoryAddress(memadd[i]));
	}
      }
      	
      if (romadd) pcidev->addResource(new MemoryAddress(romadd));
      
      return pcidev;
    }
  }

  return 0L;
}
  
 
bool  PCIScanner::readPCIHeader(uint bus, uint slot, uint func, char *buf, int count)
{
  int   pcifd, rbytes;
  char  fname[64];
    
  sprintf(fname, "/proc/bus/pci/%02x/%02x.%x", bus, slot, func);
  
  pcifd = open(fname, O_RDONLY);
    
  if (pcifd < 0) perror("open");
  else
  {
    rbytes = read(pcifd, buf, count);
    close(pcifd);
    
    if (rbytes < count) perror("read");
    else                return true;
  }

  return false;
}

