//////////////////////////////////////////////////////////////         
//      $Id: rpmInterface.cpp,v 1.49 2000/10/12 15:59:01 bero Exp $ 
//
// Author: Toivo Pedaste
//
#include "../config.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <qregexp.h>
#include <qfile.h>

#ifdef HAVE_RPM
#include <rpm/rpmlib.h>

#include <rpm/rpmio.h>

#if RPM_V1 > 3 || RPM_V2 > 0 || RPM_V3 >= 4
#    define DECLARE_HEADER_HANDLE  FD_t fd
#else 
#    define DECLARE_HEADER_HANDLE  _FD *fd
#endif

#include "rpmInstall.h"
#include "rpmVerify.h"

#if RPM_V1 < 4
#include <rpm/dbindex.h>
#endif

#include <time.h>

#include <klocale.h>
#include <kglobal.h>
#include <kiconloader.h>
 
#include "rpmInterface.h"
#include "updateLoc.h"
#include "packageInfo.h"
#include "managementWidget.h"
#include "kpackage.h"
#include "rpmutils.h"
#include "options.h"
#include "cache.h"

static param rpm_pinstall[] =  {
  {"Upgrade",TRUE,FALSE},
  {"Replace Files",FALSE,FALSE},
  {"Replace Packages",TRUE,FALSE},
  {"Check Dependencies",TRUE,TRUE},
  {"Test (do not install)",FALSE,FALSE},
  {0,FALSE,FALSE}
};

static param rpm_puninstall[] =  {
  {"Use Scripts",TRUE,TRUE},
  {"Check Dependencies",TRUE,TRUE},
  {"Test (do not uninstall)",FALSE,FALSE},
  {0,FALSE,FALSE}
};

extern Params *params;

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
RPM::RPM()
{
  char * rcfile = NULL;
  char *arch = NULL;
   head = "RPM";

  locatedialog = new Locations(i18n("Location of RPM package archives"));
  locatedialog->dLocations(7,6, this, i18n("D"),
  "Rpm","*.rpm", i18n("Location of directories containg RPM packages"));

  connect(locatedialog,SIGNAL(returnVal(LcacheObj *)),
	  this,SLOT(setAvail(LcacheObj *)));
  locatedialog->apply_slot();

  rpm_pinstall[0].name = i18n("Upgrade");
  rpm_pinstall[1].name = i18n("Replace Files");
  rpm_pinstall[2].name = i18n("Replace Packages");
  rpm_pinstall[3].name = i18n("Check Dependencies");
  rpm_pinstall[4].name = i18n("Test (do not install)");

  rpm_puninstall[0].name = i18n("Use Scripts");
  rpm_puninstall[1].name = i18n("Check Dependencies");
  rpm_puninstall[2].name = i18n("Test (do not uninstall)");

  icon = "rpm";
  pict = UserIcon(icon);
  updated_pict = UserIcon("rupdated");
  new_pict = UserIcon("rnew");

  packagePattern = "*.rpm";
  queryMsg = i18n("Querying RPM package list: ");
  typeID = "/rpm";

  QDict<QString> provides(1433,false);

  rpmErrorSetCallback(rpmErr);

  if(rpmReadConfigFiles(rcfile, arch)) {
    rpmSetup = FALSE;
  } else {
    rpmSetup = TRUE;
  }
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
RPM::~RPM()
{
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
param *RPM::initinstallOptions()
{
  return &(rpm_pinstall[0]);
}

param *RPM::inituninstallOptions()
{
  return &(rpm_puninstall[0]);
}

bool RPM::isType(char *buf, const QString & /* fname */)
{
  if ((unsigned char)buf[0] == 0355 && (unsigned char)buf[1] == 0253 &&
      (unsigned char)buf[2] == 0356 && (unsigned char)buf[3] == 0333 ) {
    return true;
  } else
    return false;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


bool RPM::parseName(QString name, QString *n, QString *v)
{
  int d1, d2, s1, s2;

  s2 = name.findRev('.');
  if (s2 > 0) {
    s1 = name.findRev('.',s2-1);
    if (s1 > 0) {
      d2 = name.findRev('-',s1-1);
      if (d2 > 0) {
	d1 = name.findRev('-',d2-1);
	if (d1 < 0)
	  d1 = d2;
	*n = name.left(d1);
	*v = name.mid(d1+1,s1-d1-1);
	return TRUE;
      }
    }
  }
  return FALSE;
}

void RPM::collectDepends(packageInfo *p, const QString &name, int src)
{
  QString dlist;
  QStringList *list = depends(name,src);

  if (list) {
    int i = 0;

    for ( QStringList::ConstIterator it = list->begin(); 
	  it != list->end(); 
	  it++, i++ ) 
    {
      if (i)
	dlist += " , ";
      dlist += *it;
    }
    if (!dlist.isNull())
      p->info->insert("unsatisfied dependencies",new QString(dlist));
    if (list)
      delete(list);
  }
}

void RPM::listInstalledPackages(QList<packageInfo> *pki)
{
  int percentage;
#if RPM_V1 < 4
  int offset;
#endif
  rpmdb db;
  Header h;
  QString prefix("");
  packageInfo *p;

  if (!rpmSetup)
    return;
    
  kpackage->setStatus(i18n("Querying RPM database for installed packages"));
  percentage=0;
  kpackage->setPercent(percentage);
  
  //  if ( rpmdbOpenForTraversal(prefix.ascii(), &db) )
  //    return;
  if ( rpmdbOpen(const_cast<char*>(prefix.ascii()), &db, O_RDONLY, 0644) )
    return;
  
#if RPM_V1 < 4
  offset = rpmdbFirstRecNum(db);
  while (offset)
    {
      percentage+=5;
      if(percentage > 100) percentage=0;
      kpackage->setPercent(percentage);
      h = rpmdbGetRecord(db, offset);
      if (!h) 
	{
	  KpMsgE(i18n("could not read database record\n"),TRUE);
	  return;
	}
      
      p = collectInfo(h);
      if (p) {
	if (!p->update(pki, typeID,TRUE))
	  delete p;
      }
      
      headerFree(h);
      offset = rpmdbNextRecNum(db, offset);
    }
#else
  rpmdbMatchIterator mi;
  char *group;
  mi = rpmdbInitIterator(db, RPMDBI_PACKAGES, NULL, 0);
  while((h = rpmdbNextIterator(mi)) != NULL) {
      percentage += 5;
      if(percentage > 100) percentage=0;
      kpackage->setPercent(percentage);
      if (!headerGetEntry(h, RPMTAG_GROUP, NULL, (void **) &group, NULL)) 
	{
	  KpMsgE(i18n("could not read database record\n"),TRUE);
	  return;
	}
      
      p = collectInfo(h);
      if (p) {
	if (!p->update(pki, typeID,TRUE))
	  delete p;
      }
  }
  rpmdbFreeIterator(mi);
#endif
  
  kpackage->setPercent(100);
  rpmdbClose(db);
  kpackage->setStatus("");
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// mode: i = query installed    u = query uninstalled
packageInfo *RPM::getPackageInfo(char mode, 
                                 const QString &name, 
                                 const QString & /*version*/ )
{
  rpmdb db;
  DECLARE_HEADER_HANDLE;
  int  rc=0, isSource;
  Header h;
  QString prefix(""); // this is the root directory -- should be configured!
  packageInfo *pki = 0;;
  
  switch(mode)
    {
      ////////////////////////////////////////////////////////////////////////
      // query an installed package!
    case 'i':
      if (rpmdbOpen(const_cast<char*>(prefix.ascii()),&db, O_RDONLY, 0644) )
	return 0;

#if RPM_V1 < 4
      dbiIndexSet matches;

      rc = findPackageByLabel(db, QFile::encodeName(name), &matches);
      if(rc==1)
	return 0;
	//	KpMsgE(i18n("Package %1 is not installed\n").arg(name),TRUE);
      else if(rc==2)
	KpMsgE(i18n("Error looking for package %1\n").arg(name),TRUE);
      else
	{
	  int i;
	  for(i=0; i<matches.count; i++)
	    {
	      if(matches.recs[i].recOffset)
		{
		  h = rpmdbGetRecord(db, matches.recs[i].recOffset);
		  if(!h)
		    KpMsgE(  i18n("Could not read database record\n"),TRUE);
		  else
		    {
		      pki = RPM::collectInfo(h);
		      headerFree(h);
		    }
		}
	    }

	  collectDepends(pki, pki->getProperty("name"),VERIFY_PACKAGE);

	  dbiFreeIndexRecord(matches);
	}
#else /* RPM 4 */
      {
      rpmdbMatchIterator mi;
      Header tmp;
      bool found=false;
      mi=rpmdbInitIterator(db, RPMDBI_LABEL, QFile::encodeName(name), 0);
      while ((tmp = rpmdbNextIterator(mi)) != NULL) {
	      found=true;
	      pki = RPM::collectInfo(tmp);
      }
      if(!found)
	      return 0; //KpMsgE(i18n("Error looking for package %1\n").arg(name),TRUE);
      collectDepends(pki, pki->getProperty("name"), VERIFY_PACKAGE);
      rpmdbFreeIterator(mi);
      }
#endif
      rpmdbClose(db);
      break;

      ////////////////////////////////////////////////////////////////////
      // query an uninstalled package      
    case 'u':
#if RPM_V1 < 4
      if((fd = fdOpen(QFile::encodeName(name), O_RDONLY, 0)) < 0) {
	  KpMsgE(i18n("Problem opening %1\n").arg(name),TRUE);
	  return pki;
      }
#else /* RPM 4 */
      if((fd = Fopen(QFile::encodeName(name), "r.fdio")) < 0) {
	  KpMsgE(i18n("Problem opening %1\n").arg(name),TRUE);
	  return pki;
      }
#endif
      if(fd!=0)	{
	  rc = rpmReadPackageHeader(fd, &h, &isSource, NULL, NULL);
#if RPM_V1 < 4
	  fdClose(fd);
#else /* RPM 4 */
	  Fclose(fd);
#endif
      }
      switch(rc)
	{
	case 0:
	  if(!h)
	    KpMsgE(i18n("Old format source packages cannot be queried!\n"),TRUE);
	  else
	    {
	      pki = RPM::collectInfo(h);
	      headerFree(h);
	      collectDepends(pki, name, VERIFY_RPM);
	    }
	  break;
	case 1:
	  KpMsgE(i18n("%1 does not appear to be a RPM package\n").arg(name),TRUE);
	  // fallthrough
	case 2:
	  KpMsgE(i18n("Query of %1 failed!\n").arg(name),TRUE);
	  return 0;
	}
      break;
    }
  if (pki)
    pki->updated = TRUE;
  return pki;
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
#define DO(x,z)                                                  \
{                                                                \
  char *k;                                                       \
  headerGetEntry(h, z, &type, (void **) &k, &count);             \
  if(k)                                                          \
    a->insert(x, new QString(QString::fromLocal8Bit(k)));        \
}								 \

packageInfo *RPM::collectInfo(Header h)
{
  int_32 type,count;
  time_t dateint;
  
  QDict<QString> *a = new QDict<QString>;
  a->setAutoDelete(TRUE);

  DO("name",RPMTAG_NAME);
  DO("version",RPMTAG_VERSION);
  DO("release",RPMTAG_RELEASE);
  DO("summary",RPMTAG_SUMMARY);
  DO("group",RPMTAG_GROUP);
  DO("distribution",RPMTAG_DISTRIBUTION);
  DO("vendor",RPMTAG_VENDOR);
  DO("packager",RPMTAG_PACKAGER);
  //  DO("source",RPMTAG_SOURCE);

  QString *vers = a->find("version");
  QString *rel = a->find("release");
  if (rel) {
    *vers += "-";
    *vers += *rel;
    a->remove("release");
  }

  char *k;


  // Get nice description
  headerGetEntry(h, RPMTAG_DESCRIPTION, &type, (void **) &k, &count);
  if(k) {
    char *p;
    for (p = k; *p!=0; p++) {	// remove newlines
      if (*p == '\n') {
	if (*(p+1) == '\n') {
	  p++;
	} else if (*(p+1) == ' ') {
	  p++;
	} else
	  *p = ' ';
      }
    }
    a->insert("description",new QString(QString::fromLocal8Bit(k)));
  }

  // Get nice install time
  headerGetEntry(h, RPMTAG_INSTALLTIME, &type, (void **) &k, &count);
  if(k)
    {
      /* this is important if sizeof(int_32) ! sizeof(time_t) */
      dateint = *(((int_32 *) k) + 0);
      QDateTime datetime;
      datetime.setTime_t(dateint);
      QString str = KGlobal::locale()->formatDateTime(datetime);
      
      a->insert("installtime", new QString(str));
    }
  
  // Get nice build time
  headerGetEntry(h, RPMTAG_BUILDTIME, &type, (void **) &k, &count);
  if(k) {
      /* this is important if sizeof(int_32) ! sizeof(time_t) */
      dateint = *(((int_32 *) k) + 0);
      QDateTime datetime;
      datetime.setTime_t(dateint);
      QString str = KGlobal::locale()->formatDateTime(datetime);
      
      a->insert("build-time", new QString(str));
  }
  
  // Get nice size
  headerGetEntry(h, RPMTAG_SIZE, &type, (void **) &k, &count);
  if(k) {
      QString s;
      a->insert("size", new QString(s.setNum(*(int_32 *)k)));
  }
  
#if RPM_V1 > 3 || RPM_V2 > 0 || RPM_V3 >= 3
  const char *error;
#else
  char *error;
#endif
  char *str;
  str = headerSprintf(h, "[%{REQUIRENAME} (%{REQUIREFLAGS:depflags}"
                         " %{REQUIREVERSION}), ]",
		      rpmTagTable, rpmHeaderFormats, &error);
  if (str) {
    QString req = str;
    req.remove(req.length() - 2, 2);
    req.replace(QRegExp("( )"),"");
    //    printf("D=%s=\n",req.data());
    a->insert("depends", new QString(req));
  }

  str = headerSprintf(h, "[%{PROVIDES}, ]",rpmTagTable,
			    rpmHeaderFormats, &error);
  if (str && *str) {
    //    printf("S=%s\n",str);
    QString prov = str;
    prov.remove(prov.length() - 2, 2);

    int s = 0, n;
    QString t;

    while ((n = prov.find(",",s)) > 0) {
      t = prov.mid(s,n-s);
      t = t.stripWhiteSpace();
      provides.insert(t,new QString(*a->find("name")));
      s = n+1;
    }
    t = prov.mid(s);
    t = t.stripWhiteSpace();
    provides.insert(t,new QString(*a->find("name")));

    a->insert("provides", new QString(prov));
  }
  //  printf("S=%s\n",str);

  packageInfo *i = new packageInfo(a,this);
  i->packageState = packageInfo::INSTALLED;
  i->fixup();
  return i;
}
#undef DO


QString RPM::provMap(QString p)
{
  QString *r = provides[p];
  if (r) {
    QString s = *r;
    //    printf("%s=>%s\n",p.data(),s.data());
    return s;
  } else {
    return p;
  }
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QStringList *RPM::getFileList(packageInfo *p)
{

  rpmdb db;
  DECLARE_HEADER_HANDLE;
  int  rc=0, isSource;
  Header h;
  QString prefix(""); // this is the root directory -- should be configured!
  QString name;
  char mode;
  
  QString fn = p->getFilename();
  if(!fn.isEmpty())
    mode = 'u';
  else
    mode = 'i';

  QStringList *filelist=NULL;

  switch(mode)
    {
      ////////////////////////////////////////////////////////////////////////
      // query an installed package!
    case 'i':
      name = p->getProperty("name");
      if (rpmdbOpen(const_cast<char*>(prefix.ascii()),&db, O_RDONLY, 0644) )
	return 0;

#if RPM_V1 < 4
      dbiIndexSet matches;

      rc = findPackageByLabel(db, QFile::encodeName(name), &matches);
      if(rc==1)
	KpMsgE(i18n("Package %1 is not installed\n").arg(name),TRUE);
      else if(rc==2)
	KpMsgE(i18n("Error looking for package %1\n").arg(name),TRUE);
      else
	{
	  int i;
	  for(i=0; i<matches.count; i++)
	    {
	      if(matches.recs[i].recOffset)
		{
		  h = rpmdbGetRecord(db, matches.recs[i].recOffset);
		  if(!h)
		    KpMsgE(i18n("Could not read database record\n"),TRUE);
		  else
		    {
		      filelist = RPM::collectFileList(h);
		      headerFree(h);
		    }
		}
	    }
	  dbiFreeIndexRecord(matches);
	}
#else /* RPM v4 */
      {
      rpmdbMatchIterator mi;
      bool found=false;
      mi=rpmdbInitIterator(db, RPMDBI_LABEL, QFile::encodeName(name), 0);
      while((h=rpmdbNextIterator(mi)) != NULL) {
	      found=true;
	      filelist = RPM::collectFileList(h);
      }
      if(!found)
	      KpMsgE(i18n("Could not read database record\n"), TRUE);
      rpmdbFreeIterator(mi);
      }
#endif
      rpmdbClose(db);
      break;

      ////////////////////////////////////////////////////////////////////
      // query an uninstalled package      
    case 'u':
      name = fn;
#if RPM_V1 < 4
      if((fd = fdOpen(name.ascii(), O_RDONLY, 0)) < 0)
	{
	  KpMsgE(i18n("ERROR opening %1\n").arg(name),TRUE);
	  return filelist;
	}
#else /* RPM 4 */
      if((fd = Fopen(name.ascii(), "r.fdio")) < 0)
	{
	  KpMsgE(i18n("ERROR opening %1\n").arg(name),TRUE);
	  return filelist;
	}
#endif

      if(fd!=0)	{
	  rc = rpmReadPackageHeader(fd, &h, &isSource, NULL, NULL);
#if RPM_V1 < 4
	  fdClose(fd);
#else /* RPM 4 */
	  Fclose(fd);
#endif
	}
      switch(rc) {
	case 0:
	  if(!h)
	    KpMsgE(i18n("Old format source packages cannot be queried\n"),TRUE);
	  else
	    {
	      filelist = RPM::collectFileList(h);
	      headerFree(h);
	    }
	  break;
	case 1:
	  KpMsgE(i18n("%1 does not appear to be a RPM package\n").arg(name),TRUE);
	  // fallthrough
	case 2:
	  KpMsgE(i18n("Query of %1 failed\n").arg(name),TRUE);
	  return filelist;
	}
      break;
    }
   
  return filelist;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QStringList* RPM::collectFileList(Header h)
{
  char **fileList;
// Code for RPM 3.0.4 or greater
#if RPM_V1 >= 3 && (RPM_V1 > 3 || RPM_V2 > 0 || RPM_V3 >= 4)
  int count2;
#else
  int_32 type, count;
#endif
  QStringList* files = new QStringList();

  kpackage->setStatus(i18n("Getting file list"));
  kpackage->setPercent(0);

// Code for RPM 3.0.4 or greater
#if RPM_V1 >= 3 && (RPM_V1 > 3 || RPM_V2 > 0 || RPM_V3 >= 4)

  rpmBuildFileList(h, (const char ***) &fileList, &count2);
  if(fileList != 0) 
    {    
      int i;
      for(i=0; i<count2; i++) 
        {
          kpackage->setPercent( (i/count2) * 100);
          files->append(fileList[i]);
        }
      free(fileList);
    }

#else

  if(headerGetEntry(h, RPMTAG_FILENAMES, &type, (void **) &fileList, &count))
    {
      int i;
      for(i=0; i<count; i++)
	{
	  kpackage->setPercent( (i/count) * 100);
	  files->append(fileList[i]);
	}
      free(fileList);
    }

#endif

  kpackage->setPercent(100);
  return files;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QStringList *RPM::depends(const QString &name, int src)
{
  kpackage->setStatus(i18n("Depends"));
  kpackage->setPercent(0);

  int verifyFlags = 0;
  verifyFlags |= VERIFY_DEPS;

  QStringList fls;
  fls.append(name);

  QStringList *result = new QStringList();

  doVerify("/", (enum verifysources) src, fls, verifyFlags, result);
  return result;  
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QStringList *RPM::verify(packageInfo *p, const QStringList &)
{
  kpackage->setStatus(i18n("Verifying"));
  kpackage->setPercent(0);

  int verifyFlags = 0;
  verifyFlags |= VERIFY_FILES |  VERIFY_SCRIPT |  VERIFY_MD5;

  QStringList *result = new QStringList();

  QStringList fls;
  fls.append( p->getProperty("name"));

  doVerify("/", VERIFY_PACKAGE, fls, verifyFlags, result);
  return result;  
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

int RPM::uninstall(int uninstallFlags, QList<packageInfo> *plist)
{
  QStringList files;
 
  packageInfo *pk;
  for (pk = plist->first(); pk != 0; pk = plist->next()) {
    files.append( pk->getProperty("name") );
  }

  int n =  doUninst(uninstallFlags,files);
  return n;
}

int RPM::uninstall(int uninstallFlags, packageInfo *p)
{
  QStringList files;
  files.append( p->getProperty("name") );

  return doUninst(uninstallFlags,files);
}

int RPM::doUninst(int uninstallFlags, const QStringList &files)
{
  int interfaceFlags = 0;
  int untransFlags = 0;
  int test = 0;
 
  if ((uninstallFlags>>0 & 1) ^ rpm_puninstall[0].invert)
    untransFlags|=RPMTRANS_FLAG_NOSCRIPTS;
  if ((uninstallFlags>>1 & 1) ^ rpm_puninstall[1].invert)
    interfaceFlags|=UNINSTALL_NODEPS;
  if ((uninstallFlags>>2 & 1) ^ rpm_puninstall[2].invert) {
    untransFlags|= RPMTRANS_FLAG_TEST;
    test = 1;
  }

  rpmEBuf = new QString();
  int ret = doUninstal("/", files, untransFlags, interfaceFlags) || test;
   if (!rpmEBuf->isEmpty())
     KpMsgE(*rpmEBuf,FALSE);
   delete rpmEBuf;
   rpmEBuf = 0;
   return ret;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
int RPM::install(int installFlags, QList<packageInfo> *plist)
{
  QStringList files;

  for (packageInfo *pk = plist->first(); pk != 0; pk = plist->next()) {
    QString fname( pk->fetchFilename() );
    if (!fname.isEmpty()) {
       files.append(fname);
    }
  }

  return doinst(installFlags, files);
}

int RPM::install(int installFlags, packageInfo *p)
{
  QStringList files;
  files.append( p->fetchFilename());
  return doinst(installFlags,files);
}

int RPM::doinst(int installFlags, const QStringList &files)
{
  int interfaceFlags = 0 , transFlags = 0, probFlags = 0, test=0;

  if ((installFlags>>0 & 1) ^ rpm_pinstall[0].invert) {
    interfaceFlags |= INSTALL_UPGRADE;
  }

  if ((installFlags>>1 & 1) ^ rpm_pinstall[1].invert) {
    probFlags|= RPMPROB_FILTER_REPLACEOLDFILES;
    probFlags|= RPMPROB_FILTER_REPLACENEWFILES;
  }
  if ((installFlags>>2 & 1) ^ rpm_pinstall[2].invert) {
    probFlags|=RPMPROB_FILTER_REPLACEPKG;
    probFlags|=RPMPROB_FILTER_OLDPACKAGE;
  }
  if ((installFlags>>3 & 1) ^ rpm_pinstall[3].invert)
    interfaceFlags|=INSTALL_NODEPS;
  if ((installFlags>>4 & 1) ^ rpm_pinstall[4].invert) {
    transFlags|=RPMTRANS_FLAG_TEST;
    test = 1;
  }
  interfaceFlags|=INSTALL_PERCENT;

  rpmEBuf = new QString();
  int ret = doInstal("/", files, transFlags,
               interfaceFlags,  probFlags, 0) || test;
   if (!rpmEBuf->isEmpty())
     KpMsgE(*rpmEBuf,FALSE);
   delete rpmEBuf;
   rpmEBuf = 0;
   return ret;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QString RPM::FindFile(const QString &name) {
 
  QString s = "";
 
  QRegExp regexp(QString("*")+name, true, true);
 
  int percentage;
#if RPM_V1 < 4
  int offset;
#endif
  rpmdb db;
  Header h;
  const char *prefix = "";
  const char ** fileList;
  const char *n;
  int_32 count, type;
  int pos, len;
 
  if (!rpmSetup)
    return "";
 
  kpackage->setStatus(i18n("Querying RPM database for file"));
  percentage=0;
  kpackage->setPercent(percentage);
 
  if ( rpmdbOpen(prefix,&db, O_RDONLY, 0644) )
    return "";
 
#if RPM_V1 < 4
  offset = rpmdbFirstRecNum(db);
  while (offset)
    {
      percentage+=5;
      if(percentage > 100) percentage=0;
      kpackage->setPercent(percentage);
      h = rpmdbGetRecord(db, offset);
      if (!h)
        {
          KpMsgE(i18n("could not read database record\n"),TRUE);
          return "";
        }
 
// Code for RPM 3.0.4 or greater
#if RPM_V1 > 3 || (RPM_V1 == 3 && (RPM_V2 > 0 || (RPM_V2 == 0 && RPM_V3 >= 4)))
      if (headerGetEntry(h, RPMTAG_BASENAMES, &type, (void **) &fileList, 
			 &count))
        {
#else
      if (headerGetEntry(h, RPMTAG_FILENAMES, &type, (void **) &fileList, 
			 &count))
        {
#endif
          for (int i = 0; i < count; i++)
            if ( (pos = regexp.match(fileList[i], 0, &len)) >= 0)
              {
                headerGetEntry(h, RPMTAG_NAME, &type, (void **) &n, &count);
 
                s += n;
                s += "\t";
                s += QString(fileList[i]).mid(pos,len);
                s += "\n";
              }
 
          free(fileList);
        }
 
      headerFree(h);
      offset = rpmdbNextRecNum(db, offset);
    }
#else /* RPM v4 */
  {
  rpmdbMatchIterator mi=rpmdbInitIterator(db, RPMDBI_LABEL, NULL, 0);
  while ((h=rpmdbNextIterator(mi))!=NULL)
    {
      percentage+=5;
      if(percentage > 100) percentage=0;
      kpackage->setPercent(percentage);

      if (headerGetEntry(h, RPMTAG_BASENAMES, &type, (void **) &fileList, 
			 &count))
        {
          for (int i = 0; i < count; i++)
            if ( (pos = regexp.match(fileList[i], 0, &len)) >= 0)
              {
                headerGetEntry(h, RPMTAG_NAME, &type, (void **) &n, &count);
 
                s += n;
                s += "\t";
                s += QString(fileList[i]).mid(pos,len);
                s += "\n";
              }
 
          free(fileList);
        }
 
    }
    rpmdbFreeIterator(mi);
    }
#endif
 
  kpackage->setPercent(100);
  rpmdbClose(db);
  kpackage->setStatus("");
 
  return s;
}                                                                              
                              

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void RPM::setLocation()
{
    locatedialog->restore();
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void RPM::setAvail(LcacheObj *slist)
{
  if (packageLoc)
    delete packageLoc;
  packageLoc = slist;
}
#else
  #include "rpmInterface.h"

  RPM::RPM(){}
  RPM::~RPM(){}
  
  bool RPM::isType(char *, const QString&){return 0;}
  param *RPM::initinstallOptions(){return 0;}
  param *RPM::inituninstallOptions(){return 0;}

  packageInfo *RPM::getPackageInfo(char, const QString &, const QString 
&){return 0;}
  QStringList *RPM::getFileList(packageInfo *){return 0;}
  QStringList *RPM::depends(const QString &, int){return 0;}
  QStringList *RPM::verify(packageInfo *, const QStringList &){return 0;}

  int RPM::uninstall(int, QList<packageInfo> *){return 0;}
  int RPM::uninstall(int, packageInfo *){return 0;}
  int RPM::doUninst(int, const QStringList &){return 0;}

  int RPM::install(int, QList<packageInfo> *){return 0;}
  int RPM::install(int, packageInfo *){return 0;}
  int RPM::doinst(int, const QStringList &){return 0;}

  QString RPM::FindFile(const QString &){return QString::null;;}
  void RPM::collectDepends(packageInfo *, const QString &, int){}
  bool RPM::parseName(QString, QString *, QString *){return 0;}

  void RPM::setLocation(){}
  void RPM::setAvail(LcacheObj *){}
  QString RPM::provMap(QString) { return QString::null; }

  packageInfo* RPM::collectInfo(Header){return 0;}
  QStringList* RPM::collectFileList(Header){return 0;}
  void RPM::listInstalledPackages(QList<packageInfo> *){}

#endif


