/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <unistd.h>

#include <qlayout.h>

#include <kdebug.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kstddirs.h>

#include "conflict.h"
#include "devicelist.h"
#include "proprestab.h"
#include "reschangedlg.h"
#include "resentry.h"


extern DeviceList  *devicelist;


ResourcePropertiesTab::ResourcePropertiesTab(Device *device, QWidget *parent, 
					     const char *name) : QWidget(parent, name),
					     			 _device(device),
                                                                 _baseconf(0L)
{
  QString   devname   = _device->deviceName();
  QString   classname = _device->deviceClassName();
  Resource  *res;
  
  _resxpm   = SmallIcon("res");
  _confxpm  = SmallIcon("conflict");
  _rootperm = !getuid();
  
  if (!classname.isEmpty()) devname += " ("+classname+")";
  
  DeviceTitle  *devtitle = new DeviceTitle(_device->bigPixmapFile(), devname, this, "devtitle");
    
  _baselabel = new QLabel(i18n("Settings based on:"), this, "baselabel");  
  _baselabel->setEnabled(_rootperm);
  _baselabel->adjustSize();
  
  _basecombo = new QComboBox(this, "basecombo");
  _basecombo->setEnabled(_rootperm);
  connect(_basecombo, SIGNAL(activated(int)), SLOT(slotConfigChanged(int)));  

  _listview = new QListView(this, "listview");
  _listview->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  _listview->setLineWidth(2);
  _listview->setAllColumnsShowFocus(true);
  _listview->addColumn(i18n("Resource type"));
  _listview->addColumn(i18n("Setting"));
  connect(_listview, SIGNAL(currentChanged(QListViewItem *)), SLOT(slotCurrentChanged(QListViewItem *)));

  _changebut = new QPushButton(i18n("Change &setting..."), this, "changebut");
  _changebut->adjustSize();
  _changebut->setEnabled(_rootperm && _listview->currentItem() && _device->numConfigs());
  connect(_changebut, SIGNAL(clicked()), SLOT(slotChangeResource()));
  
  QLabel  *conflictlabel = new QLabel(i18n("Device conflicts:"), this, "conflictlabel");
  conflictlabel->adjustSize();
  
  _conflictdisplay = new ConflictDisplay(this, "conflictdisplay");
  
  _reslist.setAutoDelete(true);
  _reslist.clear();
  
  for (res = _device->firstResource(); res; res = _device->nextResource())
    _reslist.append(res->copy());

  fillInConfigurations();
  fillInResources();
  
  QVBoxLayout  *toplayout   = new QVBoxLayout(this, 16,  2, "toplayout");
  
  toplayout->addWidget(devtitle);
  toplayout->addSpacing(16);
  
  QHBoxLayout  *combolayout = new QHBoxLayout(toplayout, 4, "combolayout");
  
  combolayout->addWidget(_baselabel);
  combolayout->addSpacing(8);
  combolayout->addWidget(_basecombo, 1);
  
  toplayout->addSpacing(4);
  toplayout->addWidget(_listview, 1);
  toplayout->addSpacing(4);
  
  QHBoxLayout  *chbutlayout = new QHBoxLayout(toplayout, 4, "chbutlayout");
  
  chbutlayout->addWidget(_changebut);
  chbutlayout->addStretch(1);
  
  toplayout->addSpacing(6);
  
  QHBoxLayout  *clablayout  = new QHBoxLayout(toplayout, 4, "clablayout");
  
  clablayout->addWidget(conflictlabel);
  clablayout->addStretch(1);
  
  toplayout->addWidget(_conflictdisplay, 1);
  toplayout->activate();
}


/*
 * Private methods
 ******************/

QList<Conflict>  ResourcePropertiesTab::checkForConflicts()
{
  QList<Conflict>  conflictlist;

  if (!(conflictlist = devicelist->conflicts(_device, _reslist, false)).isEmpty())
    _conflictdisplay->showConflicts(conflictlist);
  else _conflictdisplay->reset();
  
  return conflictlist;
}


void  ResourcePropertiesTab::fillInConfigurations()
{
  QString  combotext;
  uint     numconf  = _device->numConfigs();
  uint     conf;
  int      confindex;
  bool     rootperm = !getuid();
  
  _baseconf = _device->settingsBasedOn(&confindex);
  _basecombo->clear();
  
  if (!numconf) 
  {
    _baselabel->setEnabled(false);
    _basecombo->setEnabled(false);
  }
  else 
  {
    // This device offers different base configurations (ISAPnP, for example)
    
    _baselabel->setEnabled(rootperm);
    _basecombo->setEnabled(rootperm);
    
    for (conf = 0; conf < numconf; conf++)
    {
      combotext = i18n("Base configuration");
      combotext += QString().sprintf(" %04u", conf);
 
      _basecombo->insertItem(combotext);
    }
    
    if (_baseconf) _basecombo->setCurrentItem(confindex);
  }
} 

 
void  ResourcePropertiesTab::fillInResources()
{
  QPixmap          xpm;
  Resource         *res;
  ResourceEntry    *entry;
  QList<Conflict>  conflictlist;
  Conflict         *conflict;
  QList<Resource>  reslist = (_baseconf) ? _baseconf->nearestMatch(_reslist) : _reslist;
  
  _listview->clear();
  
  conflictlist = checkForConflicts();
  
  for (res = reslist.first(); res; res = reslist.next())
  {  
    entry = new ResourceEntry(_listview, res);
    xpm   = _resxpm;
    
    for (conflict = conflictlist.first(); conflict; conflict = conflictlist.next())
    {
      if (*(conflict->resource()) == *res) 
      {
        xpm = _confxpm;
	break;
      }
    }
    
    entry->setPixmap(0, xpm);
  } 
  
  // Why does highlighting not work at all???
  if ((entry = (ResourceEntry *) _listview->firstChild())) _listview->setCurrentItem(entry);							     
}


/*
 * Private slots
 ****************/
 
void  ResourcePropertiesTab::slotChangeResource()
{
  ResourceEntry         *entry = (ResourceEntry *) (_listview->currentItem());
  Resource              *res   = entry->resource();
  ResourceChangeDialog  *resdlg;
  
  resdlg = new ResourceChangeDialog(_baseconf, res, _device, 0L, "resdlg");
  
  if (resdlg->exec() == QDialog::Accepted)
  {
    if (!_reslist.removeRef(res)) kdDebug() << "***** Can't remove resource from device" << endl;

    entry->setResource(res = resdlg->newSetting()->copy());
    entry->setPixmap(0, (resdlg->conflict()) ? _confxpm : _resxpm);
    
    _reslist.append(res);
    
    (void) checkForConflicts();
  }
  
  delete resdlg;
} 
 
 
void  ResourcePropertiesTab::slotConfigChanged(int index)
{
  _baseconf = _device->baseConfig(index);
  
  fillInResources();
}


void  ResourcePropertiesTab::slotCurrentChanged(QListViewItem *item)
{
  _changebut->setEnabled(_rootperm && _baseconf);
}


