/*
 * krootwm.cc Part of the KDE project.
 *
 * Copyright (C) 1997 Matthias Ettrich
 *           (C) 1997 Torben Weis, weis@kde.org
 *           (C) 1998 S.u.S.E. weis@suse.de
 *
 */

//#define QT_CLEAN_NAMESPACE // now globally defined in kdebase (Werner)
#include <qdir.h>

#include <signal.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>

#include <qmessagebox.h>
#include <qtimer.h>
#include <kpopupmenu.h>
#include <kapp.h>
#include <kbookmark.h>
#include <kbookmarkmenu.h>
#include <kglobal.h>
#include <kio/job.h>
#include <kiconloader.h>
#include <klocale.h>
#include <knewmenu.h>
#include <kprocess.h>
#include <krun.h>
#include <ksimpleconfig.h>
#include <kwinmodule.h>
#include <kwin.h>
#include <kglobalsettings.h>
#include <kstddirs.h>
#include <dcopclient.h>
#include <khelpmenu.h>
#include <kstringhandler.h>
#include <kiconloader.h>
#include <kwin.h>
#include <netwm.h>
#include <kdebug.h>

#include <X11/X.h>
#include <X11/Xlib.h>

#include "krootwm.h"
#include "minicli.h"
#include "desktop.h"

#include <kmenubar.h>

KRootWm * KRootWm::s_rootWm = 0;

KRootWm::KRootWm(KDesktop* _desktop) : QObject(_desktop)
{
  s_rootWm = this;
  m_actionCollection = new KActionCollection();
  miniCli = new Minicli;
  m_pDesktop = _desktop;

  // Creates the new menu
  menuBar = 0; // no menubar yet
  menuNew = new KNewMenu( m_actionCollection, "new_menu" );
  connect(menuNew->popupMenu(), SIGNAL( aboutToShow() ),
          this, SLOT( slotFileNewAboutToShow() ) );

  bookmarks = new KActionMenu( i18n("Bookmarks"), "bookmark", m_actionCollection, "bookmarks" );
  bookmarkMenu = new KBookmarkMenu( new KBookmarkOwner(),
                                    (QPopupMenu*)bookmarks->popupMenu(),
                                    m_actionCollection,
                                    true, false );

  // The windowList and desktop menus can be part of a menubar (Mac style)
  // so we create them here
  desktopMenu = new KPopupMenu;
  desktopMenu->setFont(KGlobalSettings::menuFont());
  windowListMenu = new KPopupMenu;
  windowListMenu->setFont(KGlobalSettings::menuFont());
  windowListMenu->setCheckable(TRUE);

  // Create the actions

  m_actionCollection->insert( m_pDesktop->actionCollection()->action( "paste" ) );
  m_actionCollection->insert( m_pDesktop->actionCollection()->action( "undo" ) );
  new KAction(i18n("Help on desktop..."), "contents", 0, this, SLOT( slotHelp() ), m_actionCollection, "help" );
  new KAction(i18n("Run Command..."), "run", 0, this, SLOT( slotExec() ), m_actionCollection, "exec" );
  new KAction(i18n("Configure Background..."), "background", 0, this, SLOT( slotConfigureBackground() ),
              m_actionCollection, "configbackground" );
  new KAction(i18n("Configure Desktop..."), "configure", 0, this, SLOT( slotConfigureDesktop() ),
              m_actionCollection, "configdesktop" );
  new KAction(i18n("Disable Desktop Menu"), 0, this, SLOT( slotToggleDesktopMenu() ),
              m_actionCollection, "togglemenubar" );

  new KAction(i18n("Unclutter Windows"), 0, this, SLOT( slotUnclutterWindows() ),
              m_actionCollection, "unclutter" );
  new KAction(i18n("Cascade Windows"), 0, this, SLOT( slotCascadeWindows() ),
              m_actionCollection, "cascade" );
  new KAction(i18n("Arrange Icons"), 0, this, SLOT( slotArrangeIcons() ),
              m_actionCollection, "arrange" );
  new KAction(i18n("Line up Icons"), 0, this, SLOT( slotLineupIcons() ),
              m_actionCollection, "lineup" );
  // Icons in sync with kicker
  new KAction(i18n("Lock Screen"), "lock", 0, this, SLOT( slotLock() ),
              m_actionCollection, "lock" );
  new KAction(i18n("Logout..."), "exit", 0, this, SLOT( slotLogout() ),
              m_actionCollection, "logout" );

  connect( kapp, SIGNAL( appearanceChanged() ), this, SLOT( slotAppearanceChanged() ) );

  initConfig();
}

KRootWm::~KRootWm()
{
    delete miniCli;
    delete m_actionCollection;
    delete bookmarkMenu;
}

void KRootWm::initConfig()
{
  kdDebug() << "KRootWm::initConfig" << endl;
  KConfig *kconfig = KGlobal::config();

  // parse the configuration
  kconfig->setGroup(QString::fromLatin1("KDE"));
  globalMenuBar = kconfig->readBoolEntry(QString::fromLatin1("macStyle"), false);
  kconfig->setGroup(QString::fromLatin1("Menubar"));
  showMenuBar = globalMenuBar || kconfig->readBoolEntry(QString::fromLatin1("ShowMenubar"), false );

  // read configuration for clicks on root window
  const char * s_choices[4] = { "", "WindowListMenu", "DesktopMenu", "AppMenu" };
  leftButtonChoice = middleButtonChoice = rightButtonChoice = NOTHING;
  kconfig->setGroup("Mouse Buttons");
  QString s = kconfig->readEntry("Left", "");
  for ( int c = 0 ; c < 4 ; c ++ )
    if (s == s_choices[c])
      { leftButtonChoice = (menuChoice) c; break; }
  s = kconfig->readEntry("Middle", "WindowListMenu");
  for ( int c = 0 ; c < 4 ; c ++ )
    if (s == s_choices[c])
      { middleButtonChoice = (menuChoice) c; break; }
  s = kconfig->readEntry("Right", "DesktopMenu");
  for ( int c = 0 ; c < 4 ; c ++ )
    if (s == s_choices[c])
      { rightButtonChoice = (menuChoice) c; break; }

  buildMenus();
}


void KRootWm::slotAppearanceChanged()
{
    // weird weird weird, but in the style of the rest of hacky
    // krootwm..... Who started using dcop for the communication
    // _within_ one applicaitons?!
    kapp->dcopClient()->send( "kdesktop", "KDesktopIface", "configure()", "");
}

void KRootWm::buildMenus()
{
    kdDebug() << "KRootWm::buildMenus" << endl;

    delete menuBar;
    menuBar = 0;

    if (showMenuBar)
    {
        kdDebug() << "showMenuBar" << endl;
        QWidget* dummy = new QWidget;
        menuBar = new KMenuBar( dummy );
        disconnect( kapp, SIGNAL( appearanceChanged() ), menuBar, SLOT( slotReadConfig() ) );
        menuBar->setCaption("KDE Desktop");
    }

    if (menuBar) {
        file = new QPopupMenu;

        m_actionCollection->action("exec")->plug( file );

        file->insertSeparator();
        m_actionCollection->action("lock")->plug( file );
        m_actionCollection->action("logout")->plug( file );

        desk = new QPopupMenu;

        m_actionCollection->action("unclutter")->plug( desk );
        m_actionCollection->action("cascade")->plug( desk );
        desk->insertSeparator();
        m_actionCollection->action("lineup")->plug( desk );
        m_actionCollection->action("arrange")->plug( desk );
        desk->insertSeparator();
        m_actionCollection->action("configbackground")->plug( desk );
        m_actionCollection->action("configdesktop")->plug( desk );
        desk->insertSeparator();
        m_actionCollection->action("togglemenubar")->plug( desk );
        m_actionCollection->action("togglemenubar")->setText(i18n("Disable Desktop Menu"));

        help = new KHelpMenu(0, 0, false);
    }
    else
        m_actionCollection->action("togglemenubar")->setText(i18n("Enable Desktop Menu"));

    desktopMenu->clear();
    desktopMenu->disconnect( this );
    windowListMenu->clear();
    windowListMenu->disconnect( this );

    menuNew->plug( desktopMenu );
    bookmarks->plug( desktopMenu );
    desktopMenu->insertSeparator();

    m_actionCollection->action("undo")->plug( desktopMenu );
    m_actionCollection->action("paste")->plug( desktopMenu );
    m_actionCollection->action("help")->plug( desktopMenu );
    m_actionCollection->action("exec")->plug( desktopMenu );
    desktopMenu->insertSeparator();
    m_actionCollection->action("configbackground")->plug( desktopMenu );
    m_actionCollection->action("configdesktop")->plug( desktopMenu );
    desktopMenu->insertSeparator();
    if ( !globalMenuBar ) {
	m_actionCollection->action("togglemenubar")->plug( desktopMenu );
	desktopMenu->insertSeparator();
    }
    m_actionCollection->action("unclutter")->plug( desktopMenu );
    m_actionCollection->action("cascade")->plug( desktopMenu );
    desktopMenu->insertSeparator();
    m_actionCollection->action("lineup")->plug( desktopMenu );
    m_actionCollection->action("arrange")->plug( desktopMenu );

    desktopMenu->insertSeparator();
    m_actionCollection->action("lock")->plug( desktopMenu );
    m_actionCollection->action("logout")->plug( desktopMenu );
    connect( desktopMenu, SIGNAL( aboutToShow() ), this, SLOT( slotFileNewAboutToShow() ) );

    connect(windowListMenu, SIGNAL(activated(int)), this, SLOT(slotWindowItemActivated(int)));
    connect(windowListMenu, SIGNAL(aboutToShow()), this, SLOT(generateWindowlist()));

    if (menuBar) {
        menuBar->insertItem(i18n("File"), file);
        menuBar->insertItem(i18n("New"), menuNew->popupMenu());
        menuBar->insertItem(i18n("Bookmarks"), bookmarks->popupMenu());
        menuBar->insertItem(i18n("Desktop"), desk);
        menuBar->insertItem(i18n("Windows"), windowListMenu);
        menuBar->insertItem(i18n("Help"), help->menu());
        help->menu()->removeItemAt( 4 ); // we don't need no aboutApplication

        menuBar->setTopLevelMenu( true );
        XSetTransientForHint( qt_xdisplay(), menuBar->winId(), m_pDesktop->winId() );
        menuBar->show(); // we need to call show() as we delayed the creation with the timer
    }
}

void KRootWm::slotFileNewAboutToShow()
{
  kdDebug() << " KRootWm:: (" << this << ") slotFileNewAboutToShow() menuNew=" << menuNew << endl;
  // As requested by KNewMenu :
  menuNew->slotCheckUpToDate();
  // And set the files that the menu apply on :
  menuNew->setPopupFiles( m_pDesktop->url() );
}

/*
  Shows minicli
 */
void KRootWm::slotExec()
{
  // Move minicli to the current desktop
  NETWinInfo info( qt_xdisplay(), miniCli->winId(), qt_xrootwin(), NET::WMDesktop );
  int currentDesktop = m_pDesktop->kwinModule()->currentDesktop();
  if ( info.desktop() != currentDesktop )
      info.setDesktop( currentDesktop );

  if ( miniCli->isVisible() ) {
      miniCli->raise();
  } else {
      miniCli->move(QApplication::desktop()->width()/2 - miniCli->width()/2,
                    QApplication::desktop()->height()/2 - miniCli->height()/2);
      miniCli->show();
  }
  KWin::setActiveWindow( miniCli->winId() );
}

void KRootWm::activateMenu( menuChoice choice, const QPoint& global )
{
  switch ( choice )
  {
    case WINDOWLISTMENU:
      windowListMenu->popup(global);
      break;
    case DESKTOPMENU:
      desktopMenu->popup(global);
      break;
    case APPMENU:
    {
      // This allows the menu to disappear when clicking on the background another time
      XUngrabPointer(qt_xdisplay(), CurrentTime);
      XSync(qt_xdisplay(), False);
      // Ask kicker to showup the menu
      QByteArray data;
      QDataStream stream( data, IO_WriteOnly );
      stream << global.x();
      stream << global.y();
      kapp->dcopClient()->send( "kicker", "kickerMenuManager", "popupKMenu(int,int)", data );
      break;
    }
    case NOTHING:
      break;
  }
}

void KRootWm::mousePressed( const QPoint& _global, int _button )
{
    if (!desktopMenu) return; // initialisation not yet done
    switch ( _button ) {
    case LeftButton:
        if ( showMenuBar && menuBar )
            menuBar->raise();
        activateMenu( leftButtonChoice, _global );
        break;
    case MidButton:
        activateMenu( middleButtonChoice, _global );
        break;
    case RightButton:
        activateMenu( rightButtonChoice, _global );
        break;
    default:
        // nothing
        break;
    }
}


void KRootWm::slotArrangeIcons() {
  m_pDesktop->rearrangeIcons( true /* ask confirmation */);
}
void KRootWm::slotLineupIcons() {
  m_pDesktop->lineupIcons();
}
void KRootWm::slotConfigureBackground() {
  QStringList args;
  args.append(QString::fromLatin1("background"));
  KApplication::kdeinitExec(QString::fromLatin1("kcmshell"), args);
}

void KRootWm::slotConfigureDesktop() {
  QStringList args;
  args.append(QString::fromLatin1("desktop"));
  KApplication::kdeinitExec(QString::fromLatin1("kcmshell"), args);
}

void KRootWm::slotToggleDesktopMenu() {
  KConfig *config = KGlobal::config();
  KConfigGroupSaver saver(config, QString::fromLatin1("Menubar"));
  if (showMenuBar && menuBar)
    config->writeEntry(QString::fromLatin1("ShowMenubar"), false);
  else
    config->writeEntry(QString::fromLatin1("ShowMenubar"), true);
  config->sync();
  kapp->dcopClient()->send( "kdesktop", "KDesktopIface", "configure()", "");
}
void KRootWm::slotUnclutterWindows() {
  kapp->dcopClient()->send("kwin", "KWinInterface", "unclutterDesktop()", "");
}
void KRootWm::slotCascadeWindows() {
  kapp->dcopClient()->send("kwin", "KWinInterface", "cascadeDesktop()", "");
}
void KRootWm::slotHelp() {
  KApplication::kdeinitExec(QString::fromLatin1("khelpcenter"));
}
void KRootWm::slotPaste() {
  m_pDesktop->unselectAll(); // perhaps a hack. But required by pasteSelection() currently
  m_pDesktop->slotPaste();
}
void KRootWm::slotLock() {
  kapp->dcopClient()->send("kdesktop", "KScreensaverIface", "lock()", "");
}
void KRootWm::slotLogout() {
  kapp->requestShutDown();
}

void KRootWm::slotMenuItemActivated(int /* item */ )
{
}

void KRootWm::slotWindowItemActivated(int item){
  if (item>1000)
      KWin::setCurrentDesktop(item - 1000);
  else if ( item >= 0 ) {
    Window w = m_pDesktop->kwinModule()->windows()[item];
    KWin::setActiveWindow(w);
  }
}


void KRootWm::generateWindowlist()
{
    int i, d;
    windowListMenu->clear();
    windowListMenu->insertItem( i18n("Unclutter Windows"),
				this, SLOT( slotUnclutterWindows() ) );
    windowListMenu->insertItem( i18n("Cascade Windows"),
				this, SLOT( slotCascadeWindows() ) );

    windowListMenu->insertSeparator();

    KWinModule* kwin_module = m_pDesktop->kwinModule();
    int nd = kwin_module->numberOfDesktops();
    int cd = kwin_module->currentDesktop();
    WId active_window = kwin_module->activeWindow();


    for (d = 1; d <= nd; d++) {
        if (nd > 1)
            windowListMenu->insertItem( kwin_module->desktopName( d ), 1000 + d);

        int items = 0;

        if (!active_window && d == cd)
            windowListMenu->setItemChecked(1000 + d, TRUE);

        QValueList<WId>::ConstIterator it;
        for (it = kwin_module->windows().begin(), i = 0;
             it != kwin_module->windows().end(); ++it, ++i) {
            KWin::Info info = KWin::info( *it );
            if ( (info.desktop == d ) || (d == cd && info.onAllDesktops ) ) {
                QString title = info.visibleNameWithState();
                if ( info.windowType == NET::Normal || info.windowType == NET::Unknown ) {
                    QPixmap pm = KWin::icon(*it, 16, 16, true );
                    if ( pm.isNull() )
                        pm = defaultPixmap;
                    items++;
                    windowListMenu->insertItem( pm, QString("   ")+ KStringHandler::csqueeze(  title,25),i);
                    if (*it == active_window)
                        windowListMenu->setItemChecked(i, TRUE);
                }
            }
        }
        if (d < nd)
            windowListMenu->insertSeparator();
    }
}

#include "krootwm.moc"
