/*****************************************************************

Copyright (c) 2000 Matthias Ettrich <ettrich@kde.org>
              2000 Matthias Elter   <elter@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <kwinmodule.h>

#include "systemtrayapplet.h"
#include "systemtrayapplet.moc"

#include <X11/Xlib.h>

template class QList<KXEmbed>;

extern "C"
{
  KPanelApplet* init(QWidget *parent, const QString& configFile)
  {
    return new SystemTrayApplet(configFile, KPanelApplet::Normal,
				0, parent, "ksystemtrayapplet");
  }
}

SystemTrayApplet::SystemTrayApplet(const QString& configFile, Type type, int actions,
				   QWidget *parent, const char *name)
  : KPanelApplet(configFile, type, actions, parent, name)
{
  m_Wins.setAutoDelete(true);
  setFrameStyle(Panel | Sunken);

  // the KWinModule notifies us when tray windows are added or removed
  kwin_module = new KWinModule(this);
  connect( kwin_module, SIGNAL( systemTrayWindowAdded(WId) ),
	   this, SLOT( systemTrayWindowAdded(WId) ) );
  connect( kwin_module, SIGNAL( systemTrayWindowRemoved(WId) ),
	   this, SLOT( updateTrayWindows() ) );

  // register existing tray windows
  const QValueList<WId> systemTrayWindows = kwin_module->systemTrayWindows();
  QValueList<WId>::ConstIterator it;
  for ( it=systemTrayWindows.begin(); it!=systemTrayWindows.end(); ++it )
    systemTrayWindowAdded( *it );
}

SystemTrayApplet::~SystemTrayApplet()
{ 
  m_Wins.clear();
}

void SystemTrayApplet::systemTrayWindowAdded( WId w )
{
  KXEmbed *emb = new KXEmbed(this);
  connect(emb, SIGNAL(embeddedWindowDestroyed()), SLOT(updateTrayWindows()));
  m_Wins.append(emb);

  emb->embed(w);
  emb->resize(24, 24);
  emb->show();

  layoutTray();
  emit updateLayout();
}

void SystemTrayApplet::updateTrayWindows()
{
  KXEmbed *emb = m_Wins.first();
  while ((emb = m_Wins.current()) != 0L) {
    WId wid = emb->embeddedWinId();
    if ((wid == 0) || !kwin_module->systemTrayWindows().contains(wid))
      m_Wins.remove(emb);
    else
      m_Wins.next();
  }
  layoutTray();
  emit updateLayout();
}

int SystemTrayApplet::widthForHeight(int h) const
{
  int ret;

  if (h < 48)
    ret = m_Wins.count() * 24 + 4;
  else
    ret = ( (m_Wins.count()+1) / 2 ) * 24 + 4;

  return (ret > 28) ? ret : 28;
}

int SystemTrayApplet::heightForWidth(int w) const
{
  int ret;
  
  if (w < 48)
    ret =  m_Wins.count() * 24 + 4;
  else
    ret = ( (m_Wins.count()+1) / 2 ) * 24 + 4;
  
  return (ret > 28) ? ret : 28;
}

void SystemTrayApplet::resizeEvent( QResizeEvent* )
{
  layoutTray();
}
	
void SystemTrayApplet::layoutTray()
{
  if (m_Wins.count() == 0)
    return;

  int i;

  KXEmbed *emb;
  int col = 0;
  if (orientation() == Vertical) {
    i = 0;
    for (emb = m_Wins.first(); emb != 0L; emb = m_Wins.next()) {
      if ( (m_Wins.count() == 1) || width() < 48 )
	emb->move(width() / 2 - 12, 2 + i*24);
      else {
	emb->move(((i%2) == 0) ? 2 : width() - 26, 2 + col*24);
	if ( (i%2) != 0 )
	  ++col;
      }
      i++;
    }
  }
  else {
    i = 0;
    for (emb = m_Wins.first(); emb != 0L; emb = m_Wins.next()) {
      if ( (m_Wins.count() == 1) || height() < 48 )
	emb->move(2 + i*24, height() / 2 - 12);
      else {
	emb->move(2 + col*24, ((i%2) == 0) ? 2 : height() - 26);
	if ( (i%2) != 0 )
	  ++col;
      }
      i++;
    }
  }
  updateGeometry();
  show();
}

