/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <math.h>
#include <unistd.h>

#include <qdragobject.h>
#include <qpixmap.h>
#include <qfileinfo.h>
#include <qbitmap.h>

#include <kapp.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <kurl.h>
#include <kdebug.h>
#include <kdesktopfile.h>
#include <kmimetype.h>
#include <kprocess.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <dcopclient.h>

#include "containerarea.h"
#include "containerarea.moc"
#include "appletinfo.h"
#include "dirdrop_mnu.h"
#include "exe_dlg.h"
#include "panel.h"

#include "container_applet.h"
#include "container_button.h"

ContainerArea::ContainerArea( Orientation orient, QWidget* parent, const char* name)
    : Panner( orient, parent, name )
, DCOPObject("appletArea")
, _block_relayout(false)
, _movingAC(false)
, _moveAC(0)
, _moveOffset(QPoint(0,0))
{
    setAcceptDrops(true);
    _containers.setAutoDelete(false);

    // restore applet layout or load a default panel layout
    KConfig* config = KGlobal::config();
    config->setGroup("General");

    if(config->hasKey("Applets"))
	loadContainerConfig();
    else
	defaultContainerConfig();
}

ContainerArea::~ContainerArea()
{
    // clear applets
    removeAllContainers();
}

void ContainerArea::defaultContainerConfig()
{
    // clear applets
    removeAllContainers();

    // kmenu
    KMenuButtonContainer *kmenu = new KMenuButtonContainer(viewport());
    addContainer(kmenu);

    // window list
    WindowListButtonContainer *wlist = new WindowListButtonContainer(viewport());
    addContainer(wlist);

    // the desktop button
    DesktopButtonContainer *desktop = new DesktopButtonContainer(viewport());
    addContainer(desktop);

    // some url buttons
    URLButtonContainer *url;

    QRect r = PGlobal::panel->initialGeometry();
    int dsize;
    if (orientation() == Horizontal)
	dsize = r.width();
    else
	dsize = r.height();

    dsize -= 360;

    QStringList buttons;
    buttons << "Home.desktop";
    buttons << "System/konsole.desktop";
    buttons << "KControl.desktop";
    buttons << "Help.desktop";
    buttons << "Internet/konqbrowser.desktop";
    buttons << "Internet/KMail.desktop";
    buttons << "Editors/kwrite.desktop";
    buttons << "Office/kword.desktop";
    buttons << "Office/kspread.desktop";
    buttons << "Office/kpresenter.desktop";
    buttons << "Office/killustrator.desktop";

    int size = dsize;
    for (QStringList::ConstIterator it = buttons.begin(); it != buttons.end(); ++it) {
	size -= 42;
	if (size <= 0)
	    break;
	QString s = locate("apps", *it);
	if (s.isEmpty()) continue;

	url = new URLButtonContainer(viewport(), s);
	addContainer(url);
	size -= 42;
    }

    // pager applet
    QString df = KGlobal::dirs()->findResource("applets", "kminipagerapplet.desktop");
    InternalAppletContainer *pager = new InternalAppletContainer(AppletInfo(df) ,viewport());
    addContainer(pager);

    // taskbar applet
    df = KGlobal::dirs()->findResource("applets", "ktaskbarapplet.desktop");
    InternalAppletContainer *taskbar = new InternalAppletContainer(AppletInfo(df), viewport());
    addContainer(taskbar);

    // system tray applet
    df = KGlobal::dirs()->findResource("applets", "ksystemtrayapplet.desktop");
    InternalAppletContainer *systemTray = new InternalAppletContainer(AppletInfo(df), viewport());
    systemTray->setFreeSpace(1);
    addContainer(systemTray);

    // date applet
    df = KGlobal::dirs()->findResource("applets", "clockapplet.desktop");
    InternalAppletContainer *date = new InternalAppletContainer(AppletInfo(df), viewport());
    date->setFreeSpace(1);
    addContainer(date);

    layoutChildren();
    saveContainerConfig();
}

void ContainerArea::saveContainerConfig(bool layoutOnly)
{
    kdDebug() << "ContainerArea::saveContainerConfig()" << endl;

    KConfig *config = KGlobal::config();
    config->setGroup("General");

    // build the applet list
    QStringList alist;

    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it)
	alist.append( it.current()->appletId() );

    // write applet list (group 'panel', key 'applets')
    config->writeEntry("Applets", alist);

    // write applet config
    it.toFirst();
    for(; it.current() ; ++it)
	{
	    BaseContainer* a = it.current();

	    // set group to the applet id
	    config->setGroup(a->appletId());

	    // write positioning info
	    config->writeEntry("FreeSpace", QString("%1").arg(a->freeSpace()));

	    // write size hint
	    if(a->inherits("AppletContainer")) {
		if(orientation() == Horizontal)
		    config->writeEntry("WidthForHeightHint", QString("%1").arg(a->widthForHeight(height())));
		else
		    config->writeEntry("HeightForWidthHint", QString("%1").arg(a->heightForWidth(width())));
	    }

	    // let the applet container write custom data fields
	    if(!layoutOnly)
	    a->saveConfiguration(a->appletId());
	}

    config->sync();
}

void ContainerArea::loadContainerConfig()
{
    kdDebug() << "ContainerArea::loadContainerConfig()" << endl;

    KConfig *config = KGlobal::config();

    // clear panel
    removeAllContainers();

    // read applet list
    config->setGroup("General");
    QStringList alist = config->readListEntry("Applets");
    QStringList trusted = config->readListEntry("TrustedApplets");

    // now restore the applets
    QStringList::Iterator it = alist.begin();
    while(it != alist.end())
	{
	    // applet id
	    QString appletId(*it);
	    QString group = appletId;

	    // is there a config group for this applet?
	    if(!config->hasGroup(group))
		continue;

	    // set config group
	    config->setGroup(group);

	    // read free space
	    float fspace = (float) config->readDoubleNumEntry("FreeSpace");

	    BaseContainer* a = 0;

	    // create a matching applet container
	    if (appletId.contains("KMenuButton") > 0)
		a = new KMenuButtonContainer(viewport());
	    if (appletId.contains("DesktopButton") > 0)
		a = new DesktopButtonContainer(viewport());
	    else if (appletId.contains("WindowListButton") > 0)
		a = new WindowListButtonContainer(viewport());
	    else if (appletId.contains("URLButton") > 0)
		a = new URLButtonContainer(group, viewport());
	    else if (appletId.contains("BrowserButton") > 0)
		a = new BrowserButtonContainer(group, viewport());
	    else if (appletId.contains("ExeButton") > 0)
		a = new ExeButtonContainer(group, viewport());
	    else if (appletId.contains("Applet") > 0)
		{
		    int whint = config->readNumEntry("WidthForHeightHint", 0);
		    int hhint = config->readNumEntry("HeightForWidthHint", 0);

		    KConfig *config = KGlobal::config();
		    config->setGroup(group);

		    QString df = KGlobal::dirs()->findResource("applets", config->readEntry("DesktopFile"));
		    AppletInfo info(df);

		    if (info.isUniqueApplet() && hasInstance(&info)) {
			it++;
			continue;
		    }

		    QString configFile = config->readEntry("ConfigFile");
		    if (!configFile.isNull()) info.setConfigFile(configFile);

		    config->setGroup("General");

		    if(config->readNumEntry("SecurityLevel", 1) == 0)
			{
			    QString lib = info.library().mid(3, info.library().length());
			    bool trustedapplet = false;
			    for ( QStringList::Iterator it = trusted.begin(); it != trusted.end(); ++it )
				{
				    if ((*it) == lib)
					trustedapplet = true;
				}

			    if (trustedapplet == true)
				a = new InternalAppletContainer(info, viewport());
			    else
				a = new ExternalAppletContainer(info, viewport());
			}
		    else
			a = new InternalAppletContainer(info, viewport());
		    ((AppletContainer*)a)->setWidthForHeightHint(whint);
		    ((AppletContainer*)a)->setHeightForWidthHint(hhint);
		}

	    if (a) {
		a->setFreeSpace(fspace);
		addContainer(a);
	    }
	    it++;
	}

    layoutChildren();
}

bool ContainerArea::hasInstance(AppletInfo* info)
{
    bool found = false;

    for (QListIterator<BaseContainer> it(_containers); it.current(); ++it )
	{
	    BaseContainer *a = static_cast<BaseContainer*>(it.current());
	    if (a->inherits("AppletContainer")) {
		if (static_cast<AppletContainer*>(a)->info().library() == info->library()) {
		    found = true;
		    break;
		}
	    }
	}
    return found;
}

void ContainerArea::removeAllContainers()
{
    while ( !_containers.isEmpty() ) {
	BaseContainer* b = _containers.first();
	_containers.removeRef( b );
	delete b;
    }
}

void ContainerArea::configure()
{
	kdDebug() << "ContainerArea::configure()" << endl;
    // set background pixmap
    KConfig* config = KGlobal::config();
    config->setGroup("General");
    if (config->readBoolEntry("UseBackgroundTheme", false))
	{
	    QString bgStr = config->readEntry("BackgroundTheme", "");
	    if(!bgStr.isEmpty()){
		QPixmap bgPix(bgStr);
		if(!bgPix.isNull()){
			// Do we need to rotate the image
			QPixmap bgPixNew;

			if ( config->readBoolEntry("RotateBackground", false) &&
				 orientation() == Vertical )
			{
				// Rotate the pixmap before scaling
				QWMatrix m;
				m.rotate( -90.0 );
				bgPixNew = bgPix.xForm( m );
			}
			else
			{
				// Don't rotate the image - just copy it
				bgPixNew = bgPix;
			}

			// Scale the image but keep the same aspect ratio
			QImage bgImage = bgPixNew.convertToImage();
			double dAspect = (double)bgPixNew.width() / (double)bgPixNew.height();

			int nNewWidth, nNewHeight;
			int panelsize = PGlobal::panel->panelSize();
			if ( config->readBoolEntry("RotateBackground", false) &&
				 orientation() == Vertical )
			{
				nNewWidth  = panelsize;
				nNewHeight = (int)( (double)nNewWidth / dAspect );
			}
			else
			{
				nNewHeight = panelsize;
				nNewWidth  = (int)( (double)nNewHeight * dAspect);
			}
			//kdDebug() << "Image size = " << nNewWidth << " " << nNewHeight << endl;
			//
			QImage bgImageNew = bgImage.smoothScale( nNewWidth, nNewHeight );

			// Convert back to a QPixmap
			bgPixNew.convertFromImage( bgImageNew );

			QBrush bgBrush(colorGroup().background(), bgPixNew);
		    QPalette pal = kapp->palette();
		    pal.setBrush(QColorGroup::Background, bgBrush);
		    setPalette(pal);
		}
		else
		    {
			unsetPalette();
			kdWarning() << "Kicker: Error loading background theme pixmap\n";
		    }
	    }
	}
    else
	{
	    unsetPalette();
	}

    for (QListIterator<BaseContainer> it(_containers); it.current(); ++it )
	{
	    BaseContainer *a = it.current();
	    if (a)
		  a->configure();
	}
}

void ContainerArea::addKMenuButton()
{
    KMenuButtonContainer *b = new KMenuButtonContainer(viewport());
    addContainer(b);
    moveToFirstFreePosition(b);
    saveContainerConfig();
}

void ContainerArea::addDesktopButton()
{
    DesktopButtonContainer *b = new DesktopButtonContainer(viewport());
    addContainer(b);
    moveToFirstFreePosition(b);
    saveContainerConfig();
}

void ContainerArea::addWindowListButton()
{
    WindowListButtonContainer *b = new WindowListButtonContainer(viewport());
    addContainer(b);
    moveToFirstFreePosition(b);
    saveContainerConfig();
}

void ContainerArea::addURLButton(const QString &url)
{
    URLButtonContainer *b = new URLButtonContainer(viewport(), url);
    addContainer(b);
    moveToFirstFreePosition(b);
    saveContainerConfig();
}

void ContainerArea::addBrowserButton(const QString &startDir)
{
    BrowserButtonContainer *b = new BrowserButtonContainer( viewport(), startDir);
    addContainer(b);
    moveToFirstFreePosition(b);
    saveContainerConfig();
}

void ContainerArea::addExeButton(const QString &filePath, const QString &icon,
				 const QString &cmdLine, bool inTerm)
{
    ExeButtonContainer *b = new ExeButtonContainer(viewport(), filePath, icon, cmdLine, inTerm);
    addContainer(b);
    moveToFirstFreePosition(b);
    saveContainerConfig();
}


void ContainerArea::addApplet( const QString& desktopFile )
{
    bool internal = false;
    KConfig *c = KGlobal::config();
    c->setGroup("General");
    if(c->readNumEntry("SecurityLevel") > 1)
	internal = true;
    addAppletContainer( desktopFile, internal );
}

void ContainerArea::addAppletContainer(const QString &desktopFile, bool internal)
{
    QString df = KGlobal::dirs()->findResource("applets", desktopFile);
    AppletInfo info(df);

    if (info.isUniqueApplet() && hasInstance(&info))
	return;

    AppletContainer *a;

    if (internal)
	{
	    a = new InternalAppletContainer(info, viewport());
	    addContainer(a);
	    moveToFirstFreePosition(a);
	    saveContainerConfig();
	}
    else
	{
	    a = new ExternalAppletContainer(info, viewport());
	    connect(a, SIGNAL(docked(ExternalAppletContainer*)),
		    SLOT(slotAddExternal(ExternalAppletContainer*)));
	}
}

void ContainerArea::slotAddExternal(ExternalAppletContainer* a)
{
    addContainer(a);
    moveToFirstFreePosition(a);
    saveContainerConfig();
}

void ContainerArea::addContainer(BaseContainer* a)
{
    if (!a) return;

    setUniqueId(a);

    _containers.append(a);

    connect(a, SIGNAL(moveme(BaseContainer*) ),
	    SLOT( startContainerMove(BaseContainer*)));
    connect(a, SIGNAL(removeme(BaseContainer*) ),
	    SLOT( slotRemoveContainer(BaseContainer*)));
    connect(a, SIGNAL(requestSave()),
	    SLOT(slotSaveContainerConfig()));

    if (a->inherits("ExternalAppletContainer"))
	connect(a, SIGNAL(embeddedWindowDestroyed() ), this,
		SLOT( embeddedWindowDestroyed()));
    if (a->inherits("InternalAppletContainer")
	|| a->inherits("ExternalAppletContainer"))
	connect(a, SIGNAL(updateLayout() ), this,
		SLOT( slotLayoutChildren()));

    Direction d;
    switch (PGlobal::panel->position())
	{
	case Left:
	    d = dRight;
	    break;
	case Right:
	    d = dLeft;
	    break;
	case Top:
	    d = dDown;
	    break;
	case Bottom:
	default:
	    d = dUp;
	    break;
	}

    a->slotSetOrientation( Panner::orientation());
    a->slotSetPopupDirection(d);
    a->configure();
    addChild(a);
    a->show();
}

void ContainerArea::removeContainer(BaseContainer *a)
{
    if (a) {
	removeChild(a);
	delete a;
	_containers.removeRef(a);
    }

    updateContainerList();
    layoutChildren();
    updateArrows();
    saveContainerConfig(true);
}

void ContainerArea::setUniqueId(BaseContainer* a)
{
    QString idBase = a->appletType() + "_%1";
    QString newId;
    int i = 0;
    bool unique = false;

    while(!unique)
	{
	    i++;
	    newId = idBase.arg(i);

	    unique = true;
	    QListIterator<BaseContainer> it(_containers);
	    for(; it.current() ; ++it)
		{
		    BaseContainer* b = static_cast<BaseContainer*>(it.current());
		    if (b->appletId() == newId)
			{
			    unique = false;
			    break;
			}
		}
	}
    a->setAppletId(newId);
}

bool ContainerArea::isStretch(BaseContainer* a)
{
    if (!a->inherits("AppletContainer"))
	return false;
    return (static_cast<AppletContainer*>(a)->type() ==  KPanelApplet::Stretch);
}

void ContainerArea::startContainerMove(BaseContainer *a)
{
    if (!a) return;

    _moveAC = a;
    _movingAC = true;
    setMouseTracking(true);
    QCursor::setPos(mapToGlobal(QPoint(a->x() + a->moveOffset().x(), a->y() + a->moveOffset().y())));
    grabMouse(sizeAllCursor);

    _block_relayout = true;
    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it)
	{
	    BaseContainer* b = static_cast<BaseContainer*>(it.current());

	    if (orientation() == Horizontal)
		b->resize(b->widthForHeight(height()), height());
	    else
		b->resize(width(), b->heightForWidth(width()));
	}

    a->raise();
}

void ContainerArea::stopContainerMove(BaseContainer *b)
{
    if (_moveAC != b) return;

    releaseMouse();
    setCursor(arrowCursor);
    _movingAC = false;
    setMouseTracking(false);

    if(_moveAC->inherits("ButtonContainer"))
	static_cast<ButtonContainer*>(_moveAC)->completeMoveOperation();

    _moveAC = 0;
    _block_relayout = false;

    updateContainerList();
    layoutChildren();
    saveContainerConfig(true);
}

void ContainerArea::mouseReleaseEvent(QMouseEvent *)
{
    if (_movingAC && _moveAC)
	stopContainerMove(_moveAC);
}

void ContainerArea::mouseMoveEvent(QMouseEvent *ev)
{
    if (!(_movingAC && _moveAC)) {
	Panner::mouseMoveEvent(ev);
	return;
    }

    int s;
    if (orientation() == Horizontal)
	s = width();
    else
	s = height();

    if (ev->state() & ShiftButton && s >= minimumUsedSpace()) {

	if (orientation() == Horizontal) {
	    int oldX = _moveAC->x() + _moveAC->moveOffset().x();
	    int x = ev->pos().x();
	    moveContainerPush(_moveAC, x - oldX);
	}
	else if (orientation() == Vertical) {
	    int oldY = _moveAC->y() + _moveAC->moveOffset().y();
	    int y = ev->pos().y();
	    moveContainerPush(_moveAC, y - oldY);
	}
    }
    else {

	if (orientation() == Horizontal) {
	    int oldX = _moveAC->x() + _moveAC->moveOffset().x();
	    int x = ev->pos().x();
	    moveContainerSwitch(_moveAC, x - oldX);
	}
	else if (orientation() == Vertical) {
	    int oldY = _moveAC->y() + _moveAC->moveOffset().y();
	    int y = ev->pos().y();
	    moveContainerSwitch(_moveAC, y - oldY);
	}
    }
}

void ContainerArea::moveContainerSwitch(BaseContainer* moving, int distance)
{
    // horizontal panel
    if (orientation() == Horizontal) {

	if (distance > 0) { // left to right

	    _containers.findRef(moving);
	    BaseContainer *next = _containers.next();
	    BaseContainer *last = moving;

	    while (next) {

		// 'moving' has completely passed applet 'next'.
		if ( next->x() + next->width() <= moving->x() + distance ) {
		    next->move(next->x() - moving->width(), next->y());
		    last = next;
		    next = _containers.next();
		    continue;
		}

		// 'next' has not been completely passed by 'moving', but
		// still may be covered by it.
		int switchMargin = 0;

		// calculate the position and width of the 'virtual' container
		// containing 'moving' and 'next'.
		int tx = next->x() - moving->width();
		int twidth = moving->width() + next->width();

		// determine the middle of the containers.
		int tmiddle = tx + twidth/2;
		int movingMiddle = moving->x() + distance + moving->width()/2;

		// move 'next' from the right side of the virtual container to
		// the left side if the middle of 'moving' has moved far enough
		// to the left, i.e. past the middle of the virtual container
		// plus the switchMargin. The switchMargin prevents rapidly
		// switching when 'moving' and 'next' have the same size.
		if (movingMiddle >= tmiddle + switchMargin) {
		    next->move(next->x() - moving->width(), next->y());
		    // store 'next', because it may become null in the next
		    // step.
		    last = next;
		    next = _containers.next();
		    continue;
		}

		// 'moving' doesn't cover 'next', and hasn't passed it. Then
		// we know that this also yields for the rest of the applets,
		// so leave the loop.
		break;
	    }

	    int newX;

	    if (last != moving) {
		newX = QMAX(last->x() + last->width(), moving->x() + distance);
		moving->move(newX, moving->y());

		// Move 'moving' to its new position in the container list.
		_containers.removeRef(moving);
		_containers.insert( _containers.findRef(last) + 1, moving );
	    }
	    else
		if (next && moving->x() + distance >= next->x() - moving->width())
		    newX = next->x() - moving->width();
		else
		    newX = moving->x() + distance;

	    // Make sure the container isn't moved outside of the panel.
	    newX = QMIN(newX, width() - moving->width());
	    moving->move(newX, moving->y());
	}

	else if (distance < 0) { // right to left

	    _containers.findRef(moving);
	    BaseContainer *prev = _containers.prev();
	    BaseContainer *last = moving;

	    while (prev) {

		if ( moving->x() + distance + moving->width() <= prev->x() ) {
		    prev->move(prev->x() + moving->width(), prev->y());
		    last = prev;
		    prev = _containers.prev();
		    continue;
		}

		int switchMargin = 0;

		// calculate the position and width of the 'virtual' container
		// containing 'moving' and 'prev'.
		int tx = prev->x();
		int twidth = moving->width() + prev->width();

		// determine the middle of the containers.
		int tmiddle = tx + twidth/2;
		int movingMiddle = moving->x() + distance + moving->width()/2;

		// move a from the left side of the virtual container to the
		// right side if the middle of 'moving' has moved past the
		// middle of the virtual container plus the switchMargin. The
		// switchMargin prevents rapidly switching when 'moving' and
		// 'prev' have the same size.
		if (movingMiddle <= tmiddle + switchMargin) {
		    prev->move(prev->x() + moving->width(), prev->y());
		    last = prev;
		    prev = _containers.prev();
		    continue;
		}

		break;
	    }

	    int newX;

	    if (last != moving) {
		newX = QMIN(last->x() - moving->width(), moving->x() + distance);
		// Move 'moving' to its new position in the container list.
		_containers.removeRef(moving);
		_containers.insert( _containers.findRef(last), moving );
	    }
	    else
		if (prev && moving->x() + distance < prev->x() + prev->width())
		    newX = prev->x() + prev->width();
		else
		    newX = moving->x() + distance;

	    // Make sure the container isn't moved outside of the panel.
	    newX = QMAX(newX, 0);
	    moving->move(newX, moving->y());
	}
    }

    // vertical panel
    else if (orientation() == Vertical) {

	if (distance > 0) { // top to bottom

	    _containers.findRef(moving);
	    BaseContainer *next = _containers.next();
	    BaseContainer *last = moving;

	    while (next) {

		if ( next->y() + next->height() <= moving->y() + distance) {
		    next->move(next->x(), next->y() - moving->height());
		    last = next;
		    next = _containers.next();
		    continue;
		}

		int switchMargin = 0;

		// calculate the position and height of the 'virtual' container
		// containing 'moving' and 'next'.
		int ty = next->y() - moving->height();
		int theight = moving->height() + next->height();

		// determine the middle of the containers.
		int tmiddle = ty + theight/2;
		int movingMiddle = moving->y() + distance + moving->height()/2;

		// move 'next' from the bottom of the virtual container to the
		// top side if the middle of 'moving' has moved past the middle
		// of the virtual container plus the switchMargin. The
		// switchMargin prevents rapidly switching when 'moving' and
		// 'next' have the same size.
		if (movingMiddle >= tmiddle + switchMargin) {
		    next->move(next->x(), next->y() - moving->height());
		    last = next;
		    next = _containers.next();
		    continue;
		}

		break;
	    }

	    int newY;

	    if (last != moving) {
		newY = QMAX(last->y() + last->height(), moving->y() + distance);

		// Move 'moving' to its new position in the container list.
		_containers.removeRef(moving);
		_containers.insert( _containers.findRef(last) + 1, moving );
	    }
	    else
		if (next && moving->y() + distance >= next->y() - moving->height())
		    newY = next->y() - moving->height();
		else
		    newY = moving->y() + distance;

	    // Make sure the container isn't moved outside of the panel.
	    newY = QMIN(newY, height() - moving->height());
	    moving->move(moving->x(), newY);
	}

	else if (distance < 0) { // bottom to top

	    _containers.findRef(moving);
	    BaseContainer *prev = _containers.prev();
	    BaseContainer *last = moving;

	    while (prev) {

		if ( moving->y() + moving->height() <= prev->y() ) {
		    prev->move(prev->x(), prev->y() + moving->height());
		    last = prev;
		    prev = _containers.prev();
		    continue;
		}

		int switchMargin = 0;

		// calculate the position and height of the 'virtual' container
		// containing 'moving' and 'prev'.
		int ty = prev->y();
		int theight = moving->height() + prev->height();

		// determine the middle of the containers.
		int tmiddle = ty + theight/2;
		int movingMiddle = moving->y() + distance + moving->height()/2;

		// move 'prev' from the top of the virtual container to the
		// bottom if the middle of 'moving' has moved past the middle
		// of the virtual container plus the switchMargin. The
		// switchMargin prevents rapidly switching when 'moving' and
		// 'prev' have the same size.
		if (movingMiddle <= tmiddle + switchMargin) {
		    prev->move(prev->x(), prev->y() + moving->height());
		    last = prev;
		    prev = _containers.prev();
		    continue;
		}

		break;
	    }

	    int newY;

	    if (last != moving) {
		newY = QMIN(last->y() - moving->height(), moving->y() + distance);
		moving->move(moving->x(), newY);

		// Move 'moving' to its new position in the container list.
		_containers.removeRef(moving);
		_containers.insert( _containers.findRef(last), moving );
	    }
	    else
		if (prev && moving->y() + distance < prev->y() + prev->height())
		    newY = prev->y() + prev->height();
		else
		    newY = moving->y() + distance;

	    // Make sure the container isn't moved outside of the panel.
	    newY = QMAX(newY, 0);
	    moving->move(moving->x(), newY);
	}
    }
}

void ContainerArea::moveContainerPush(BaseContainer* a, int distanceRequest)
{
    // Point the iterator 'it' to 'a'.
    QListIterator<BaseContainer> it(_containers);
    while (it.current() && it.current() != a)
	++it;

    moveContainerPushRec(it, distanceRequest);
}

int ContainerArea::moveContainerPushRec(QListIterator<BaseContainer> it, int distanceRequest)
{
    int distanceAvailable, distanceDisposed;
    BaseContainer* a = it.current();
    BaseContainer* b;

    if (orientation() == Horizontal) {

	if (distanceRequest < 0) {
	    // Find the previous container. If it exists, determine the
	    // distance between the two containers.
	    b = --it;
	    if (!b)
		distanceAvailable = -a->x();
	    else {
		distanceAvailable = b->x() - a->x() + b->width();
		if (distanceRequest - distanceAvailable < 0)
		    distanceAvailable += moveContainerPushRec(it, distanceRequest - distanceAvailable);
	    }
	    distanceDisposed = QMAX(distanceRequest, distanceAvailable);
	}
	else if (distanceRequest > 0) {
	    // Find the next container. If it exists, determine the distance
	    // between the two containers.
	    b = ++it;
	    if (!b)
		distanceAvailable = width() - a->x() - a->width();
	    else {
		distanceAvailable = b->x() - a->x() - a->width();
		if (distanceRequest - distanceAvailable > 0)
		    distanceAvailable += moveContainerPushRec(it, distanceRequest - distanceAvailable);
	    }
	    distanceDisposed = QMIN(distanceRequest, distanceAvailable);
	}
	else
	    return 0;

	a->move(a->x() + distanceDisposed, a->y());
	return distanceDisposed;
    }
    else if (orientation() == Vertical) {

	if (distanceRequest < 0) {
	    // Find the previous container. If it exists, determine the
	    // distance between the current and the previous container.
	    // If this distance is not enough to comply to the request,
	    // try to push the previous container.
	    b = --it;
	    if (!b)
		distanceAvailable = -a->y();
	    else {
		distanceAvailable = b->y() - a->y() + b->height();
		if (distanceRequest - distanceAvailable < 0)
		    distanceAvailable += moveContainerPushRec(it, distanceRequest - distanceAvailable);
	    }
	    distanceDisposed = QMAX(distanceRequest, distanceAvailable);
	}
	else if (distanceRequest > 0) {
	    // Find the next container. If it exists, determine the distance
	    // between the current and the next container. If this distance is
	    // not enough to comply to the request, try to push the next
	    // container.
	    b = ++it;
	    if (!b)
		distanceAvailable = height() - a->y() - a->height();
	    else {
		distanceAvailable = b->y() - a->y() - a->height();
		if (distanceRequest - distanceAvailable > 0)
		    distanceAvailable += moveContainerPushRec(it, distanceRequest - distanceAvailable);
	    }
	    distanceDisposed = QMIN(distanceRequest, distanceAvailable);
	}
	else
	    return 0;

	a->move(a->x(), a->y() + distanceDisposed);
	return distanceDisposed;
    }

    return 0;
}

int ContainerArea::position()
{
    return static_cast<int>(PGlobal::panel->position());
}

int ContainerArea::popupDirection()
{
    Direction dir;
    switch (position())
	{
	case Left:
	    dir = dRight;
	    break;
	case Right:
	    dir = dLeft;
	    break;
	case Top:
	    dir = dDown;
	    break;
	case Bottom:
	default:
	    dir = dUp;
	    break;
	}
    return dir;
}

int ContainerArea::orientation()
{
    return static_cast<int>(PGlobal::panel->orientation());
}

void ContainerArea::slotLayoutChildren()
{
    layoutChildren();
}

void ContainerArea::embeddedWindowDestroyed()
{
    if (sender() && sender()->inherits("ExternalAppletContainer"))
	removeContainer((ExternalAppletContainer*)sender());
}

void ContainerArea::layoutChildren()
{
    if (_block_relayout) return;

    QSize newsize = size();
    int mus = minimumUsedSpace();

    if (orientation() == Horizontal) {
	if (newsize.width() < mus)
	    newsize.setWidth(mus);
    }
    else {
	if (newsize.height() < mus)
	    newsize.setHeight(mus);
    }
    resizeContents(newsize.width()+1, newsize.height()+1);

    int pos = 0;

    int occupiedspace = 0;
    int freespace = totalFreeSpace();

    for (BaseContainer *a = _containers.first(); a != 0; a = _containers.next())
	{
	    a->slotSetOrientation((Orientation)orientation());
	    a->slotSetPopupDirection((Direction)popupDirection());

	    // get pointer to the nextapplet
	    BaseContainer *next = _containers.next();

	    // reset current
	    _containers.find(a);

	    double fspace = a->freeSpace()*freespace;
	    if ((fspace - floor(fspace)) > 0.5)
		fspace += 1;
	    pos = static_cast<int>(fspace) + occupiedspace;

	    if (orientation() == Horizontal) {
		moveChild(a, pos, 0);
		int w = a->widthForHeight(height());
		if (isStretch(a)) {
		    if (next)
			a->resize(w + (next->freeSpace() - a->freeSpace())*freespace, height());
		    else
			a->resize(width() - a->x(), height());
		}
		else
		    a->resize(w, height());
		occupiedspace += w;
	    }
	    else {
		moveChild(a, 0, pos);
		int h = a->heightForWidth(width());
		if (isStretch(a)) {
		    if (next)
			a->resize(width(), h + (next->freeSpace() - a->freeSpace())*freespace);
		    else
			a->resize(width(), height() - a->y());
		}
		else
		    a->resize(width(), h);
		occupiedspace += h;
	    }
	}
}

void ContainerArea::dragEnterEvent(QDragEnterEvent *ev)
{
    ev->accept(QUriDrag::canDecode(ev));
}

void ContainerArea::dropEvent(QDropEvent *ev)
{
    QStringList uriList;

    if (QUriDrag::decodeToUnicodeUris(ev, uriList)) {

        QStringList::ConstIterator it(uriList.begin());

        for (; it != uriList.end(); ++it) {

            BaseContainer* a;
            KURL url(*it);

            // see if it's a executable or directory
            if(url.isLocalFile() && !KDesktopFile::isDesktopFile(url.path()))
		{
		    QFileInfo fi(url.path());
		    if(fi.isDir())  // directory
			{
			    PanelDirDropMenu mnu;
			    switch(mnu.exec(mapToGlobal(ev->pos()))){
			    case PanelDirDropMenu::Browser:
				a = new BrowserButtonContainer(viewport(), url.path());
				break;
			    case PanelDirDropMenu::Url:
			    default:
				a = new URLButtonContainer(viewport(), *it);
				break;
			    }
			}
		    else if(fi.isExecutable())  // non-KDE executable
			{
			    QString pixmapFile;
			    KMimeType::pixmapForURL(url.path(), 0, KIcon::Panel, 0,
						    KIcon::DefaultState, &pixmapFile);
			    PanelExeDialog dlg(url.path(), pixmapFile,
					       QString::null, false, 0);
			    if(dlg.exec() == QDialog::Accepted){
				// KIconloader returns a full path, we only want name
				QFileInfo iconfi(dlg.icon());
				a = new ExeButtonContainer(viewport(), url.path(), iconfi.fileName(),
							   dlg.commandLine(),
							   dlg.useTerminal());
			    }
			    else
				break;
			}
		    else { // some unknown local file
			a = new URLButtonContainer(viewport(), *it);
		    }
		}
            else { // a internet URL or desktop file.
                a= new URLButtonContainer(viewport(), *it);
            }
            addContainer(a);
            moveToFirstFreePosition(a);
            saveContainerConfig();
        }
    }
}

void ContainerArea::moveToFirstFreePosition(BaseContainer* a)
{
    Orientation orient = Panner::orientation();

    int w = a->widthForHeight(height());
    int h = a->heightForWidth(width());

    bool stretch = false;
    bool found = false;

    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it)
	{
	    BaseContainer* b = static_cast<BaseContainer*>(it.current());

	    int space = relativeContainerPos(b);

	    if (orient == Horizontal) {
		if (space >= w)
		    {
			if(stretch)
			    moveChild(a, b->pos().x() - w, a->y());
			else
			    moveChild(a, b->pos().x() - space, a->y());
			found = true;
			break;
		    }
	    }
	    else {
		if (space >= h)
		    {
			if(stretch)
			    moveChild(a, b->pos().x() - h, a->y());
			else
			    moveChild(a, a->x(), b->pos().y() - space);
			found = true;
			break;
		    }
	    }
	    stretch = isStretch(b);
	}

    if (found)
	updateContainerList();
    layoutChildren();
}

BaseContainer* ContainerArea::coversContainer(BaseContainer *a, bool strict)
{
    BaseContainer *b;
    QListIterator<BaseContainer> it(_containers);

    for(; it.current() ; ++it)
	{
	    b = (BaseContainer*)it.current();

	    if (b == a) continue;

	    if ( orientation() == Horizontal ) {
		int bl, br;
		if (strict) {
		    bl = b->x();
		    br = b->x() + b->width();
		}
		else {
		    bl = b->x() + 10;
		    br = b->x() + b->width() - 10;
		}

		int btnl = a->x();
		int btnr = btnl + a->width();

		if ((btnl >= bl) && (btnl <= br))
		    return b;
		else if ((btnr >= bl) && (btnr <= br))
		    return b;
	    }
	    else {
		int bt, bb;
		if (strict) {
		    bt = b->y();
		    bb = b->y() + b->height();
		}
		else {
		    bt = b->y() + 10;
		    bb = b->y() + b->height() - 10;
		}
		int btnt = a->y();
		int btnb = btnt + a->height();

		if ((btnt >= bt) && (btnt <= bb))
		    return b;
		else if ((btnb >= bt) && (btnb <= bb))
		    return b;
	    }
	}
    return 0;
}

void ContainerArea::updateContainerList()
{
    QList<BaseContainer> sorted;

    while(!_containers.isEmpty())
	{
	    BaseContainer *b = 0;
	    int pos = 9999;

	    QListIterator<BaseContainer> it(_containers);

	    for(; it.current() ; ++it)
		{
		    BaseContainer* a = static_cast<BaseContainer*>(it.current());

		    if(orientation() == Horizontal)
			{
			    if (a->x() < pos) {
				b = a;
				pos = a->x();
			    }
			}
		    else
			{
			    if (a->y() < pos) {
				b = a;
				pos = a->y();
			    }
			}
		}

	    if (b) {
		sorted.append(b);
		_containers.remove(b);
	    }
	}
    _containers = sorted;

    float freespace = totalFreeSpace();
    float fspace = 0;

    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it)
	{
	    fspace += relativeContainerPos(it.current());
	    if (fspace < 0) fspace = 0;
	    it.current()->setFreeSpace(fspace/freespace);
	}
}

int ContainerArea::totalFreeSpace()
{
    int availablespace;
    int usedspace = 0;

    if(orientation() == Horizontal) {
	if(contentsWidth() > width())
	    availablespace = contentsWidth();
	else
	    availablespace = width();
    }
    else {
	if (contentsHeight() > height())
	    availablespace = contentsHeight();
	else
	    availablespace = height();
    }

    // calculate used space
    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it)
	{
	    BaseContainer* a = static_cast<BaseContainer*>(it.current());

	    int space;
	    if(orientation() == Horizontal)
		space = a->widthForHeight(height());
	    else
		space = a->heightForWidth(width());

	    if (space > 0)
		usedspace += space;
	}

    int freespace = availablespace - usedspace;
    if (freespace < 0) freespace = 0;

    return freespace;
}

int ContainerArea::minimumUsedSpace()
{
    int usedspace = 0;

    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it)
	{
	    BaseContainer* a = static_cast<BaseContainer*>(it.current());

	    int space;
	    if(orientation() == Horizontal)
		space = a->widthForHeight(height());
	    else
		space = a->heightForWidth(width());

	    if (space > 0)
		usedspace += space;
	}
    return usedspace;
}

int ContainerArea::relativeContainerPos(BaseContainer* b)
{
    if (!b) return 0;
    if (!_containers.contains(b)) return 0;

    uint pos = 0;

    QListIterator<BaseContainer> it(_containers);
    for(; it.current() ; ++it) {
	BaseContainer* a = static_cast<BaseContainer*>(it.current());

	if(orientation() == Horizontal) {
	    if (a == b)  {
		int p = b->x() - pos;
		if (b < 0) return 0;
		return p;
	    } else
		pos = a->x() + a->widthForHeight(height());
	} else  {
	    if (a == b) {
		int p = b->y() - pos;
		if (b < 0) return 0;
		return p;
	    }
	    else
		pos = a->y() + a->heightForWidth(width());
	}
    }
    return 0;
}

void ContainerArea::slotSaveContainerConfig()
{
    saveContainerConfig();
}

void ContainerArea::slotRemoveContainer(BaseContainer* a)
{
    removeContainer(a);
}
