/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/


#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#include <kapp.h>
#include <qlayout.h>
#include <qscrollview.h>
#include <qvbox.h>
#include <qcursor.h>
#include <qevent.h>
#include <qtimer.h>
#include <qfile.h>

#include <kglobal.h>
#include <kglobalsettings.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kwinmodule.h>
#include <dcopclient.h>
#include <kwin.h>
#include <kstddirs.h>

#include "panel.h"
#include "panel.moc"
#include "panelop_mnu.h"
#include "containerarea.h"
#include "containerareabox.h"
#include "hidebutton.h"
#include "pluginmgr.h"
#include "userrectsel.h"

#include <X11/Xlib.h>
const int XFocusOut = FocusOut;
const int XFocusIn = FocusIn;
#undef FocusOut
#undef FocusIn
#undef KeyPress
#undef KeyRelease

extern void qt_set_sane_enterleave( bool b );
extern Time qt_x_time;

Panel::Panel() : QWidget(0, "Panel", WStyle_Customize | WStyle_NoBorderEx)
  , DCOPObject("Panel")
  , _showHBs(true)
  , _HBwidth(10)
  , _autoHide(false)
  , _autoHidden(false)
  , _userHidden(false)
  , _hideAnim(true)
  , _autoHideAnim(true)
  , _autoHideDelay(3)
  , _hideAnimSpeed(5)
  , _autoHideAnimSpeed(50)
  , _block_user_input(false)
  , _layout(0)
  , _opMnu(0)
  ,_faked_activation(false)
{
  qt_set_sane_enterleave( true ); // enable enter/leave propagation

  // kicker lives in the dock
  KWin::setType( winId(), NET::Dock );
  KWin::setState( winId(), NET::StaysOnTop | NET::Sticky );
  KWin::setOnAllDesktops( winId(), TRUE );

  // instantiate plugin manager and kwin module
  PGlobal::pluginmgr = new KickerPluginManager();
  PGlobal::kwin_module = new KWinModule(this );
  connect( PGlobal::kwin_module, SIGNAL( workAreaChanged() ), this, SLOT( workAreaChanged() ) );

  // let PGlobal know we are there
  PGlobal::panel = this;

   // left/top hide button
  _ltHB = new HideButton(position(), this);
  _ltHB->installEventFilter( this );
  connect( _ltHB, SIGNAL( clicked() ), this, SLOT( hideLeft() ) );

   // right/bottom hide button
  _rbHB = new HideButton( position(), this );
  _rbHB->installEventFilter( this );
  connect( _rbHB, SIGNAL( clicked() ), this, SLOT( hideRight() ) );

  // applet area box
  _appletAreaBox = new ContainerAreaBox(this);
  _appletAreaBox->enableX11EventFilter(true);
  _appletAreaBox->setFrameStyle(QFrame::WinPanel  | QFrame::Raised);
  _appletAreaBox->setLineWidth(1);
  _appletAreaBox->installEventFilter( this );

  // applet area
  _appletArea = new ContainerArea( orientation(), _appletAreaBox );
  _appletArea->setFrameStyle( QFrame::NoFrame );
  _appletArea->viewport()->installEventFilter( this );

  // instantiate the autohide timer
  _autohideTimer = new QTimer(this);
  connect(_autohideTimer, SIGNAL(timeout()), SLOT(autoHideTimeout()));

  installEventFilter( this ); // for mouse event handling

  // initialise
  configure();
}

Panel::~Panel()
{
  kdDebug() << "Panel::~Panel()" << endl;
  delete _opMnu;
  writeConfig();
}

bool Panel::process(const QCString &fun, const QByteArray &data, QCString& replyType,
                                        QByteArray &replyData)
{
  if ( fun == "configure()" )
        {
          replyType = "void";
          configure( );
          return true;
        }
  else if ( fun == "restart()" )
  {
    replyType = "void";
    // do this on a timer to give us time to return true
    QTimer::singleShot(1000, this, SLOT(slotRestart()));
    return true;
  }
  if ( DCOPObject::process( fun, data, replyType, replyData ) )
        return true;
  return false;
}

void Panel::configure()
{
  KConfig *config = KGlobal::config();
  config->reparseConfiguration();
  emit configurationChanged();
  config->setGroup("General");

  _autoHide = config->readBoolEntry("AutoHidePanel", false);
  _hideAnim = config->readBoolEntry("HideAnimation", true);
  _autoHideAnim = config->readBoolEntry("AutoHideAnimation", true);
  _autoHideDelay = config->readNumEntry("AutoHideDelay", 3);
  _hideAnimSpeed = config->readNumEntry("HideAnimationSpeed", 50);
  _autoHideAnimSpeed = config->readNumEntry("AutoHideAnimationSpeed", 5);

  _size = static_cast<Size>(config->readNumEntry("Size", Normal));
  _pos = static_cast<Position>(config->readNumEntry("Position", Bottom));
  _showHBs = config->readBoolEntry("ShowHideButtons", true);
  _HBwidth = config->readNumEntry("HideButtonSize", 10);
  if (_HBwidth < 3) _HBwidth = 3;
  if (_HBwidth > 24) _HBwidth = 24;

  Direction dir;
  switch (_pos)
    {
    case Left:
      dir = dRight;
      break;
    case Right:
      dir = dLeft;
      break;
    case Top:
      dir = dDown;
      break;
    case Bottom:
    default:
      dir = dUp;
      break;
    }

  emit popupDirectionChanged(dir);
  emit orientationChanged(orientation());

  _panelsize = PGlobal::sizeValue(_size);
  resetLayout();

  _ltHB->configure();
  _rbHB->configure();
  _appletArea->configure();

  // setup auto-hide timer
  if (_autoHide)
    resetAutoHideTimer();
  else {
    _autohideTimer->stop();
    autoHide(false);
  }

  updateWindowManager();
}

void Panel::setPosition(Position p)
{
    if ( p != _pos ) {
	_pos = p;

	Direction dir;
	switch (_pos)
	    {
	    case Left:
		dir = dRight;
		break;
	    case Right:
		dir = dLeft;
		break;
	    case Top:
		dir = dDown;
		break;
	    case Bottom:
	    default:
		dir = dUp;
		break;
	    }

	emit popupDirectionChanged(dir);
	emit orientationChanged(orientation());
	resetLayout();
	updateWindowManager();
    }
}

void Panel::setSize(Size s)
{
  _size = s;
  _panelsize = PGlobal::sizeValue(s);
  resetLayout();
  updateWindowManager();
}

void Panel::writeConfig()
{
  KConfig *config = KGlobal::config();
  config->setGroup("General");

  config->writeEntry("Position", static_cast<int>(_pos));
  config->writeEntry("Size", static_cast<int>(_size));

  config->sync();
}


/*
  React dynamically on work-area changes
 */
void Panel::workAreaChanged()
{
    // let's not retrigger the full size kicker after an
    // auto hide - Lotzi B.
    // ... or a user hide - Matthias E.
    if (_autoHidden || _userHidden )
        return;
    if ( initialGeometry() != geometry() )
        resetLayout();
}

void Panel::resetLayout()
{
  writeConfig();

  delete _layout;

  setGeometry( initialGeometry() );

  // layout
  _layout = new QBoxLayout( this, isHorizontal()?QBoxLayout::LeftToRight
                           : QBoxLayout::TopToBottom, 0, 0);
  _layout->setResizeMode( QLayout::FreeResize );

  // left/top hide button
  if (isHorizontal())
    {
      _ltHB->setPosition(Left);
      _ltHB->setFixedSize( _HBwidth, rect().height() );
    }
  else
    {
      _ltHB->setPosition(Top);
      _ltHB->setFixedSize( rect().width(), _HBwidth );
    }
  if (_showHBs)
    {
      _ltHB->show();
      _layout->addWidget( _ltHB );
    }
  else
    _ltHB->hide();

  // applet area
  _appletArea->setOrientation(orientation());
  _layout->addWidget(_appletAreaBox);

  // right/bottom hide button
  if ( isHorizontal() )
    {
      _rbHB->setPosition(Right);
      _rbHB->setFixedSize( _HBwidth, rect().height() );
    }
  else
    {
      _rbHB->setPosition(Bottom);
      _rbHB->setFixedSize( rect().width(), _HBwidth );
    }
  if (_showHBs)
    {
      _rbHB->show();
      _layout->addWidget( _rbHB );
    }
  else
    _rbHB->hide();

  _layout->activate();
  updateGeometry();

  _autoHidden = false;
  _userHidden = false;
}

QRect Panel::initialGeometry()
{
    QValueList<WId> exclude;
    exclude.append(winId());
    QRect a = PGlobal::kwin_module->workArea(exclude);

    switch( _pos ) {
    case Left:
        return QRect(  a.left(), a.top(), _panelsize, a.bottom()+1 - a.top() );
    case Right:
        return QRect(  a.right() - _panelsize + 1, a.top(), _panelsize, a.bottom()+1 - a.top() );
    case Top:
        return QRect(  a.left(), a.top(), a.width(), _panelsize );
    case Bottom:
        return QRect(  a.left(), a.bottom() - _panelsize + 1, a.width(), _panelsize );
    }
    return QRect(  a.left(), a.bottom() - _panelsize + 1, a.width(), _panelsize );
}

/* 1 is the initial speed, hide_show_animation is the top speed. */
#define PANEL_SPEED(x, c) (int)((1.0-2.0*fabs((x)-(c)/2.0)/c)*_hideAnimSpeed+1.0)
#define PANEL_AUTOSPEED(x, c) (int)((1.0-2.0*fabs((x)-(c)/2.0)/c)*_autoHideAnimSpeed+1.0)

void Panel::hideLeft()
{
    animatedHide(true);
}

void Panel::hideRight()
{
    animatedHide(false);
}

void Panel::animatedHide(bool left)
{
    blockUserInput(true);
    QRect geom(initialGeometry());

    if (geom != geometry()) {

	// SHOW

	if ( isVertical() ) {

	    int strut = _ltHB->height();

	    if (_hideAnim) {

		for (int i = geom.height()-strut; i>0;i-=PANEL_SPEED(i,geom.height()) ) {
		    move(geom.x(), left ? geom.y() + i : geom.y() - i);

		    qApp->syncX();
		    qApp->processEvents();
		}
	    }
	}
	else {
	    int strut = _ltHB->width();
	    if(_hideAnim)
		{
		    for (int i = geom.width()-strut; i>0;i-=PANEL_SPEED(i,geom.width()) )
			{
			    move(left ? geom.x() + i : geom.x() - i, geom.y());

			    qApp->syncX();
			    qApp->processEvents();
			}
		}
	}
	move(geom.x(), geom.y());

	_userHidden = false;
	updateWindowManager();
	raise();
    }
    else {
	
	// HIDE
	
	lower();
	_userHidden = true;
	// Don't bother autohiding when the user's done it already.
	_autohideTimer->stop();

	if ( isVertical() ) {
	    int strut = _ltHB->height();
	    if(_hideAnim)
		{
		    for (int i = 0;
			 i < geom.height() - strut;
			 i += PANEL_SPEED(i,geom.height()))
			{
			    move(geom.x(), left ? geom.y() - i : geom.y() + i);

			    qApp->syncX();
			    qApp->processEvents();
			}
		}
	    move(geom.x(), left ? strut + geom.top() - geom.height() : geom.bottom() - strut + 1);
	}
	else {
	    int strut = _ltHB->width();
	    if(_hideAnim)   {
		for (int i = 0;
		     i < geom.width() - strut;
		     i += PANEL_SPEED(i,geom.width())) {
		    move(left ? geom.x() - i : geom.x() + i, geom.y());

		    qApp->syncX();
		    qApp->processEvents();
		}
	    }
	    move(left ? strut + geom.left() - geom.width() : geom.right() - strut + 1, geom.y());
	}
	updateWindowManager();
    }
    blockUserInput( FALSE );
}

void Panel::resetAutoHideTimer()
{
  if (_autoHide)
    _autohideTimer->start(_autoHideDelay * 1000, true);
}

void Panel::autoHideTimeout()
{
  QRect r = rect();
  r.moveTopLeft(pos());
  if (!r.contains(QCursor::pos())) {
    autoHide(true);
  }
  resetAutoHideTimer();
}

void Panel::autoHide(bool hide)
{
    if (_userHidden) // Don't autohide when the user has hidden us already.
	return;

    if (KApplication::activePopupWidget()) // Is there a popup somewhere ?
	return;

    // check for foreign popup menus
    if ( XGrabPointer( qt_xdisplay(), winId(), true, ButtonPressMask, GrabModeAsync, 
		       GrabModeAsync, None, None, CurrentTime ) != Success )
	return;
    XUngrabPointer( qt_xdisplay(), CurrentTime );
	

    if(_appletArea->inMoveOperation()) // is the user currently moving a container around?
	return;

    if (hide && !_autoHidden) //HIDE
	_autoHidden = true;
    else if  (!hide && _autoHidden) // SHOW
	_autoHidden = false;
    else
        return;

    blockUserInput(true);
    QRect geom(initialGeometry());
    QPoint oldpos;
    QPoint newpos;

    if (hide) // HIDE
        {
	    oldpos = QPoint(geom.left(), geom.top());
	    _appletAreaBox->enableX11EventFilter(false);

	    switch( _pos ) {
	    case Left:
                newpos = QPoint(geom.left() - geom.width() + 1, geom.top());
                break;
	    case Right:
                newpos = QPoint(geom.left() + geom.width() - 1, geom.top());
                break;
	    case Top:
                newpos = QPoint(geom.left(), geom.top() - geom.height() + 1);
                break;
	    case Bottom:
	    default:
                newpos = QPoint(geom.left(), geom.top() + geom.height() - 1);
                break;
	    }
        }
    else
        {
	    _appletAreaBox->enableX11EventFilter(true);
	    oldpos = pos();
	    newpos = QPoint(geom.left(), geom.top());
        }

    if (hide)
	lower();

    if(_autoHideAnim)
        {
	    switch( _pos ) {
	    case Left:
	    case Right:
                for (int i = 0; i < abs(newpos.x() - oldpos.x());
		     i += PANEL_AUTOSPEED(i,abs(newpos.x() - oldpos.x())))
		    {
                        if (newpos.x() > oldpos.x())
			    move(oldpos.x() + i, newpos.y());
                        else
			    move(oldpos.x() - i, newpos.y());
                        qApp->syncX();
                        qApp->processEvents();
		    }
                break;
	    case Top:
	    case Bottom:
	    default:
                for (int i = 0; i < abs(newpos.y() - oldpos.y());
		     i += PANEL_AUTOSPEED(i,abs(newpos.y() - oldpos.y())))
		    {
                        if (newpos.y() > oldpos.y())
			    move(newpos.x(), oldpos.y() + i);
                        else
			    move(newpos.x(), oldpos.y() - i);
                        qApp->syncX();
                        qApp->processEvents();
		    }
                break;
	    }
        }

    if(!hide)
	raise();

    move(newpos);
    blockUserInput(false);
    updateWindowManager();
}

bool Panel::eventFilter( QObject* o, QEvent * e)
{
  if (_autoHidden) {
    switch ( e->type() ) {
    case QEvent::MouseButtonPress:
    case QEvent::MouseButtonRelease:
    case QEvent::MouseButtonDblClick:
    case QEvent::MouseMove:
    case QEvent::KeyPress:
    case QEvent::KeyRelease:
      return true; // ignore;
    case QEvent::Enter:
      autoHide(false);
      break;
    /* Why does the following not work? Wilco
    case QEvent::DragEnter:
      autoHide(false);
      break;
    */
    default:
      break;
    }
    return false;
  }

  if ( _block_user_input ) {
    switch ( e->type() ) {
    case QEvent::MouseButtonPress:
    case QEvent::MouseButtonRelease:
    case QEvent::MouseButtonDblClick:
    case QEvent::MouseMove:
    case QEvent::KeyPress:
    case QEvent::KeyRelease:
      return true; // ignore;
    default:
      break;
    }
    return false;
  }

  switch ( e->type() ) {
  case QEvent::Enter:
      if  ( o == this && !isActiveWindow() ) {
	  XEvent ev;
	  memset(&ev, 0, sizeof(ev));
	  ev.xfocus.display = qt_xdisplay();
	  ev.xfocus.type = XFocusIn;
	  ev.xfocus.window = winId();
	  ev.xfocus.mode = NotifyNormal;
	  ev.xfocus.detail = NotifyAncestor;
	  Time time = qt_x_time;
	  qt_x_time = 1;
	  qApp->x11ProcessEvent( &ev );
	  qt_x_time = time;
	  _faked_activation = true;
	  setFocus(); // to avoid having a blinking cursor or a focus rectangle somewhere else
      }
      break;
  case QEvent::Leave:
      if ( o == this && _faked_activation && isActiveWindow() ) {
	  XEvent ev;
	  memset(&ev, 0, sizeof(ev));
	  ev.xfocus.display = qt_xdisplay();
	  ev.xfocus.type = XFocusIn;
	  ev.xfocus.window = winId();
	  ev.xfocus.mode = NotifyNormal;
	  ev.xfocus.detail = NotifyAncestor;
	  Time time = qt_x_time;
	  qt_x_time = 1;
	  qApp->x11ProcessEvent( &ev );
	  qt_x_time = time;
	  _faked_activation = false;
      }
  case QEvent::WindowDeactivate:
      _faked_activation = false;
      break;
  case QEvent::MouseButtonPress:
    {
      _autohideTimer->stop();
      QMouseEvent* me = (QMouseEvent*) e;
      if ( me->button() == LeftButton )
        {
          _last_lmb_press = me->pos();
        }
      else if ( me->button() == RightButton )
        {
          if(!_opMnu)
            _opMnu = new PanelOpMenu();
          _opMnu->exec(getPopupPosition(_opMnu));
        }
      resetAutoHideTimer();
    }
    break;
  case QEvent::MouseMove:
    {
      QMouseEvent* me = (QMouseEvent*) e;
      if ( (me->state() & LeftButton) == LeftButton ) {
        QPoint p( me->pos() - _last_lmb_press );
        if ( !geometry().contains(me->globalPos()) &&
            p.manhattanLength() > KGlobalSettings::dndEventDelay() ) {
          moveMe();
          return true;
        }
      }
    }
    break;
  default:
    break;
  }
  return false;
}

QCStringList Panel::functions()
{
    QCStringList funcs = DCOPObject::functions();
    funcs << "void configure()";
    return funcs;
}

void Panel::blockUserInput( bool block )
{
  if ( block == _block_user_input )
    return;

  // If we don't want any user input to be possible we should catch mouse
  // events and such. Therefore we install an eventfilter and let the
  // eventfilter discard those events.
  if ( block )
    qApp->installEventFilter( this );
  else
    qApp->removeEventFilter( this );

  _block_user_input = block;
}

void Panel::moveMe()
{
  _autohideTimer->stop();
  hide();
  QApplication::syncX();
  QValueList<QRect> rects;

  QValueList<WId> exclude;
  exclude.append(winId());
  QRect a = PGlobal::kwin_module->workArea(exclude);
  //QRect r(QApplication::desktop()->geometry());

  rects.append( QRect(  a.left(), a.top(), _panelsize, a.height() ) );
  rects.append( QRect(  a.right() - _panelsize + 1, a.top(), _panelsize, a.height() ) );
  rects.append( QRect(  a.left(), a.top(), a.width(), _panelsize ) );
  rects.append( QRect(  a.left(), a.bottom() - _panelsize + 1, a.width(), _panelsize ) );
  Position newpos = (Position) UserRectSel::select(rects, _pos);
  setPosition(newpos);
  show();

  // sometimes the HB's are not reset correctly
  _ltHB->setDown(false);
  _rbHB->setDown(false);

  resetAutoHideTimer();

  updateWindowManager();
}

void Panel::updateWindowManager()
{
/*
  // Set the relevant properties on the window.
    int w = width();
    int h = height();

    if ( _userHidden )
        w = h = 0;
    else if ( _autoHide )
        w = h = 1;

    switch (_pos) {
    case Top:     KWin::setStrut( winId(), 0, 0,  h, 0 ); break;
    case Bottom:  KWin::setStrut( winId(), 0, 0, 0, h ); break;
    case Right:   KWin::setStrut( winId(), 0, w, 0, 0 ); break;
    case Left:    KWin::setStrut( winId(), w, 0, 0, 0 ); break;
    }
*/
    // Set the relevant properties on the window.
    int w = width();
    int h = height();

    QRect r(QApplication::desktop()->geometry());

    QRect geom = initialGeometry();

    if ( _userHidden )
        w = h = 0;
    else if ( _autoHide ) // use _autoHidden for auto adjustment, maybe configurable in the future
        w = h = 1;

    switch (_pos) {
    case Top:     KWin::setStrut( winId(), 0, 0,  geom.y() + h, 0 ); break;
    case Bottom:  KWin::setStrut( winId(), 0, 0, 0, (r.bottom() - geom.y() - height()) + h); break;
    case Right:   KWin::setStrut( winId(), 0, (r.right() - geom.x() - width()) + w, 0, 0 ); break;
    case Left:    KWin::setStrut( winId(), geom.x() + w, 0, 0, 0 ); break;
    }
}

void Panel::closeEvent( QCloseEvent *e )
{
  e->ignore();
}

QPoint Panel::getPopupPosition(QPopupMenu *menu)
{
  QPoint p(0,0);

  switch (_pos) {
  case Top:
    p = mapToGlobal(QPoint(0, height()));
    p.setX(QCursor::pos().x());
    break;
  case Bottom:
    p = mapToGlobal(QPoint(0, 0 - menu->height()));
    p.setX(QCursor::pos().x());
    break;
  case Right:
    p = mapToGlobal(QPoint(0-menu->width(), 0));
    p.setY(QCursor::pos().y());
    break;
  case Left:
    p = mapToGlobal(QPoint(width(), 0));
    p.setY(QCursor::pos().y());
    break;
  }

  return p;
}

void Panel::slotRestart()
{
  char ** o_argv = new char*[2];
  o_argv[0] = strdup("kicker");
  o_argv[1] = 0L;

  PGlobal::panel->writeConfig();
  delete PGlobal::panel;
  execv(QFile::encodeName(locate("exe", "kdeinit_wrapper")), o_argv);
  exit(1);
}
