/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qdragobject.h>

#include <kapp.h>
#include <kstddirs.h>
#include <kdebug.h>
#include <kdesktopfile.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmimetype.h>
#include <krun.h>
#include <kservicegroup.h>
#include <ksycoca.h>
#include <ksycocaentry.h>
#include <kservice.h>
#include <kservicegroup.h>

#include "service_mnu.h"
#include "service_mnu.moc"

const int idStart = 4242;

PanelServiceMenu::PanelServiceMenu(
  const QString & label,
  const QString & relPath,
  QWidget * parent,
  const char * name
)
  : PanelMenu(label, parent, name),
    relPath_(relPath),
    loaded_(false),
    opPopup_(0),
    clearOnClose_(false)
{
    merge_ = KGlobal::config()->readBoolEntry("MergeKDEDirs", true);
    setAcceptDrops(true);

    connect(
        KSycoca::self(),  SIGNAL(databaseChanged()),
        this,             SLOT(slotClearOnClose())
        );
}


PanelServiceMenu::~PanelServiceMenu()
{
    delete opPopup_;
    opPopup_ = 0;
}

void PanelServiceMenu::initialize()
{
    // Set the startposition outside the panel, so there is no drag initiated
    // when we use drag and click to select items. A drag is only initiated when
    // you click to open the menu, and then press and drag an item.
    startPos_ = QPoint(-1,-1);

    if (init) return;
    init = true;

    // We ask KSycoca to give us all services (sorted).

    KServiceGroup::Ptr root = KServiceGroup::group(relPath_);
    KServiceGroup::List list = root->entries(true);

    if (list.isEmpty()) {
	setItemEnabled(insertItem(i18n("No entries")), false);
	return;
    }

    KServiceGroup::List::ConstIterator it = list.begin();

    int id = idStart;

    KIconLoader *loader = KGlobal::instance()->iconLoader();
    for (; it != list.end(); ++it) {

	KSycocaEntry * e = *it;

	if (e->isType(KST_KServiceGroup)) {

	    KServiceGroup::Ptr g(static_cast<KServiceGroup *>(e));
	    QString groupCaption = g->caption();

	    // Avoid adding empty groups.
	    KServiceGroup::Ptr subMenuRoot = KServiceGroup::group(g->relPath());
	    if (subMenuRoot->entries().count() == 0)
		continue;

	    // Ignore dotfiles.
	    if ((g->name().at(0) == '.'))
		continue;

	    // Item names may contain ampersands. To avoid them being converted
	    // to accelators, replace them with two ampersands.
	    groupCaption.replace(QRegExp("&"), "&&");

	    PanelServiceMenu * m =
		newSubMenu(g->name(), g->relPath(), this, g->name().utf8());

	    int newId = insertItem(SmallIconSet(g->icon()), groupCaption, m, id++);
	    entryMap_.insert(newId, static_cast<KSharedPtr<KSycocaEntry> >(g));

	} else {

	    KService::Ptr s(static_cast<KService *>(e));
	    QString serviceName = s->name();

	    // check for NoDisplay
	    if ((s->property("NoDisplay")).asBool())
		continue;

	    // Ignore dotfiles.
	    if ((serviceName.at(0) == '.'))
		continue;

	    // Item names may contain ampersands. To avoid them being converted
	    // to accelators, replace them with two ampersands.
	    serviceName.replace(QRegExp("&"), "&&");
	    QIconSet iconSet;
	    iconSet.setPixmap(loader->loadIcon(s->icon(), KIcon::Small, 0, KIcon::DefaultState, 0L, true), QIconSet::Small, QIconSet::Normal);
	    iconSet.setPixmap(loader->loadIcon(s->icon(), KIcon::Small, 0, KIcon::ActiveState, 0L, true), QIconSet::Small, QIconSet::Active);
	    int newId = insertItem(iconSet, serviceName, id++);
	    entryMap_.insert(newId, static_cast<KSharedPtr<KSycocaEntry> >(s));
	}
    }
}

void PanelServiceMenu::slotExec(int id)
{
    kdDebug() << "PanelServiceMenu::slotExec(" << id << ")" << endl;
    if (!entryMap_.contains(id)) {
	kdDebug() << "Cannot find service with menu id " << id << endl;
	return;
    }

    KSycocaEntry * e = entryMap_[id];

    kdDebug() << "Running `" << e->name() << "'" << endl;

    KService::Ptr service = static_cast<KService *>(e);

    kapp->propagateSessionManager();
    KRun::run(*service, KURL::List());
}

void PanelServiceMenu::mousePressEvent(QMouseEvent * ev)
{
    startPos_ = ev->pos();
    PanelMenu::mousePressEvent(ev);
}

void PanelServiceMenu::mouseMoveEvent(QMouseEvent * ev)
{
    PanelMenu::mouseMoveEvent(ev);

    if ( (ev->state() & LeftButton ) != LeftButton )
	return;

    QPoint p = ev->pos() - startPos_;
    if (p.manhattanLength() <= QApplication::startDragDistance() )
	return;

    int id = idAt(startPos_);

    // Don't drag items we didn't create.
    if (id < idStart)
	return;

    if (!entryMap_.contains(id)) {
	kdDebug() << "Cannot find service with menu id " << id << endl;
	return;
    }

    KSycocaEntry * e = entryMap_[id];

    kdDebug() << "Dragging `" << e->name() << "'" << endl;

    KService::Ptr service = static_cast<KService *>(e);

    QString filePath;

    QPixmap icon;

    switch (e->sycocaType()) {

    case KST_KService:
	icon     = static_cast<KService *>(e)->pixmap(KIcon::Small);
	filePath = static_cast<KService *>(e)->desktopEntryPath();
	break;

    case KST_KServiceGroup:
	icon     = KGlobal::iconLoader()->
		   loadIcon(static_cast<KServiceGroup *>(e)->icon(), KIcon::Small);
	filePath = static_cast<KServiceGroup *>(e)->relPath();
	break;

    default:
	return;
	break;
    }

    // If the path to the desktop file is relative, try to get the full
    // path from KStdDirs.

    QString path = (filePath[0] == '/') ? filePath : locate("apps", filePath);

    QUriDrag * d(new QUriDrag(this));

    d->setPixmap(icon);
    kdDebug() << "dragging. path: " << path << endl;
    d->setFilenames(QStringList(path));
    d->dragCopy();
    close();
    
    // Set the startposition outside the panel, so there is no drag initiated
    // when we use drag and click to select items. A drag is only initiated when
    // you click to open the menu, and then press and drag an item.
    startPos_ = QPoint(-1,-1);
}

void PanelServiceMenu::dragEnterEvent(QDragEnterEvent *e)
{
    e->accept();
}

void PanelServiceMenu::dragMoveEvent(QDragMoveEvent *e)
{
    e->accept();
}

PanelServiceMenu *PanelServiceMenu::newSubMenu(const QString & label, const QString & relPath,
					       QWidget * parent, const char * name)
{
    return new PanelServiceMenu(label, relPath, parent, name);
}

void PanelServiceMenu::slotClearOnClose()
{
  kdDebug() << "database changed. Will clear on next close." << endl;
  clearOnClose_ = true;
}

void PanelServiceMenu::closeEvent(QCloseEvent *)
{
  if (clearOnClose_) {
    clearOnClose_ = false;
    slotClear();
  }
}
